/*===========================================================================
SOLAR :: ..\common\join.c

Original Author: Kevin Houle <kjhoule@iowegia.des-moines.ia.us>
This software module has been placed into the public domain.
===========================================================================*/

/* Header Files */
#include <dir.h>
#include <string.h>
#include <stdio.h>

/* Local Defines */
#define WAFFLE_SUBSC        "JOIN"
#define SOLAR_SUBSC   "NEWSRC.HPG"

/* External Data */
extern char user_path[MAXPATH];           /* Full path to user directory */
extern char _slrerr[80];                  /* Global error buffer         */

/*
 * Function: int join2newsrc()
 * Purpose : Convert Waffle's JOIN file to NEWSRC.HPG format.
 * Return  : 0 on success, 1 on error, -1 on fatal. set _slrerr.
 */

int join2newsrc()
{
  FILE *wafjoin = NULL;
  FILE *slrjoin = NULL;
  char wafjoinpath[MAXPATH];
  char slrjoinpath[MAXPATH];
  char wafjoinbuf[80];
  char slrjoinbuf[80];
  char *p;

  /* Build path to Waffle's JOIN and Solar's
     NEWSRC.HPG file for this user. */

  strcpy(wafjoinpath, user_path);
  strcat(wafjoinpath, "\\");
  strcat(wafjoinpath, WAFFLE_SUBSC);

  strcpy(slrjoinpath, user_path);
  strcat(slrjoinpath, "\\");
  strcat(slrjoinpath, SOLAR_SUBSC);

  /* Open Waffle JOIN for read, and Solar
     NEWSRC.HPG for write. */

  if ((wafjoin = fopen(wafjoinpath,"rt")) == NULL)
  {
    /* User has no JOIN file. This could be OK */

    sprintf(_slrerr, "%s file not found",wafjoinpath);
    goto ErrorExit;
  }

  if ((slrjoin = fopen(slrjoinpath,"wt")) == NULL)
  {
    /* This is _not_ OK */

    sprintf(_slrerr, "error opening %s for write", slrjoinpath);
    goto FatalExit;
  }

  /* Process for each line in JOIN file */

  while (fgets(wafjoinbuf,80,wafjoin) != NULL)
  {
    /* Truncate string with NULL, not linefeed */

    wafjoinbuf[strlen(wafjoinbuf) - 1] = '\0';

    /* Parse token before first space, this
       should be the message area name */

    if ((p = strtok(wafjoinbuf," ")) == NULL)
    {
      /* JOIN is corrupt */

      sprintf(_slrerr,"invalid %s file format",wafjoinpath);
      goto FatalExit;
    }

    /* Store message area name, and append
       a colon to the name */

    strcpy(slrjoinbuf,p);
    strcat(slrjoinbuf,":");

    /* Check for a last read message number
       in JOIN */

    if ((p = strtok(NULL," ")) != NULL)
    {
      /* Convert to 1-x format */

      strcat(slrjoinbuf," 1-");
      strcat(slrjoinbuf,p);
    }

    /* Write converted message area line
       to NEWSRC.HPG file */

    fprintf(slrjoin,"%s\n",slrjoinbuf);
  }

  /* Close the files */

  fclose(slrjoin);
  fclose(wafjoin);

/* Things went just fine */
GoodExit:
  return 0;

/* Something needed wasn't there,
   but it might be OK. */
ErrorExit:
  if (wafjoin) fclose(wafjoin);
  if (slrjoin) fclose(slrjoin);
  return 1;

/* Something really bad happened */
FatalExit:
  if (wafjoin) fclose(wafjoin);
  if (slrjoin) fclose(slrjoin);
  return -1;
}

/*
 * Function: int newsrc2join()
 * Purpose : Convert NEWSRC.HPG to Waffle's JOIN format. This kinda
 *           sucks to have to use this, as specific message number
 *           information retained in NEWSRC.HPG is lost going to JOIN.
 * Return  : 0 on success, 1 on error, -1 on fatal. set _slrerr.
*/

int newsrc2join()
{
  FILE *wafjoin = NULL;
  FILE *slrjoin = NULL;
  char wafjoinpath[MAXPATH];
  char slrjoinpath[MAXPATH];
  char wafjoinbuf[80];
  char slrjoinbuf[1024];
  int x = 0;
  char *p;

  /* Build path to Waffle's JOIN and Solar's
     NEWSRC.HPG file for this user. */

  strcpy(wafjoinpath, user_path);
  strcat(wafjoinpath, "\\");
  strcat(wafjoinpath, WAFFLE_SUBSC);

  strcpy(slrjoinpath, user_path);
  strcat(slrjoinpath, "\\");
  strcat(slrjoinpath, SOLAR_SUBSC);

  /* Open NEWSRC.HPG for read */

  if ((slrjoin = fopen(slrjoinpath,"rt")) == NULL)
  {
    /* This might be OK */

    sprintf(_slrerr, "file %s not found",wafjoinpath);
    goto ErrorExit;
  }

  /* Open JOIN for write */

  if ((wafjoin = fopen(wafjoinpath,"wt")) == NULL)
  {
    /* This is not OK */

    sprintf(_slrerr, "error opening %s for write", slrjoinpath);
    goto FatalExit;
  }

  /* Process each line in NEWSRC.HPG */

  while (fgets(slrjoinbuf,80,slrjoin) != NULL)
  {
    /* Truncate with NULL not linefeed */

    slrjoinbuf[strlen(slrjoinbuf) - 1] = '\0';

    /* Parse out message area name */

    if ((p = strtok(slrjoinbuf,":")) == NULL)
    {
      /* NEWSRC.HPG is corrupt */

      sprintf(_slrerr,"invalid %s file format",slrjoinpath);
      goto FatalExit;
    }

    /* Check to see if message area is marked
       for no processing. If so, drop area on
       the floor. */

    if (p[0] != '!')
		{
      /* Save message area name */

      strcpy(wafjoinbuf,p);

      if ((p = strtok(NULL,"-")) != NULL)
			{
				if ((p = strtok(NULL,"-")) != NULL)
				{
					strcat(wafjoinbuf," ");
					strcat(wafjoinbuf,p);
				}
			}
      fprintf(wafjoin,"%s\n",wafjoinbuf);
		}
	}
  fclose(slrjoin);
  fclose(wafjoin);

GoodExit:
  return 0;
ErrorExit:
  if (wafjoin) fclose(wafjoin);
  if (slrjoin) fclose(slrjoin);
  return 1;
FatalExit:
	if (wafjoin) fclose(wafjoin);
  if (slrjoin) fclose(slrjoin);
  return -1;
}

