/*===========================================================================
SOLAR slrreply :: Module ..\slrreply\config.c
Original author : Kevin Houle <kjhoule@iowegia.des-moines.ia.us>

This source code has been placed into the public domain.
===========================================================================*/

/* Header Files */
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <dir.h>
#include <time.h>
#include <values.h>
#include "config.h"

/*
 * Function: int load_static(char username[10])
 * Purpose : Load parameters from Waffle's static file.
 * Return  : 0 on success, non-zero on error and set _slrerr.
*/

int load_static(char username[10])
{
  FILE *static_file = NULL;

  char buffer[128];
  char delimiter = ':';
  char *path = NULL;

  if ((path = getenv("WAFFLE")) == NULL)
  {
    strcpy(_slrerr,"load_static(): environment variable WAFFLE not set.");
    goto ErrorExit;
  }
  strcpy(static_path,path);
  if (verbose == YES)
    RPrintf("\nWaffle static file: %s\n",static_path);


  if ((static_file = fopen(static_path,"rt")) == NULL)
  {
    sprintf(_slrerr,"load_static(): error opening %s for reading.",static_path);
    goto ErrorExit;
  }

  while (fgets(buffer,128,static_file) != NULL)
  {
    if (strnicmp(buffer,"USER",4) == 0)
    {
      strcpy(user_path,extract_parm(buffer,delimiter));
      strcat(user_path,"\\");
      strcat(user_path, username);
      if (verbose == YES)
        RPrintf("User Path: %s\n",user_path);
    }

    /* Set path to Solar's configuration file.
       Give 'solar95' priority. */

    if (strnicmp(buffer,"SOLAR95",7) == 0)
		{
			strcpy(config_path,extract_parm(buffer,delimiter));
      if (verbose == YES)
        RPrintf("Solar config path: %s\n",config_path);
		}

    if ((strnicmp(buffer,"SOLAR",5) == 0) && (strcmp(config_path,"NONE") == 0))
		{
			strcpy(config_path,extract_parm(buffer,delimiter));
      if (verbose == YES)
        RPrintf("Solar config path: %s\n",config_path);
		}

    /* Get path to temporary directory. This only gets used if
       'solarwork' is not set in Solar's configuration file. */

    if (strnicmp(buffer,"TEMPORARY",9) == 0)
    {
      if (strcmp(temp_path,"NONE") == 0)
      {
        strcpy(temp_path,extract_parm(buffer,delimiter));
        if (verbose == YES)
          RPrintf("Temp path: %s\n",temp_path);
      }
    }

    /* Store path to Waffle root directory plus some other stuff */

    if (strnicmp(buffer,"WAFFLE",6) == 0)
    {
        strcpy(waffle_path,extract_parm(buffer,delimiter));

        strcpy(log_path,waffle_path);
        strcat(log_path,"\\admin\\solar");
        if (verbose == YES)
          RPrintf("Log file: %s\n",log_path);
        strcpy(system_path,waffle_path);
        strcat(system_path,"\\system");
        if (verbose == YES)
          RPrintf("System path: %s\n",system_path);
        strcpy(extern_path,waffle_path);
        strcat(extern_path,"\\extern");
        if (verbose == YES)
          RPrintf("Extern path: %s\n",extern_path);
    }

    if (strnicmp(buffer,"NODE",4) == 0)
    {
      strcpy(domain_name,extract_parm(buffer,delimiter));
      if (verbose == YES)
        RPrintf("System DNS Name: %s\n",domain_name);
    }

    if (strnicmp(buffer,"UUCPNAME",8) == 0)
    {
      strcpy(uucp_name,extract_parm(buffer,delimiter));
      if (verbose == YES)
        RPrintf("System UUCP Name: %s\n",uucp_name);
    }

    if (strnicmp(buffer,"ORGAN",5) == 0)
    {
      strcpy(organization,full_line(buffer));
      if (verbose == YES)
        RPrintf("Organization: %s\n",organization);
    }

    if (strnicmp(buffer,"BACKBONE",8) == 0)
    {
      strcpy(backbone,extract_parm(buffer,delimiter));
      if (verbose == YES)
        RPrintf("USENET Backbone: %s\n",backbone);
    }

    if (strnicmp(buffer,"TIMEZONE",8) == 0)
    {
      strcpy(tz,extract_parm(buffer,delimiter));
      if (verbose == YES)
        RPrintf("Timezone: %s\n",tz);
    }

    /* Check the driver, use fossil only if selected on command line
       and in static file 'driver' parameter. */

    if (strnicmp(buffer,"DRIVER",6) == 0)
    {
      if (use_fossil == YES)
      {
        if (strnicmp(extract_parm(buffer,delimiter),"FOSSIL",6) == 0)
        {
          use_fossil = YES;
          if (verbose == YES) RPrintf("Fossil support enabled.\n");
        }
        else
        {
          use_fossil = NO;
          if (verbose == YES) RPrintf("Fossil support disabled.\n");
        }
      }
      else
      {
        if (verbose == YES) RPrintf("Fossil support disabled.\n");
      }
    }

    /* Load comm device, in case we are using a fossil driver. */

    if (strnicmp(buffer,"DEVICE",6) == 0)
    {
      port = atoi(extract_parm(buffer,delimiter)) - 1;
      if (verbose == YES)
        RPrintf("COM Port: %u\n",port);
    }
  }
  fclose(static_file);

  if (verbose == YES)
    RPrintf("Finished reading static, validating...\n");

  if (strcmp(user_path,"NONE") == 0)
  {
    strcpy(_slrerr,"load_static(): user: parameter not found in static file.");
    goto ErrorExit;
  }
  if (strcmp(domain_name,"NONE") == 0)
  {
    strcpy(_slrerr,"load_static(): node: parameter not found in static file.");
    goto ErrorExit;
  }
  if (strcmp(uucp_name,"NONE") == 0)
  {
    strcpy(_slrerr,"load_static(): uucpname: parameter not found in static file.");
    goto ErrorExit;
  }
  if (strcmp(temp_path,"NONE") == 0)
  {
    strcpy(_slrerr,"load_static(): temporary: parameter not found in static file.");
    goto ErrorExit;
  }
  if (strcmp(waffle_path,"NONE") == 0)
  {
    strcpy(_slrerr,"load_static(): waffle: parameter not found in static file.");
    goto ErrorExit;
  }
  if (strcmp(tz,"NONE") == 0)
  {
    strcpy(_slrerr,"load_static(): timezone: parameter not found in static file.");
    goto ErrorExit;
  }
  if ((use_fossil == YES) && (port == 99))
  {
    strcpy(_slrerr,"load_static(): device: parameter not found in static file.");
    goto ErrorExit;
  }

  if (verbose == YES)
    RPrintf("Static file load OK\n");

GoodExit:
  return 0;
ErrorExit:
  return 1;
}

/*
 * Function: load_config()
 * Purpose : Load Solar's configuration file.
 * Return  : 0 on success, non-zero on error and set _slrerr.
*/

int load_config()
{
  FILE *config_file = NULL;
  char buf[128];
  char delimiter = '=';
  char *path = NULL;

  /* Get path to config file if not already set */

  if (strcmp(config_path,"NONE") == 0)
  {
    if ((path = getenv("SOLAR95")) == NULL)
    {
      if ((path = getenv("SOLAR")) == NULL)
      {
        strcpy(_slrerr,"load_config(): environment variable SOLAR not set.");
        goto ErrorExit;
      }
    }
    strcpy(config_path,path);
  }

  if (verbose == YES)
    RPrintf("\nSolar config file: %s\n",config_path);

  if ((config_file = fopen(config_path,"rt")) == NULL)
  {
    sprintf(_slrerr,"load_config(): error opening %s for reading.",config_file);
    goto ErrorExit;
  }

  while (fgets(buf,128,config_file) != NULL)
  {
    /* Override the default temporary work directory */

    if (strnicmp(buf,"SOLARWORK",9) == 0)
    {
      strcpy(temp_path,extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Temp path set to: %s\n",temp_path);
    }

    /* Set Solar System home path */

    if (strnicmp(buf,"SOLARPATH",9) == 0)
    {
      strcpy(solar_path,extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Solar path set to: %s\n",solar_path);
    }

    /* Set swap option for slrreply.exe */

    if (strnicmp(buf,"SWAP-SLRREPLY",13) == 0)
    {
      diskswap = convert_to_number(extract_parm(buf,delimiter));
			if (verbose == YES)
			{
        if (diskswap == YES)
          RPrintf("Disk Swap set to YES\n");
				else
          RPrintf("Disk Swap set to NO\n");
      }
    }

    /* Set path to swap directory for SPAWNO */

    if (strnicmp(buf,"SOLARSWAP",9) == 0)
		{
      strcpy(swap_path,extract_parm(buf,delimiter));
			if (verbose == YES)
        RPrintf("Swap path set to: %s\n",swap_path);
		}

    /* Override the default log file path */

    if (strnicmp(buf,"LOGFILE",7) == 0)
    {
      strcpy(log_path,extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Log file set to: %s\n",log_path);
    }

    if (strnicmp(buf,"LOST+FOUND",10) == 0)
    {
      strcpy(lost_path,extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Lost+Found directory: %s\n",lost_path);
    }

    if (strnicmp(buf,"SPAM-MAX",8) == 0)
    {
      spam_max = atol(extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Maximim SPAM count: %lu\n",spam_max);
    }

    if (strnicmp(buf,"MAX-MAIL-SIZE",13) == 0)
    {
      _maxmailsize = atol(extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Maximum mail size: %lu\n",_maxmailsize);
    }

    if (strnicmp(buf,"MAX-NEWS-SIZE",13) == 0)
    {
      _maxnewssize = atol(extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Maximum news size: %lu\n",_maxnewssize);
    }

    if (strnicmp(buf,"HOST-DOMAIN",11) == 0)
    {
      strcpy(domain_name,extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("System DNS Name: %s\n",domain_name);
    }
  }
  fclose(config_file);

  if (verbose == YES)
    RPrintf("Finished reading config, validating...\n");

  if (strcmp(solar_path,"NONE") == 0)
  {
    strcpy(_slrerr,"load_config(): solar: parameter not found in config file.");
    goto ErrorExit;
  }

  if (verbose == YES)
    RPrintf("Config file load OK\n\n");

GoodExit:
  return 0;
ErrorExit:
  return 1;
}

/*
 * Function: char full_line(char line[128])
 * Purpose : Extracts entire line found after a space.
 * Return  : Returns the line found.
*/

char *full_line(char line[128])
{
  int  idx  = 0;
  int  idx2 = 0;
  static char path[MAXPATH];

  while (line[idx++] != ':');
  while (line[idx] == ' ') idx++;
  while ((line[idx] != '\0') && (line[idx] != '\n'))
  {
    path[idx2++] = line[idx];
    idx++;
  }
  path[idx2] = '\0';
  return path;
}

/*
 * Function: int load_programs()
 * Purpose : Load News and Mail programs
 * Return  : zero on success, non-zero on error
*/

int load_programs()
{
  char path[80];
  char buf[80];
  int  x = 0, y = 0;
  char *program = NULL;
  FILE *sysfile = NULL;

  strcpy(path, extern_path);
  strcat(path, "\\");
  strcat(path, WAFFLE_SYSTEM);

  if ((sysfile = fopen(path,"rt")) == NULL)
  {
    if (verbose == YES)
      RPrintf("Waffle file %s not found\n",path);
    goto GoodExit;
  }

  if (verbose == YES)
    RPrintf("Reading %s\n",WAFFLE_SYSTEM);

  while (fgets(buf,80,sysfile) != NULL)
  {
    if (strnicmp(buf,"NEWS",4) == 0)
    {
      y = 0;
      program = strstr(buf,"=");
      for (x = 0; x < strlen(program); x++)
      {
        if (program[x] != '"')
          news_program[y++] = program[x];
        if (program[x] == ' ')
          break;
      }
      news_program[y] = '\0';
    }

    if (strnicmp(buf,"MAIL",4) == 0)
    {
      y = 0;
      program = strstr(buf,"=");
      for (x = 0; x < strlen(program); x++)
      {
        if (program[x] != '"')
          mail_program[y++] = program[x];
        if (program[x] == ' ')
          break;
      }
      mail_program[y] = '\0';
    }
  }
  fclose(sysfile);

GoodExit:
  if (verbose == YES)
  {
    RPrintf("News program: %s\n",news_program);
    RPrintf("Mail program: %s\n",mail_program);
  }
  return 0;
}
