/*===========================================================================
SOLAR slrxfer v0.95.3 :: Module ..\slrxfer\slrxfer.c
Original Author: Kevin Houle <kjhoule@iowegia.des-moines.ia.us>

This software module has been placed in the public domain.
===========================================================================*/

/* Header Files */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dir.h>
#include <process.h>
#include <time.h>
#include "codedate.h"
#include "slrxfer.h"

int main(int argc, char *argv[])
{
	extern    char    *optarg;
  unsigned  char    x;
  const     char    *optionS = "u:U:vVsSrRfF";
  char              orig_path[MAXPATH];
  char              username[10];
  int               exit_code = x_success;
  long              packet_size;
  clock_t           end_ticks, start_ticks;
  long              cps;

  /* Store original directory */

  getcwd(orig_path, MAXPATH);

  /* Set default values for globals */

  strcpy(temp_path,"NONE");
  strcpy(user_path,"NONE");
  strcpy(log_path,"NONE");
  strcpy(uucpname,"NONE");
  strcpy(config_path,"NONE");
  strcpy(swap_path,".");
  strcpy(protocol,"NONE");
  strcpy(username,"NONE");

  diskswap      = YES;
  use_fossil    = NO;
	verbose       = NO;
	mode          = SEND;
  use_fossil    = NO;
  port          = 99;

  /* Parse command line */

  while ((x = getopt(argc, argv, optionS)) != '\xFF')
	{
    switch (toupper(x)) {

      case '?'  : usage();
                  exit_code = x_usage;
                  goto ExitMain;

      case 'U'  : if ((strcmp(user_path,"NONE")) == 0)
                  {
                    strcpy(username,optarg);
                    break;
                  }
                  else
                  {
                    strcpy(_slrerr,"main(): cannot specify username twice.");
                    exit_code = x_fatal;
                    goto ExitMain;
                  }
      case 'V'  : verbose = YES;
                  break;
      case 'S'  : mode = SEND;
                  break;
      case 'R'  : mode = RECEIVE;
                  break;
      case 'F'  : use_fossil = YES;
									break;
    }
  }

  /* Check to see if username was set. If
     not, try to set from PCBDOOR.SYS or
     DOORS.SYS file. Then load static. */

  if (strcmp(username,"NONE") == 0)
	{
    if (read_doors() != 0)
		{
      usage();
      exit_code = x_usage;
      goto ExitMain;
    }
  }

  /* If verbose mode, show what we know so far. */

  if (verbose == YES)
  {
    RPrintf("\nSOLAR v%s <> slrxfer (%s)\n\n",VERSION,CODE_DATE);
    RPrintf("Packet transfer for user: %s\n",username);
    RPrintf("Transfer mode is ");
    if (mode == SEND)
    {
      RPrintf("SEND\n");
    }
    else
    {
			if (mode == RECEIVE)
        RPrintf("RECEIVE\n");
      else
        RPrintf("unknown!\n");
    }
  }

  /* Load configuration files. */

	if ((load_static(username) != 0)    || \
			(load_config() != 0)            || \
			(user_defaults() != 0)          || \
			(set_directory(temp_path) != 0))
	{
    exit_code = x_fatal;
		goto ExitMain;
	}

  if (verbose == YES)
    RPrintf("Default directory set to %s\n",temp_path);

  /* Take appropriate action based on mode */

  switch (mode) {

    case SEND  : /* Record the starting time */

							start_ticks = clock();

							/* Send the packet */

							if (xfer_packet("SEND") == 0)
							{
                /* Record the ending time */

                end_ticks = clock();

                /* Get packet size and do rough speed calculation */

                packet_size = get_packet_size();
                if ((end_ticks - start_ticks) > 0)
                {
                  cps = packet_size / ((end_ticks - start_ticks) / CLK_TCK);
                }
								else
                {
									cps = packet_size;
                }

                /* Log the rough speed calculation */

                sprintf(logbuf,"%s %4.1f min %lu cps %lu bytes",username,((end_ticks - start_ticks) / CLK_TCK) / 60,cps,packet_size);
                logit("slrxfer");

                /* Cleanup. */

								clean_up();
              }
              else /* The transfer failed */
              {
                /* Try to save the packet, if so, then cleanup. */

                if (copy_packet() == 0) clean_up();

                /* This is an error, BTW */

                exit_code = x_fatal;
                goto ExitMain;
              }
              break;

    case RECEIVE  : /* Record starting time */

              start_ticks = clock();

              /* Transfer the incoming packet */

							if (xfer_packet("RECEIVE") == 0)
							{
								/* Get the packet size */

								packet_size = get_packet_size();

                /* Check to see if it is a real packet */

                if (packet_size > 0L)
                {
                  /* Record ending time and do some rough time calcs */

                  end_ticks = clock();
                  if ((end_ticks - start_ticks) > 0)
                  {
                    cps = packet_size / ((end_ticks - start_ticks) / CLK_TCK);
									}
                  else
                  {
                    cps = packet_size;
                  }

                  /* Log the rough time calcs */

                  sprintf(logbuf,"%s %4.1f min %lu cps %lu bytes",username,((end_ticks - start_ticks) / CLK_TCK)/60,cps,packet_size);
                  logit("slrxfer");
                }
              }
              else
              {
                /* Transfer failed. Clean temp directory */

                clean_up();
                exit_code = x_fatal;
                goto ExitMain;
              }
              break;
  }

ExitMain:
  set_directory(orig_path);
  switch (exit_code) {
    case x_success  : break;
    case x_usage    : break;
    case x_fatal    : fprintf(stderr,"\n%s\n",_slrerr);
                      sprintf(logbuf,"%s",_slrerr);
                      logit("slrxfer");
                      break;
  }
  return exit_code;
}

/*
 * Function: xfer_packet()
 * Purpose : Call transfer protocol to send/receive packet.
 * Return  : zero on success, non-zero on error and set _slrerr.
*/

int xfer_packet(char *function)
{
  struct ffblk ffblk;

  if (load_transfer_parms(function) != 0) goto ErrorExit;

  if (stricmp(function,"SEND") == 0)
	{
    if (findfirst("*.*",&ffblk,0) != 0)
    {
      sprintf(_slrerr,"xfer_packet(): no packet found - nothing to send.");
      goto ErrorExit;
    }
    if (verbose == YES)
      RPrintf("Attempting to send packet %s\n",ffblk.ff_name);

    if (diskswap == YES)
    {
      if (verbose == YES)
        RPrintf("Initializing swap to %s\n",swap_path);
      init_SPAWNO(swap_path,SWAP_DISK);
    }

    if (spawnlp(P_WAIT,program,program,options,ffblk.ff_name,NULL) != 0)
		{
      sprintf(_slrerr,"xfer_packet(): %s transfer unsuccessful!",protocol);
      goto ErrorExit;
		}
    if (verbose == YES)
      RPrintf("Packet sent successfully\n");
  }
	else
	{
    if (verbose == YES)
      RPrintf("Attempting to receive packet\n");

    if (diskswap == YES)
    {
      if (verbose == YES)
        RPrintf("Initializing swap to %s\n",swap_path);
      init_SPAWNO(swap_path,SWAP_DISK);
    }

    if (spawnlp(P_WAIT,program,program,options,NULL) != 0)
		{
      sprintf(_slrerr,"xfer_packet(): %s transfer unsuccessful!",protocol);
      goto ErrorExit;
    }
    if (verbose == YES)
      RPrintf("Packet received successfully\n");
  }
GoodExit:
  return 0;
ErrorExit:
	return 1;
}

/*
 * Function: load_transfer_parms()
 * Purpose : Load the appropriate transfer parameters from the Solar
 *           configuration file.
 * Return  : zero on success, non-zero on error and set _slrerr.
*/

int load_transfer_parms(char *function)
{
	FILE *config_file = fopen(config_path,"rt");

	int idx  = 0;

	char buf[128];
	char search_string[20];
	char *p = NULL;

  if (!config_file)
	{
    sprintf(_slrerr,"load_transfer_parms(): error opening %s",config_path);
    goto ErrorExit;
  }
  if (verbose == YES)
    RPrintf("Scanning %s\n",config_path);

	strcpy(program,"NONE");
	options[0] = '\0';
	if (stricmp(function,"SEND") == 0)
	{
		strcpy(search_string,"send-");
	}
	if (stricmp(function,"RECEIVE") == 0)
	{
		strcpy(search_string,"receive-");
	}
  strcat(search_string,protocol);

  if (verbose == YES)
    RPrintf("Looking for %s\n",search_string);

	while (fgets(buf,128,config_file) != NULL)
	{

    if ((strnicmp(buf,"send",4) == 0) || (strnicmp(buf,"receive",7) == 0))
		{

      if ((p = strtok(buf,"=")) == NULL)
			{
        strcpy(_slrerr,"load_transfer_parms(): error in protocol parameter.");
        goto ErrorExit;
			}

      if (stricmp(p,search_string) == 0)
			{

        if ((p = strtok(NULL,"=")) == NULL)
        {
          strcpy(_slrerr,"load_transfer_parms(): error in protocol parameter.");
          goto ErrorExit;
        }

        idx = 0;
        while ((p[idx] != '\n') && (p[idx] != '\0'))
					buf[idx] = p[idx++];
        buf[idx] = '\0';

        if ((p = strtok(buf," ")) != NULL)
				{

          idx = 0;
          while ((p[idx] != '\n') && (p[idx] != '\0'))
            program[idx] = p[idx++];
          program[idx] = '\0';

          if ((p = strtok(NULL,"=")) != NULL)
          {
            idx = 0;
            while ((p[idx] != '\n') && (p[idx] != '\0'))
							options[idx] = p[idx++];
						options[idx] = '\0';
          }
          else
          {
            options[0] = '\0';
          }
        }
        else
        {
          strcpy(_slrerr,"load_transfer_parms(): error in protocol parameter.");
          goto ErrorExit;
        }
				break;
			}
    }
  }
  fclose(config_file);

  if (verbose == YES)
    RPrintf("Scan complete\n");

  if (strcmp(program,"NONE") == 0)
	{
    strcpy(_slrerr,"load_transfer_parms(): selected protocol not configured.");
    goto ErrorExit;
  }
  if (verbose == YES)
    RPrintf("Command loaded: %s %s\n",program, options);

GoodExit:
  return 0;
ErrorExit:
  if (config_file) fclose(config_file);
  return 1;
}

/*
 * Function: get_packet_size()
 * Purpose : Find size of packet in temp directory. Assumes the
 *           packet is the only file in the directory.
 * Return  : The size of the packet in bytes.
*/

long get_packet_size()
{
  struct ffblk ffblk;

  if (findfirst("*.*",&ffblk,0) != 0)
	{
		return 0L;
	}
  return ffblk.ff_fsize;
}

/*
 * Function: void usage()
 * Purpose : Display slrxfer command line usage.
 * Return : N/A
*/

void usage()
{
  RPrintf("\nSOLAR v%s <> slrxfer (%s)\n",VERSION,CODE_DATE);
  RPrintf("\nUsage: slrxfer -u username {-s|-r} [-v] [-f]\n");
}

/*
Function: int copy_packet();
Purpose : After an unsuccessful download transfer, copy packet
          to user's directory so download may be continued another time.
          This assumes the packet is the only file in the temp directory.
Return  : 0 on success, non-zero on error.
*/

int copy_packet()
{
	struct ffblk ffblk;
	FILE *orig_file = NULL;
	FILE *dest_file = NULL;
	char orig_path[MAXPATH];
	char dest_path[MAXPATH];
	unsigned char movebuf;

  if (findfirst("*.*",&ffblk,0) == 0)
	{
		strcpy(orig_path,temp_path);
		strcat(orig_path,"\\");
		strcat(orig_path,ffblk.ff_name);
		strcpy(dest_path,user_path);
		strcat(dest_path,"\\");
		strcat(dest_path,ffblk.ff_name);

    RPrintf("copying %s to %s\n",orig_path,dest_path);

		if ((orig_file = fopen(orig_path,"rb")) == NULL) goto ErrorExit;
		if ((dest_file = fopen(dest_path,"wb")) == NULL) goto ErrorExit;
    movebuf = getc(orig_file);
    while (feof(orig_file) == 0)
		{
			putc(movebuf,dest_file);
      movebuf = getc(orig_file);
    }
  }

ExitFunct:
	if (orig_file) fclose(orig_file);
	if (dest_file) fclose(dest_file);
	return 0;
ErrorExit:
	if (orig_file) fclose(orig_file);
	if (dest_file) fclose(dest_file);
	return 1;
}

/*
 * Function: clean_up()
 * Purpose : Clean temporary directory
 * Return  : N/A
*/

void clean_up()
{
	struct ffblk ffblk;
  int done = 0;

  done = findfirst("*.*",&ffblk,0);
	while (!done)
	{
    unlink(ffblk.ff_name);
		done = findnext(&ffblk);
	}
  return;
}

