{
Rafe's Digital Display V1.03
----------------------------

Please see the file README.TXT for full information!

Additional property summary:

AutoSize - If true the display will automatically adjust it's
           height to suit the "Numbers" bitmap.
FormatWidth - This is the number of digits displayed. If
	      "Value" has less digits than this it is padded
	      with 0's.
Gap - The number of pixels between digits.
Margin - The number of pixels to leave between the borders and
         the actual digits.
Numbers - This has to be a bitmap containing the numbers
	  0-9 in a filmstrip fashion where all the numbers
          occupy the same width. The display also supports
          a - (for negative numbers) if added after the 9.
          NOTE: If you do use - then you will need to
          manually alter "NumberWidth".
 NumberWidth - The width of each number in the "Numbers"
               bitmap. This number is automatically
               calculated whenever the bitmap changes so will
               not normally need to be changed.
 Value - The value to display!

E-Mail: rafe@rcai.demon.co.uk
WWW: http://www.rcai.demon.co.uk/
Surface Mail: 89 Sharland Close, Grove, Oxon. OX12 0AF. UK.
}

unit Rcadigi;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls;

type
  TRCADigital = class(TPanel)
  private
    FNumbers : TBitmap;
    FNumberWidth : Word;
    FValue : Integer;
    FMargin : Word;
    FAutoSize : Boolean;
    FGap : Integer;
    FFormatWidth : Word;
    procedure SetNumbers (Value : TBitmap);
    procedure SetNumberWidth (Value : Word);
    procedure SetValue (Value : Integer);
    procedure SetMargin (Value : Word);
    procedure SetAutoSize (Value : Boolean);
    procedure SetGap (Value : Integer);
    procedure SetFormatWidth (Value : Word);
  protected
    ValueString : String;
    procedure Paint; override;
    procedure SetHeight;
    procedure GetSides (var TopSide, LeftSide, RightSide : Integer);
    procedure WMSize (var Message : TWMSize); message WM_Size;
  public
    constructor Create (AOwner : TComponent); override;
    destructor Destroy; override;
  published
    property Numbers : TBitmap read FNumbers write SetNumbers;
    property NumberWidth : Word read FNumberWidth write SetNumberWidth default 0;
    property Value : Integer read FValue write SetValue default 0;
    property Margin : Word read FMargin write SetMargin default 2;
    property AutoSize : Boolean read FAutoSize write SetAutoSize default true;
    property Gap : Integer read FGap write SetGap default 0;
    property FormatWidth : Word read FFormatWidth write SetFormatWidth default 1;
  end;

procedure Register;

implementation

{ constructor / destructor }

constructor TRCADigital.Create (AOwner : TComponent);
begin
  inherited Create (AOwner);
  FNumbers:=TBitmap.Create;
  { panel properties }
  BevelInner:=bvLowered;
  BevelWidth:=3;
  Color:=clBlack;
  { our properties }
  ValueString:='0';
  FNumberWidth:=0;
  FValue:=0;
  FMargin:=2;
  FAutoSize:=True;
  FGap:=0;
  FFormatWidth:=1;
end;

destructor TRCADigital.Destroy;
begin
  FNumbers.Free;
  inherited Destroy;
end;

{ properties }

procedure TRCADigital.SetNumbers (Value : TBitmap);
begin
  FNumbers.Assign(Value);
  if FNumbers<>nil then
  begin
    FNumberWidth:=FNumbers.Width div 10;
    SetHeight;
    Invalidate;
  end;
end;

procedure TRCADigital.SetNumberWidth (Value : Word);
begin
  FNumberWidth:=Value;
  Invalidate;
end;

procedure TRCADigital.SetValue (Value : Integer);
begin
  FValue:=Value;
  ValueString:=Format ('%.'+IntToStr (FFormatWidth)+'d', [FValue]);
  Invalidate;
end;

procedure TRCADigital.SetMargin (Value : Word);
begin
  FMargin:=Value;
  SetHeight;
  Invalidate;
end;

procedure TRCADigital.SetAutoSize (Value : Boolean);
begin
  FAutoSize:=Value;
  SetHeight;
  Invalidate;
end;

procedure TRCADigital.SetGap (Value : Integer);
begin
  FGap:=Value;
  Invalidate;
end;

procedure TRCADigital.SetFormatWidth (Value : Word);
begin
  FFormatWidth:=Value;
  SetValue (FValue);
end;

{ support procedures }

procedure TRCADigital.WMSize (var Message : TWMSize);
begin
  inherited;
  SetHeight;
end;

procedure TRCADigital.Paint;
var
  DigitLeft, BitmapLeft, TopSide, LeftSide, RightSide, L : Integer;
begin
  inherited Paint;
  if FNumbers<>nil then { something to draw with }
  begin
    GetSides (TopSide, LeftSide, RightSide); { get drawing area }
    DigitLeft:=RightSide;
    for L:=Length (ValueString) downto 1 do { draw value }
    begin
      Dec (DigitLeft, FNumberWidth);
      BitmapLeft:=0;
      if DigitLeft>LeftSide then
      begin
        case ValueString[L] of
          '-' : BitmapLeft:=10*FNumberWidth;
          '0'..'9' : BitmapLeft:=(ord (ValueString[L])-48)*FNumberWidth;
        end; { case }
        Canvas.CopyRect (Rect (DigitLeft, TopSide, DigitLeft+FNumberWidth, TopSide+FNumbers.Height),
                         FNumbers.Canvas,
                         Rect (BitmapLeft, 0, BitmapLeft+FNumberWidth, FNumbers.Height));
      end; { if }
      DigitLeft:=DigitLeft-Gap; { add the gap if there is one }
    end; { for }
  end; { if }
end;

procedure TRCADigital.GetSides (var TopSide, LeftSide, RightSide : Integer);
begin
  LeftSide:=0;
  RightSide:=Width;
  TopSide:=0;
  if BevelInner<>bvNone then
  begin
    Inc (LeftSide, BevelWidth);
    Dec (RightSide, BevelWidth);
    Inc (TopSide, BevelWidth);
  end; { if }
  if BevelOuter<>bvNone then
  begin
    Inc (LeftSide, BevelWidth);
    Dec (RightSide, BevelWidth);
    Inc (TopSide, BevelWidth);
  end; { if }
  Inc (LeftSide, BorderWidth);
  Dec (RightSide, BorderWidth);
  Inc (TopSide, BorderWidth);
  Inc (LeftSide, Margin);
  Dec (RightSide, Margin);
  Inc (TopSide, Margin);
end;

procedure TRCADigital.SetHeight;
var
  T, L, R : Integer;
begin
  if FAutoSize then
  begin
    GetSides (T, L, R);
    Height:=(T*2)+FNumbers.Height;
  end;
end;

procedure Register;
begin
  RegisterComponents('Rafe', [TRCADigital]);
end;

end.
