// SQUASH32 by bill buckels 1995
// Scale a 256 color bitmap to a 3:2 ratio of its original size
// using color averaging and a user defined color clipping algorithm.

// We could have made this more versatile by implementing
// floating scale algorithms but too many colors
// would be created in most images to display in a 256 color palette
// and we would need to quantize colors and further lose color
// information until the colors are reduced to the display resolution.

/*
    3:2 area reduction fixed scale

    The Color Scaling Algorithm applies the following convolution
    to the RGB gun values in the image to scale the image:

    Multiplier X RGB  - Sum -   Divide by 9 = Average Color

      PIXEL                            PIXEL
     _______________                  _______________
  R  | 4  | 2  | 4 |               R  |4+2+2+|2+4+1+|
  A  |____|____|___|               A  |1 /9  |2 /9  |
  S  | 2  | 1  | 2 |     To        S  |______|______|
  T  |____|____|___|               T  |2+1+4+|1+2+2+|
  E  | 4  | 2  | 4 |               E  |2 /9  |4 /9  |
  R  |____|____|___|               R  |______|______|

  Typically the color resolution of the result will be greater
  than the palette even with the fixed scale reduction.

  A color loss algorithm is applied to the RGB gun values after
  scaling. Color resolution is lost by 2's compliments, by
  shifting the gun value right then left.

  This value is set to 4 by default but is user definable.

*/

#include <io.h>
#include <fcntl.h>
#include <malloc.h>
#include <dos.h>
#include <stdio.h>
#include <string.h>

/* Bitmap Header structures */
typedef struct tagRGBQUAD
{
    unsigned char    rgbBlue;
    unsigned char    rgbGreen;
    unsigned char    rgbRed;
    unsigned char    rgbReserved;
} RGBQUAD;

typedef struct tagBITMAPINFOHEADER
{
    unsigned long   biSize;
    unsigned long   biWidth;
    unsigned long   biHeight;
    unsigned        biPlanes;
    unsigned        biBitCount;
    unsigned long   biCompression;
    unsigned long   biSizeImage;
    unsigned long   biXPelsPerMeter;
    unsigned long   biYPelsPerMeter;
    unsigned long   biClrUsed;
    unsigned long   biClrImportant;
} BITMAPINFOHEADER;

// constants for the biCompression field
#define BI_RGB      0L
#define BI_RLE8     1L
#define BI_RLE4     2L

typedef struct tagBITMAPINFO
{
    BITMAPINFOHEADER bmiHeader;
    RGBQUAD          bmiColors[256];
} BITMAPINFO;


typedef struct tagBITMAPFILEHEADER
{
    unsigned        bfType;
    unsigned long   bfSize;
    unsigned        bfReserved1;
    unsigned        bfReserved2;
    unsigned long   bfOffBits;
} BITMAPFILEHEADER;


BITMAPFILEHEADER BitMapFileHeader;
BITMAPINFO       bmp;

// basic palette mapping globals
long bmpcount[256];
long outcount[256];

unsigned char rgbinfo[256][3];
unsigned char outinfo[256][3];

// flag for too many colors
int toomany=0;

// read 3 lines, write 2
// average the color
// first pass is average in the x axis
// second pass is average in the y axis
// clip colors on the way out
unsigned char outbuffer[800];
unsigned buf1[800][3];
unsigned buf2[800][3];
unsigned buf3[800][3];

int shifter=4;

void partsize(unsigned char *ptr1,unsigned char *ptr2,unsigned char *ptr3,
              unsigned xres,FILE *fp2)
{
    unsigned red,green,blue;
    unsigned char outred,outgreen,outblue;
    unsigned char c;
    unsigned i,j,found,packet;
    int x, x1;

    xres = (xres/3)*3;    // 3 pixel groups

    // Pass 1 - absorb the middle pixels into the pixels before and after
    //          accumulate the right and left pixels into shorter rasters
    //          preapare the convolution ratios during accumulation
    i=0;
    for(x1=0;x1<xres;x1++)
    {

       if(x1%3==1)continue;     // skip middle pixels - these are absorbed
       if(x1%3==0)x=x1+1;       // if we are on the left side, add 1
       else x = x1-1;           // if we are on the right side, subtract 1

       c= ptr1[x1];             // top raster
       red   = rgbinfo[c][0];
       green = rgbinfo[c][1];
       blue  = rgbinfo[c][2];   // 4 x main pixel
       buf1[i][0] = red<<2;
       buf1[i][1] = green<<2;
       buf1[i][2] = blue<<2;
       c= ptr1[x];
       red   = rgbinfo[c][0];
       green = rgbinfo[c][1];
       blue  = rgbinfo[c][2];   // 2 x middle pixel
       buf1[i][0] += red<<1;
       buf1[i][1] += green<<1;
       buf1[i][2] += blue<<1;

       c= ptr2[x1];             // middle raster
       red   = rgbinfo[c][0];
       green = rgbinfo[c][1];
       blue  = rgbinfo[c][2];   // 2 x main pixel
       buf2[i][0] = red<<1;
       buf2[i][1] = green<<1;
       buf2[i][2] = blue<<1;
       c= ptr2[x];
       red   = rgbinfo[c][0];
       green = rgbinfo[c][1];
       blue  = rgbinfo[c][2];   // 1 x middle pixel
       buf2[i][0] += red;
       buf2[i][1] += green;
       buf2[i][2] += blue;

       c= ptr3[x1];            // bottom raster
       red   = rgbinfo[c][0];
       green = rgbinfo[c][1];
       blue  = rgbinfo[c][2];  // 4 x main pixel
       buf3[i][0] = red<<2;
       buf3[i][1] = green<<2;
       buf3[i][2] = blue<<2;
       c= ptr3[x];
       red   = rgbinfo[c][0];
       green = rgbinfo[c][1];
       blue  = rgbinfo[c][2];  // 2 x middle pixel
       buf3[i][0] += red<<1;
       buf3[i][1] += green<<1;
       buf3[i][2] += blue<<1;
       i++;
    }

    xres = (xres/3)*2;    // 2 pixel groups
    packet=xres;
    while((packet%4)!=0)packet++;

    // Pass 2 - absorb the middle raster into the raster above and below
    for(i=0;i<xres;i++)
    {

       buf1[i][0] += buf2[i][0];
       buf1[i][1] += buf2[i][1];
       buf1[i][2] += buf2[i][2];

       buf3[i][0] += buf2[i][0];
       buf3[i][1] += buf2[i][1];
       buf3[i][2] += buf2[i][2];
    }

    // Passes 3 and 4 - write top and bottom raster buffers
    // after the rgb value is converted to a palette index
    
    for(x1=0;x1<xres;x1++)
    {
       red    =   buf1[x1][0]/9;
       green  =   buf1[x1][1]/9;
       blue   =   buf1[x1][2]/9;

       outred     = (unsigned char)red;
       outgreen   = (unsigned char)green;
       outblue    = (unsigned char)blue;
       if(shifter)
       {
       outred     = ((outred >> shifter)<<shifter);
       outgreen   = ((outgreen>>shifter)<<shifter);
       outblue    = ((outblue>>shifter)<<shifter);
       }
       found = 0;

       // check for an entry position in the current palette
       for(j=0;j<256;j++)
       {
         // if an unused entry use it... highwater mark
         if(outcount[j]<1L)
         {
           found++;
           outcount[j]++;
           outinfo[j][0] = outred;
           outinfo[j][1] = outgreen;
           outinfo[j][2] = outblue;
           break;
         }

         // if we have a match use that
         if(outred   == outinfo[j][0] &&
            outgreen == outinfo[j][1] &&
            outblue  == outinfo[j][2])
            {
             found++;
             outcount[j]++;
             outinfo[j][0] = outred;
             outinfo[j][1] = outgreen;
             outinfo[j][2] = outblue;
             break;
            }

       }
       // if more than 256 colors we have too many colors - don't bother
       // otherwise use the index position for this color
       if(!found)toomany = 1;
       else outbuffer[x1] = (unsigned char )j;

    }
    // regardless, write to the file...
    fwrite(outbuffer,packet,1,fp2);

    for(x1=0;x1<xres;x1++)
    {
       red    =   buf3[x1][0]/9;
       green  =   buf3[x1][1]/9;
       blue   =   buf3[x1][2]/9;
       outred     = (unsigned char)red;
       outgreen   = (unsigned char)green;
       outblue    = (unsigned char)blue;
       if(shifter)
       {
       outred     = ((outred >> shifter)<<shifter);
       outgreen   = ((outgreen>>shifter)<<shifter);
       outblue    = ((outblue>>shifter)<<shifter);
       }
       found = 0;

       // check for an entry position in the current palette
       for(j=0;j<256;j++)
       {
         // if an unused entry use it... highwater mark
         if(outcount[j]<1L)
         {
           found++;
           outcount[j]++;
           outinfo[j][0] = outred;
           outinfo[j][1] = outgreen;
           outinfo[j][2] = outblue;
           break;
         }

         // if we have a match use that
         if(outred   == outinfo[j][0] &&
            outgreen == outinfo[j][1] &&
            outblue  == outinfo[j][2])
            {
             found++;
             outcount[j]++;
             outinfo[j][0] = outred;
             outinfo[j][1] = outgreen;
             outinfo[j][2] = outblue;
             break;
            }

       }
       if(!found)toomany = 1;
       else outbuffer[x1] = (unsigned char )j;

    }
    fwrite(outbuffer,packet,1,fp2);

}

// command line help
char *usage[]={
"Scale a 256 color bitmap to a 3:2 ratio of its original size",
"using color averaging and a user defined color clipping ratio.",
"The Color Scaling Algorithm applies the following convolution",
"to the RGB gun values in the image to scale the image:",
"",
" Multiplier X RGB  - Sum - Divide by 9 = Average Color",
" PIXEL                            PIXEL",
"    _______________                  _______________",
" R  | 4  | 2  | 4 |               R  |4+2+2+|2+4+1+|",
" A  |____|____|___|               A  |1 /9  |2 /9  |",
" S  | 2  | 1  | 2 |     To        S  |______|______|",
" T  |____|____|___|               T  |2+1+4+|1+2+2+|",
" E  | 4  | 2  | 4 |               E  |2 /9  |4 /9  |",
" R  |____|____|___|               R  |______|______|",
"",
" Typically the color resolution of the result will be greater than the",
" palette even with the fixed scale reduction. A color loss algorithm is",
" applied to the RGB gun values after scaling. Color resolution is lost",
" by 2's compliments, by shifting the gun value right then left.",
" This value is set to 4 by default but is user definable.",
"!"};

int main(int argc, char **argv)
{

    long colors;
    unsigned char infile[128], outfile[128];
    static unsigned char src1[800],src2[800],src3[800];
    FILE *fp,*fp2;
    int status,i;
    unsigned x,bytesperline;
    unsigned char c;

    unsigned long hres,vres,seeksize;
    unsigned xres;

    puts("SQUASH32 Copyright (C) 1995 by Bill Buckels");
    if(argc==1 || argc>4)
    {
   puts(
   "Usage is : \"SQUASH32 [BMP256.IN] [BMP256.OUT] (optional-shift (1-8))\"");
   puts("For Automatic name generation...");
   puts(
   "Usage is : \"SQUASH32 [BMP256.BMP] (optional-shift (1-8))\"");
   puts(
   "Output is: \"[BMP256.BMW]\"");
   puts(
   "For Help : \"SQUASH32 ?\"");
   puts(
   "Returns  : 0 = Success, 1 = Invalid, 2 - 8 = Failure (suggested shift)");
   return 1;
    }

    if(argc==2 && argv[1][0] == '?')
    {
       for(i=0;usage[i][0]!='!';i++)puts(usage[i]);
       return 0;

    }
    strcpy(infile,argv[1]);
    strcpy(outfile,argv[2]);
    if((fp=fopen(infile,"rb"))==NULL)
    {
        perror(infile);
        return 1;
    }
    if(argc==4)
    {
        shifter=atoi(argv[3]);
    }

    if(argc<4)
    {   // automatic name generation

        if(argc==2)
        {
          strcpy(outfile,"4");
        }
        else
        {
          strcpy(outfile,argv[2]);
        }

        if(atoi(outfile)>0 && atoi(outfile)<9 && strlen(outfile)<2)
        {
            shifter=atoi(outfile);
            strcpy(outfile,infile);
            i=0;
            while(outfile[i]!=0)
            {
                if(outfile[i]=='.')outfile[i]=0;
                else i++;
            }
            strcat(outfile,".bmw");
        }
    }

    // read the header stuff into the appropriate structures
    fread((char *)&BitMapFileHeader.bfType,
	             sizeof(BITMAPFILEHEADER),1,fp);
    fread((char *)&bmp.bmiHeader.biSize,
                 sizeof(BITMAPINFO),1,fp);

    status = 0;
    if(bmp.bmiHeader.biCompression!=BI_RGB)status++;
    if(bmp.bmiHeader.biWidth>800L)status++;
    if(bmp.bmiHeader.biHeight>600L)status++;
    if(bmp.bmiHeader.biPlanes!=1)status++;
    if(bmp.bmiHeader.biBitCount!=8)status++;
    if(status)
     {
        puts("Not A Supported 256 Color .BMP!");
        if(bmp.bmiHeader.biCompression!=BI_RGB)
          puts("Not an RGB Format .BMP!");
        if(bmp.bmiHeader.biWidth>800L)
          puts("Maximum Width of 800 pixels exceeded!");
        if(bmp.bmiHeader.biHeight>600L)
          puts("Maximum Height of 600 Rasters exceeded!");
        if(bmp.bmiHeader.biPlanes!=1)
          puts("File Has More Than 1 Plane of Image Data!");
        if(bmp.bmiHeader.biBitCount!=8)
          puts("File Must Be 8 Bits Per Pixel!");
        puts("Exiting!");
        fclose(fp);
        return 1;
     }

    if((fp2=fopen(outfile,"wb"))==NULL)
    {
        perror(outfile);
        fclose(fp);
        return 1;
    }

    // palette stuff
    for(i=0;i<256;i++)
    {

      rgbinfo[i][0]=bmp.bmiColors[i].rgbRed;
      rgbinfo[i][1]=bmp.bmiColors[i].rgbGreen;
      rgbinfo[i][2]=bmp.bmiColors[i].rgbBlue;
      outinfo[i][0]=0;
      outinfo[i][1]=0;
      outinfo[i][2]=0;
    }

    // zero our colour count arrays
    memset((char *)&bmpcount[0],0,1024);
    memset((char *)&outcount[0],0,1024);
    for(x=0;x<800;x++)
    {
       for(i=0;i<3;i++)
       {
         buf1[x][i]=0;
         buf2[x][i]=0;
         buf3[x][i]=0;
       }
    }
    hres = bmp.bmiHeader.biWidth;
    vres = bmp.bmiHeader.biHeight;
    vres = (vres/3) * 3;
    xres = (unsigned )hres;

    // dw boundaries
    bytesperline = xres;
    while((bytesperline%4)!=0)bytesperline++;

    printf("Now Reading 256 Color Windows Bitmap : %s\n",infile);
    printf("Height %ld\n",vres);
    printf("Width  %ld\n",hres);
    fwrite((char *)&BitMapFileHeader.bfType,
                 sizeof(BITMAPFILEHEADER),1,fp2);
    fwrite((char *)&bmp.bmiHeader.biSize,
                 sizeof(BITMAPINFO),1,fp2);

    seeksize =  0L;
    seeksize += BitMapFileHeader.bfOffBits;
    printf("       <- Scanline");
    for(i=0;i<vres;i++)
        {
        printf("\r%d",i+1);
        fseek(fp,seeksize,SEEK_SET);
        fread(src1,xres,1,fp);
        for(x=0;x<xres;x++)
        {
            c=src1[x];
            bmpcount[c]+=1;
        }
        seeksize += bytesperline;
        i++;
        if(i<vres)
        {
        printf("\r%d",i+1);
        fseek(fp,seeksize,SEEK_SET);
        fread(src2,xres,1,fp);
        for(x=0;x<xres;x++)
        {
            c=src2[x];
            bmpcount[c]+=1;
        }
        seeksize += bytesperline;
        }
        else
        {
         break;
        }
        i++;
        if(i<vres)
        {
        printf("\r%d",i+1);
        fseek(fp,seeksize,SEEK_SET);
        fread(src3,xres,1,fp);
        for(x=0;x<xres;x++)
        {
            c=src3[x];
            bmpcount[c]+=1;
        }
        seeksize += bytesperline;
        }
        else
        {
         break;
        }
        partsize(src1,src2,src3,xres,fp2);
        if(toomany)break;

    }
    puts("");
    fclose(fp);
    fclose(fp2);
    colors=0;

    // do a colour count
    for(x=0;x<256;x++)
    {
      if(bmpcount[x]!=0L)colors++;
    }
    printf("Input File contains %ld colors.\n",colors);

    if(toomany)
    {
       puts("Not Enough Colors! Output File needs more than 256 colors!");
       puts("Try Running Again Using The Following Command Line :");
       printf("SQUASH32 %s %s %d\n",infile,outfile,shifter+1);
       remove(outfile);
       return shifter+1;
    }
    colors=0L;
    for(x=0;x<256;x++)
    {
      if(outcount[x]!=0L)colors++;
    }

    if((fp2=fopen(outfile,"rb+w"))==NULL)
    {
        perror(outfile);
        fclose(fp);
        return 1;
    }
    seeksize = filelength(fileno(fp2));

    // new palette stuff
    for(i=0;i<256;i++)
    {
      bmp.bmiColors[i].rgbRed =    outinfo[i][0];
      bmp.bmiColors[i].rgbGreen =  outinfo[i][1];
      bmp.bmiColors[i].rgbBlue =   outinfo[i][2];
      bmp.bmiColors[i].rgbReserved = 0;
    }
    // new header stuff
    BitMapFileHeader.bfOffBits = 0L;
    BitMapFileHeader.bfOffBits +=sizeof(BITMAPFILEHEADER);
    BitMapFileHeader.bfOffBits +=sizeof(BITMAPINFO);
    BitMapFileHeader.bfSize =  seeksize;
    seeksize -= BitMapFileHeader.bfOffBits;
    bmp.bmiHeader.biWidth         = (xres/3)*2;
    bmp.bmiHeader.biHeight        = (vres/3)*2;
    bmp.bmiHeader.biClrUsed       = 256L;
    bmp.bmiHeader.biClrImportant  = colors;
    bmp.bmiHeader.biSizeImage     = seeksize;
    fseek(fp2,0L,SEEK_SET);
    fwrite((char *)&BitMapFileHeader.bfType,
                 sizeof(BITMAPFILEHEADER),1,fp2);
    fwrite((char *)&bmp.bmiHeader.biSize,
                 sizeof(BITMAPINFO),1,fp2);
    fclose(fp2);
    printf("Output File %s created! Done!\n",outfile);
    printf("Output File contains %ld colors.\n",colors);
    printf("Height %ld\n",bmp.bmiHeader.biHeight);
    printf("Width  %ld\n",bmp.bmiHeader.biWidth);

return 0;
}
