/* ------------------------------------------------------------------------ */
/* WSQUASH.C (C) Copyright Bill Buckels 1995-1999                           */
/* All Rights Reserved.                                                     */
/*                                                                          */
/* Licence Agreement                                                        */
/* -----------------                                                        */
/*                                                                          */
/* You have a royalty-free right to use, modify, reproduce and              */
/* distribute this source code in any way you find useful,                  */
/* provided that you agree that Bill Buckels has no warranty obligations    */
/* or liability resulting from said distribution in any way whatsoever.     */
/* If you don't agree, remove this source code from your computer now.      */
/*                                                                          */
/* Written by   : Bill Buckels                                              */
/*                589 Oxford Street                                         */
/*                Winnipeg, Manitoba, Canada R3M 3J2                        */
/*                                                                          */
/* Email: bbuckels@escape.ca                                                */
/* WebSite: http://www.escape.ca/~bbuckels                                  */
/*                                                                          */
/* Date Written : November 1995                                             */
/* Purpose      : Squash Windows Bitmaps While Retaining Detail.            */
/* Revision     : 1.0 First Release                                         */
/* ------------------------------------------------------------------------ */
/* Written in Large Model Microsoft C Version 6.00a                         */
/* ------------------------------------------------------------------------ */

#include <windows.h>
#include <mmsystem.h>
#include <commdlg.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <conio.h>
#include <stdarg.h>
#include <malloc.h>
#include "wsquash.h"

int AboutWSQUASH(HWND);
int WarnSquash(HWND);
int OpenBMP(HWND);
int LoadPic(LPSTR,HWND);
int ShowPic(void);
int SaveBMP(HWND);
int ClearScreen(HWND);
int partsize(unsigned,FILE *);
int Squash32(LPSTR,HWND);
void Vga2Svga(unsigned char *, unsigned char huge *, unsigned);
void Mono2Svga(unsigned char *, unsigned char huge *, unsigned);

long far PASCAL WndProc (HWND, WORD, WORD, LONG) ;

// allow up to 256 colors
typedef struct tagMYBITMAPINFO
{
    BITMAPINFOHEADER bmiHeader;
    RGBQUAD          bmiColors[256];
} MYBITMAPINFO;

typedef struct tagLOGPAL
{
    WORD         Version;
    WORD         Entries;
    PALETTEENTRY Entry[256];
}LOGPAL;

BITMAPFILEHEADER bmfhead;
MYBITMAPINFO     bmp;
BITMAPFILEHEADER bmfsave;
MYBITMAPINFO     bmpout;

LOGPAL         myLogPal;                   // logical palette
HPALETTE       hpalCurrent = NULL;         // Handle to current palette
HPALETTE       hpalOld     = NULL;

// basic palette mapping globals
static long bmpcount[256];
static long outcount[256];

static unsigned char rgbinfo[256][3];
static unsigned char outinfo[256][3];

// flag for too many colors
static int toomany=0,shifter=4;

// read 3 lines, write 2
// average the color
// first pass is average in the x axis
// second pass is average in the y axis
// clip colors on the way out
static unsigned char outbuffer[800];
static unsigned buf1[800][3];
static unsigned buf2[800][3];
static unsigned buf3[800][3];
static unsigned char src1[800], src2[800], src3[800];

HANDLE hInst;                              // handle to instance
HDC hdcMine;

GLOBALHANDLE bighandle;
unsigned char huge *bigbuffer;
char lpCaption[51];      // stringstore buffer for short string resource
char lpMessage[250];     // stringstore buffer for long string resource


unsigned  xwidth,ywidth,ywidth2;
LONG  lxwidth,lywidth;
LONG  bigsize;
static char szErrorBuf[256]="\0";

unsigned bmpwidth,bmpheight;

// Global Flags For Editing Functions
BOOL STARTED =FALSE;
BOOL SQUASH  =FALSE;

#define MAXFILENAME 256          // maximum length of file pathname
#define MAXCUSTFILTER 40         // maximum size of custom filter buffer

char *title="Select/Enter BMP Input File For Load";
char *savetitle ="Select/Enter BMP Output File For Save";

char *lpCopyRight="WSQUASH\251 v1.0 Copyright Bill Buckels 1995-1999";

// new variables for common dialogs
char far *szFilterSpec;
static char szImgSpec[128]
          = "BMP Files(*.BMP)\0*.BMP\0"; // filter string for dir. listings

char szCustFilterSpec[MAXCUSTFILTER];        // custom filter buffer

OPENFILENAME ofn;             // struct. passed to GetOpenFileName

static char PathName[128] = "\0";
static char FileName[128] = "\0";

#define LOAD_ACTION 1
#define SAVE_ACTION 2

int initDialog(HWND hWnd, int iAction)
{
    // fill in non-variant fields of OPENFILENAME struct.
    ofn.lStructSize       = sizeof(OPENFILENAME);
    ofn.hwndOwner         = hWnd;
    ofn.lpstrFilter       = szFilterSpec;
    ofn.lpstrCustomFilter = szCustFilterSpec;
    ofn.nMaxCustFilter	  = MAXCUSTFILTER;
    ofn.nFilterIndex      = 1;

    if (iAction == SAVE_ACTION)
      strcpy(FileName, "temp.bmp");
    else
      FileName[0] = 0;

    ofn.lpstrFile         = FileName;
    ofn.nMaxFile          = MAXFILENAME;
    ofn.lpstrInitialDir   = PathName;
    ofn.Flags             = OFN_HIDEREADONLY;
    ofn.lpfnHook          = NULL;

    if (iAction == SAVE_ACTION)
      ofn.lpstrTitle = savetitle;
    else
      ofn.lpstrTitle = title;

return 0;
}


int PASCAL WinMain (HANDLE hInstance, HANDLE hPrevInstance,
                    LPSTR lpszCmdParam, int nCmdShow)
     {
     HWND        hWnd;
     MSG         msg ;
     WNDCLASS    wc ;
     WORD        wParam;

     hInst=hInstance;            // handle to instance

     if (!hPrevInstance)
          {
          wc.style         = CS_HREDRAW | CS_VREDRAW ;
          wc.lpfnWndProc   = WndProc ;
          wc.cbClsExtra    = 0 ;
          wc.cbWndExtra    = 0 ;
          wc.hInstance     = hInstance ;
          wc.hIcon   = LoadIcon(hInstance, "MyIcon");
          wc.hCursor       = LoadCursor (NULL, IDC_ARROW) ;
          wc.hbrBackground = GetStockObject (BLACK_BRUSH) ;
          wc.lpszMenuName  = "WSQUASH";
          wc.lpszClassName = "WSQUASH" ;

          RegisterClass (&wc) ;
	  }


     xwidth = 800;
     ywidth = 601;
     lxwidth = (LONG) xwidth;
     lywidth = (LONG) ywidth;
     bigsize = lxwidth * lywidth;

     hWnd = CreateWindow ("WSQUASH",
              lpCopyRight,
              WS_MAXIMIZE|WS_MINIMIZEBOX|
              WS_CAPTION|WS_POPUP|WS_BORDER|
			  WS_SYSMENU,
              0,0,
              800,600,
              NULL, NULL, hInstance, NULL) ;

     ShowWindow (hWnd, SW_SHOWMAXIMIZED) ;
     UpdateWindow (hWnd) ;
     SetCursor(LoadCursor(NULL,IDC_ARROW));

     while (GetMessage (&msg, NULL, 0, 0))
     {
          TranslateMessage (&msg) ;
          DispatchMessage (&msg) ;
     }
     return msg.wParam ;
}


long far PASCAL WndProc (HWND hWnd, WORD message, WORD wParam, LONG lParam)
{
       PAINTSTRUCT ps;
       unsigned mx, my;

     switch (message)
	  {

       case WM_CREATE:

     hdcMine    =NULL;
     bighandle  =NULL;
     szErrorBuf[0] = 0;

     if(wParam=GetSystemMenu(hWnd,FALSE))
            {
              DeleteMenu(wParam, SC_SIZE,        MF_BYCOMMAND);
              DeleteMenu(wParam, SC_RESTORE,     MF_BYCOMMAND);
              DeleteMenu(wParam, SC_NEXTWINDOW,  MF_BYCOMMAND);
              DeleteMenu(wParam, SC_PREVWINDOW,  MF_BYCOMMAND);
            }

     if((hdcMine=GetDC(hWnd))==NULL)
     {
        strcat(szErrorBuf,"Unable to create device context! ");
     }
     else
     {
       if((bighandle  = GlobalAlloc(GMEM_MOVEABLE,bigsize))==NULL)
       {
        strcat(szErrorBuf,"Unable to allocate primary memory block! ");
       }
    }


     if(hdcMine == NULL || bighandle == NULL)
     {
          if(hdcMine   !=NULL)ReleaseDC(hWnd,hdcMine);
          if(bighandle !=NULL)GlobalFree(bighandle);
          bighandle =NULL;
          hdcMine   =NULL;
          MessageBox(hWnd,szErrorBuf,"Unable to Continue...",
                     MB_ICONSTOP | MB_OK);
          PostMessage(hWnd,WM_COMMAND,IDM_Exit,0L);

     }
     else
     {
       PostMessage(hWnd,WM_COMMAND,IDM_Warning,0L);
     }
     return 0L;

         case WM_SYSCOMMAND :

                      if((wParam &0xfff0)==SC_MAXIMIZE)
                      {
                        if(STARTED == TRUE)ShowPic(); return 0L;
                       }
                      break;

       case WM_COMMAND:
                        switch(wParam)
                        {
                          case  IDM_Warning:
                       if (WarnSquash(hWnd)==IDNO)
                       {
                         PostMessage(hWnd,WM_COMMAND,IDM_Exit,0L);
                        }
                        return 0L;


                          case  IDM_Open : OpenBMP(hWnd);
                                           return 0L;

                          case  IDM_Reset: ClearScreen(hWnd);
                                           if(hpalCurrent!=NULL)
                                             DeleteObject(hpalCurrent);
                                           hpalCurrent = NULL;
                                           SQUASH  = FALSE;
                                           return 0L;

                          case  IDM_Save : SaveBMP(hWnd);
                                           return 0L;

                          case  IDM_Exit : DestroyWindow(hWnd);
                                           return 0L;

                          case  IDM_About: AboutWSQUASH(hWnd);
                                           return 0L;
                        }
                       break;

      case WM_PAINT:

           // our paint method...
           // restore the damaged area of the client window
           // by redrawing the bitmap
           BeginPaint(hWnd,&ps);
           if(STARTED == TRUE)ShowPic();
           EndPaint(hWnd,&ps);
           return 0L;

      case WM_DESTROY:
           STARTED = FALSE;
           if(bighandle !=NULL)GlobalFree(bighandle);
           if(hpalCurrent!=NULL)
           {
             SelectPalette (hdcMine, hpalOld, 0);
             DeleteObject(hpalCurrent);
             hpalCurrent=NULL;
            }
           if(hdcMine!=NULL)ReleaseDC(hWnd,hdcMine);
           bighandle = NULL;
           hdcMine   = NULL;
           PostQuitMessage (0);
           return 0L;
        }

     return DefWindowProc (hWnd, message, wParam, lParam) ;
}

int OpenBMP(HWND hWnd)
{

    szFilterSpec=(char far *)&szImgSpec[0];
    initDialog(hWnd, LOAD_ACTION);
    if(GetOpenFileName ((LPOPENFILENAME)&ofn))
           {
            LoadPic(ofn.lpstrFile,hWnd);
            }

return 0;
}

int SaveBMP(HWND hWnd)
{
    int retval;

    if(STARTED==FALSE)
    {
     MessageBox(hWnd,"A .BMP File Has Not Been Loaded!",lpCopyRight,
                     MB_ICONEXCLAMATION);
     return 0;
    }

    if(SQUASH==FALSE)
    {
     MessageBox(hWnd,"Unable to Squash Current Image!\n\n"
                     "WSQUASH supports 2, 16, and 256 Color .BMP Files Only!",
                     lpCopyRight,
                     MB_ICONEXCLAMATION);
     return 0;
    }

    szFilterSpec=(char far *)&szImgSpec[0];
    initDialog(hWnd, SAVE_ACTION);
    if(GetOpenFileName ((LPOPENFILENAME)&ofn))
           {
             shifter = 4;
             for(;;)
             {
               retval = Squash32(ofn.lpstrFile,hWnd);
               if(retval<0)break;
               if(retval>0)
               {
                shifter=retval;
                continue;
               }
               LoadPic(ofn.lpstrFile,hWnd);
               break;
             }
            }

return 0;
}

// translate a 16 color line  to an svga line
void Vga2Svga(unsigned char *src, unsigned char huge *dest,unsigned width)
{
     int i,j;

     j=0;
     for(i=0;i<width;i++)
     {
        dest[j] = src[i]>>4;  j++;
        dest[j] = src[i]&0xf; j++;
     }
}

// translate a 2 color line to an svga line
void Mono2Svga(unsigned char *src, unsigned char huge *dest,unsigned width)
{
     int i,j,k;
     unsigned char msk[]={0x80,0x40,0x20,0x10,0x8,0x4,0x2,0x1};

     j=0;
     for(i=0;i<width;i++)
     {

        for(k=0;k<8;k++)
        {
            if(src[i]&msk[k])dest[j] = 1; // foreground
            else dest[j] = 0;             // background
            j++;
        }
     }
}


char oldbmp[128]="";
char szScanLine[832];

// load a non compressed bitmap up to the size of the screen
// and up to 256 colors
int LoadPic(LPSTR mypic, HWND hWnd)
{

    FILE *bmpfile;
    unsigned char *ptr;
    unsigned long lcount;
    int i;
    unsigned char huge *MyPtr;
    unsigned wHres, wVres, bytesperline;

    int iFormat,
        iPalColors;

    if((bmpfile=fopen(mypic,"rb"))==NULL)
    {
     MessageBeep(MB_ICONEXCLAMATION);
     MessageBox(hWnd,"Problems Were Encountered Opening The File.",
                     mypic,MB_ICONEXCLAMATION);
     return -1;
     }

    // now we are committed
    ClearScreen(hWnd);
    fread((char *)&bmfhead.bfType,
                   sizeof(BITMAPFILEHEADER),1,bmpfile);
    fread((char *)&bmp.bmiHeader.biSize,
                   sizeof(MYBITMAPINFO),1,bmpfile);

    ptr=(char *)&bmfhead.bfType;
    // check header
    if(ptr[0] != 'B' || ptr[1]  != 'M' ||
       bmp.bmiHeader.biPlanes   !=  1  ||
       bmp.bmiHeader.biBitCount >   8  ||
       bmp.bmiHeader.biCompression != BI_RGB)
        {
         fclose(bmpfile);
         MessageBeep(MB_ICONEXCLAMATION);
         MessageBox(hWnd,"Format is not supported...",
                     mypic,MB_ICONEXCLAMATION);
         SQUASH = FALSE;
         if(hpalCurrent!=NULL)DeleteObject(hpalCurrent);
         hpalCurrent=NULL;
         oldbmp[0]=0;
         return -2;
        }

    bmpwidth = (unsigned)bmp.bmiHeader.biWidth;
    bmpheight = (unsigned)bmp.bmiHeader.biHeight;

    if(bmpwidth > 800 || bmpheight > 600)
    {    fclose(bmpfile);
         MessageBeep(MB_ICONEXCLAMATION);
         MessageBox(hWnd,"File is too large to display...",
                     mypic,MB_ICONEXCLAMATION);
         if(hpalCurrent!=NULL)DeleteObject(hpalCurrent);
         hpalCurrent=NULL;
         SQUASH = FALSE;
         oldbmp[0]=0;
         return -3;
    }

    SetCursor(LoadCursor(NULL,IDC_WAIT));

    // create the new palette from the palette entries
    // in the bmp file

    SQUASH = TRUE;
    iFormat = 0;

    if(bmp.bmiHeader.biPlanes==1 && bmp.bmiHeader.biBitCount==8)
    {
       iFormat        = SVGA;
       iPalColors     = SVGA;
       bytesperline = bmpwidth;
       while((bytesperline%4)!=0)bytesperline++;
    }
    if(bmp.bmiHeader.biPlanes==1 && bmp.bmiHeader.biBitCount==4)
    {
       iFormat       = VGA;
       iPalColors    = VGA;
       bytesperline = bmpwidth;
       while((bytesperline%8)!=0)bytesperline++;
       bytesperline /= 2;

    }
    if(bmp.bmiHeader.biPlanes==1 && bmp.bmiHeader.biBitCount==1)
    {
       iFormat       = MONO;
       iPalColors    = MONO;
       bytesperline = bmpwidth;
       while((bytesperline%32)!=0)bytesperline++;
       bytesperline /= 8;

    }

    wHres = (unsigned)bmpwidth;
    while((wHres%4)!=0)wHres++;
    wVres = (unsigned)bmpheight;

    if(hpalCurrent!=NULL)DeleteObject(hpalCurrent);
    myLogPal.Version = 0x300;
    myLogPal.Entries = 256;
    for(i=0;i<iPalColors;i++)
    {
      myLogPal.Entry[i].peRed  =  bmp.bmiColors[i].rgbRed;
      myLogPal.Entry[i].peGreen=  bmp.bmiColors[i].rgbGreen;
      myLogPal.Entry[i].peBlue =  bmp.bmiColors[i].rgbBlue;
      myLogPal.Entry[i].peFlags=0;
    }
    for(i=iPalColors;i<256;i++)
    {
      myLogPal.Entry[i].peRed  =
      myLogPal.Entry[i].peGreen=
      myLogPal.Entry[i].peBlue =
      myLogPal.Entry[i].peFlags=0;
    }

    hpalCurrent=CreatePalette((PLOGPALETTE )&myLogPal.Version);

    if((bigbuffer = GlobalLock(bighandle))==NULL)
     {
            SetCursor(LoadCursor(NULL,IDC_ARROW));
            bigbuffer=NULL;
            fclose(bmpfile);
            MessageBox(hWnd,
                  "Memory Lock Error!\n"
                  "Unable to globally lock a buffer!",
                  "Unable to lock...",
                  MB_OK | MB_ICONSTOP);
            oldbmp[0]=0;
           STARTED = FALSE;
           SQUASH  = FALSE;
           if(hpalCurrent!=NULL)DeleteObject(hpalCurrent);
           hpalCurrent = NULL;
           oldbmp[0]=0;
           return -5;
     }

    fseek(bmpfile,(long)bmfhead.bfOffBits,SEEK_SET);


    // SVGA doesn't need banding during the read...
    // But we need to band mono and vga since we are expanding to
    // 256 colors...

    if (iFormat == MONO || iFormat == VGA) {
      lcount  = 0L;
      for (i = 0; i < wVres; i++) {
        fread(szScanLine, bytesperline,1,bmpfile);

        // we use a huge to properly traverse segment boundaries
        // not and issue in WIN32 of course, but this is 16 bit
        // code with 64K segment boundaries.

        MyPtr = (unsigned char huge *)&bigbuffer[lcount];
        lcount += wHres;

        switch(iFormat)
        {
            case MONO:  Mono2Svga(szScanLine, MyPtr, bytesperline);
                               break;
            case VGA:   Vga2Svga(szScanLine, MyPtr, bytesperline);
                              break;
        }
      }

      // now override original settings on mono and vga bitmaps
      // all .BMP's expand to 256 colors
      bmp.bmiHeader.biSize= 40L;
      // (long) sizeof(MYBITMAPINFO);
      // bmp.bmiHeader.biWidth=nochange;
      // bmp.bmiHeader.biHeight=nochange;
      // bmp.bmiHeader.biPlanes=nochange;
      bmp.bmiHeader.biBitCount=8;
      // bmp.bmiHeader.biCompression=nochange;
      bmp.bmiHeader.biSizeImage=(long)wHres;
      bmp.bmiHeader.biSizeImage*=wVres;
      // bmp.bmiHeader.biXPelsPerMeter=;
      // bmp.bmiHeader.biYPelsPerMeter=;
      bmp.bmiHeader.biClrUsed=256;
      bmp.bmiHeader.biClrImportant=0;
    }
    else {

      // 256 colors only... one shot read...
      MyPtr = bigbuffer;
      fread(MyPtr,bytesperline,wVres,bmpfile);
    }

    fclose(bmpfile);
    GlobalUnlock(bighandle);
    bigbuffer = NULL;
    STARTED = TRUE;

    if(ShowPic()!=0)
    {
      SetCursor(LoadCursor(NULL,IDC_ARROW));
      MessageBox(hWnd,"Memory Lock Error!\n"
                 "Unable to globally lock a buffer!",
                 "Unable to lock...",
                 MB_OK | MB_ICONSTOP);

      STARTED = FALSE;
      SQUASH  = FALSE;
      if(hpalCurrent!=NULL)DeleteObject(hpalCurrent);
      hpalCurrent = NULL;
      oldbmp[0]=0;
      return -5;
    }
    strcpy(oldbmp,mypic);
    SetCursor(LoadCursor(NULL,IDC_ARROW));

return 0;
}

// copy a file into view
int ShowPic()
{
   // set the custom palette... save the windows palette
   if(hpalCurrent!=NULL)
   {
   hpalOld=SelectPalette (hdcMine, hpalCurrent, 0);
   RealizePalette(hdcMine);
   }

   // lock the memory
   if((bigbuffer = GlobalLock(bighandle))==NULL)
   {
            SelectPalette (hdcMine, hpalOld, 0);
            bigbuffer  =NULL;
            return -5;
   }

   // draw the bitmap on the screen
   SetDIBitsToDevice(hdcMine,0,0,
                             bmpwidth,bmpheight,
                             0, 0, 0, bmpheight, // height
                  (LPSTR)&bigbuffer[0],
                  (PBITMAPINFO)&bmp.bmiHeader.biSize,
                             DIB_RGB_COLORS);
   // unlock the memory
   GlobalUnlock(bighandle);

   // restore the windows palette
   if(hpalCurrent!=NULL)
   {
   SelectPalette (hdcMine, hpalOld, 0);
   }

bigbuffer=NULL;
return 0;
}

// clear the client area of the window to black
int ClearScreen(HWND hWnd)
{
    HBRUSH hPrevBrush, hClearBrush;// brush to clear client area
    RECT rcClientArea;   // data structure of xy coordinates for rectangle
    HPEN hPrevPen, hClearPen;      // pen to clear area

if(hdcMine==NULL)return 0;
STARTED = FALSE;                                // reset image status
hClearBrush = CreateSolidBrush(RGB(0,0,0));     // create brush
hClearPen   = CreatePen(PS_SOLID,1,RGB(0,0,0)); // create pen
hPrevBrush = SelectObject(hdcMine,hClearBrush); // select brush
hPrevPen   = SelectObject(hdcMine,hClearPen);   // select pen
GetClientRect(hWnd,&rcClientArea);              // fetch bounding coordinates
FillRect(hdcMine,&rcClientArea,hClearBrush);    // filled rectangle
SelectObject(hdcMine,hPrevPen);                 // deselect pen
SelectObject(hdcMine,hPrevBrush);               // deselect brush
DeleteObject(hClearBrush);                      // delete brush
DeleteObject(hClearPen);                        // delete pen

return 0;                                       // return to caller
}

int partsize(unsigned xoutres,FILE *fp)
{
    unsigned red,green,blue;
    unsigned char outred,outgreen,outblue;
    unsigned char c;
    unsigned i,j,found,packet;
    int x, x1;

    xoutres = (xoutres/3)*3;    // 3 pixel groups

    // Pass 1 - absorb the middle pixels into the pixels before and after
    //          accumulate the right and left pixels into shorter rasters
    //          prepare the convolution ratios during accumulation
    i=0;
    for(x1=0;x1<xoutres;x1++)
    {

       if(x1%3==1)continue;     // skip middle pixels - these are absorbed
       if(x1%3==0)x=x1+1;       // if we are on the left side, add 1
       else x = x1-1;           // if we are on the right side, subtract 1

       c= src1[x1];             // top raster
       red   = rgbinfo[c][0];
       green = rgbinfo[c][1];
       blue  = rgbinfo[c][2];   // 4 x main pixel
       buf1[i][0] = red<<2;
       buf1[i][1] = green<<2;
       buf1[i][2] = blue<<2;
       c= src1[x];
       red   = rgbinfo[c][0];
       green = rgbinfo[c][1];
       blue  = rgbinfo[c][2];   // 2 x middle pixel
       buf1[i][0] += red<<1;
       buf1[i][1] += green<<1;
       buf1[i][2] += blue<<1;

       c= src2[x1];             // middle raster
       red   = rgbinfo[c][0];
       green = rgbinfo[c][1];
       blue  = rgbinfo[c][2];   // 2 x main pixel
       buf2[i][0] = red<<1;
       buf2[i][1] = green<<1;
       buf2[i][2] = blue<<1;
       c= src2[x];
       red   = rgbinfo[c][0];
       green = rgbinfo[c][1];
       blue  = rgbinfo[c][2];   // 1 x middle pixel
       buf2[i][0] += red;
       buf2[i][1] += green;
       buf2[i][2] += blue;

       c= src3[x1];            // bottom raster
       red   = rgbinfo[c][0];
       green = rgbinfo[c][1];
       blue  = rgbinfo[c][2];  // 4 x main pixel
       buf3[i][0] = red<<2;
       buf3[i][1] = green<<2;
       buf3[i][2] = blue<<2;
       c= src3[x];
       red   = rgbinfo[c][0];
       green = rgbinfo[c][1];
       blue  = rgbinfo[c][2];  // 2 x middle pixel
       buf3[i][0] += red<<1;
       buf3[i][1] += green<<1;
       buf3[i][2] += blue<<1;
       i++;
    }

    xoutres = (xoutres/3)*2;    // 2 pixel groups
    packet=xoutres;
    while((packet%4)!=0)packet++;

    // Pass 2 - absorb the middle raster into the raster above and below
    for(i=0;i<xoutres;i++)
    {

       buf1[i][0] += buf2[i][0];
       buf1[i][1] += buf2[i][1];
       buf1[i][2] += buf2[i][2];

       buf3[i][0] += buf2[i][0];
       buf3[i][1] += buf2[i][1];
       buf3[i][2] += buf2[i][2];
    }

    // Passes 3 and 4 - write top and bottom raster buffers
    // after the rgb value is converted to a palette index
    
    for(x1=0;x1<xoutres;x1++)
    {
       red    =   buf1[x1][0]/9;
       green  =   buf1[x1][1]/9;
       blue   =   buf1[x1][2]/9;

       outred     = (unsigned char)red;
       outgreen   = (unsigned char)green;
       outblue    = (unsigned char)blue;
       if(shifter)
       {
       outred     = ((outred >> shifter)<<shifter);
       outgreen   = ((outgreen>>shifter)<<shifter);
       outblue    = ((outblue>>shifter)<<shifter);
       }
       found = 0;

       // check for an entry position in the current palette
       for(j=0;j<256;j++)
       {
         // if an unused entry use it... highwater mark
         if(outcount[j]<1L)
         {
           found++;
           outcount[j]++;
           outinfo[j][0] = outred;
           outinfo[j][1] = outgreen;
           outinfo[j][2] = outblue;
           break;
         }

         // if we have a match use that
         if(outred   == outinfo[j][0] &&
            outgreen == outinfo[j][1] &&
            outblue  == outinfo[j][2])
            {
             found++;
             outcount[j]++;
             outinfo[j][0] = outred;
             outinfo[j][1] = outgreen;
             outinfo[j][2] = outblue;
             break;
            }

       }
       // if more than 256 colors we have too many colors - don't bother
       // otherwise use the index position for this color
       if(!found)toomany = 1;
       else outbuffer[x1] = (unsigned char )j;

    }
    // regardless, write to the file...
    fwrite(outbuffer,packet,1,fp);

    for(x1=0;x1<xoutres;x1++)
    {
       red    =   buf3[x1][0]/9;
       green  =   buf3[x1][1]/9;
       blue   =   buf3[x1][2]/9;
       outred     = (unsigned char)red;
       outgreen   = (unsigned char)green;
       outblue    = (unsigned char)blue;
       if(shifter)
       {
       outred     = ((outred >> shifter)<<shifter);
       outgreen   = ((outgreen>>shifter)<<shifter);
       outblue    = ((outblue>>shifter)<<shifter);
       }
       found = 0;

       // check for an entry position in the current palette
       for(j=0;j<256;j++)
       {
         // if an unused entry use it... highwater mark
         if(outcount[j]<1L)
         {
           found++;
           outcount[j]++;
           outinfo[j][0] = outred;
           outinfo[j][1] = outgreen;
           outinfo[j][2] = outblue;
           break;
         }

         // if we have a match use that
         if(outred   == outinfo[j][0] &&
            outgreen == outinfo[j][1] &&
            outblue  == outinfo[j][2])
            {
             found++;
             outcount[j]++;
             outinfo[j][0] = outred;
             outinfo[j][1] = outgreen;
             outinfo[j][2] = outblue;
             break;
            }

       }
       if(!found)toomany = 1;
       else outbuffer[x1] = (unsigned char )j;

    }
    fwrite(outbuffer,packet,1,fp);

return 0;

}

int Squash32(LPSTR outfilename, HWND hWnd)
{
    FILE *fp;
    int i;
    long dwCtr, houtres, voutres, seeksize, colors;
    unsigned xoutres,x,outbytes;
    unsigned char c;

    i=0;
    while(outfilename[i]!=0)
    {
      if(outfilename[i]=='.')outfilename[i]=0;
      else i++;
    }
    if(outfilename[i]==0)
    {
        outfilename[i]   ='.';
        outfilename[i+1] ='B';
        outfilename[i+2] ='M';
        outfilename[i+3] ='P';
        outfilename[i+4] = 0 ;
    }
    if((fp=fopen(outfilename,"rb"))!=NULL)
     {
       fclose(fp);
       if(MessageBox(hWnd,"BMP file Already Exists.\n\nOverwrite?",
                     outfilename,MB_YESNO | MB_ICONQUESTION)==IDNO)return -1;

       oldbmp[0]=0;
     }


    // copy the header info from the input file into the work structures
    memcpy((char *)&bmfsave.bfType,(char *)&bmfhead.bfType,
           sizeof(BITMAPFILEHEADER));
    memcpy((char *)&bmpout.bmiHeader.biSize,(char *)&bmp.bmiHeader.biSize,
           sizeof(MYBITMAPINFO));

    toomany = 0;

    // palette stuff
    for(i=0;i<256;i++)
    {
      rgbinfo[i][0]=bmpout.bmiColors[i].rgbRed;
      rgbinfo[i][1]=bmpout.bmiColors[i].rgbGreen;
      rgbinfo[i][2]=bmpout.bmiColors[i].rgbBlue;
      outinfo[i][0]=0;
      outinfo[i][1]=0;
      outinfo[i][2]=0;
      outcount[i]  =0L;
      bmpcount[i]  =0L;
    }

    // zero our colour count arrays
    memset((char *)&bmpcount[0],0,1024);
    memset((char *)&outcount[0],0,1024);
    for(x=0;x<800;x++)
    {
       for(i=0;i<3;i++)
       {
         buf1[x][i]=0;
         buf2[x][i]=0;
         buf3[x][i]=0;
       }
    }

    houtres = bmpout.bmiHeader.biWidth;
    voutres = bmpout.bmiHeader.biHeight;
    voutres = (voutres/3) * 3;
    xoutres = (unsigned )houtres;
    outbytes = xoutres;
    while((outbytes%4)!=0)outbytes++;

    if((bigbuffer = GlobalLock(bighandle))==NULL)
    {
     MessageBox(hWnd,"Memory Lock Error!\n"
                     "Unable to globally lock a buffer!",
                     "Unable to lock...",
                     MB_OK | MB_ICONSTOP);

      bigbuffer  =NULL;
      return -1;
    }

    if((fp=fopen(outfilename,"wb"))==NULL)
    {
       MessageBox(hWnd,"File Creation Error! Cannot Continue.",outfilename,
                  MB_OK | MB_ICONSTOP);
       GlobalUnlock(bighandle);
       bigbuffer = NULL;
       return -1;
    }
    SetCursor(LoadCursor(NULL,IDC_WAIT));
    seeksize = 0L;
    fwrite((char *)&bmfsave.bfType,sizeof(BITMAPFILEHEADER),1,fp);
    fwrite((char *)&bmpout.bmiHeader.biSize,sizeof(MYBITMAPINFO),1,fp);
    for(i=0;i<voutres;i++)
        {
        dwCtr = seeksize;
        for(x=0;x<xoutres;x++)
        {
            src1[x] = bigbuffer[dwCtr];
            c=src1[x];
            bmpcount[c]+=1;
            dwCtr+=1;
        }
        seeksize += outbytes;
        dwCtr = seeksize;
        i++;
        if(i<voutres)
        {
        for(x=0;x<xoutres;x++)
        {
            src2[x] = bigbuffer[dwCtr];
            c=src2[x];
            bmpcount[c]+=1;
            dwCtr+=1;
        }
        seeksize += outbytes;
        dwCtr = seeksize;
        }
        else
        {
         break;
        }
        i++;
        if(i<voutres)
        {
        for(x=0;x<xoutres;x++)
        {
            src3[x] = bigbuffer[dwCtr];
            c=src3[x];
            bmpcount[c]+=1;
            dwCtr+=1;
        }
        seeksize += outbytes;
        }
        else
        {
         break;
        }
        partsize(xoutres,fp);
        if(toomany)break;

    }
    fclose(fp);
    GlobalUnlock(bighandle);
    bigbuffer = NULL;

    if(toomany)
    {
       SetCursor(LoadCursor(NULL,IDC_ARROW));
       remove(outfilename);
       if(MessageBox(hWnd,"Unable To Create File! File Removed!\n"
                  "Too Many Colors!\n\n"
                  "Do You Want To Try Fewer Colors?",
                   outfilename,
                   MB_YESNO|MB_ICONQUESTION)==IDNO)return -1;

        shifter +=1;
        return shifter;
    }

    // do a colour count
    colors=0L;
    for(x=0;x<256;x++)
    {
      if(outcount[x]!=0L)colors++;
    }

    if((fp=fopen(outfilename,"rb+w"))==NULL)
    {
      SetCursor(LoadCursor(NULL,IDC_ARROW));
      MessageBox(hWnd,"Unable To Write To File!\nUnable to Create Header!",
                 outfilename,MB_OK|MB_ICONSTOP);
      return -1;
    }
    seeksize = filelength(fileno(fp));

    // new palette stuff
    for(i=0;i<256;i++)
    {
      bmpout.bmiColors[i].rgbRed =    outinfo[i][0];
      bmpout.bmiColors[i].rgbGreen =  outinfo[i][1];
      bmpout.bmiColors[i].rgbBlue =   outinfo[i][2];
      bmpout.bmiColors[i].rgbReserved = 0;
    }

    // new header stuff
    bmfsave.bfOffBits = 0L;
    bmfsave.bfOffBits +=sizeof(BITMAPFILEHEADER);
    bmfsave.bfOffBits +=sizeof(MYBITMAPINFO);
    bmfsave.bfSize =  seeksize;
    seeksize -= bmfsave.bfOffBits;
    bmpout.bmiHeader.biWidth         = (xoutres/3)*2;
    bmpout.bmiHeader.biHeight        = (voutres/3)*2;
    bmpout.bmiHeader.biClrUsed       = 256L;
    bmpout.bmiHeader.biClrImportant  = colors;
    bmpout.bmiHeader.biSizeImage     = seeksize;

    fseek(fp,0L,SEEK_SET);
    fwrite((char *)&bmfsave.bfType,sizeof(BITMAPFILEHEADER),1,fp);
    fwrite((char *)&bmpout.bmiHeader.biSize,sizeof(MYBITMAPINFO),1,fp);
    fclose(fp);
    SetCursor(LoadCursor(NULL,IDC_ARROW));

return 0;
}

int AboutWSQUASH(HWND hWnd)
{
    MessageBox(hWnd,
    "WSQUASH(C) is a special purpose Image Size Reduction Utility Program "
    "which attempts to preserve detail (especially in text screens) by using "
    "a process called \"Squashing\".\n\n"
    "If you reduce a 256 color Windows .BMP file in size using conventional "
    "scaling techniques, you will lose detail when shrinking your image. "
    "SQUASHING (scaling by even factors) increases the quality of the "
    "reduced image.\n\nGeometric objects "
    "are almost preserved.\nWSQUASH uses a color loss method "
    "called Color Clipping resulting in a uniform appearance, "
    "and reduces your image by 3:2 each time it is processed.\n\n"
    "WSQUASH(C) is written by\n\n"
    "\tBill Buckels\n"
    "\t589 Oxford Street\n"
    "\tWinnipeg, Mb, Cdn R3M 3J2\n\n"
    "\tEmail: bbuckels@escape.ca\n"
    "\tInternet: http://www.escape.ca/~bbuckels\n\n"
    "WSQUASH(C) is written in Microsoft C using Standard Windows API Calls. "
    "It comes with open source, and may be used and distributed freely.",
    lpCopyRight, MB_OK | MB_ICONINFORMATION);
    return 0;

}

int WarnSquash(HWND hWnd)
{
    return MessageBox(hWnd,
    "WSQUASH(C) is a special purpose Image Size Reduction Utility Program "
    "which attempts to preserve detail (especially in text screens) by using "
    "a process called \"Squashing\".\n\n"
    "WSQUASH(C) is written by\n\n"
    "\tBill Buckels\n"
    "\t589 Oxford Street\n"
    "\tWinnipeg, Mb, Cdn R3M 3J2\n\n"
    "\tEmail: bbuckels@escape.ca\n"
    "\tInternet: http://www.escape.ca/~bbuckels\n\n"
    "WSQUASH(C) is distributed as FreeWare. "
    "The Author Assumes No Liability Whatsoever "
    "For The Use of This Program. "
    "If you do not agree with these terms quit now!\n\n"
    "Do You Want To Continue?",
    lpCopyRight,MB_YESNO | MB_ICONQUESTION);
}
