unit TrayIcon;

{ TrayIcon v1.0
  Copyright (c) 1996 James Roberts-Thomson (com3roberjd@ntu.ac.uk)

  Based on TrayIcon v1.1, Tempest Software and
           TrayIcon v1.3, Peter Ness (102347.710@compuserve.com)

  A Delphi2 component designed to make creating SystemTray icons easier.
  Adding this component to a form, and when the control is active, the
  application will minimize to the SystemTray instead of the TaskBar.
  A popup menu may be made available (via a right-click on the icon).

  The component operates in two modes - if the "SeparateIcon" property is
  False (default), then the SysTray icon only appears when/if the application
  is minimized.  Setting the "SeparateIcon" property true allows both taskbar
  and SystemTray icons to appear, and also a SystemTray icon when the
  application isn't minimized.

  Requires Delphi 2.0 (32-bit), and either Windows95 or Windows-NT v4.0
  Component will compile and run with NT v3.51; but functionality is
  automatically disabled (NT v3.51 doesn't have a SystemTray).

  This component is Freeware, as were those it was based on.  You may freely
  use this component in any Application (be it Commercial, Shareware, or
  Freeware) without fee or royalty, provided this copyright notice remains
  intact.  Some minor form of recognition for commercial or shareware software
  would be nice.
 }

interface

uses
  Windows, Messages, SysUtils, ShellApi, Classes, Graphics, Controls, Forms, Menus;

Const WM_CallBack_Message = WM_User + 1;

type
  TTrayIcon = class(TComponent)
  private
    {Properties}
    fActive:Boolean;
    fHint:String;
    fIcon:TIcon;
    fPopupMenu:TPopupMenu;
    fSeparateIcon:Boolean;

    {events}
    fOnClick:TNotifyEvent;
    fOnDblClick:TNotifyEvent;
    fOnRightClick:TMouseEvent;
    fOnMinimize:TNotifyEvent;
    fOnRestore:TNotifyEvent;

    {Internal variables}
    fData:TNotifyIconData;
    fWindowHandle:hwnd;
    fWinVer4:Boolean;
    fMinimized:Boolean;
    fNoTrayIcon:Boolean;

  protected
    procedure SetActive(Value:Boolean);
    procedure SetHint(Value:String);
    procedure SetIcon(Icon:TIcon);
    procedure SetSeparateIcon(Value:Boolean);

    procedure AddIconToTray;
    procedure RemoveIconFromTray;
    procedure UpdateTrayIcon;
    procedure WndProc(var Msg:TMessage);
    procedure HandleRightClick(Sender:TObject);
    procedure HandleMinimize(Sender:TObject);
    procedure HandleRestore(Sender:TObject);

  public
    constructor Create(Owner:TComponent); override;
    destructor Destroy; override;

  published
    property Active:Boolean read fActive write SetActive;
    property Hint:string read fHint write SetHint;
    property Icon:TIcon read fIcon write SetIcon;
    property PopupMenu:TPopupmenu read fPopupMenu write fPopupMenu;
    property SeparateIcon:Boolean read fSeparateIcon write SetSeparateIcon;

    property OnClick:TNotifyEvent read fOnClick write fOnClick;
    property OnDblClick:TNotifyEvent read fOnDblClick write fOnDblClick;
    property OnRightClick:TMouseEvent read fOnRightClick write fOnRightClick;
    property OnMinimize:TNotifyEvent read fOnMinimize write fOnMinimize;
    property OnRestore:TNotifyEvent read fOnRestore write fOnRestore;
  end;

procedure Register;

implementation

{Create the Component}
constructor TTrayIcon.Create(Owner:TComponent);
var Hint:String;
    OSVerInfo:TOSVersionInfo;
    WindowPlacement:TWindowPlacement;
begin
     {Call inherited create method}
     Inherited Create(Owner);

     {Create the fIcon object, and assign the Application Icon to it}
     fIcon:=TIcon.Create;
     fIcon.Assign(Application.Icon);

     GetVersionEx(OSVerInfo);
     if OSVerInfo.dwMajorVersion > 3
     then
         fWinVer4:=True
     else
         fWinVer4:=False;

     fNoTrayIcon:=True;

     if not (csDesigning in ComponentState)
     then
         {At RunTime *only*, perform the following:}
         begin
              FillChar(fData, SizeOf(fData), 0);

              fWindowHandle:=AllocateHWnd(WndProc);

              fData.cbSize:=SizeOf(fData);
              fData.wnd:=fWindowHandle;
              fData.hIcon:=fIcon.Handle;
              fData.uFlags:=NIF_Icon OR NIF_Message;
              fData.uCallbackMessage:=WM_CallBack_Message;

              if fHint = ''
              then
                  Hint:=Application.Title
              else
                  Hint:=fHint;

              if Hint <> ''
              then
                  begin
                       fData.uFlags:=fData.uFlags OR NIF_Tip;
                       StrPLCopy(fData.szTip,Hint,SizeOf(fData.szTip)-1);
                  end;

              Application.OnMinimize:=HandleMinimize;
              Application.OnRestore:=HandleRestore;

              FillChar(WindowPlacement,SizeOf(WindowPlacement),0);
              WindowPlacement.length:=SizeOf(WindowPlacement);
              GetWindowPlacement(Application.Handle,@WindowPlacement);
              if WindowPlacement.showCmd = SW_ShowMinimized
              then
                  fMinimized:=True
              else
                  fMinimized:=False;

              if fActive and fMinimized
              then
                  AddIconToTray;
         end;
end;

{Destroy the Component}
destructor TTrayIcon.Destroy;
begin
     if fActive
     then
         RemoveIconFromTray;

     if not (csDesigning in ComponentState)
     then
         DeAllocateHWnd(FWindowHandle);

     fIcon.Free;
     inherited Destroy;
end;

procedure TTrayIcon.SetSeparateIcon(Value:Boolean);
begin
     if fSeparateIcon <> Value
     then
         fSeparateIcon:=Value;

     if not (csDesigning in ComponentState)
     then
         case fSeparateIcon
         of
           False:if fActive and (NOT fMinimized)
                 then
                     RemoveIconFromTray;
           True:if fActive
                then
                    AddIconToTray;
         end;
end;

procedure TTrayIcon.SetActive(Value:Boolean);
begin
     if fActive <> Value
     then
         fActive:=Value;

     if not (csDesigning in ComponentState)
     then
         if fActive and (fMinimized xor fSeparateIcon)
         then
             AddIconToTray
         else
             RemoveIconFromTray;
end;

procedure TTrayIcon.SetHint(Value:String);
begin
     if fHint <> Value
     then
         begin
              fHint:=Value;

              if not (csDesigning in ComponentState)
              then
                  begin
                       StrPLCopy(fData.szTip,fHint,SizeOf(fData.szTip)-1);
                       if fHint <> ''
                       then
                           fData.uFlags:=fData.uFlags OR NIF_Tip
                       else
                           fData.uFlags:=fData.uFlags AND NOT NIF_Tip;
                       UpdateTrayIcon;
                  end;
         end;
end;

procedure TTrayIcon.SetIcon(Icon:TIcon);
begin
     if fIcon <> Icon
     then
         begin
              fIcon.Assign(Icon);
              fData.hIcon:=Icon.Handle;
              UpdateTrayIcon;
         end;
end;

procedure TTrayIcon.AddIconToTray;
begin
     if fActive AND fWinVer4 AND fNoTrayIcon
     then
         if not Shell_NotifyIcon(NIM_Add,@fData)
         then
             raise EOutOfResources.Create('Cannot create shell notification icon')
         else
             fNoTrayIcon:=False;
end;

procedure TTrayIcon.RemoveIconFromTray;
begin
     if fWinVer4
     then
         Shell_NotifyIcon(NIM_Delete,@fData);
     fNoTrayIcon:=True;
end;

procedure TTrayIcon.UpdateTrayIcon;
begin
     if (fActive AND fWinVer4) AND not (csDesigning in ComponentState)
     then
         Shell_NotifyIcon(NIM_Modify,@fData);
end;

procedure TTrayIcon.WndProc(var Msg:TMessage);
begin
     with Msg
     do
       begin
            if msg = WM_CallBack_Message
            then
                case lParam
                of
                  WM_LButtonDblClk : if Assigned(fOnDblClick) then fOnDblClick(Self);
                  WM_LButtonUp     : if Assigned(fOnClick) then fOnClick(Self);
                  WM_RButtonUp     : HandleRightClick(Self);
                end
            else
                Result:=DefWindowProc(fWindowHandle,Msg,wParam,lParam);
       end;
end;

procedure TTrayIcon.HandleRightClick(Sender:TObject);
var CursorPos:TPoint;
begin
     if Assigned(fPopupMenu) AND ((NOT IsWindowVisible(Application.Handle)) OR
                                  fSeparateIcon)
     then
         begin
              GetCursorPos(CursorPos);
              fPopupMenu.Popup(CursorPos.X,CursorPos.Y);
         end;

     if Assigned(fOnRightClick)
     then
         fOnRightClick(Sender,mbRight,[],CursorPos.X,CursorPos.Y);
end;

procedure TTrayIcon.HandleMinimize(Sender:TObject);
begin
     if fActive
     then
         begin
              ShowWindow(Application.Handle,SW_Hide);
              if fNoTrayIcon
              then
                  AddIconToTray;
         end;
     fMinimized:=True;
     if Assigned(fOnMinimize)
     then
         fOnMinimize(Sender);
end;

procedure TTrayIcon.HandleRestore(Sender:TObject);
begin
     if fActive
     then
         begin
              ShowWindow(Application.Handle,SW_Restore);
              if not fSeparateIcon
              then
                  RemoveIconFromTray;
         end;
     if Assigned(fOnRestore)
     then
         fOnRestore(Sender);
     fMinimized:=False;
end;

procedure Register;
begin
  RegisterComponents('3rd Party', [TTrayIcon]);
end;

end.
