//
// Win-GZ.c -- Windows gzip/gunzip
//
// Version: 1.2
//
// Copyright (C) 2003 by Bob Crispen <crispen@hiwaay.net> doing business as
//   Foxy Software
//
// Portions of this code are based on minigzip.c, Copyright (C) 1995-1998
//   Jean-loup Gailly, distributed with zlib 1.1.3 by Jean-loup Gailly and
//   Mark Adler, at ftp://ftp.cdrom.com/pub/infozip/zlib/
//
// License:
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//   (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; if not, write to the Free Software
//   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Documentation:
//   The license is contained in /doc/copying-2.0.txt
//   Instructions are contained in /doc/readme.txt
// 
// Compiling notes:
//   Link with zlib.lib
//     Note: a port of zlib for lcc can be found at
//     http://home.hiwaay.net/~crispen/src/zlib_lcc.zip
//   Resource file: wingz.rc, requires wingzres.h
//   Compiles under lcc.  No guarantees for other systems.
//
// Known deficiencies:
//   1. If a file size is larger than a DWORD can contain,
//      the progress bar is likely to behave incorrectly.
//

#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>
#include <commdlg.h>
#include <ctype.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "zlib.h"
#include <sys/stat.h>
#include "wingzres.h"

// Constants
#define	GZIP_MAGIC     "\037\213" /* Magic header for gzip files, 1F 8B */
#define	OLD_GZIP_MAGIC "\037\236" /* Magic header for gzip 0.5 = freeze 1.x */
#define BUFLEN      16384

// Global variables
#define KEEP_ORIGINAL 0
#define DELETE_ORIGINAL 1
#define NAME_AS_ORIGINAL 2
int eDoWithOriginal;		// What to do with original file (pseudo enum)

BOOL fInputIsGzipped;		// The file is already gzipped

OPENFILENAME ofn;		// Struct for file dialog
char szFilePath[_MAX_PATH+1];	// Pathname of open (source) file
char szFileName[_MAX_FNAME+_MAX_EXT+1];	// Filename of open (source) file
char szFileDir[_MAX_PATH+1];	// Directory portion of path
char szStatus[_MAX_FNAME+_MAX_EXT+200];	// Send this to status bar
char szOutFilePath[_MAX_PATH+4];	// Output file path
HWND hwndStatusBar;		// Handle for status bar
HWND hwndProgressBar;		// Handle for progress bar
HWND hwndGzipButton;		// Handle for "gzip" button
HWND hwndGunzipButton;		// Handle for "gunzip" button
HWND hwndOpenButton;		// Handle for "select file"
DWORD dFileSizeHigh;		// Size of file in bytes == high doubleword
DWORD dFileSizeLow;		// Size of file in bytes == low doubleword
HINSTANCE ghInstance;		// Instance flag
FILE* src;
FILE* dest;

// Forward references
static BOOL CALLBACK DialogFunc(HWND, UINT, WPARAM, LPARAM);
static BOOL CALLBACK AboutDlgProc(HWND, UINT, WPARAM, LPARAM);
void processCmdLine();
void getDirectory();
void checkSourceFile(FILE*);
void doGzip(HWND);
void doGunzip(HWND);

//
// Win main just registers a class of the same type as the dialog class,
// gets the command line argumements, calls DialogBox, and exits.
// The return value is the return value of the dialog procedure.
//
int APIENTRY WinMain(
    HINSTANCE hinst, HINSTANCE hinstPrev, LPSTR lpCmdLine, int nCmdShow)
{
    WNDCLASS wc;

    memset(&wc,0,sizeof(wc));
    wc.lpfnWndProc = DefDlgProc;
    wc.cbWndExtra = DLGWINDOWEXTRA;
    wc.hInstance = hinst;
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = (HBRUSH) (COLOR_WINDOW + 1);
    wc.lpszClassName = "wingz";
    RegisterClass(&wc);
    InitCommonControls();

    // Save the current instance so we can get the icon
    ghInstance = hinst;

    return DialogBox(hinst,
	MAKEINTRESOURCE(IDD_MAINDIALOG), NULL, (DLGPROC) DialogFunc);
}

//
// Process the command line arguments
//
void processCmdLine(HWND hwndDlg)
{
    LPTSTR szCmdLine;
    char *ptr;
    char *ptr2 = szFilePath;

    szCmdLine = GetCommandLine();
    ptr = szCmdLine;

    // Default option radio button
    eDoWithOriginal = KEEP_ORIGINAL;

    memset(szFilePath, 0, sizeof(szFilePath));
    memset(szFileName, 0, sizeof(szFileName));
    memset(szFileDir, 0, sizeof(szFileDir));

    // Skip past the program name
    if (szCmdLine[0] == '"') {
	if ((ptr=strchr(&szCmdLine[1], '"')) == NULL) {
	    // Must be a weird command line.  Forget it.
	    return;
	}
	ptr++;
    } else {
	// Haven't seen unquoted program names in the wild, but just
	// in case
	for (; ((!isspace(*ptr)) && (*ptr != '\0')); ptr++);
    }
    if (*ptr == '\0') return;

    // Step ahead to the first non-whitespace character
    for (; isspace(*ptr) && (*ptr != '\0'); ptr++);

    // Check for options
    while (*ptr != '\0') {
	if ((strncmp(ptr, "-u", 2)) == 0) {
	    eDoWithOriginal = DELETE_ORIGINAL;
	    for (; !isspace(*ptr) && (*ptr != '\0'); ptr++);
	    for (; isspace(*ptr) && (*ptr != '\0'); ptr++);
	    continue;
    	}
	if ((strncmp(ptr, "-v", 2)) == 0) {
	    eDoWithOriginal = NAME_AS_ORIGINAL;
	    for (; !isspace(*ptr) && (*ptr != '\0'); ptr++);
	    for (; isspace(*ptr) && (*ptr != '\0'); ptr++);
	    continue;
    	}
	// Quoted filename
	if (*ptr == '"') {
	    for (++ptr; ((*ptr != '"') && (*ptr != '\0')); ptr++) {
		*ptr2++ = *ptr;
	    }
	    break;
	}
	// Naked filename
	for (; (*ptr != '\0'); ptr++) {
	    *ptr2++ = *ptr;
	}
    }
}

//
// Initialization code
//
static int InitializeApp(HWND hDlg, WPARAM wParam, LPARAM lParam)
{
    HICON icon;
    static char szFilter[] =
	"All Files (*.*)\0*.*\0" \
	"VRML Files (*.wrl)\0*.wrl\0\0";

    // Process the command line parameters
    processCmdLine(hDlg);

    // If there's a file path, get the name and directory
    if (strlen(szFilePath) > 0) {
	GetFileTitle(szFilePath, szFileName, sizeof(szFileName));
	getDirectory();
    }

    // Set default option on radio buttons
    switch (eDoWithOriginal) {
	case KEEP_ORIGINAL:
	    CheckRadioButton(hDlg, IDKEEPORIGINAL, IDNAMEASORIGINAL,
		IDKEEPORIGINAL);
	    break;
	case DELETE_ORIGINAL:
	    CheckRadioButton(hDlg, IDKEEPORIGINAL, IDNAMEASORIGINAL,
		IDDELETEORIGINAL);
	    break;
	case NAME_AS_ORIGINAL:
	    CheckRadioButton(hDlg, IDKEEPORIGINAL, IDNAMEASORIGINAL,
		IDNAMEASORIGINAL);
	    break;
    }

    // Set up the status bar
    hwndStatusBar = CreateStatusWindow(
	WS_CHILD | WS_VISIBLE | WS_CLIPSIBLINGS | CCS_BOTTOM,
	"Ready", hDlg, IDSTATUSBAR);

    // Set up for the file open dialog
    ofn.lStructSize       = sizeof(OPENFILENAME);
    ofn.hwndOwner	 = hDlg;
    ofn.hInstance	 = NULL;
    ofn.lpstrFilter       = szFilter;
    ofn.lpstrCustomFilter = NULL;
    ofn.nMaxCustFilter    = 0;
    ofn.nFilterIndex      = 0;
    ofn.lpstrFile	 = NULL;    // Set in Open and Close functions
    ofn.nMaxFile	  = _MAX_PATH;
    ofn.lpstrFileTitle    = NULL;    // Set in Open and Close functions
    ofn.nMaxFileTitle     = _MAX_FNAME + _MAX_EXT;
    ofn.lpstrInitialDir   = NULL;
    ofn.lpstrTitle	= NULL;
    ofn.Flags		  = 0;    // Set in Open and Close functions
    ofn.nFileOffset       = 0;
    ofn.nFileExtension    = 0;
    ofn.lpstrDefExt       = ".*";
    ofn.lCustData	 = 0L;
    ofn.lpfnHook	  = NULL;
    ofn.lpTemplateName    = NULL;

    // Create the icon
    icon=LoadIcon(ghInstance, MAKEINTRESOURCE(ID_ICON));
    SetClassLong(hDlg, GCL_HICON, (LONG)icon);

    // Set up the scale of the progress bar (0-100)
    hwndProgressBar = GetDlgItem(hDlg, IDPROGRESS);
    SendMessage(hwndProgressBar, PBM_SETRANGE, 0, MAKELPARAM(0, 100));
    SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)0, 0L);

    // Get the handles for both gzip and gunzip buttons and
    // disable them until the user has opened a file
    hwndGzipButton = GetDlgItem(hDlg, IDGZIP);
    hwndGunzipButton = GetDlgItem(hDlg, IDGUNZIP);
    EnableWindow(hwndGzipButton, FALSE);
    EnableWindow(hwndGunzipButton, FALSE);

    // Set the initial focus on the "Select File" button
    hwndOpenButton = GetDlgItem(hDlg, IDOPEN);

    // If there's already a file, open it and check it
    if (strlen(szFilePath) > 0) {
	src = fopen(szFilePath, "rb");
	checkSourceFile(src);
    }

    return 1;
}


//
// Main dialog function
//
static BOOL CALLBACK DialogFunc(
    HWND hwndDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
    switch (msg) {
    // This message means the dialog is started but not yet visible.
    // Do All initializations here
    case WM_INITDIALOG:
	InitializeApp(hwndDlg,wParam,lParam);
	return TRUE;
    case WM_COMMAND:
	switch (LOWORD(wParam)) {
	    case IDEXIT:
		EndDialog(hwndDlg,1);
		return TRUE;
	    case IDOPEN:
		// Open file dialog
		memset(szFilePath, 0, sizeof(szFilePath));
		memset(szFileName, 0, sizeof(szFileName));
		memset(szFileDir, 0, sizeof(szFileDir));
		ofn.lpstrFile      = szFilePath;
		ofn.lpstrFileTitle = szFileName;
		ofn.Flags	  = OFN_HIDEREADONLY | OFN_CREATEPROMPT;
		if (GetOpenFileName(&ofn)) {
		    src = fopen(szFilePath, "rb");
		    checkSourceFile(src);
		    return TRUE;
		} else {
		    SendMessage(hwndStatusBar,
			SB_SETTEXT, (WPARAM)0, (LPARAM)(LPSTR)"Ready");
		    return TRUE;
		}
	    case IDGZIP:
		// Gzip the file
		doGzip(hwndDlg);
		return TRUE;
	    case IDGUNZIP:
		// Gunzip the file
		doGunzip(hwndDlg);
		return TRUE;
	    case IDKEEPORIGINAL:
		// Set option to keep original file
		eDoWithOriginal = KEEP_ORIGINAL;
		return TRUE;
	    case IDDELETEORIGINAL:
		// Set option to delete original file
		eDoWithOriginal = DELETE_ORIGINAL;
		return TRUE;
	    case IDNAMEASORIGINAL:
		// Set option to delete original and
		// name new file as old
		eDoWithOriginal = NAME_AS_ORIGINAL;
		return TRUE;
	    case IDM_ABOUT:
		// Pop up the "About" box
		DialogBox(ghInstance, MAKEINTRESOURCE(IDD_ABOUTBOX),
		    hwndDlg, AboutDlgProc);
	}
	break;
    case WM_CLOSE:
	EndDialog(hwndDlg,0);
	return TRUE;
    }
    return FALSE;
}

//
// "About" box functions
//
// Handle messages to "About" dialog
static BOOL CALLBACK AboutDlgProc (
    HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
    switch (iMsg) {
	case WM_INITDIALOG:
	    return TRUE;
	case WM_COMMAND:
	    switch (LOWORD (wParam)) {
		case IDOK:
		case IDCANCEL:
		    EndDialog (hDlg, 0);
		    return TRUE;
	    }
	    break;
    }
    return FALSE;
}

//////////////////////////////////////////////////////////////////////////////
//
// Auxiliary functions
//
//////////////////////////////////////////////////////////////////////////////

//
// Get the directory name from the file path
//
void getDirectory()
{
    char* pathend;
    char* pathptr;
    int i;

    pathend = strstr(szFilePath, szFileName);
    i=0;
    for (pathptr=szFilePath; pathptr != pathend; pathptr++)
	szFileDir[i++] = *pathptr;
}

//
// Check the source file and see if it's openable.  Also check to see
// whether it's already gzipped.
//
void checkSourceFile(FILE* src)
{
    char magic[2];
    int c;
    HANDLE hf;

    getDirectory();
    fInputIsGzipped = FALSE;
    if (src == 0) {
	sprintf(szStatus, "Can't open %s", szFilePath);
	SendMessage(hwndStatusBar, SB_SETTEXT, (WPARAM)0,
	    (LPARAM)(LPSTR)szStatus);
	return;
    } else {
	sprintf(szStatus, "%s ", szFileName);

	// Check whether the file is gzipped already
	c = getc(src);
	magic[0] = c;
	c = getc(src);

	// If the file is empty, can't gzip it
	if (c == EOF) {
	    strcat(szStatus, "(empty) in ");
	    strcat(szStatus, szFileDir);
	    SendMessage(hwndStatusBar, SB_SETTEXT, (WPARAM)0,
		(LPARAM)(LPSTR)szStatus);
	    fclose(src);
	    return;
	}

	// Use the magic number to see if the file is gzipped
	magic[1] = c;
	if ((memcmp(magic, GZIP_MAGIC, 2) == 0) ||
	    (memcmp(magic, OLD_GZIP_MAGIC, 2) == 0)) {
	    strcat(szStatus, "(gzipped) in ");
	    fInputIsGzipped = TRUE;
	} else {
	    strcat(szStatus, "(not gzipped) in ");
	}

	strcat(szStatus, szFileDir);
	SendMessage(hwndStatusBar, SB_SETTEXT, (WPARAM)0,
	    (LPARAM)(LPSTR)szStatus);
	fclose(src);

	// Get the file size
	hf = CreateFile(szFilePath, GENERIC_READ, FILE_SHARE_READ,
	    (LPSECURITY_ATTRIBUTES)0, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL,
	    (HANDLE)0);
	dFileSizeLow = GetFileSize(hf, &dFileSizeHigh);
	CloseHandle(hf);

	// Disable the inappropriate button
	if (fInputIsGzipped) {
	    EnableWindow(hwndGzipButton, FALSE);
	    EnableWindow(hwndGunzipButton, TRUE);
	} else {
	    EnableWindow(hwndGzipButton, TRUE);
	    EnableWindow(hwndGunzipButton, FALSE);
	}
	return;
    }
}

//
// gzip the file using the zlib routines
//
void doGzip(HWND hwndDlg)
{
    gzFile out;
    int len;
    int err;
    struct stat ss;
    char buf[BUFLEN];
    int mbrtn;
    unsigned long disp;
    double dReadSoFar;

    dReadSoFar = 0;
    SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)0, 0L);
    EnableWindow(hwndGzipButton, FALSE);

    // Create the output file path by tacking on ".gz" unless the
    // user has asked us to give it the same name as the original.
    // In that case give it a temporary name.
    if (eDoWithOriginal == NAME_AS_ORIGINAL) {
	sprintf(szOutFilePath, "%sgztemp", szFileDir);
    } else {
	sprintf(szOutFilePath, "%s.gz", szFilePath);
    }
    // There are two possible problems:
    //   a) The resulting pathname may be too long
    if (strlen(szOutFilePath) > MAX_PATH) {
	sprintf(szStatus, "Name of file + directories too long\r\nPlease shorten it and try again");
	MessageBox(hwndDlg, szStatus, "Name too long",
	    MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
	SendMessage(hwndStatusBar,
	    SB_SETTEXT, (WPARAM)0, (LPARAM)(LPSTR)"Ready");
	return;
    }

    //   b) The file might already exist
    if ((stat(szOutFilePath, &ss)) == 0) {
	sprintf(szStatus, "%s exists\r\nOK to Overwrite?", szOutFilePath);
	mbrtn = MessageBox(hwndDlg, szStatus, "File Exists",
	    MB_OKCANCEL | MB_ICONQUESTION | MB_DEFBUTTON2 | MB_APPLMODAL);
	if (mbrtn == IDCANCEL) {
	    SendMessage(hwndStatusBar,
		SB_SETTEXT, (WPARAM)0, (LPARAM)(LPSTR)"Ready");
	    SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)0, 0L);
	    return;
	}
    }

    // Now open the files
    if ((src = fopen(szFilePath, "rb")) == NULL) {
	sprintf(szStatus, "Can't open %s", szFilePath);
	MessageBox(hwndDlg, szStatus, "fopen() Error",
	    MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
	SendMessage(hwndStatusBar,
	    SB_SETTEXT, (WPARAM)0, (LPARAM)(LPSTR)"Ready");
	SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)0, 0L);
	return;
    }
    if ((out = gzopen(szOutFilePath, "wb6 ")) == NULL) {
	sprintf(szStatus, "Can't open %s", szOutFilePath);
	MessageBox(hwndDlg, szStatus, "gzopen() Error",
	    MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
	fclose(src);
	SendMessage(hwndStatusBar,
	    SB_SETTEXT, (WPARAM)0, (LPARAM)(LPSTR)"Ready");
	SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)0, 0L);
	return;
    }
    for(;;) {
	len = fread(buf, 1, sizeof(buf), src);
	if (ferror(src)) {
	    sprintf(szStatus, "Error reading %s", szFilePath);
	    MessageBox(hwndDlg, szStatus, "fread() Error",
		MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
	    gzclose(out);
	    fclose(src);
	    SendMessage(hwndStatusBar,
		SB_SETTEXT, (WPARAM)0, (LPARAM)(LPSTR)"Ready");
	    SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)0, 0L);
	    return;
	}
	if (len == 0) break;

	if (gzwrite(out, buf, (unsigned)len) != len) {
	    sprintf(szStatus, gzerror(out, &err));
	    MessageBox(hwndDlg, szStatus, "fread() Error",
		MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
	    fclose(src);
	    SendMessage(hwndStatusBar,
		SB_SETTEXT, (WPARAM)0, (LPARAM)(LPSTR)"Ready");
	    SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)0, 0L);
	    return;
	}
	// Update the status bar and progress bar
	dReadSoFar += len;
	if (dFileSizeLow > 0) {
	    disp = (int)(((double)(100*dReadSoFar))/dFileSizeLow);
	} else {
	    disp = 0;
	}
	SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)disp, 0L);
	sprintf(szStatus, "Gzipping %s %d%%", szFileName, (int)disp);
	SendMessage(hwndStatusBar, 
	    SB_SETTEXT, (WPARAM)0, (LPARAM)(LPSTR)szStatus);
    }
    if (gzclose(out) != Z_OK) {
	sprintf(szStatus, "Close %s failed\r\nCheck if disk full",
	    szFilePath);
	MessageBox(hwndDlg, szStatus, "gzclose() Error",
	    MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
	fclose(src);
	SendMessage(hwndStatusBar,
	    SB_SETTEXT, (WPARAM)0, (LPARAM)(LPSTR)"Ready");
	SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)0, 0L);
	return;
    }
    fclose(src);

    // Now let's take care of cleanup.
    switch (eDoWithOriginal) {
	// If the user wants to keep the original file, we're done.
	case KEEP_ORIGINAL:
	    sprintf(szStatus, "Gzipped file: %s\n", szOutFilePath);
	    break;
	// If the user wants to delete the original, unlink it
	case DELETE_ORIGINAL:
	    if (unlink(szFilePath)) {
		sprintf(szStatus, "Can't delete %s\r\nGzipped file is %s",
		    szFilePath, szOutFilePath);
		MessageBox(hwndDlg, szStatus, "Can't delete original file",
		    MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
		sprintf(szStatus, "Gzipped file: %s\n", szOutFilePath);
	    } else {
		sprintf(szStatus, "Gzipped file: %s\n", szOutFilePath);
	    }
	    break;
	case NAME_AS_ORIGINAL:
	    if (unlink(szFilePath)) {
		sprintf(szStatus, "Can't delete %s\r\nGzipped file is %s",
		    szFilePath, szOutFilePath);
		MessageBox(hwndDlg, szStatus, "Can't delete original file",
		    MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
		sprintf(szStatus, "Gzipped file: %s\n", szOutFilePath);
	    } else {
		if (rename(szOutFilePath, szFilePath)) {
		    sprintf(szStatus,
			"Can't rename file!\r\nGzipped file is %s",
			szOutFilePath);
		    MessageBox(hwndDlg, szStatus, "rename() Error",
			MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
		    sprintf(szStatus, "Gzipped file: %s\n", szOutFilePath);
		} else {
		    sprintf(szStatus, "Gzipped file: %s\n", szFilePath);
		}
	    }
	    break;
    }
    SendMessage(hwndStatusBar, SB_SETTEXT, (WPARAM)0, (LPARAM)(LPSTR)szStatus);
    SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)0, 0L);
}

//
// gunzip the file using the zlib routines
//
void doGunzip(HWND hwndDlg)
{
    gzFile in;
    struct stat ss;
    int mbrtn;
    int len, err;
    int namelen;
    BOOL fTempUsed;
    unsigned long disp;
    double dReadSoFar;
    char buf[BUFLEN];

    dReadSoFar = 0;
    SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)0, 0L);
    EnableWindow(hwndGunzipButton, FALSE);

    fTempUsed=FALSE;
    namelen=strlen(szFileName);
    memset(szOutFilePath, 0, sizeof(szOutFilePath));

    // If the file is named "xxx.gz", output to "xxx", except in the
    // case where the user wants the output file to have the same name
    // as the input file.  In that case, we save to a temp file.
    if (eDoWithOriginal == NAME_AS_ORIGINAL) {
	sprintf(szOutFilePath, "%sgztemp", szFileDir);
    } else {
	if (namelen > 3) {
	    namelen = strlen(szFilePath);
	    if (strnicmp(szFilePath+namelen-3, ".gz", 3) == 0) {
		strncpy(szOutFilePath, szFilePath, namelen-3);
	    }
	}
    }

    // If the file doesn't end in ".gz" we may be in trouble.  Let's see.
    if (strlen(szOutFilePath) == 0) {
	sprintf(szOutFilePath, "%sgztemp", szFileDir);
	switch (eDoWithOriginal) {
	    case KEEP_ORIGINAL:
	    // If we're keeping the original, we have to save it in a
	    // different file name.  Better get the user's OK.
		sprintf(szStatus,
		    "%s doesn't end in '.gz'\r\nShall I gunzip it into gztemp?",
		    szFileName);
		mbrtn = MessageBox(hwndDlg, szStatus,
		    "File doesn't end in '.gz'",
		    MB_OKCANCEL | MB_ICONQUESTION | MB_DEFBUTTON2 |
		    MB_APPLMODAL);
		if (mbrtn == IDCANCEL) {
		    SendMessage(hwndStatusBar,
			SB_SETTEXT, (WPARAM)0, (LPARAM)(LPSTR)"Ready");
		    SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)0, 0L);
		    return;
		}
		break;
	    case DELETE_ORIGINAL:
	    // If we're deleting the original, we can recycle the name.
	    // In this case, the user should probably have checked "Name
	    // as original", but let's not punish him
		fTempUsed = TRUE;
		break;
	    case NAME_AS_ORIGINAL:
	    // If we're renaming the output file as the original, it'll
	    // just fall out when we rename the file.  Don't bother the
	    // user with these trifles.
		break;
	}
    }

    // Check to see if the intended output file exists
    if ((stat(szOutFilePath, &ss)) == 0) {
	sprintf(szStatus, "%s exists\r\nOK to Overwrite?", szOutFilePath);
	mbrtn = MessageBox(hwndDlg, szStatus, "File Exists",
	    MB_OKCANCEL | MB_ICONQUESTION | MB_DEFBUTTON2 | MB_APPLMODAL);
	if (mbrtn == IDCANCEL) {
	    SendMessage(hwndStatusBar,
		SB_SETTEXT, (WPARAM)0, (LPARAM)(LPSTR)"Ready");
	    SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)0, 0L);
	    return;
	}
    }

    // Now open the files
    if ((in = gzopen(szFilePath, "rb")) == NULL) {
	sprintf(szStatus, "Can't open %s", szFilePath);
	MessageBox(hwndDlg, szStatus, "gzopen() Error",
	    MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
	SendMessage(hwndStatusBar,
	    SB_SETTEXT, (WPARAM)0, (LPARAM)(LPSTR)"Ready");
	SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)0, 0L);
	return;
    }
    if ((dest = fopen(szOutFilePath, "wb")) == NULL) {
	sprintf(szStatus, "Can't open %s", szOutFilePath);
	MessageBox(hwndDlg, szStatus, "fopen() Error",
	    MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
	gzclose(in);
	SendMessage(hwndStatusBar,
	    SB_SETTEXT, (WPARAM)0, (LPARAM)(LPSTR)"Ready");
	SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)0, 0L);
	return;
    }

    // Gunzip the file one buffer at a time
    for (;;) {
	len = gzread(in, buf, sizeof(buf));
	if (len < 0) {
	    sprintf(szStatus, gzerror(in, &err));
	    MessageBox(hwndDlg, szStatus, "gzread() Error",
		MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
	    SendMessage(hwndStatusBar,
		SB_SETTEXT, (WPARAM)0, (LPARAM)(LPSTR)"Ready");
	    SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)0, 0L);
	    return;
	}
	if (len == 0) break;

	// Write the buffer
	if ((int)fwrite(buf, 1, (unsigned)len, dest) != len) {
	    sprintf(szStatus, "Write %s failed\r\nCheck if disk full",
		szOutFilePath);
	    MessageBox(hwndDlg, szStatus, "fwrite() Error",
		MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
	    fclose(dest);
	    gzclose(in);
	    SendMessage(hwndStatusBar,
		SB_SETTEXT, (WPARAM)0, (LPARAM)(LPSTR)"Ready");
	    SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)0, 0L);
	    return;
	}

	// Update the progress bar and status bar
	dReadSoFar += len;
	if (dFileSizeLow > 0) {
	    disp = (int)(((double)(100*dReadSoFar))/dFileSizeLow);
	} else {
	    disp = 0;
	}
	SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)disp, 0L);
	sprintf(szStatus, "Gunzipping %s %d%%", szFileName, (int)disp);
	SendMessage(hwndStatusBar, 
	    SB_SETTEXT, (WPARAM)0, (LPARAM)(LPSTR)szStatus);
    }

    // Be anal and report close errors
    if (fclose(dest)) {
	sprintf(szStatus, "Close %s failed\r\nCheck if disk full",
	    szOutFilePath);
	MessageBox(hwndDlg, szStatus, "fclose() Error",
	    MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
	gzclose(in);
	SendMessage(hwndStatusBar,
	    SB_SETTEXT, (WPARAM)0, (LPARAM)(LPSTR)"Ready");
	SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)0, 0L);
	return;
    }

    if (gzclose(in) != Z_OK) {
	sprintf(szStatus, "Close %s failed\r\nCheck if disk full",
	    szFilePath);
	MessageBox(hwndDlg, szStatus, "gzclose() Error",
	    MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
	SendMessage(hwndStatusBar,
	    SB_SETTEXT, (WPARAM)0, (LPARAM)(LPSTR)"Ready");
	SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)0, 0L);
	return;
    }

    // Now let's take care of cleanup.
    switch (eDoWithOriginal) {
	// If the user wants to keep the original file, we're done.
	case KEEP_ORIGINAL:
	    sprintf(szStatus, "Gunzipped file: %s\n", szOutFilePath);
	    break;
	// If the user wants to delete the original, unlink it
	case DELETE_ORIGINAL:
	    if (unlink(szFilePath)) {
		sprintf(szStatus, "Can't delete %s\r\nGunzipped file is %s",
		    szFilePath, szOutFilePath);
		MessageBox(hwndDlg, szStatus, "Can't delete original file",
		    MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
		sprintf(szStatus, "Gunzipped file: %s\n", szOutFilePath);
	    }
	    // If we used a temporary filename because the filename didn't
	    // end in ".gz", we have to rename the unzipped file
	    if (fTempUsed) {
		if (rename(szOutFilePath, szFilePath)) {
		    sprintf(szStatus,
			"Can't rename file!\r\nGunzipped file is %s",
			szOutFilePath);
		    MessageBox(hwndDlg, szStatus, "rename() Error",
			MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
		    sprintf(szStatus, "Gunzipped file: %s\n", szOutFilePath);
		} else {
		    sprintf(szStatus, "Gunzipped file: %s\n", szFilePath);
		}
	    } else {
		sprintf(szStatus, "Gunzipped file: %s\n", szFilePath);
	    }
	    break;
	case NAME_AS_ORIGINAL:
	    if (unlink(szFilePath)) {
		sprintf(szStatus, "Can't delete %s\r\nGunzipped file is %s",
		    szFilePath, szOutFilePath);
		MessageBox(hwndDlg, szStatus, "Can't delete original file",
		    MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
		sprintf(szStatus, "Gunzipped file: %s\n", szOutFilePath);
	    } else {
		if (rename(szOutFilePath, szFilePath)) {
		    sprintf(szStatus,
			"Can't rename file!\r\nGunzipped file is %s",
			szOutFilePath);
		    MessageBox(hwndDlg, szStatus, "rename() Error",
			MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
		    sprintf(szStatus, "Gunzipped file: %s\n", szOutFilePath);
		} else {
		    sprintf(szStatus, "Gunzipped file: %s\n", szFilePath);
		}
	    }
	    break;
    }
    SendMessage(hwndStatusBar, SB_SETTEXT, (WPARAM)0, (LPARAM)(LPSTR)szStatus);
    SendMessage(hwndProgressBar, PBM_SETPOS, (WPARAM)0, 0L);
}
