// ------------------------------- //
// -------- Start of File -------- //
// ------------------------------- //
// ----------------------------------------------------------- // 
// C++ Source Code File Name: sreg101.cpp 
// Compiler Used: MSVC, BCC32, GCC, HPUX aCC, SOLARIS CC
// Produced By: glNET Software
// File Creation Date: 10/13/1999 
// Date Last Modified: 06/27/2001
// Copyright (c) 2001 glNET Software
// ----------------------------------------------------------- // 
// ------------- Program Description and Details ------------- // 
// ----------------------------------------------------------- // 
/*
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.
 
This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  
USA

Code used to generate software registration numbers using A-C-E
level one encryption and a magic number sequence.
*/
// ----------------------------------------------------------- // 
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include "sreg101.h"

srnGenerator::srnGenerator()
{

}

srnGenerator::srnGenerator(const UString &pname, const UString &user_name)
{
  GenRegString(pname, user_name);
} 

srnGenerator::srnGenerator(char *pname, char *user_name)
{
  GenRegString(pname, user_name);
} 

srnGenerator::srnGenerator(const char *pname, const char *user_name)
{ 
  GenRegString(pname, user_name);
}

srnGenerator::~srnGenerator()
{

}

srnGenerator::srnGenerator(const srnGenerator &ob)
{
  srnCopy(ob);
}

srnGenerator srnGenerator::operator=(const srnGenerator &ob)
{
  srnCopy(ob);
  return *this;
}

srnIntType srnGenerator::GetMagicNumber()
{
  return magic_number;
}

char *srnGenerator::GetRegString()
{
  return (char *)reg_string;
}

const char *srnGenerator::GetRegString() const
{
  return (const char *)reg_string;
}

char *srnGenerator::GetRegCode()
{
  return reg_code.c_str();
}

const char *srnGenerator::GetRegCode() const
{
  return reg_code.c_str();
}

void srnGenerator::srnCopy(const srnGenerator &ob)
{
  magic_number = ob.magic_number;
  reg_code = ob.reg_code;
  int i;
  for(i = 0; i < srnMaxNameChars; i++) reg_string[i] = ob.reg_string[i];
  for(i = 0; i < srnTableSize; i++) srn_table[i] = ob.srn_table[i];
}


void srnGenerator::srnInit()
// Set initial values for the srnGenerator.
{
  srnTableInit();
  ClearRegString();
}

void srnGenerator::ClearRegString()
{
  for(int i = 0; i < srnMaxNameChars; i++) reg_string[i] = '\0';
}

void srnGenerator::srnTableInit()
// Initalize the table used to generate magic numbers. Table
// values are calculated by a byte-wise 32-bit operation based
// on a specifed polynomial. NOTE: In this representation the
// coefficient of x^0 is stored in the MSB of the 32-bit word
// and the coefficient of x^31 is stored in the LSB. 
{
  int i,n;
  srnIntType val32;
  for (i = 0; i < srnTableSize; i++) {
    val32=i;
    for (n = 1; n < 9; n++) {
      if (val32 & 1)
	val32 = (val32 >> 1) ^ srnPolynomial;
      else
	val32 = val32 >> 1;
    }
    srn_table[i] = val32;
  }
}

void srnGenerator::GenRegString(const char *pname, const char *user_name)
{
  srnInit(); // Reset all values
  unsigned len = strlen(pname);
  unsigned char sbuf[srnMaxNameChars];

  // Ensure the maximum number of characters are used per name
  if(len >= srnMaxNameChars) {
    memmove(sbuf, pname, srnMaxNameChars);
    len = srnMaxNameChars;
  }
  else {
    memmove(sbuf, pname, len);
    while(len < srnMaxNameChars) {
      sbuf[len] = sbuf[0]+len; // Fill in the missing characters
      len++;
    }
  }

  unsigned i;
  int offset = 15;
  unsigned random_offset = (unsigned)sbuf[strlen(pname)]/8;

  // Generate a magic number based on the user name
  magic_number = GenMagicNumber(pname, user_name);
  
  // Encrypt once using random and non-random offsets
  for(i = 0; i < len; i++) {
    sbuf[i] = sbuf[i]+random_offset; 
    ACE1encrypt(sbuf[i], offset++);
  }

  // Encrypt again without offsets to remove any non-alphanumerics
  for(i = 0; i < len; i++) {
    ACE1encrypt(sbuf[i]);
    reg_string[i] = sbuf[i];  
  }

  reg_string[srnMaxNameChars] = 0; // Ensure null termination

  // Generate a composite string that includes the Registration
  // String and the Magic Number Sequence
  GenRegCode(); 
}

void srnGenerator::GenRegString(char *pname, char *user_name)
{
  GenRegString((const char *)pname, (const char *)user_name);
}

void srnGenerator::GenRegString(const UString &pname, const UString &user_name)

{
  GenRegString(pname.c_str(), user_name.c_str());
}

srnIntType srnGenerator::GenMagicNumber(const UString &pname,
					const UString &user_name)
// Generate a magic number based on the program's name. This
// function is called by the srnGenerator::GenRegString() but
// can be used by the application to validate magic numbers.
{
  UString sbuf(pname); sbuf += user_name;
  srnIntType val32 = 0xffffffffL;
  unsigned n = sbuf.length();
  char *p = sbuf.c_str();
  
  while(n--) {
    val32 = srn_table[(val32 ^ (*p++)) & 0xFF] ^ ((val32>>8) & 0x00ffffffL);
  }
  return val32 ^ 0xffffffffL;
}

srnIntType srnGenerator::GenMagicNumber(const char *pname,
					   const char *user_name)
{
  UString pn(pname); UString un(user_name);
  return GenMagicNumber(pn, un);
}

srnIntType srnGenerator::GenMagicNumber(char *pname, char *user_name)
{
  UString pn(pname); UString un(user_name);
  return GenMagicNumber(pn, un);
}

int srnGenerator::Validate(const UString &regCode, const UString &pname,
			   const UString &user_name)
// Validate a software registration string and magic number. Returns
// true if the software registration string matches the program name,
// user name, and the magic number sequence.
{
  UString rs(regCode); UString mn(regCode);
  char sep[1]; sep[0] = srnSegmentSeparator; sep[1] = 0;
  unsigned offset = rs.Find((char *)sep);
  if(offset == -1) return 0;
  rs.DeleteAt(offset, (rs.length() - offset));
  mn.DeleteAt(0, (offset + 1));
  GenRegString(pname, user_name);
  unsigned long m_number;
  sscanf(mn.c_str(), "%u", &m_number);
  if(strcmp(reg_string, rs.c_str()) != 0) return 0;
  if(m_number != magic_number) return 0;
  return 1;
}

int srnGenerator::Validate(const char *regCode, const char *pname,
			   const char *user_name)
{
  UString rc(regCode); UString pn(pname); UString un(user_name);
  return Validate(rc, pn, un);
}
  
int srnGenerator::Validate(char *regCode, char *pname, char *user_name)
{
  UString rc(regCode); UString pn(pname); UString un(user_name);
  return Validate(rc, pn, un);
}

void srnGenerator::GenRegCode()
// Generate a registration code that contains the
// registration string, segment separator, and magic
// number sequence. This function is called by the
// srnGenerator::GenRegString() function.
{
  char mn_str[255];
  reg_code.DeleteAt(0, reg_code.length());
  reg_code += GetRegString();
  reg_code += srnSegmentSeparator;
  sprintf(mn_str, "%u", (unsigned long)GetMagicNumber());
  reg_code += mn_str;
}

// ---------------------------------------------------------------------
// A-C-E encryption routines.
// ---------------------------------------------------------------------
int srnGenerator::ACE1encrypt(unsigned char &c, int offset)
// ---------------------------------------------------------------------
// A-C-E level one alphanumeric encryption.
// ---------------------------------------------------------------------
// A-C-E alphanumeric encryption routine rules: (1) All non-printable chars
// will be treated as letter 'g' or disallowed depending on the application.
// (2) All letters will be evaluated according to case during the
// encryption process. (3) All even numbers will be converted to odd
// numbers: 0=9, 2=7, 4=5, 6=3, 8=1. (4) Odd numbers (except 5) will be 
// converted to letters: 1='o', 3='I', 7='q', 9=W'. (4) Every other letter
// from the beginning of the alphabet will be converted to even numbered
// letters starting from 'Y' to 'O': B=Y, b=k , D=W, d=i, F=U, f=g , H=S,
// h=e, J=Q, j=c, L=O, l=a, and from 'A' to 'K': Z=A, z=o , X=C, x=q , V=E,
// v=s , T=G, t=u ,R=I, r=w, P=K, p=y. (6) The letter 'N' will be converted
// to the number 7. (7) The number 5 will be converted to letter 'G'. (8)
// All omitted letters will retain the case of the letters they have been
// substituted with. (9) The letters A, C, E, G, I, K, M, O, Q, S, U, W,
// and Y will be retain their original case in reverse order: A=Y, C=W, E=U,
// G=S, I=Q, K=O, M=7, O=K, Q=I, S=G, U=E, W=C, and Y=A. (10) The letters a,
// c, e, g, i, k, m, o, q, s, u, w, and y will be converted to odd numbers:
// a=9, c=7, e=3, g=1, i=5, k=5, m=5, o=5, q=5, s=9, u=7, w=3, and y=1. (11)
// All final values will be offset by a random value according to the number
// of times it occurs in the string. 
{
  switch(c) {
    case 'a':
      c = '9' + offset; 
      break;
    case 'A':
      c = 'Y' + offset; 
      break;
    case 'B':
      c = 'Y' + offset; 
      break;
    case 'b':
      c = 'k' + offset; 
      break;
    case 'c':
      c = '7' + offset; 
      break;
    case 'C':
      c = 'W' + offset; 
      break;
    case 'D':
      c = 'W' + offset; 
      break;
    case 'd':
      c = 'i' + offset; 
      break;
    case 'e':
      c = '3' + offset;
      break;
    case 'E':
      c = 'U' + offset;
      break;
    case 'F': 
      c = 'U' + offset;
      break;
    case 'f': 
      c = 'g' + offset;
      break;
    case 'g':
      c = '1' + offset;
      break;
    case 'G':
      c = 'S' + offset;
      break;
    case 'H':
      c = 'S' + offset;
      break;
    case 'h':
      c = 'e' + offset;
      break;
    case 'i':
      c = '5' + offset;
      break;
    case 'I':
      c = 'Q' + offset;
      break;
    case 'J':
      c = 'Q' + offset;
      break;
    case 'j':
      c = 'c' + offset;
      break;
    case 'k':
      c = '5' + offset;
      break;
    case 'K':
      c = 'O' + offset;
      break;
    case 'L':
      c = 'O' + offset;
      break;
    case 'l':
      c = 'a' + offset;
      break;
    case 'm':
      c = '5' + offset;
      break;
    case 'M':
      c = '7' + offset;
      break;
    case 'n':
      c = '7' + offset;
      break;
    case 'N':
      c = '7' + offset;
      break;
    case 'o':
      c = '5' + offset;
      break;
    case 'O':
      c = 'K' + offset;
      break;
    case 'P':
      c = 'k' + offset;
      break;
    case 'p':
      c = 'y' + offset;
      break;
    case 'q':
      c = '5' + offset;
      break;
    case 'Q':
      c = 'I' + offset;
      break;
    case 'R':
      c = 'I' + offset;
      break;
    case 'r':
      c = 'w' + offset;
      break;
    case 's':
      c = '9' + offset;
      break;
    case 'S':
      c = 'G' + offset;
      break;
    case 'T':
      c = 'G' + offset;
      break;
    case 't':
      c = 'u' + offset;
      break;
    case 'u':
      c = '7' + offset;
      break;
    case 'U':
      c = 'E' + offset;
      break;
    case 'V':
      c = 'E' + offset;
      break;
    case 'v':
      c = 's' + offset;
      break;
    case 'w':
      c = '3' + offset;
      break;
    case 'W':
      c = 'C' + offset;
      break;
    case 'X': 
      c = 'C' + offset;
      break;
    case 'x': 
      c = 'q' + offset;
      break;
    case 'y':
      c = '1' + offset;
      break;
    case 'Y':
      c = 'A' + offset;
      break;
    case 'Z':
      c = 'A' + offset;
      break;
    case 'z':
      c = 'o' + offset;
      break;
    case '1':
      c = 'o' + offset;
      break;
    case '3':
      c = 'I' + offset;
      break;
    case '7':
      c = 'q' + offset;
      break;
    case '9':
      c ='W' + offset;
      break;
    case '5':
      c = 'G' + offset;
      break;
    case '0':
      c = '9' + offset;
      break;
    case '2':
      c = '7' + offset;
      break;
    case '4':
      c = '5' + offset;
      break;
    case '6':
      c = '3' + offset;
      break;
    case '8':
      c = '1' + offset;
      break;
    default:
      break;
  }

  if(!isalnum(c)) return ACE1Aencrypt(c, offset);
  return 1;
}

int srnGenerator::ACE1Aencrypt(unsigned char &c, int offset)
// Special rules for non-alphanumerics: (1) All non-alphanumeric
// characters will be converted to even numbers or ever other letter
// of the alphabet depending on its keyboard location: !=Z, @=x, #=U,
// $=t, %=R, ^=p, &=B, *=d, (=F, )=h, -=J, _=l, ==N, +=0, [=8, {=4, ]=2,
// }=z, \=X, |=u, ;=T, :=r, '=P, "=b, <=D, ,=f, >=H, .=j, `=L, ~=n, /=0,
// ?=8. 
{
  int is_printable = 1; // Return true if 'c' is printable
  switch(c) {
    case '!':
      c = 'Z' + offset;
      break;
    case '@':
      c = 'x' + offset;
      break;
    case '#':
      c = 'U' + offset;
      break;
    case '$':
      c = 't' + offset;
      break;
    case '%':
      c = 'R' + offset;
      break;
    case '^':
      c = 'p' + offset;
      break;
    case '&':
      c = 'B' + offset;
      break;
    case '*':
      c = 'd' + offset;
      break;
    case '(':
      c = 'F' + offset;
      break;
    case ')':
      c = 'h' + offset;
      break;
    case '-':
      c = 'J' + offset;
      break;
    case '_':
      c = 'l' + offset;
      break;
    case '=':
      c = 'N' + offset;
      break;
    case '+':
      c = '0' + offset;
      break;
    case '[':
      c = '8' + offset;
      break;
    case '{':
      c = '4' + offset;
      break;
    case ']':
      c = '2' + offset;
      break;
    case '}':
      c = 'z' + offset;
      break;
    case '\\':
      c = 'X' + offset;
      break;
    case '|':
      c = 'u' + offset;
      break;
    case ';':
      c = 'T' + offset;
      break;
    case ':':
      c = 'r' + offset;
      break;
    case '\'':
      c = 'P' + offset;
      break;
    case '\"':
      c = 'b' + offset;
      break;
    case '<':
      c = 'D' + offset;
      break;
    case ',':
      c = 'f' + offset;
      break;
    case '>':
      c = 'H' + offset;
      break;
    case '.':
      c = 'j' + offset;
      break;
    case '`':
      c = 'L' + offset;
      break;
    case '~':
      c = 'n' + offset;
      break;
    case '/':
      c = '0' + offset;
      break;
    case '?':
      c = '8' + offset;
      break;
    default:
      c = 'g' + offset;
      is_printable = 0;
      break;
  }
  return is_printable;
}
// ----------------------------------------------------------- //
// ------------------------------- //
// --------- End of File --------- //
// ------------------------------- //
