{$INCLUDE cHeader.inc}
unit cMaths;

interface

uses
  // System units
  Math,
  SysUtils,
  Classes,

  // Simple types (L0)
  cStream;              // ATStream



{                                                                              }
{ Maths unit v0.23 (L0)                                                        }
{                                                                              }
{   A collection of base mathematical functions.                               }
{                                                                              }
{                                                                              }
{ This unit is copyrighted  1995-1999 by David Butler (david@e.co.za)         }
{                                                                              }
{ I invite you to use this unit, free of charge.                               }
{ I invite you to distibute this unit, but it must be for free.                }
{ I also invite you to contribute to its development, but do not distribute    }
{ a modified copy of this file, send modifications, suggestions and bug        }
{ reports to david@e.co.za                                                     }
{                                                                              }
{                                                                              }
{ Revision history:                                                            }
{   1995-96     v0.01  Wrote statistical and actuarial functions.              }
{   1998/03     v0.02  Added Solver and SecantSolver.                          }
{   1998/10     v0.03  Removed functions now found in Delphi 3's Math unit     }
{                      Uses Delphi's math exceptions (eg EOverflow,            }
{                      EInvalidArgument, etc)                                  }
{   1999/08/29  v0.04  Added ASet, TRangeSet, TFlatSet, TSparseFlatSet.        }
{   1999/09/27  v0.05  Added TMatrix, TVector.                                 }
{                      Added BinomialCoeff.                                    }
{   1999/10/02  v0.06  Added TComplex.                                         }
{   1999/10/03  v0.07  Added DerivateSolvers.                                  }
{                      Completed TMatrix.                                      }
{   1999/10/04  v0.08  Added TLifeTable.                                       }
{   1999/10/05  v0.09  T3DPoint                                                }
{   1999/10/06  v0.10  Transform matrices.                                     }
{   1999/10/13  v0.11  TRay, TSphere                                           }
{   1999/10/14  v0.12  TPlane                                                  }
{   1999/10/26  v0.13  Upgraded to Delphi 4. Compared the assembly code of the }
{                      new dynamic arrays with that of pointers to arrays.     }
{                      Its basically the same. Converted all PRealList type    }
{                      references to RealArray type.                           }
{   1999/10/30  v0.14  Added TVector.StdDev                                    }
{                      Changed some functions to the same name (since Delphi   }
{                      now supports overloading).                              }
{                      Removed Min and Max functions (now in Math).            }
{   1999/11/04  v0.15  Added TVector.Pos, TVector.Append                       }
{                      855 lines interface, 3071 lines implementation.         }
{   1999/11/07  v0.16  Added RandomSeed function.                              }
{                      Added assembly bit functions.                           }
{   1999/11/10  v0.17  Added hashing functions. Coded XOR8 in assembly.        }
{                      Added MD5 hashing.                                      }
{   1999/11/11  v0.18  Added EncodeBase.                                       }
{   1999/11/21  v0.19  Added TComplex.Power                                    }
{   1999/11/25  v0.20  Moved TRay, TSphere and TPlane to cRayTrace.            }
{                      Added Primes.                                           }
{   1999/11/26  v0.21  Added Rational numbers (can convert to/from TReal).     }
{                      Added GCD.                                              }
{                      Added RealArray/IntegerArray functions.                 }
{   1999/11/27  v0.22  Replaced GCD algorithm with a faster one.               }
{                      Added SI constants.                                     }
{                      Added TMatrix.Normalise, TMatrix.Multiply (Row, Value)  }
{   1999/12/01  v0.23  Added RandomUniform.                                    }
{                                                                              }
{                                                                              }
{                                                                              }
{                                                                              }
{ To-do's:                                                                     }
{   * Create a TSet class that dynamically maintains an list of TRangeSets and }
{     TSparseFlatSets.                                                         }
{   * Random number generators from probability density functions.             }
{   * Finish HugeInt functions. Add HugeFloat functions.                       }
{   * CRC16/32 in assembly.                                                    }
{   * TMAC (RFC2104) to be applied to MD5 to get variable key (currently       }
{     fixed).                                                                  }
{   * Add user/machine/other random states to RandomSeed.                      }
{   * Rectangular -> Polar                                                     }
{   * Integer vector                                                           }
{   * Regression functions                                                     }
{                                                                              }
{                                                                              }
{                                                                              }



{                                                                              }
{ Mathematical constants                                                       }
{                                                                              }
const
  Pi        = 3.14159265358979323846 +           { Pi                          }
              0.26433832795028841971e-20 +       {                             }
              0.69399375105820974944e-40 +       {                             }
              0.59230781640628620899e-60 +       {                             }
              0.86280348253421170679e-80 +       {                             }
              0.82148086513282306647e-100 +      {                             }
              0.09384460955058223172e-120 +      {                             }
              0.53594081284811174502e-140 +      {                             }
              0.84102701938521105559e-160 +      {                             }
              0.64462294895493038196e-180;       {                             }
  PiSq      = 9.869604401089358618834491;        { Pi^2                        }
  PiE       = 22.45915771836104547342715;        { Pi^e                        }
  E         = 2.718281828459045235360287;        { e                           }
  E2        = 7.389056098930650227230472;        { e^2                         }
  ExpM2     = 0.13533528323661269189;            { e^-2                        }
  Ln10      = 2.3025850929940456840179915;       { Ln (10)                     }
  Ln2       = 0.6931471805599453094172321;       { Ln (2)                      }
  LnPi      = 1.144729885849400174143427;        { Ln (Pi)                     }
  LnSqrt2Pi = 0.91893853320467274178;            { Ln (Sqrt (2 * Pi))          }
  LogE      = 0.4342944819032518276511289;       { Log (e)                     }
  Log2      = 0.3010299956639811952137389;       { Log (2)                     }
  LogPi     = 0.4971498726941338543512683;       { Log (Pi)                    }
  Sqrt2     = 1.4142135623730950488;             { Sqrt (2)                    }
  Sqrt10    = 3.1622776601683793320;             { Sqrt (10)                   }
  SqrtPi    = 1.772453850905516027298167;        { Sqrt (Pi)                   }
  Sqrt2Pi   = 2.506628274631000502415765;        { Sqrt (2 * Pi)               }
  OneDegree = 0.017453292519943295769237;        { 1 degree in rad, Pi / 180   }
  OneRad    = 57.295779513082320876798155;       { 1 rad in degrees, 180 / Pi  }



{                                                                              }
{ Real / Integer                                                               }
{   Two fundemental number types are defined here, reals (TReal) and           }
{     integers (TInteger).                                                     }
{   Real comparisons are done using RealEqual and RealZero which returns True  }
{     if the numbers are within RealCompareTolerance.                          }
{                                                                              }
type
  TReal = Extended;
  {   Type       Range	                Significant digits  Size in bytes      }
  {   Single     1.5e45 .. 3.4e38	78	            4                  }
  {   Double     5.0e324 .. 1.7e308	1516	            8                  }
  {   Extended   3.6e4951 .. 1.1e4932	1920	            10                 }
  TInteger = Integer;
  {   Type       Range	                   Format                              }
  {   Shortint	 128..127	           signed 8-bit                        }
  {   Smallint	 32768..32767	           signed 16-bit                       }
  {   LongInt    2147483648..2147483647   signed 32-bit = Integer (CPU type)  }
  {   Int64	 2^63..2^631	           signed 64-bit                       }

const
  MaxReal    = MaxExtended;
  MaxInteger = MaxInt;

var
  RealCompareTolerance : TReal = 1e-14;

Function  RealEqual (const X, Y : TReal) : Boolean;
Function  RealZero (const X : TReal) : Boolean;



{                                                                              }
{ RealArray / IntegerArray                                                     }
{   PosNext finds the next occurance of Find in V, -1 if it was not found.     }
{     Searches from item [PrevPos + 1], ie PrevPos = -1 to find first          }
{     occurance.                                                               }
{   Intersection (V1, V2) returns elements in both V1 and V2.                  }
{                                                                              }
type
  IntegerArray = Array of TInteger;
  RealArray = Array of TReal;

Function  PosNext (const Find : TReal; const V : RealArray; const PrevPos : Integer = -1;
          const IsSortedAscending : Boolean = False) : Integer; overload;
Function  PosNext (const Find : TInteger; const V : IntegerArray; const PrevPos : Integer = -1;
          const IsSortedAscending : Boolean = False) : Integer; overload;
Procedure Concat (var V : RealArray; const R : TReal); overload;
Procedure Concat (var V : IntegerArray; const R : TInteger); overload;
Function  Intersection (const V1, V2 : RealArray; const IsSortedAscending : Boolean = False) : RealArray; overload;
Function  Intersection (const V1, V2 : IntegerArray; const IsSortedAscending : Boolean = False) : IntegerArray; overload;
Procedure Delete (var V : RealArray; const Idx, Count : Integer); overload;
Procedure Delete (var V : IntegerArray; const Idx, Count : Integer); overload;



{                                                                              }
{ Miscellaneous functions                                                      }
{   RealToStr returns R as a string in the shortest non-scientific form.       }
{   Cond returns TrueValue if Expr=True else returns FalseValue.               }
{   Sgn returns sign of argument, +1 or -1.                                    }
{                                                                              }
Function  RealToStr (const R : TReal) : String;

Procedure Swap (var X, Y : TInteger); overload;
Procedure Swap (var X, Y : TReal); overload;

Function  Cond (const Expr : Boolean; const TrueValue, FalseValue : TInteger) : TInteger; overload;
Function  Cond (const Expr : Boolean; const TrueValue, FalseValue : TReal) : TReal; overload;
Function  Cond (const Expr : Boolean; const TrueValue, FalseValue : String) : String; overload;

Function  Sgn (const R : TReal) : TReal; overload;
Function  Sgn (const I : TInteger) : TInteger; overload;



{                                                                              }
{ SI conversion constants                                                      }
{                                                                              }
const
  // Distance
  Meter_per_Inch    = 0.0254;
  Meter_per_Foot    = 0.3048;
  Meter_per_Yard    = 0.9144;
  Meter_per_Rod     = 5.029;    // Also refered to as a Pole or a Perch
  Meter_per_Furlong = 201.168;
  Meter_per_Mile    = 1609.4;
  Meter_per_League  = 4830;



{                                                                              }
{ Trigonometric functions                                                      }
{   Delphi's Math unit includes most commonly used trigonometric functions.    }
{                                                                              }
Function  ATan360 (const X, Y : TReal) : TReal;
Function  InverseTangentDeg (const X, Y : TReal) : TReal;
Function  InverseTangentRad (const X, Y : TReal) : TReal;
Function  InverseSinDeg (const Y, R : TReal) : TReal;
Function  InverseSinRad (const Y, R : TReal) : TReal;
Function  InverseCosDeg (const X, R : TReal) : TReal;
Function  InverseCosRad (const X, R : TReal) : TReal;
Function  DMSToReal (const Degs, Mins, Secs : TReal) : TReal;
Procedure RealToDMS (const X : TReal; var Degs, Mins, Secs : TReal);
Function  Distance (const X1, Y1, X2, Y2 : TReal) : TReal;
Procedure PolarToRectangular (const R, Theta : TReal; var X, Y : TReal);
Procedure RectangularToPolar (const X, Y : TReal; var R, Theta : TReal);



{                                                                              }
{ Primes                                                                       }
{   IsPrime returns True if N is a prime number.                               }
{   IsPrimeFactor returns True if F is a prime factor of N.                    }
{   PrimeFactors returns an array of prime factors of N, sorted in acending    }
{   order.                                                                     }
{   GCD returns the Greatest Common Divisor of N1 and N2.                      }
{                                                                              }
Function  IsPrime (const N : TInteger) : Boolean;
Function  IsPrimeFactor (const F, N : TInteger) : Boolean;
Function  PrimeFactors (const N : TInteger) : IntegerArray;
Function  GCD (const N1, N2 : TInteger) : TInteger;



{                                                                              }
{ Rational numbers                                                             }
{   Class that represents a rational number (Numerator / Denominator pair)     } 
{                                                                              }
type
  TRational = class
    private
    FT, FN : TInteger;

    protected
    Procedure Simplify;

    public
    Constructor Create; overload;
    Constructor Create (const Numerator : Integer; const Denominator : TInteger = 1); overload;
    Constructor Create (const R : TReal); overload;

    Property Numerator : TInteger read FT;
    Property Denominator : TInteger read FN;

    Function  GetAsString : String;
    Procedure SetAsString (const S : String);
    Property  AsString : String read GetAsString write SetAsString;

    Function  GetAsReal : TReal;
    Procedure SetAsReal (const R : TReal);
    Property  AsReal : TReal read GetAsReal write SetAsReal;

    Procedure Assign (const R : TRational); overload;
    Procedure Assign (const R : TReal); overload;
    Procedure Assign (const Numerator : TInteger; const Denominator : TInteger = 1); overload;
    Procedure AssignZero;
    Procedure AssignOne;

    Function  Duplicate : TRational;

    Function  IsEqual (const R : TRational) : Boolean; reintroduce; overload;
    Function  IsEqual (const Numerator : TInteger; const Denominator : TInteger = 1) : Boolean; reintroduce; overload;
    Function  IsEqual (const R : TReal) : Boolean; reintroduce; overload;
    Function  IsZero : Boolean;
    Function  IsOne : Boolean;

    Procedure Add (const R : TRational); overload;
    Procedure Add (const V : TReal); overload;
    Procedure Add (const V : TInteger); overload;

    Procedure Subtract (const R : TRational); overload;
    Procedure Subtract (const V : TReal); overload;
    Procedure Subtract (const V : TInteger); overload;

    Procedure Negate;
    Procedure Abs;
    Function  Sgn : TInteger;

    Procedure Multiply (const R : TRational); overload;
    Procedure Multiply (Const V : TReal); overload;
    Procedure Multiply (const V : TInteger); overload;

    Procedure Reciprocal;
    Procedure Divide (const R : TRational); overload;
    Procedure Divide (const V : TReal); overload;
    Procedure Divide (const V : TInteger); overload;

    Procedure Sqrt;
    Procedure Sqr;
    Procedure Power (const R : TRational); overload;
    Procedure Power (const V : TInteger); overload;
    Procedure Power (const V : TReal); overload;
  end;



{                                                                              }
{ Complex numbers                                                              }
{   Class that represents a complex number (Real + i * Imag)                   }
{                                                                              }
type
  EComplex = class (Exception);
  TComplex = class
    private
    FReal,
    FImag  : TReal;

    public
    Constructor Create (const TheRealPart : TReal = 0.0; const TheImaginaryPart : TReal = 0.0);

    Property  RealPart : TReal read FReal write FReal;
    Property  ImaginaryPart : TReal read FImag write FImag;

    Function  GetAsString : String;
    Procedure SetAsString (const S : String);
    Property  AsString : String read GetAsString write SetAsString;

    Procedure Assign (const C : TComplex); overload;
    Procedure Assign (const V : TReal); overload;
    Procedure AssignZero;
    Procedure AssignI;                                                          // i^2 = -1

    Function  Duplicate : TComplex;

    Function  IsEqual (const C : TComplex) : Boolean; reintroduce; overload;
    Function  IsEqual (const R, I : TReal) : Boolean; reintroduce; overload;
    Function  IsZero : Boolean;
    Function  IsI : Boolean;

    Procedure Add (const C : TComplex); overload;
    Procedure Add (const V : TReal); overload;
    Procedure Subtract (const C : TComplex); overload;
    Procedure Subtract (const V : TReal); overload;
    Procedure Multiply (const C : TComplex); overload;
    Procedure Multiply (Const V : TReal); overload;
    Procedure MultiplyI;
    Procedure MultiplyMinI;
    Procedure Divide (const C : TComplex); overload;
    Procedure Divide (const V : TReal); overload;
    Procedure Negate;

    Function  Modulo : TReal;
    Function  Denom : TReal;
    Procedure Conjugate;
    Procedure Inverse;

    Procedure Sqrt;
    Procedure Exp;
    Procedure Ln;
    Procedure Sin;
    Procedure Cos;
    Procedure Tan;
    Procedure Power (const C : TComplex);
  end;



{                                                                              }
{ Vector                                                                       }
{   General purpose vector (dynamic array wrapper) class.                      }
{                                                                              }
type
  EVector = class (Exception);
  TVector = class
  private
    protected
    FData : RealArray;

    public
    Constructor Create (const Values : array of TReal); overload;
    Constructor Create (const Values : RealArray); overload;
    Constructor Create (const Values : IntegerArray); overload;

    Function  GetCount : Integer;
    Procedure SetCount (const NewCount : Integer);
    Property  Count : Integer read GetCount write SetCount;

    Procedure SetItem (const Idx : Integer; const Value : TReal);
    Function  GetItem (const Idx : Integer) : TReal;
    Property  Item [const Idx : Integer] : TReal read GetItem write SetItem; default;

    Function  GetAsString : String;
    Procedure SetAsString (const S : String);
    Property  AsString : String read GetAsString write SetAsString;

    Procedure Assign (const V : TVector); overload;
    Procedure Assign (const V : TReal); overload;
    Procedure Assign (const V : array of TReal); overload;
    Procedure Assign (const V : RealArray); overload;
    Procedure Assign (const V : IntegerArray); overload;
    Procedure Assign (const V, Increment : TReal; const Count : Integer); overload;  // [V, V+Inc, V+2*Inc, ... ]

    Procedure Append (const V : TVector); overload;
    Procedure Append (const V : TReal); overload;

    procedure Delete(const Idx : Integer; const Count : Integer = 1);

    Function  Duplicate : TVector; overload;
    Function  Duplicate (const LoIdx, HiIdx : Integer) : TVector; overload;

    Function  IsEqual (const V : TVector) : Boolean; reintroduce; overload;
    Function  IsZero : Boolean;

    Function  Intersection (const V : TVector; const IsSortedAscending : Boolean = False): TVector;
    { If both vectors are sorted ascending set IsSortedAscending True for time }
    { o(n) instead of o(n^2).                                                  }

    Function  Pos (const V : TReal; const PrevPos : Integer = -1; const IsSortedAscending : Boolean = False) : Integer;
    Procedure InvertOrder;

    Procedure Sort;

    { Mathematical functions                                                   }
    Procedure Add (const V : TVector); overload;
    Procedure Add (const V : TVector; const Factor : TReal); overload;
    Procedure Add (const Value : TReal); overload;
    Procedure Multiply (const V : TVector); overload;
    Procedure Multiply (const Value : TReal); overload;
    Function  DotProduct (const V : TVector) : TReal;
    Function  Norm : TReal;
    Procedure Invert;
    Procedure SquareValues;
    Function  Angle (const V : TVector) : TReal;                                // UV=|U||V|Cos

    { Statistical functions                                                    }
    Function  Sum : TReal; overload;
    Function  Sum (const LoIdx, HiIdx : Integer) : TReal; overload;
    Function  MaxValue : TReal;
    Function  MinValue : TReal;
    Function  Mean : TReal;
    Function  HarmonicMean : TReal;
    Function  GeometricMean : TReal;
    Procedure Normalize;                                                        // Divide each element with Sum
    Function  Median : TReal;
    Function  Variance : TReal;
    Function  StdDev (var Mean : TReal) : TReal;                                // Sample StdDev
    Function  SumOfSquares : TReal;
    Procedure SumAndSquares (var Sum, SumOfSquares : TReal);
    Function  TotalVariance : TReal;
    Function  PopulationVariance : TReal;
    Function  PopulationStdDev : TReal;                                         // Population StdDev
  end;



{                                                                              }
{ Matrix                                                                       }
{                                                                              }
type
  EMatrix = class (Exception);
  TMatrix = class
    private
    FColCount : Integer;
    FRows     : Array of RealArray;

    Function  GetRow (const Row : Integer) : TVector;
    // Returns a *reference* to a row in the matrix as a TVector. Caller must free.

    protected
    Function  GetAsString : String; virtual;

    public
    Constructor CreateSquare (const N : Integer);
    Constructor CreateUnity (const N : Integer);
    Constructor CreateDiagonal (const D : TVector);                             // D in diagonal

    Procedure SetSize (const Rows, Cols : Integer);

    Function  GetRowCount : Integer;
    Procedure SetRowCount (const NewRowCount : Integer);
    Procedure SetColCount (const NewColCount : Integer);
    Property  ColCount : Integer read FColCount write SetColCount;
    Property  RowCount : Integer read GetRowCount write SetRowCount;

    Procedure SetItem (const Row, Col : Integer; const Value : TReal);
    Function  GetItem (const Row, Col : Integer) : TReal;
    Property  Item [const Row, Col : Integer] : TReal read GetItem write SetItem; default;

    Property  AsString : String read GetAsString;

    Procedure Assign (const M : TMatrix); overload;
    Procedure Assign (const V : TVector); overload;
    Procedure Assign (const Value : TReal); overload;
    Procedure AssignRow (const Row : Integer; const V : TVector);
    Procedure AssignCol (const Col : Integer; const V : TVector);
    Procedure AssignRowValues (const Row : Integer; const Values : Array of TReal);

    Function  Duplicate : TMatrix; overload;
    Function  Duplicate (const R1, C1, R2, C2 : Integer) : TMatrix; overload;
    Function  DuplicateRow (const Row : Integer) : TVector;
    Function  DuplicateCol (const Col : Integer) : TVector;
    Function  DuplicateDiagonal : TVector;

    { Mathematical functions                                                   }
    Function  IsEqual (const M : TMatrix) : Boolean; overload;
    Function  IsEqual (const V : TVector) : Boolean; overload;
    Function  IsUnity : Boolean;
    Function  IsZero : Boolean;

    Function  Transposed : TMatrix;
    Procedure Transpose;                                                        // swap rows/cols
    Procedure Add (const M : TMatrix);
    Procedure AddRows (const I, J : Integer; const Factor : TReal);             // inc (row i, row j * Factor)
    Procedure SwapRows (const I, J : Integer);
    Procedure Multiply (const Value : TReal); overload;                         // multiply matrix with constant value
    Procedure Multiply (const Row : Integer; const Value : TReal); overload;    // multiply row with constant value
    Procedure Multiply (const M : TMatrix); overload;
    Function  Multiplied (const M : TMatrix) : TMatrix;

    Function  Normalise (const M : TMatrix = nil) : TReal;                      // make diagonal 1's by multiplying each row with a factor. also applies operations to M (if specified)

    Function  Trace : TReal;                                                    // sum of diagonal
    Function  IsSquare : Boolean;
    Function  IsOrtogonal : Boolean;                                            // X'X = I
    Function  IsIdempotent : Boolean;                                           // XX = X
    Function  SolveMatrix (var M : TMatrix) : TReal;                            // Returns determinant
    Function  Determinant : TReal;
    Procedure Inverse;
    Function  SolveLinearSystem (const V : TVector) : TVector;
  end;



{                                                                              }
{ T3DPoint                                                                     }
{  Stores a (x,y,z)-value which can represent a point or a vector in 3D        }
{    space.                                                                    }
{  Internally it inherits from TVector so all the vector operations are        }
{    available.                                                                }
{  A point is represented as [x,y,z,1] and a vector as [x,y,z,0]. The 4th      }
{    element is needed when multiplying with transformation matrices (see      }
{    "3D Transformation matrices") to preserve scale.                          }
{                                                                              }
type
  E3DPoint = class (Exception);
  T3DPoint = class (TVector)
    private
    Function GetX : TReal;
    Function GetY : TReal;
    Function GetZ : TReal;

    Procedure SetX (const NewX : TReal);
    Procedure SetY (const NewY : TReal);
    Procedure SetZ (const NewZ : TReal);

    public
    Constructor CreatePoint (const X, Y, Z : TReal);
    Constructor CreateVector (const X, Y, Z : TReal);

    Function  Duplicate : T3DPoint;

    { Transformations                                                          }
    Procedure RotateX (const Angle : TReal);
    Procedure RotateY (const Angle : TReal);
    Procedure RotateZ (const Angle : TReal);
    Procedure RotateXYZ (const XAngle, YAngle, ZAngle : TReal);
    Procedure RotateVector (const NX, NY, NZ, Angle : TReal);

    Procedure Scale (const XScale, YScale, ZScale : TReal);
    Procedure Origin (const XOrigin, YOrigin, ZOrigin : TReal);
    Procedure CrossProduct (const P : T3DPoint);
    Procedure Homogenize;

    { Parallel projections                                                     }
    { Angle typically 30 or 45                                                 }
    Procedure CavalierProject (const Angle : TReal; var X, Y : TReal);          // (x,y)=(x+z*cos(Angle),y+z*sin(Angle))
    Procedure CabinetProject (const Angle : TReal; var X, Y : TReal);           // (x,y)=(x+z/2*cos(Angle),y+z/2*sin(Angle))

    { Perspective projections                                                  }
    { Zv = distance from origin of z-axis vanishing point                      }
    { Xv = distance from origin of x-axis vanishing point                      }
    Procedure OnePointPerspectiveProject (const Angle, Zv : TReal; var X, Y : TReal);
    Procedure TwoPointPerspectiveProject (const Angle, Xv, Zv : TReal; var X, Y : TReal);

    Property X : TReal read GetX write SetX;
    Property Y : TReal read GetY write SetY;
    Property Z : TReal read GetZ write SetZ;
  end;



{                                                                              }
{ 3D Transformation matrices                                                   }
{   Multiply with a T3DPoint to transform. Transform matrices can also be      }
{   multiplied with each other before being applied to a T3DPoint.             }
{   All are 4x4 matrices.                                                      }
{                                                                              }
Function OriginAndScaleTransform (const TX, TY, TZ, SX, SY, SZ : TReal) : TMatrix;
{ Translates origin with (TX, TY, TZ) and scale by (SX, SY, SZ)                }
Function XRotateTransform (const Angle : TReal) : TMatrix;
Function YRotateTransform (const Angle : TReal) : TMatrix;
Function ZRotateTransform (const Angle : TReal) : TMatrix;
Function XYZRotateTransform (const XAngle, YAngle, ZAngle : TReal) : TMatrix;
{ Rotate around x, y and z-axis                                                }



{                                                                              }
{ Combinatoric functions                                                       }
{                                                                              }
Function  Factorial (const N : TInteger) : TReal;
Function  Combinations (const N, C : TInteger) : TReal;
Function  Permutations (const N, P : TInteger) : TReal;


{                                                                              }
{ Statistical functions                                                        }
{                                                                              }
Function  GammLn (X : TReal) : TReal;
{ Returns the natural logarithm of the gamma function with paramater X         }
Function  BinomialCoeff (N, R : TInteger) : TReal;
{ Returns the binomial coeff for the bin (n) distribution                      }

Function  RandomSeed : LongWord;
{ Returns a random seed value, based on the Windows counter, the CPU counter   }
{ and the current date/time.                                                   }
Function  RandomUniform : LongWord;
{ Returns a random number that is uniformly distributed. See implementation    }
{ for specifics.                                                               }
Function  RandomUniformF : TReal;
{ Returns a uniformly distributed real number between 0 and 1.                 }

{ ========================================== Cumulative distribution functions }
Function  CummNormal (const u, s, X : TReal) : TReal;
{ CumNormal returns the area under the N(u,s) distribution.                    }
Function  CummNormal01 (const X : TReal) : TReal;
{ CumNormal01 returns the area under the N(0,1) distribution.                  }
Function  CummChiSquare (const Chi, Df : TReal) : TReal;
{ CumChiSquare returns the area under the X^2 (Chi-squared) (Chi, Df)          }
{ distribution.                                                                }
Function  CumF (const f, Df1, Df2 : TReal) : TReal;
{ CumF returns the area under the F (f, Df1, Df2) distribution.                }
Function  CummPoisson (const X : TInteger; const u : TReal) : TReal;
{ CummPoison returns the area under the Poi(u)-distribution.                   }

{ =============================== Inverse of cumulative distribution functions }
Function  InvCummNormal01 (Y0 : TReal) : TReal;
{     InvCummNormal01 returns position on X-axis that gives cummulative area   }
{       of Y0 under the N(0,1) distribution.                                   }
Function  InvCummNormal (const u, s, Y0 : TReal) : TReal;
{     InvCummNormal returns position on X-axis that gives cummulative area     }
{       of Y0 under the N(u,s) distribution.                                   }




{                                                                              }
{ Computer maths                                                               }
{                                                                              }
Function SwapBits (const Value : LongWord) : LongWord;
{ Reverse the bit order of a LongWord.                                         }
Function LSBit (const Value : Integer): Integer;
{ Returns bit number (0-31) of least significant bit set.                      }
Function MSBit (const Value : Integer): Integer;
{ Returns bit number (0-31) of most significant bit set.                       }
Function SwapEndian (const Value : LongWord) : LongWord;
{ Swap between little and big endian formats.                                  }

Function DecodeBase (const S : String; const Base : Byte) : TInteger;
{ Converts string S of Base to an integer.                                     }
{ Uses an alphabeth of up to 36 characters (0-9A-Z)                            }
Function EncodeBase (const I : TInteger; const Base : Byte) : String;
{ Converts number I to Base.                                                   }
{ Uses an alphabeth of up to 36 characters (0-9A-Z)                            }

const
  b64_MIME     = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/';
  b64_UUEncode = ' !"#$%&''()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\]^_';

Function DecodeBase64 (const S, Alphabet : String) : String;
{ Converts a base 64 string using Alphabet (64 characters for values 0-63) to  }
{ a binary string.                                                             }



{                                                                              }
{ Hashing functions                                                            }
{   The CRC16 function is the CCITT V.41 polynomial hashing function used for  }
{     calculating CRCs in communications. "This polynomial traps 100% of       }
{     1 bit, 2 bit, odd numbers of bit errors, 100% of <= 16 bit burst errors  }
{     and over 99% of all other errors." - See implementation for more detail. }
{   MD5 is an Internet standard authenticity hashing function, that was        }
{     produced by RSA Data Security, Inc. and placed in the public domain.     }
{     It returns a 128-bit (4 LongWords) digest. IMAC (another Internet        }
{     standard) needs to be implemented for variable keys.                     }
{   XOR and Checksum is general purpose hashing functions.                     }
{   See the implementation notes for speed comparisons.                        }
{                                                                              }
{   Hash is the general purpose hashing function for hash tables. It returns   }
{   a value in the range 0..Slots-1. It's currently implemented using CRC32.   }
{                                                                              }
type
  LongWordArray = Array of LongWord;

Function CalcCCITT_CRC16 (const Data : String) : Word; overload;
Function CalcCCITT_CRC16 (const Octet : Byte; const CRC16 : Word) : Word; overload;
Function CalcCRC32 (const Octet : Byte; const CRC32 : LongWord) : LongWord; overload;
Function CalcCRC32 (const Data : String) : LongWord; overload;
Function CalcChecksum32 (const Data : String) : LongWord;
Function CalcXOR8 (const Data : String) : Byte;                                 // Blazingly fast, +/- 1.2 clock ticks per character on P166MMX
Function CalcXOR16 (const Data : String) : Word;
Function CalcXOR32 (const Data : String) : LongWord;
Function CalcMD5 (const Data : AStream) : LongWordArray; overload;
Function CalcMD5 (const Data : String) : LongWordArray; overload;

Function Hash (const S : String; const Slots : TInteger = MaxInteger) : TInteger;


{                                                                              }
{ Actuarial functions                                                          }
{                                                                              }
Function ForceAsI (d : TReal) : TReal;
{ Force of intrest d as an intrest rate, i                                     }
Function DiscountAsI (d : TReal) : TReal;
{ Discount rate, d, as intrest rate, i                                         }
Function VAsI (v : TReal) : TReal;
{ Discounted value of i                                                        }
Function IAsDiscount (i : TReal) : TReal;
{ Interest rate, i, as a discount rate, d                                      }
Function IAsForce (i : TReal) : TReal;
{ Interest rate, i, as a force of Interest                                     }
Function IAsV (i : TReal) : TReal;
{ Interest rate, i, discounted                                                 }
{ ie Present value of future payment of 1 at end of period 1                   }
Function v (i, n : TReal) : TReal;
{ Interest rate, i, discounted for n periods                                   }
{ ie Present value of future payment of 1 at end of period n                   }
Function an (i, n : TReal) : TReal;
{ Present value of an annuity payable in arrear for n periods discounted with  }
{ a nominal yearly interest rate of i, an                                      }
Function aDOTn (i, n : TReal) : TReal;
{ a"n                                                                          }
Function sn (i, n : TReal) : TReal;
{ Future value of annuity payable in arrear, sn                                }
Function sDOTn (i, n : TReal) : TReal;
{ s"n                                                                          }
Function aCONTn (i, n : TReal) : TReal;
{ Present value of annuity paid continiously at force of d                     }
Function Ian (i, n : TReal) : TReal;
{ Present value of an increasing annuity payable in arrear, Ian                }
Function IaDOTn (i, n : TReal) : TReal;
{ Ia"n                                                                         }
Function IaCONTn (i, n : TReal) : TReal;
{ Present value of an increasing annuity, paid continuously                    }
Function ICONTaCONTn (i, n : TReal) : TReal;
{ Continuously increasing annuity, paid continuously                           }
Function ip (i, p : TReal) : TReal;
{ i(p), interest rate per period for p periods                                 }
Function dp (i, p : TReal) : TReal;
{ d(p)                                                                         }
Function apn (i, p, n : TReal) : TReal;
{ Present value of an annuity payable in arrear p times per period for         }
{ n periods discounted with a nominal yearly interest rate of i, a(p)n         }
Function aDOTpn (i, p, n : TReal) : TReal;
{ Present value of an annuity payable in advance p times per period for        }
{ n periods discounted with a nominal yearly interest rate of i, a"(p)n        }
Function spn (i, p, n : TReal) : TReal;
{ Future value of an annuity payable in arrear p times per period for          }
{ n periods accumulated at a nominal yearly interest rate of i, s(p)n          }
Function sDOTpn (i, p, n : TReal) : TReal;
{ Future value of an annuity payable in advance p times per period for         }
{ n periods accumulated at a nominal yearly interest rate of i, s"(p)n         }



{                                                                              }
{ TLifeTable                                                                   }
{                                                                              }
type
  TLifeTable = class (TVector)
    Function l (const x : Integer) : TReal;
    Function d (const x, n : Integer) : TReal;
    { = lx - l(x+n)                                                            }

    Function p (const x, n : Integer) : TReal;
    { m|nPx  Probability of survival for n periods for individual aged x       }
    Function q (const x, n, m : Integer) : TReal;
    { m|nQx  Probibility of mortality for n periods for individual aged x with }
    {        death deferred for m periods.                                     }
    {        = (l(x+m) - l(x+m+n)) / lx                                        }
    {        = mPx nQ(x+m)                                                     }

    Function Dx (const i : TReal; const x : Integer) : TReal;
    { Dx = v^x * lx                                                            }
    Function Nx (const i : TReal; const x : Integer) : TReal;
    { Nx = Sum (Di)  i = x..infinity                                           }
    Function Sx (const i : TReal; const x : Integer) : TReal;
    { Sx = Sum (Ni)  i = x..infinity                                           }
    Function Cx (const i : TReal; const x : Integer) : TReal;
    { Cx = v^(x+1) * dx                                                        }
    Function Mx (const i : TReal; const x : Integer) : TReal;
    { Mx = Sum (Ci)  i = x..infinity                                           }
    Function Rx (const i : TReal; const x : Integer) : TReal;
    { Rx = Sum (Mi)  i = x..infinity                                           }
  end;

const
  { A1967-70 Select Life table }
  A1967T70S : Array [0..80] of TReal = (
    34481.408, 34456.927, 34433.841, 34412.836, 34393.221, 34375.681, 34359.181,
    34344.063, 34329.638, 34315.907, 34303.210, 34290.518, 34277.830, 34264.461,
    34250.070, 34232.259, 34209.439, 34179.680, 34143.368, 34109.166, 34076.957,
    34046.610, 34017.983, 33990.921, 33965.254, 33940.795, 33917.341, 33894.668,
    33872.531, 33650.662, 33827.764, 33806.514, 33783.557, 33759.503, 33733.924,
    33706.352, 33676.272, 33643.122, 33606.286, 33565.089, 33518.794, 33466.599,
    33407.624, 33340.915, 33265.431, 33180.042, 33083.523, 32974.549, 32851.686,
    32713.392, 32558.008, 32383.756, 32188.740, 31970.942, 31728.226, 31458.342,
    31158.931, 30827.543, 30461.645, 30058.648, 29615.936, 29130.898, 28600.975,
    28023.708, 27396.808, 26718.225, 25986.236, 25199.536, 24357.348, 23459.538,
    22506.732, 21500.445, 20443.198, 19338.635, 18191.617, 17008.294, 15796.140,
    14563.940, 13321.717, 12080.592, 10852.568);



{                                                                              }
{ Numerical solvers                                                            }
{                                                                              }
type
  fx = Function (const x : TReal) : TReal;

Function SecantSolver (const f : fx; const y, Guess1, Guess2 : TReal) : TReal;
{ Uses Secant method to solve for x in f(x) = y                                }

Function NewtonSolver (const f, df : fx; const y, Guess : TReal) : TReal;
{ Uses Newton's method to solve for x in f(x) = y.                             }
{ df = f'(x).                                                                  }
{ Note: This implementation does not check if the solver goes on a tangent     }
{       (which can happen with certain Guess values)                           }

Function FirstDerivative (const f : fx; const x : TReal) : TReal;
{ Returns the value of f'(x)                                                   }
{ Uses (-f(x+2h) + 8f(x+h) - 8f(x-h) + f(x-2h)) / 12h                          }

Function SecondDerivative (const f : fx; const x : TReal) : TReal;
{ Returns the value of f''(x)                                                  }
{ Uses (-f(x+2h) + 16f(x+h) - 30f(x) + 16f(x-h) - f(x-2h)) / 12h^2             }

Function ThirdDerivative (const f : fx; const x : TReal) : TReal;
{ Returns the value of f'''(x)                                                 }
{ Uses (f(x+2h) - 2f(x+h) + 2f(x-h) - f(x-2*h)) / 2h^3                         }

Function FourthDerivative (const f : fx; const x : TReal) : TReal;
{ Returns the value of f''''(x)                                                }
{ Uses (f(x+2h) - 4f(x+h) + 6f(x) - 4f(x-h) + f(x-2h)) / h^4                   }

Function SimpsonIntegration (const f : fx; const a, b : TReal; N : TInteger) : TReal;
{ Returns the area under f from a to b, by applying Simpson's 1/3 Rule with    }
{ N subdivisions.                                                              }



{                                                                              }
{ Sets                                                                         }
{                                                                              }

{                                                                              }
{ ASet                                                                         }
{   Abstract base class for set implementations.                               }
{                                                                              }
type
  ASet = class
    Procedure Clear; virtual; abstract;
    Procedure Invert; virtual; abstract;

    Function GetBit (const Idx : Integer) : Boolean; virtual; abstract;
    Function GetRange (const Low, High : Integer; const Value : Boolean) : Boolean; virtual; abstract;
    Procedure SetBit (const Idx : Integer; const Value : Boolean); virtual; abstract;
    Procedure SetRange (const Low, High : Integer; const Value : Boolean); virtual; abstract;
  end;



{                                                                              }
{ TRange                                                                       }
{   Store Low..High value                                                      }
{                                                                              }
type
  ERange = class (Exception);
  TRange = class
    protected
    FLow, FHigh : Integer;

    public
    Constructor Create (const Low, High : Integer);

    Function HasElement (const I : Integer) : Boolean;
    { True if I in Low..High                                                   }
    Function Overlap (const Low, High : Integer) : Boolean;
    { True if overlapping ranges                                               }
    Function Touch (const Low, High : Integer) : Boolean;
    { True if touching ranges                                                  }
    Function Inside (const Low, High : Integer) : Boolean;
    { True if inside range                                                     }
    Procedure Merge (const Low, High : Integer);
    { Merge adjacent/overlapping ranges.                                       }
    { Raises exception if ranges are not adjacent/overlapping                  }
  end;



{                                                                              }
{ TRangeSet                                                                    }
{   A set (stored as a list of ranges)                                         }
{   Pros: Good memory usage if set bits occur sparsely in ranges               }
{   Cons: Bad memory usage if bits are randomly distributed                    }
{                                                                              }
type
  TRangeSet = class (ASet)
    FRanges : TList;

    Constructor Create;
    Destructor Destroy; override;
    Procedure Clear; override;
    Procedure Invert; override;
    Procedure SetRange (const Low, High : Integer; const Value : Boolean); override;
    Function GetBit (const Idx : Integer) : Boolean; override;
    Function GetRange (const Low, High : Integer; const Value : Boolean) : Boolean; override;
    Procedure SetBit (const Idx : Integer; const Value : Boolean); override;
  end;



{                                                                              }
{ TFlatSet                                                                     }
{   A flat set (stored as a flat piece of memory, Delphi's TBits)              }
{   Pros: Fast allocation/deallocation.                                        }
{         1 bit of memory for each entry.                                      }
{   Cons: Bad memory usage if bits are randomly distributed                    }
{                                                                              }
type
  TFlatSet = class (ASet)
    FBits : TBits;

    Constructor Create;
    Destructor Destroy; override;
    Procedure Clear; override;
    Procedure Invert; override;
    Procedure SetRange (const Low, High : Integer; const Value : Boolean); override;
    Function GetBit (const Idx : Integer) : Boolean; override;
    Function GetRange (const Low, High : Integer; const Value : Boolean) : Boolean; override;
    Procedure SetBit (const Idx : Integer; const Value : Boolean); override;
  end;



{                                                                              }
{ TSparseFlatSet                                                               }
{   A flat set (stored as a array of pointers to Delphi sets of Byte)          }
{   Pros: Good performance for randomly distributed bits, especially if they   }
{   appear in little clumps.                                                   }
{   Cons: Bad performance for ranges.                                          }
{                                                                              }
type
  TPointerArray = Array [0..MaxLongInt div 256] of Pointer;
  PPointerArray = ^TPointerArray;
  TDelphiSet = Set of Byte; // 256 elements
  PDelphiSet = ^TDelphiSet;
const
  EmptyDelphiSet : TDelphiSet = [];
  CompleteDelphiSet : TDelphiSet = [0..255];
type
  TSparseFlatSet = class (ASet)
    FSetList        : PPointerArray;
    FSetListEntries : Integer;

    Destructor Destroy; override;
    Procedure Clear; override;
    Procedure Invert; override;
    Function GetBit (const Idx : Integer) : Boolean; override;
    Procedure SetBit (const Idx : Integer; const Value : Boolean); override;
    Procedure SetRange (const Low, High : Integer; const Value : Boolean); override;
  end;



implementation

uses
  // System units
  Windows,              // GetTickCount

  // Simple types (L0)
  cStrings;



{                                                                              }
{ Miscellaneous functions                                                      }
{                                                                              }
Function RealToStr (const R : TReal) : String;
  Begin
    Result := TrimRight (TrimRight (Format ('%18.18f', [R]), ['0']), ['.']);
  End;

Function RealEqual (const X, Y : TReal) : Boolean;
  Begin
    Result := Abs (X - Y) <= RealCompareTolerance;
  End;

Function RealZero (const X : TReal) : Boolean;
  Begin
    Result := Abs (X) <= RealCompareTolerance;
  End;

Function Cond (const Expr : Boolean; const TrueValue, FalseValue : TInteger) : TInteger;
  Begin
    if Expr then
      Result := TrueValue else
      Result := FalseValue;
  End;

Function Cond (const Expr : Boolean; const TrueValue, FalseValue : TReal) : TReal;
  Begin
    if Expr then
      Result := TrueValue else
      Result := FalseValue;
  End;

Function Cond (const Expr : Boolean; const TrueValue, FalseValue : String) : String;
  Begin
    if Expr then
      Result := TrueValue else
      Result := FalseValue;
  End;

Function Sgn (const R : TReal) : TReal;
  Begin
    if R < 0.0 then
      Result := -1.0 else
      Result := 1.0;
  End;

Function Sgn (const I : TInteger) : TInteger;
  Begin
    if I < 0 then
      Result := -1 else
      Result := 1;
  End;

Procedure Swap (var X, Y : TInteger);
var F : TInteger;
  Begin
    F := X;
    X := Y;
    Y := F;
  End;

Procedure Swap (var X, Y : TReal);
var F : TReal;
  Begin
    F := X;
    X := Y;
    Y := F;
  End;

Function Distance (const X1, Y1, X2, Y2 : TReal) : TReal;
var DX, DY : TReal;
  Begin
    DX := X1 - X2;
    DY := Y1 - Y2;
    Result := Sqrt (DX * DX + DY * DY);
  End;

Procedure RealToDMS (const X : TReal; var Degs, Mins, Secs : TReal);
var Y : TReal;
  Begin
    Degs := Int (X);
    Y := Frac (X) * 60.0;
    Mins := Int (Y);
    Secs := Frac (Y) * 60.0;
  End;

Function DMSToReal (const Degs, Mins, Secs : TReal) : TReal;
  Begin
    Result := Degs + Mins / 60.0 + Secs / 3600.0;
  End;

Function CanonicalForm (const Theta : TReal) : TReal;                           {-PI < theta <= PI}
  Begin
    if Abs (Theta) > Pi then
       Result := Round (Theta / (Pi * 2)) * 2 * Pi;
  End;

Procedure PolarToRectangular (const R, Theta : TReal; var X, Y : TReal);
var S, C : Extended;
  Begin
    SinCos (Theta, S, C);
    X := R * C;
    Y := R * S;
  End;

Procedure RectangularToPolar (const X, Y : TReal; var R, Theta : TReal);
  Begin
    if RealZero (X) then
      if RealZero (Y) then
        R := 0.0 else
        if Y > 0.0 then
          R := Y else
          R := -Y else
      R := Sqrt (Sqr (X) + Sqr (Y));
    Theta := ArcTan2 (Y, X);
  End;

Function PosNext (const Find : TReal; const V : RealArray; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : TReal;
  Begin
    if IsSortedAscending then
      begin
        if Max (PrevPos + 1, 0) = 0 then // find first, binary search
          begin
            L := 0;
            H := Length (V) - 1;
            Repeat
              I := (L + H) div 2;
              D := V [I];
              if RealEqual (D, Find) then
                begin
                  While (I > 0) and RealEqual (V [I - 1], Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Length (V) - 1 then
            Result := -1 else
            if RealEqual (V [PrevPos + 1], Find) then
              Result := PrevPos + 1 else
              Result := -1;
      end else
      begin // linear search
        For I := Max (PrevPos + 1, 0) to Length (V) - 1 do
          if RealEqual (V [I], Find) then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;

Function PosNext (const Find : TInteger; const V : IntegerArray; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : TReal;
  Begin
    if IsSortedAscending then
      begin
        if Max (PrevPos + 1, 0) = 0 then // find first, binary search
          begin
            L := 0;
            H := Length (V) - 1;
            Repeat
              I := (L + H) div 2;
              D := V [I];
              if D = Find then
                begin
                  While (I > 0) and (V [I - 1] = Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Length (V) - 1 then
            Result := -1 else
            if V [PrevPos + 1] = Find then
              Result := PrevPos + 1 else
              Result := -1;
      end else // linear search
      begin
        For I := Max (PrevPos + 1, 0) to Length (V) - 1 do
          if V [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;

Procedure Concat (var V : RealArray; const R : TReal);
var L : Integer;
  Begin
    L := Length (V);
    SetLength (V, L + 1);
    V [L] := R;
  End;

Procedure Concat (var V : IntegerArray; const R : TInteger);
var L : Integer;
  Begin
    L := Length (V);
    SetLength (V, L + 1);
    V [L] := R;
  End;

Function Intersection (const V1, V2 : RealArray; const IsSortedAscending : Boolean) : RealArray;
var I, J, L, LV : Integer;
  Begin
    SetLength (Result, 0);
    if IsSortedAscending then
      begin
        I := 0;
        J := 0;
        L := Length (V1);
        LV := Length (V2);
        While (I < L) and (J < LV) do
          begin
            While (I < L) and (V1 [I] < V2 [J]) do
              Inc (I);
            if I < L then
              begin
                if RealEqual (V1 [I], V2 [J]) then
                  Concat (Result, V1 [I]);
                While (J < LV) and (RealEqual (V2 [J], V1 [I]) or (V2 [J] < V1 [I])) do
                  Inc (J);
              end;
          end;
      end else
      For I := 0 to Length (V1) - 1 do
        if (PosNext (V1 [I], V2) >= 0) and (PosNext (V1 [I], Result) = -1) then
          Concat (Result, V1 [I]);
  End;

Function Intersection (const V1, V2 : IntegerArray; const IsSortedAscending : Boolean) : IntegerArray;
var I, J, L, LV : Integer;
  Begin
    SetLength (Result, 0);
    if IsSortedAscending then
      begin
        I := 0;
        J := 0;
        L := Length (V1);
        LV := Length (V2);
        While (I < L) and (J < LV) do
          begin
            While (I < L) and (V1 [I] < V2 [J]) do
              Inc (I);
            if I < L then
              begin
                if V1 [I] = V2 [J] then
                  Concat (Result, V1 [I]);
                While (J < LV) and (V2 [J] <= V1 [I]) do
                  Inc (J);
              end;
          end;
      end else
      For I := 0 to Length (V1) - 1 do
        if (PosNext (V1 [I], V2) >= 0) and (PosNext (V1 [I], Result) = -1) then
          Concat (Result, V1 [I]);
  End;

Procedure Delete (var V : RealArray; const Idx, Count : Integer);
var I, J, L : Integer;
  Begin
    L := Length (V);
    if (Idx >= L) or (Idx + Count <= 0) or (L = 0) or (Count = 0) then
      exit;
    I := Max (Idx, 0);
    J := Min (Count, L - I);
    Move (V [I + J], V [I], J * Sizeof (TReal));
    SetLength (V, L - J);
  End;

Procedure Delete (var V : IntegerArray; const Idx, Count : Integer);
var I, J, L : Integer;
  Begin
    L := Length (V);
    if (Idx >= L) or (Idx + Count <= 0) or (L = 0) or (Count = 0) then
      exit;
    I := Max (Idx, 0);
    J := Min (Count, L - I);
    Move (V [I + J], V [I], J * Sizeof (TInteger));
    SetLength (V, L - J);
  End;


{                                                                              }
{ TRational                                                                    }
{                                                                              }
Constructor TRational.Create (const Numerator, Denominator : TInteger);
  Begin
    inherited Create;
    Assign (Numerator, Denominator);
  End;

Constructor TRational.Create;
  Begin
    inherited Create;
    AssignZero;
  End;

Constructor TRational.Create (const R : TReal);
  Begin
    inherited Create;
    Assign (R);
  End;

Procedure TRational.Simplify;
var I : TInteger;
  Begin
    if FN < 0 then
      begin
        FT := -FT;
        FN := -FN;
      end;
    if (FT = 1) or (FN = 1) or (FT = 0) then
      exit;

    I := GCD (FT, FN);
    FT := FT div I;
    FN := FN div I;
  End;

Procedure TRational.Assign (const Numerator, Denominator : TInteger);
  Begin
    if Denominator = 0 then
      raise EDivByZero.Create ('Invalid rational number');

    FT := Numerator;
    FN := Denominator;
    if FN <> 1 then
      Simplify;
  End;

Procedure TRational.Assign (const R : TRational);
  Begin
    FT := R.FT;
    FN := R.FN;
  End;


{ See http://forum.swarthmore.edu/dr.math/faq/faq.fractions.html for an        }
{ explanation on how to convert decimal numbers to fractions.                  }
Procedure TRational.Assign (const R : TReal);

  // Pre: Abs (R) < 1.0
  Function CalcFrac (const R : TReal; const Level : Integer = 1) : TRational;
  var I : TReal;
      Z : TInteger;
    Begin
      if RealZero (R) or (Level = 12) then  // 0 (if Level = 12 we get an approximation)
        Result := TRational.Create else
      if RealEqual (R, 1.0) then            // 1
        begin
          Result := TRational.Create;
          Result.AssignOne;
        end else
      if RealZero (Frac (R * 1e8)) then     // terminating decimal (<8)
        Result := TRational.Create (Trunc (R * 1e8), 100000000) else
        begin                               // recursive process
          I := 1.0 / R;
          Result := CalcFrac (Frac (I), Level + 1);
          Z := Trunc (I);
          Result.Add (Z);
          Result.Reciprocal;
        end;
    End;

var T : TRational;
    Z : TInteger;

  Begin
    T := CalcFrac (Frac (R));
    try
      Z := Trunc (R);
      T.Add (Z);
      Assign (T);
    finally
      T.Free;
    end;
  End;

Procedure TRational.AssignOne;
  Begin
    FT := 1;
    FN := 1;
  End;

Procedure TRational.AssignZero;
  Begin
    FT := 0;
    FN := 1;
  End;

Function TRational.IsEqual (const Numerator, Denominator : TInteger) : Boolean;
var R : TRational;
  Begin
    R := TRational.Create (Numerator, Denominator);
    Result := IsEqual (R);
    R.Free;
  End;

Function TRational.IsEqual (const R : TRational) : Boolean;
  Begin
    Result := (FT = R.FT) and (FN = R.FN);
  End;

Function TRational.IsEqual (const R : TReal) : Boolean;
  Begin
    Result := RealEqual (R, GetAsReal);
  End;

Function TRational.IsOne : Boolean;
  Begin
    Result := (FT = 1) and (FN = 1);
  End;

Function TRational.IsZero : Boolean;
  Begin
    Result := FT = 0;
  End;

Function TRational.Duplicate : TRational;
  Begin
    Result := TRational.Create (FT, FN);
  End;

Procedure TRational.SetAsReal (const R : TReal);
  Begin
    Assign (R);
  End;

Procedure TRational.SetAsString (const S : String);
var F : Integer;
  Begin
    F := Pos ('/', S);
    if F = 0 then
      Assign (StrToFloat (S)) else
      Assign (StrToInt (CopyLeft (S, F - 1)), StrToInt (CopyFrom (S, F + 1)));
  End;

Function TRational.GetAsReal : TReal;
  Begin
    Result := FT / FN;
  End;

Function TRational.GetAsString : String;
  Begin
    Result := IntToStr (FT) + '/' + IntToStr (FN);
  End;

Procedure TRational.Add (const R : TRational);
  Begin
    FT := FT * R.FN + R.FT;
    FN := FN * R.FN;
    Simplify;
  End;

Procedure TRational.Add (const V : TInteger);
  Begin
    Inc (FT, FN * V);
  End;

Procedure TRational.Add (const V : TReal);
  Begin
    Assign (GetAsReal + V);
  End;

Procedure TRational.Subtract (const V : TReal);
  Begin
    Assign (GetAsReal - V);
  End;

Procedure TRational.Subtract (const R : TRational);
  Begin
    FT := FT * R.FN - R.FT;
    FN := FN * R.FN;
    Simplify;
  End;

Procedure TRational.Subtract (const V : TInteger);
  Begin
    Dec (FT, FN * V);
  End;

Procedure TRational.Negate;
  Begin
    FT := -FT;
  End;

Procedure TRational.Abs;
  Begin
    FT := System.Abs (FT);
    FN := System.Abs (FN);
  End;

Function TRational.Sgn : TInteger;
  Begin
    if cMaths.Sgn (FT) = cMaths.Sgn (FN) then
      Sgn := 1 else
      Sgn := -1;
  End;

Procedure TRational.Divide (const V : TInteger);
  Begin
    if V = 0 then
      raise EDivByZero.Create ('Division by zero');

    FN := FN * V;
    Simplify;
  End;

Procedure TRational.Divide (const R : TRational);
  Begin
    if R.FT = 0 then
      raise EDivByZero.Create ('Rational division by zero');

    FT := FT * R.FN;
    FN := FN * R.FT;
    Simplify;
  End;

Procedure TRational.Divide (const V : TReal);
  Begin
    Assign (GetAsReal / V);
  End;

Procedure TRational.Reciprocal;
  Begin
    if FT = 0 then
      raise EDivByZero.Create ('Rational division by zero');

    Swap (FT, FN);
  End;

Procedure TRational.Multiply (const R : TRational);
  Begin
    FT := FT * R.FT;
    FN := FN * R.FN;
    Simplify;
  End;

Procedure TRational.Multiply (const V : TInteger);
  Begin
    FT := FT * V;
    Simplify;
  End;

Procedure TRational.Multiply (const V : TReal);
  Begin
    Assign (GetAsReal * V);
  End;

Procedure TRational.Power (const R : TRational);
  Begin
    Assign (Math.Power (GetAsReal, R.GetAsReal));
  End;

Procedure TRational.Power (const V : TInteger);
var T, N : Extended;
  Begin
    T := FT;
    N := FN;
    FT := Round (IntPower (T, V));
    FN := Round (IntPower (N, V));
  End;

Procedure TRational.Power (const V : TReal);
  Begin
    Assign (Math.Power (FT, V) / Math.Power (FN, V));
  End;

Procedure TRational.Sqrt;
  Begin
    Assign (System.Sqrt (FT / FN));
  End;

Procedure TRational.Sqr;
  Begin
    FT := System.Sqr (FT);
    FN := System.Sqr (FN);
  End;



{                                                                              }
{ TComplex                                                                     }
{                                                                              }
Constructor TComplex.Create (const TheRealPart, TheImaginaryPart : TReal);
  Begin
    inherited Create;
    FReal := TheRealPart;
    FImag := TheImaginaryPart;
  End;

Function TComplex.IsI : Boolean;
  Begin
    Result := RealZero (FReal) and RealEqual (FImag, 1.0);
  End;

Function TComplex.IsZero : Boolean;
  Begin
    Result := RealZero (FReal) and RealZero (FImag);
  End;

Function TComplex.IsEqual (const C : TComplex) : Boolean;
  Begin
    Result := RealEqual (FReal, C.FReal) and RealEqual (FImag, C.FImag);
  End;

Function TComplex.IsEqual (const R, I : TReal) : Boolean;
  Begin
    Result := RealEqual (FReal, R) and RealEqual (FImag, I);
  End;

Procedure TComplex.AssignZero;
  Begin
    FReal := 0.0;
    FImag := 0.0;
  End;

Procedure TComplex.AssignI;
  Begin
    FReal := 0.0;
    FImag := 1.0;
  End;

Procedure TComplex.Assign (const C : TComplex);
  Begin
    FReal := C.FReal;
    FImag := C.FImag;
  End;

Procedure TComplex.Assign (const V : TReal);
  Begin
    FReal := V;
    FImag := 0.0;
  End;

Function TComplex.Duplicate : TComplex;
  Begin
    Result := TComplex.Create (FReal, FImag);
  End;

Function TComplex.GetAsString : String;
  Begin
    Result := '(' + RealToStr (FReal) + ',' + RealToStr (FImag) + ')';
  End;

Procedure TComplex.SetAsString (const S : String);
var F, G, H : Integer;
  Begin
    F := Pos ('(', S);
    G := Pos (',', S);
    H := Pos (')', S);
    if (F <> 1) or (H <> Length (S)) or (G < F) or (G > H) then
      raise EConvertError.Create ('Can not convert string to complex number');
    FReal := StrToFloat (CopyRange (S, F + 1, G - 1));
    FImag := StrToFloat (CopyRange (S, G + 1, H - 1));
  End;

Procedure TComplex.Add (const C : TComplex);
  Begin
    FReal := FReal + C.FReal;
    FImag := FImag + C.FImag;
  End;

Procedure TComplex.Add (const V : TReal);
  Begin
    FReal := FReal + V;
  End;

Procedure TComplex.Subtract (const C : TComplex);
  Begin
    FReal := FReal - C.FReal;
    FImag := FImag - C.FImag;
  End;

Procedure TComplex.Subtract (const V : TReal);
  Begin
    FReal := FReal - V;
  End;

Procedure TComplex.Multiply (const C : TComplex);
var R : TReal;
  Begin
    R := FReal;
    FReal := R * C.FReal - FImag * C.FImag;
    FImag := R * C.FImag + FImag * C.FReal;
  End;

Procedure TComplex.Multiply (const V : TReal);
  Begin
    FReal := FReal * V;
    FImag := FImag * V;
  End;

Procedure TComplex.MultiplyI;
var R : TReal;
  Begin
    R := FReal;
    FReal := -FImag;
    FImag := R;
  End;

Procedure TComplex.MultiplyMinI;
var R : TReal;
  Begin
    R := FReal;
    FReal := FImag;
    FImag := -R;
  End;

Function TComplex.Denom : TReal;
  Begin
    Result := Sqr (FReal) + Sqr (FImag);
  End;

Procedure TComplex.Divide (const C : TComplex);
var R, D : TReal;
  Begin
    D := Denom;
    if RealZero (D) then
      raise EDivByZero.Create ('Complex division by zero') else
      begin
        R := FReal;
        FReal := (R * C.FReal + FImag * C.FImag) / D;
        FImag := (FImag * C.FReal - FReal * C.FImag) / D;
      end;
  End;

Procedure TComplex.Divide (const V : TReal);
var D : TReal;
  Begin
    D := Denom;
    if RealZero (D) then
      raise EDivByZero.Create ('Complex division by zero') else
      begin
        FReal := (FReal * V) / D;
        FImag := (FImag * V) / D;
      end;
  End;

Procedure TComplex.Negate;
  Begin
    FReal := -FReal;
    FImag := -FImag;
  End;

Procedure TComplex.Conjugate;
  Begin
    FImag := -FImag;
  End;

Procedure TComplex.Inverse;
var D : TReal;
  Begin
    D := Denom;
    if RealZero (D) then
      raise EDivByZero.Create ('Complex division by zero');
    FReal := FReal / D;
    FImag := - FImag / D;
  End;

Procedure TComplex.Exp;
var ExpZ : TReal;
    S, C : Extended;
  Begin
    ExpZ := System.Exp (FReal);
    SinCos (FImag, S, C);
    FReal := ExpZ * C;
    FImag := ExpZ * S;
  End;

Procedure TComplex.Ln;
var ModZ : TReal;
  Begin
    ModZ := Denom;
    if RealZero (ModZ) then
      raise EDivByZero.Create ('Complex log zero');
    FReal := System.Ln (ModZ);
    FImag := ArcTan2 (FReal, FImag);
  End;

Procedure TComplex.Power (const C : TComplex);
  Begin
    if not IsZero then
      begin
        Ln;
        Multiply (C);
        Exp;
      end else
      if C.IsZero then
        Assign (1.0) else      { lim a^a = 1 as a-> 0 }
        AssignZero;            { 0^a = 0 for a <> 0   }
  End;

Function TComplex.Modulo : TReal;
  Begin
    Result := System.Sqrt (Denom);
  End;

Procedure TComplex.Sqrt;
var Root, Q : TReal;
  Begin
    if not RealZero (FReal) or not RealZero (FImag) then
      begin
        Root := System.Sqrt (0.5 * (Abs (FReal) + Modulo));
        Q := FImag / (2.0 * Root);
        if FReal >= 0.0 then
          begin
            FReal := Root;
            FImag := Q;
          end else
          if FImag < 0.0 then
            begin
              FReal := - Q;
              FImag := - Root;
            end else
            begin
              FReal := Q;
              FImag := Root;
            end;
      end;
  End;

Procedure TComplex.Cos;
  Begin
    FReal := System.Cos (FReal) * Cosh (FImag);
    FImag := -System.Sin (FReal) * Sinh (FImag);
  End;

Procedure TComplex.Sin;
  Begin
    FReal := System.Sin (FReal) * Cosh (FImag);
    FImag := -System.Cos (FReal) * Sinh (FImag);
  End;

Procedure TComplex.Tan;
var CCos : TComplex;
  Begin
    CCos := TComplex.Create (FReal, FImag);
    try
      CCos.Cos;
      if CCos.IsZero then
        raise EDivByZero.Create ('Complex division by zero');
      self.Sin;
      self.Divide (CCos);
    finally
      CCos.Free;
    end;
  End;



{                                                                              }
{ Primes                                                                       }
{                                                                              }
const
  PrimeCacheLimit = 65537; // 8K lookup table. Note: Sqr(65537) > MaxLongInt

var
  PrimeSet : TFlatSet = nil;

{ Initializes the bit-array lookup of prime numbers using Sieve's algorithm.   }
Procedure InitPrimeSet;
var I, J : Integer;
  Begin
    PrimeSet := TFlatSet.Create;
    PrimeSet.SetRange (2, PrimeCacheLimit, True);
    PrimeSet.SetRange (0, 1, False);
    For I := 2 to Trunc (Sqrt (PrimeCacheLimit)) do
      if PrimeSet.GetBit (I) then
        For J := 2 to PrimeCacheLimit div I do
          PrimeSet.SetBit (I * J, False);
  End;

{ For small values of N (<=PrimeCacheLimit), uses a bit-array lookup. For      }
{ larger values, tries to find a prime factor.                                 }
Function IsPrime (const N : TInteger) : Boolean;
var I : Integer;
  Begin
    if N < 0 then
      Result := IsPrime (-N) else
    if N < 2 then
      Result := False else
      begin
        if not Assigned (PrimeSet) then // initialize look-up table
          InitPrimeSet;
        if N <= PrimeCacheLimit then // do look-up
          Result := PrimeSet.GetBit (N) else
          begin // calculate
            For I := 2 to Trunc (Sqrt (N)) do
              if ((I > PrimeCacheLimit) or PrimeSet.GetBit (I)) and (N mod I = 0) then
                begin
                  Result := False;
                  exit;
                end;
            Result := True;
          end;
      end;
  End;

Function PrimeFactors (const N : TInteger) : IntegerArray;
var I, J, L : Integer;
  Begin
    SetLength (Result, 0);
    if N < 0 then
      Result := PrimeFactors (-N) else
    if N = 1 then
      exit else
      begin
        if not Assigned (PrimeSet) then // initialize look-up table
          InitPrimeSet;

        L := 0;
        J := N;
        For I := 2 to Trunc (Sqrt (N)) do
          if ((I > PrimeCacheLimit) or PrimeSet.GetBit (I)) and (N mod I = 0) then
            begin // I is a prime factor
              Inc (L);
              SetLength (Result, L);
              Result [L - 1] := I;

              Repeat
                J := J div I;
                if J = 1 then // no more factors
                  exit;
              Until J mod I <> 0;
            end;
      end;
  End;

Function IsPrimeFactor (const F, N : TInteger) : Boolean;
  Begin
    Result := (N mod F = 0) and IsPrime (F);
  End;

Function GCD (const N1, N2 : TInteger) : TInteger;
var X, Y, I, J : Integer;
  Begin
    X := N1;
    Y := N2;
    if X < Y then
      Swap (X, Y);
    While (X <> 1) and (X <> 0) and (Y <> 1) and (Y <> 0) do
      begin
        J := (X - Y) mod Y;
        if J = 0 then
          begin
            Result := Y;
            exit;
          end;
        X := Y;
        Y := J;
      end;
    Result := 1;
  End;



{                                                                              }
{ TVector                                                                      }
{                                                                              }
Constructor TVector.Create (const Values : array of TReal);
  Begin
    inherited Create;
    Assign (Values);
  End;

Constructor TVector.Create (const Values : RealArray);
  Begin
    inherited Create;
    FData := Values;
  End;

Constructor TVector.Create (const Values : IntegerArray);
  Begin
    inherited Create;
    Assign (Values);
  End;

Procedure TVector.SetCount (const NewCount : Integer);
  Begin
    SetLength (FData, NewCount);
  End;

Function TVector.GetCount : Integer;
  Begin
    Result := Length (FData);
  End;

Procedure TVector.Assign (const V : TVector);
  Begin
    FData := Copy (V.FData);
  End;

Procedure TVector.Assign (const V : TReal);
var I : Integer;
  Begin
    For I := 0 to Length (FData) - 1 do
      FData [I] := V;
  End;

Procedure TVector.Assign (const V : array of TReal);
var I : Integer;
  Begin
    SetCount (High (V) + 1);
    For I := 0 to High (V) do
      FData [I] := V [I];
  End;

Procedure TVector.Assign (const V : IntegerArray);
var I : Integer;
  Begin
    SetCount (High (V) + 1);
    For I := 0 to High (V) do
      FData [I] := V [I];
  End;

Procedure TVector.Assign (const V : RealArray);
  Begin
    FData := Copy (V);
  End;

Procedure TVector.Assign (const V, Increment : TReal; const Count : Integer);
var I : Integer;
  Begin
    SetCount (Count);
    For I := 0 to Count - 1 do
      FData [I] := V + I * Increment;
  End;

Procedure TVector.Append (const V : TVector);
var I, L : Integer;
  Begin
    L := Length (FData);
    SetCount (L + Length (V.FData));
    For I := 0 to Length (V.FData) - 1 do
      FData [L + I] := V.FData [I];
  End;

Procedure TVector.Append (const V : TReal);
var L : Integer;
  Begin
    L := Length (FData);
    SetCount (L + 1);
    FData [L] := V;
  End;

Procedure TVector.SetItem (const Idx : Integer; const Value : TReal);
  Begin
    FData [Idx] := Value;
  End;

Function TVector.GetItem (const Idx : Integer) : TReal;
  Begin
    Result := FData [Idx];
  End;

Function TVector.Duplicate : TVector;
  Begin
    Result := TVector.Create;
    TVector (Result).FData := Copy (FData);
  End;

Function TVector.IsEqual (const V : TVector) : Boolean;
var I : Integer;
  Begin
    if Length (V.FData) <> Length (FData) then
      begin
        Result := False;
        exit;
      end;
    For I := 0 to Length (FData) - 1 do
      if FData [I] <> V.FData [I] then
        begin
          Result := False;
          exit;
        end;
    Result := True;
  End;

Function TVector.IsZero : Boolean;
var I : Integer;
  Begin
    For I := 0 to Length (FData) - 1 do
      if not RealZero (FData [I]) then
        begin
          Result := False;
          exit;
        end;
    Result := True;
  End;

Function TVector.Duplicate (const LoIdx, HiIdx : Integer) : TVector;
  Begin
    Result := TVector.Create;
    Result.FData := Copy (FData, LoIdx, HiIdx - LoIdx + 1);
  End;

Procedure TVector.Delete (const Idx, Count : Integer);
  Begin
    cMaths.Delete (FData, Idx, Count);
  End;

Function TVector.Intersection (const V : TVector; const IsSortedAscending : Boolean) : TVector;
  Begin
    Result := TVector.Create (cMaths.Intersection (FData, V.FData, IsSortedAscending));
  End;

Function TVector.Pos (const V : TReal; const PrevPos : Integer = -1; const IsSortedAscending : Boolean = False) : Integer;
  Begin
    Result := PosNext (V, FData, PrevPos, IsSortedAscending);
  End;

Procedure TVector.InvertOrder;
var I, J : Integer;
    R : TReal;
  Begin
    For I := 0 to Length (FData) div 2 - 1 do
      begin
        J := Length (FData) - I - 1;
        R := FData [0];
        FData [0] := FData [J];
        FData [J] := R;
      end;
  End;

Function TVector.GetAsString : String;
var I : Integer;
  Begin
    Result := '';
    For I := 0 to Length (FData) - 1 do
      Result := Result + RealToStr (FData [I]) + Cond (I = Length (FData) - 1, '', ',');
  End;

Procedure TVector.SetAsString (const S : String);
var F, G : Integer;
  Begin
    F := 1;
    SetCount (0);
    While F <= Length (S) do
      begin
        G := 0;
        While (F + G <= Length (S)) and (S [F + G] <> ',') do
          Inc (G);
        SetCount (Length (FData) + 1);
        if G = 0 then
          FData [Length (FData) - 1] := 0.0 else
          FData [Length (FData) - 1] := StrToFloat (Copy (S, F, G));
        Inc (F, G + 1);
      end;
  End;

{ Mathematical functions                                                       }
Procedure TVector.Add (const V : TVector; const Factor : TReal);
var I : Integer;
  Begin
    if Length (V.FData) <> Length (FData) then
      raise EVector.Create ('Vector size mismatch');

    For I := 0 to Length (FData) - 1 do
      FData [I] := FData [I] + V.FData [I] * Factor;
  End;

Procedure TVector.Add (const V : TVector);
  Begin
    Add (V, 1.0);
  End;

Procedure TVector.Add (const Value : TReal);
var I : Integer;
  Begin
    For I := 0 to Length (FData) - 1 do
      FData [I] := FData [I] + Value;
  End;

Procedure TVector.Multiply (const V : TVector);
var I : Integer;
  Begin
    if V.Count <> Length (FData) then
      raise EVector.Create ('Vector size mismatch');

    For I := 0 to Length (FData) - 1 do
      FData [I] := FData [I] * V.FData [I];
  End;

Procedure TVector.Multiply (const Value : TReal);
var I : Integer;
  Begin
    For I := 0 to Length (FData) - 1 do
      FData [I] := FData [I] * Value;
  End;

Procedure TVector.SquareValues;
var I : Integer;
  Begin
    For I := 0 to Length (FData) - 1 do
      FData [I] := Sqr (FData [I]);
  End;

Function TVector.Angle (const V : TVector) : TReal;
  Begin
    Result := ArcCos (DotProduct (V) / (Norm * V.Norm));
  End;

Function TVector.DotProduct (const V : TVector) : TReal;
var I : Integer;
  Begin
    if V.Count <> Length (FData) then
      raise EVector.Create ('Vector size mismatch');

    Result := 0.0;
    For I := 0 to Length (FData) - 1 do
      Result := Result + FData [I] * V.FData [I];
  End;

Function TVector.Norm : TReal;
  Begin
    Result := Sqrt (DotProduct (self));
  End;

Procedure TVector.Invert;
var I : Integer;
  Begin
    For I := 0 to Length (FData) - 1 do
      FData [I] := 1.0 / FData [I];
  End;

{ Statistical functions                                                        }
Procedure TVector.Sort;

  Procedure QuickSort (L, R : Integer);
  var I, J : Integer;
      M    : TReal;
    Begin
      Repeat
        I := L;
        J := R;
        M := FData [(L + R) shr 1];
        Repeat
          While FData [I] < M do
            Inc (I);
          While FData [J] > M do
            Dec (J);
          if I <= J then
            begin
              Swap (FData [I], FData [J]);
              Inc (I);
              Dec (J);
            end;
        Until I > J;
        if L < J then
          QuickSort (L, J);
        L := I;
      Until I >= R;
    End;

var I : Integer;
  Begin
    I := Length (FData);
    if I > 0 then
      QuickSort (0, I - 1);
  End;

Function TVector.Sum (const LoIdx, HiIdx : Integer) : TReal;
var I : Integer;
  Begin
    Result := 0.0;
    For I := Max (Integer (0), LoIdx) to Min (Length (FData) - 1, HiIdx) do
      Result := Result + FData [I];
  End;

Function TVector.Sum : TReal;
  Begin
    Result := Sum (0, Length (FData) - 1);
  End;

Function TVector.MaxValue : TReal;
var I : Integer;
  Begin
    if Length (FData) = 0 then
      raise EVector.Create ('No maximum');

    Result := FData [0];
    For I := 1 to Length (FData) - 1 do
      if FData [I] > Result then
        Result := FData [I];
  End;

Function TVector.MinValue : TReal;
var I : Integer;
  Begin
    if Length (FData) = 0 then
      raise EVector.Create ('No minimum');

    Result := FData [0];
    For I := 1 to Length (FData) - 1 do
      if FData [I] < Result then
        Result := FData [I];
  End;

Function TVector.Mean : TReal;
  Begin
    if Length (FData) = 0 then
      raise Exception.Create ('No mean');

    Result := Sum / Length (FData);
  End;

Function TVector.StdDev (var Mean : TReal) : TReal;
var S    : TReal;
    I, N : Integer;
  Begin
    N := Length (FData);
    if N = 0 then
      raise Exception.Create ('No standard deviation');

    if N = 1 then
      begin
        Mean := FData [0];
        Result := FData [0];
      end else
      begin
        Mean := self.Mean;
        S := 0.0;
        For I := 0 to N - 1 do
          S := S + Sqr (Mean - FData [I]);
        Result := Sqrt (S / (N - 1));
      end;
  End;

Procedure TVector.Normalize;
var S : TReal;
    I : Integer;
  Begin
    S := Norm;
    For I := 0 to Length (FData) - 1 do
      FData [I] := FData [I] / S;
  End;

Function TVector.HarmonicMean : TReal;
var I : Integer;
  Begin
    if Length (FData) = 0 then
      raise Exception.Create ('No harmonic mean');

    Result := 0.0;
    For I := 0 to Length (FData) - 1 do
      if FData [I] < 0.0 then
        raise Exception.Create ('Vector contains negative values') else
        Result := Result + 1.0 / FData [I];
    Result := Length (FData) / Result;
  End;

Function TVector.GeometricMean : TReal;
var I : Integer;
  Begin
    if Length (FData) = 0 then
      raise Exception.Create ('No geometric mean');

    Result := 1.0;
    For I := 0 to Length (FData) - 1 do
      if FData [I] < 0.0 then
        raise Exception.Create ('Vector contains negative values') else
        Result := Result * FData [I];
    Result := Power (Result, 1.0 / Length (FData));
  End;

Function TVector.Median : TReal;
var V : TVector;
  Begin
    if Length (FData) = 0 then
      raise EVector.Create ('No median');

    V := TVector (Duplicate);
    try
      V.Sort;
      Result := V.FData [(Length (V.FData) - 1) div 2];
    finally
      V.Free;
    end;
  End;

Function TVector.SumOfSquares : TReal;
var I : Integer;
  Begin
    Result := 0.0;
    For I := 0 to Length (FData) - 1 do
      Result := Result + Sqr (FData [I]);
  End;

Procedure TVector.SumAndSquares (var Sum, SumOfSquares : TReal);
var I : Integer;
    D : TReal;
  Begin
    Sum := 0.0;
    SumOfSquares := 0.0;
    For I := 0 to Length (FData) - 1 do
      begin
        D := FData [I];
        Sum := Sum + D;
        SumOfSquares := SumOfSquares + Sqr (D);
      end;
  End;

Function TVector.TotalVariance : TReal;
var Sum, SumSquares : TReal;
  Begin
    SumAndSquares (Sum, SumSquares);
    Result := SumSquares - Sqr (Sum) / Length (FData);
  End;

Function TVector.Variance : TReal;
  Begin
    Result := TotalVariance / (Length (FData) - 1);
  End;

Function TVector.PopulationVariance : TReal;
  Begin
    Result := TotalVariance / Length (FData);
  End;

Function TVector.PopulationStdDev : TReal;
  Begin
    Result := Sqrt (PopulationVariance);
  End;



{                                                                              }
{ TMatrix                                                                      }
{                                                                              }
Constructor TMatrix.CreateSquare (const N : Integer);
  Begin
    inherited Create;
    SetSize (N, N);
  End;

Constructor TMatrix.CreateUnity (const N : Integer);
var I : Integer;
  Begin
    inherited Create;
    SetSize (N, N);
    For I := 0 to N - 1 do
      FRows [I, I] := 1.0;
  End;

Constructor TMatrix.CreateDiagonal (const D : TVector);
var I, N : Integer;
  Begin
    inherited Create;
    N := Length (D.FData);
    SetSize (N, N);
    For I := 0 to N - 1 do
      FRows [I, I] := D.FData [I];
  End;

Procedure TMatrix.SetRowCount (const NewRowCount : Integer);
var I, OldRowCount : Integer;
  Begin
    OldRowCount := Length (FRows);
    if OldRowCount = NewRowCount then
      exit;

    SetLength (FRows, NewRowCount);
    if OldRowCount < NewRowCount then
      For I := OldRowCount to NewRowCount - 1 do
        SetLength (FRows [I], FColCount);
  End;

Function TMatrix.GetRowCount : Integer;
  Begin
    Result := Length (FRows);
  End;

Procedure TMatrix.SetColCount (const NewColCount : Integer);
var I : Integer;
  Begin
    if FColCount = NewColCount then
      exit;

    For I := 0 to Length (FRows) - 1 do
      SetLength (FRows [I], NewColCount);
    FColCount := NewColCount;
  End;

Procedure TMatrix.SetSize (const Rows, Cols : Integer);
  Begin
    SetLength (FRows, 0);
    SetColCount (Cols);
    SetRowCount (Rows);
  End;

Procedure TMatrix.SetItem (const Row, Col : Integer; const Value : TReal);
  Begin
    FRows [Row, Col] := Value;
  End;

Function TMatrix.GetItem (const Row, Col : Integer) : TReal;
  Begin
    Result := FRows [Row, Col];
  End;

Function TMatrix.GetRow (const Row : Integer) : TVector;
  Begin
    Result := TVector.Create;
    Result.FData := FRows [Row]; // reference
  End;

Function TMatrix.GetAsString : String;
var I, J : Integer;
  Begin
    Result := '';
    For I := 0 to Length (FRows) - 1 do
      begin
        Result := Result + '(';
        For J := 0 to FColCount - 1 do
          Result := Result + RealToStr (GetItem (I, J)) + Cond (J = FColCount - 1, '', ',');
        Result := Result + ')' + Cond (I = Length (FRows) - 1, '', ',');
      end;
  End;

Procedure TMatrix.Assign (const M : TMatrix);
var I : Integer;
  Begin
    SetSize (M.RowCount, M.ColCount);
    For I := 0 to Length (FRows) - 1 do
      FRows [I] := Copy (M.FRows [I]);
  End;

Procedure TMatrix.Assign (const Value : TReal);
var I, J : Integer;
  Begin
    For I := 0 to Length (FRows) - 1 do
      begin
        if Length (FRows [I]) <> FColCount then
          SetLength (FRows [I], FColCount);
        For J := 0 to FColCount - 1 do
          FRows [I, J] := Value;
      end;
  End;

Procedure TMatrix.AssignRowValues (const Row : Integer; const Values : Array of TReal);
var I : Integer;
  Begin
    For I := 0 to High (Values) do
      FRows [Row, I] := Values [I];
  End;

Procedure TMatrix.AssignRow (const Row : Integer; const V : TVector);
  Begin
    SetColCount (Length (V.FData));
    FRows [Row] := Copy (V.FData);
  End;

Procedure TMatrix.AssignCol (const Col : Integer; const V : TVector);
var I : Integer;
  Begin
    SetRowCount (Length (V.FData));
    For I := 0 to Length (FRows) - 1 do
      SetItem (I, Col, V.FData [I]);
  End;

Procedure TMatrix.Assign (const V : TVector);
  Begin
    SetSize (1, Length (V.FData));
    FRows [0] := Copy (V.FData);
  End;

Function TMatrix.Duplicate : TMatrix;
  Begin
    Result := TMatrix.Create;
    Result.Assign (self);
  End;

Function TMatrix.Duplicate (const R1, C1, R2, C2 : Integer) : TMatrix;
var I : Integer;
    _R1, _C1, _R2, _C2 : Integer;
  Begin
    Result := TMatrix.Create;

    _R1 := Max (R1, 0);
    _R2 := Min (R2, Length (FRows));
    _C1 := Max (C1, 0);
    _C2 := Min (C2, FColCount);

    if (_R1 > _R2) or (_C1 > _C2) then
      exit;

    Result.SetSize (R2 - R1 + 1, C2 - C1 + 1);
    For I := R1 to R2 do
      Result.FRows [I - R1] := Copy (FRows [I], C1, C2 - C1 + 1);
  End;

Function TMatrix.DuplicateRow (const Row : Integer) : TVector;
  Begin
    Result := TVector.Create;
    try
      Result.FData := Copy (FRows [Row]);
      SetLength (Result.FData, FColCount);
    except
      Result.Free;
      raise;
    end;
  End;

Function TMatrix.DuplicateCol (const Col : Integer) : TVector;
var I : Integer;
  Begin
    if (Col > FColCount - 1) or (Col < 0) then
      raise EMatrix.Create ('Column index out of range');

    Result := TVector.Create;
    SetLength (Result.FData, Length (FRows));
    For I := 0 to Length (FRows) - 1 do
      if Length (FRows [I]) >= Col then
        Result.FData [I] := FRows [I, Col];
  End;

Function TMatrix.DuplicateDiagonal : TVector;
var I : Integer;
  Begin
    if not IsSquare then
      raise EMatrix.Create ('Not square');

    Result := TVector.Create;
    SetLength (Result.FData, Length (FRows));
    For I := 0 to Length (FRows) - 1 do
      if Length (FRows [I]) >= I then
        Result.FData [I] := FRows [I, I];
  End;

Function TMatrix.IsSquare : Boolean;
  Begin
    Result := Length (FRows) = FColCount;
  End;

Function TMatrix.IsZero : Boolean;
var I, J : Integer;
  Begin
    For I := 0 to Length (FRows) - 1 do
      For J := 0 to Length (FRows [I]) do
        if not RealZero (FRows [I, J]) then
          begin
            Result := False;
            exit;
          end;
    Result := True;
  End;

Function TMatrix.IsUnity : Boolean;
var I, J : Integer;
    R    : TReal;
  Begin
    if not IsSquare then
      begin
        Result := False;
        exit;
      end;

    For I := 0 to Length (FRows) - 1 do
      For J := 0 to Length (FRows [I]) - 1 do
        begin
          R := FRows [I, J];
          if ((J = I) and not RealEqual (R, 1.0)) or
             ((J <> I) and not RealZero (R)) then
            begin
              Result := False;
              exit;
            end;
        end;
    Result := True;
  End;

Function TMatrix.IsEqual (const M : TMatrix) : Boolean;
var I, J   : Integer;
  Begin
    if (Length (FRows) <> Length (M.FRows)) or (FColCount <> M.FColCount) then
      begin
        Result := False;
        exit;
      end;

    For I := 0 to Length (FRows) - 1 do
      For J := 0 to FColCount - 1 do
        if not RealEqual (FRows [I, J], M.FRows [I, J]) then
          begin
            Result := False;
            exit;
          end;
    Result := True;
  End;

Function TMatrix.IsEqual (const V : TVector) : Boolean;
var I : Integer;
  Begin
    if (Length (FRows) = 1) and (Length (V.FData) = FColCount) then
      begin
        For I := 0 to FColCount - 1 do
          if not RealEqual (V.FData [I], FRows [0, I]) then
            begin
              Result := False;
              exit;
            end;
        Result := True;
      end else
      Result := False;
  End;

Procedure TMatrix.SwapRows (const I, J : Integer);
var P : RealArray;
  Begin
    if Max (I, J) > Length (FRows) - 1 then
      raise EMatrix.Create ('Row index out of range');

    P := FRows [I];
    FRows [I] := FRows [J];
    FRows [J] := P;
  End;

Procedure TMatrix.AddRows (const I, J : Integer; const Factor : TReal);
var F : Integer;
  Begin
    if Max (I, J) > Length (FRows) - 1 then
      raise EMatrix.Create ('Row index out of range');

    For F := 0 to Min (Length (FRows [J]), FColCount) - 1 do
      FRows [I, F] := FRows [I, F] + FRows [J, F] * Factor;
  End;

Function TMatrix.Transposed : TMatrix;
var I, J : Integer;
  Begin
    Result := TMatrix.Create;
    Result.SetSize (FColCount, Length (FRows) - 1);
    For I := 0 to Length (Result.FRows) - 1 do
      For J := 0 to Result.FColCount - 1 do
        if I <= Length (FRows [J]) - 1 then
          Result.FRows [I, J] := FRows [J, I];
  End;

Procedure TMatrix.Transpose;
var M : TMatrix;
  Begin
    M := Transposed;
    try
      Assign (M);
    finally
      M.Free;
    end;
  End;

Procedure TMatrix.Add (const M : TMatrix);
var R, I, J : Integer;
  Begin
    R := RowCount;
    if (M.RowCount <> R) or (M.ColCount <> ColCount) then
      raise EMatrix.Create ('Matrix size mismatch');

    For I := 0 to R - 1 do
      For J := 0 to FColCount - 1 do
        FRows [I, J] := FRows [I, J] + M.FRows [I, J];
  End;

Function TMatrix.Multiplied (const M : TMatrix) : TMatrix;
var I, J, K : Integer;
    R       : TReal;
  Begin
    if ColCount <> M.RowCount then
      raise EMatrix.Create ('Matrix size mismatch');

    Result := TMatrix.Create;
    Result.SetSize (RowCount, M.ColCount);
    For I := 0 to Result.RowCount - 1 do
      For J := 0 to Result.ColCount - 1 do
        begin
          R := 0.0;
          For K := 0 to ColCount - 1 do
            R := R + FRows [I, K] * M.FRows [K, J];
          Result.FRows [I, J] := R;
        end;
  End;

Procedure TMatrix.Multiply (const M : TMatrix);
var NM : TMatrix;
  Begin
    NM := Multiplied (M);
    try
      Assign (NM);
    finally
      NM.Free;
    end;
  End;

Procedure TMatrix.Multiply (const Row : Integer; const Value : TReal);
var I : Integer;
  Begin
    For I := 0 to FColCount - 1 do
      FRows [Row, I] := FRows [Row, I] * Value
  End;

Procedure TMatrix.Multiply (const Value : TReal);
var I : Integer;
  Begin
    For I := 0 to RowCount - 1 do
      Multiply (I, Value);
  End;

Function TMatrix.Trace : TReal;
var I : Integer;
  Begin
    if not IsSquare then
      raise EMatrix.Create ('Matrix not square');
    Result := 0.0;
    For I := 0 to RowCount - 1 do
      Result := Result + FRows [I, I];
  End;

Function TMatrix.IsOrtogonal : Boolean;
var M : TMatrix;
  Begin
    M := Duplicate;
    try
      M.Transpose;
      M.Multiply (self);
      Result := M.IsUnity;
    finally
      M.Free;
    end;
  End;

Function TMatrix.IsIdempotent : Boolean;
var M : TMatrix;
  Begin
    M := Duplicate;
    try
      M.Multiply (M);
      Result := M.IsEqual (M);
    finally
      M.Free;
    end;
  End;

Function TMatrix.Normalise (const M : TMatrix = nil) : TReal;
var I : Integer;
    R : TReal;
  Begin
    Result := 1.0;
    For I := 0 to RowCount - 1 do
      begin
        R := GetItem (I, I);
        Result := Result * R;
        if not RealZero (R) then
          begin
            R := 1.0 / R;
            Multiply (I, R);
            if Assigned (M) then
              M.Multiply (I, R);
          end;
      end;
  End;

Function TMatrix.SolveMatrix (var M : TMatrix) : TReal;
var I, J   : Integer;
    R      : TReal;
  Begin
    Result := 1.0;
    For I := 0 to RowCount - 1 do
      begin
        J := 0;
        While J < RowCount do
          if not RealZero (GetItem (I, J))  then
            break else
            Inc (J);
        if J = RowCount then
          begin
            Result := 0.0;
            exit;
          end;
        SwapRows (I, J);
        M.SwapRows (I, J);
        if Odd (M.ColCount) then
          Result := -Result;

        For J := I + 1 to RowCount - 1 do
          begin
            R := -(GetItem (J, I) / GetItem (I, I));
            AddRows (J, I, R);
            M.AddRows (J, I, R);
          end;
      end;

    For I := RowCount - 1 downto 0 do
      For J := I - 1 downto 0 do
        begin
          R := -(GetItem (J, I) / GetItem (I, I));
          AddRows (J, I, R);
          M.AddRows (J, I, R);
        end;

    Result := Normalise (M);
  End;

Function TMatrix.Determinant : TReal;
var A, B : TMatrix;
  Begin
    if not IsSquare then
      raise EMatrix.Create ('No determinant');

    A := Duplicate; try
    B := TMatrix.CreateUnity (RowCount);
    try
      Result := A.SolveMatrix (B);
    finally B.Free; end;
    finally A.Free; end;
  End;

Procedure TMatrix.Inverse;
var A : TMatrix;
  Begin
    if not IsSquare then
      raise EMatrix.Create ('No inverse');

    A := TMatrix.CreateUnity (RowCount);
    try
      if RealZero (SolveMatrix (A)) then
        raise EMatrix.Create ('Can not invert');
      Assign (A);
    finally
      A.Free;
    end;
  End;

Function TMatrix.SolveLinearSystem (const V : TVector) : TVector;
var C, M : TMatrix;
  Begin
    if not IsSquare or (V.Count <> RowCount) then
      raise EMatrix.Create ('Not a linear system');
    C := Duplicate; try
    M := TMatrix.Create;
    try
      M.Assign (V);
      if RealZero (C.SolveMatrix (M)) then
        raise EMatrix.Create ('Can not solve this system');
      Result := TVector.Create;
      Result.Assign (M.GetRow (0));
    finally M.Free; end;
    finally C.Free; end;
  End;



{                                                                              }
{ 3D Transformation matrices                                                   }
{                                                                              }
Function OriginAndScaleTransform (const TX, TY, TZ, SX, SY, SZ : TReal) : TMatrix;
  Begin
    Result := TMatrix.CreateSquare (4);
    Result.AssignRowValues (0, [ SX, 0.0, 0.0, -TX]);
    Result.AssignRowValues (1, [0.0,  SY, 0.0, -TY]);
    Result.AssignRowValues (2, [0.0, 0.0,  SZ, -TZ]);
    Result.AssignRowValues (3, [0.0, 0.0, 0.0, 1.0]);
  End;

Function XRotateTransform (const Angle : TReal) : TMatrix;
var S, C : Extended;
  Begin
    SinCos (Angle, S, C);
    Result := TMatrix.CreateSquare (4);
    Result.AssignRowValues (0, [1.0, 0.0, 0.0, 0.0]);
    Result.AssignRowValues (1, [0.0, C  , -S , 0.0]);
    Result.AssignRowValues (2, [0.0, S  , C  , 0.0]);
    Result.AssignRowValues (3, [0.0, 0.0, 0.0, 1.0]);
  End;

Function YRotateTransform (const Angle : TReal) : TMatrix;
var S, C : Extended;
  Begin
    SinCos (Angle, S, C);
    Result := TMatrix.CreateSquare (4);
    Result.AssignRowValues (0, [C  , 0.0, -S , 0.0]);
    Result.AssignRowValues (1, [0.0, 1.0, 0.0, 0.0]);
    Result.AssignRowValues (2, [S  , 0.0, C  , 0.0]);
    Result.AssignRowValues (3, [0.0, 0.0, 0.0, 1.0]);
  End;

Function ZRotateTransform (const Angle : TReal) : TMatrix;
var S, C : Extended;
  Begin
    SinCos (Angle, S, C);
    Result := TMatrix.CreateSquare (4);
    Result.AssignRowValues (0, [C  , -S , 0.0, 0.0]);
    Result.AssignRowValues (1, [S  , C  , 0.0, 0.0]);
    Result.AssignRowValues (2, [0.0, 0.0, 1.0, 0.0]);
    Result.AssignRowValues (3, [0.0, 0.0, 0.0, 1.0]);
  End;

Function XYZRotateTransform (const XAngle, YAngle, ZAngle : TReal) : TMatrix;
var SX, CX, SY, CY, SZ, CZ : Extended;
    SXSY, CXSY : TReal;
  Begin
    Result := TMatrix.CreateSquare (4);
    SinCos (XAngle, SX, CX);
    SinCos (YAngle, SY, CY);
    SinCos (ZAngle, SZ, CZ);
    SXSY := SX * SY;
    CXSY := CX * SY;
    Result.AssignRowValues (0, [        CY*CZ,         CY*SZ,   -SY, 0.0]);
    Result.AssignRowValues (1, [SXSY*CZ-CX*SZ, SXSY*SZ+CX*CZ, SX*CY, 0.0]);
    Result.AssignRowValues (2, [CXSY*CZ+SX*SZ, CXSY*SZ-SX*CZ, CX*CY, 0.0]);
    Result.AssignRowValues (3, [          0.0,           0.0,   0.0, 1.0]);
  End;



{                                                                              }
{ T3DPoint                                                                     }
{                                                                              }
Constructor T3DPoint.CreateVector (const X, Y, Z : TReal);
  Begin
    inherited Create ([X, Y, Z, 0.0]);
  End;

Constructor T3DPoint.CreatePoint (const X, Y, Z : TReal);
  Begin
    inherited Create ([X, Y, Z, 1.0]);
  End;

Function T3DPoint.Duplicate : T3DPoint;
  Begin
    Result := T3DPoint.Create ([FData [0], FData [1], FData [2], FData [3]]);
  End;

Procedure T3DPoint.SetX (const NewX : TReal);
  Begin
    FData [0] := NewX;
  End;

Procedure T3DPoint.SetY (const NewY : TReal);
  Begin
    FData [1] := NewY;
  End;

Procedure T3DPoint.SetZ (const NewZ : TReal);
  Begin
    FData [2] := NewZ;
  End;

Function T3DPoint.GetX : TReal;
  Begin
    Result := FData [0];
  End;

Function T3DPoint.GetY : TReal;
  Begin
    Result := FData [1];
  End;

Function T3DPoint.GetZ : TReal;
  Begin
    Result := FData [2];
  End;

Procedure T3DPoint.RotateX (const Angle : TReal);
var S, C : Extended;
    Y, Z : TReal;
  Begin
    SinCos (Angle, S, C);
    Y := FData [1];
    Z := FData [2];
    FData [1] := C * Y - S * Z;
    FData [2] := S * Y + C * Z;
  End;

Procedure T3DPoint.RotateY (const Angle : TReal);
var S, C : Extended;
    X, Z : TReal;
  Begin
    SinCos (Angle, S, C);
    X := FData [0];
    Z := FData [2];
    FData [0] := C * X - S * Z;
    FData [2] := S * X + C * Z;
  End;

Procedure T3DPoint.RotateZ (const Angle : TReal);
var S, C : Extended;
    X, Y : TReal;
  Begin
    SinCos (Angle, S, C);
    X := FData [0];
    Y := FData [1];
    FData [0] := C * X - S * Y;
    FData [1] := S * X + C * Y;
  End;

Procedure T3DPoint.RotateXYZ (const XAngle, YAngle, ZAngle : TReal);
var SX, CX, SY, CY, SZ, CZ : Extended;
    F1, F2, YCX, ZSX,
    X, Y, Z                : TReal;
  Begin
    X := FData [0];
    Y := FData [1];
    Z := FData [2];
    SinCos (XAngle, SX, CX);
    SinCos (YAngle, SY, CY);
    SinCos (ZAngle, SZ, CZ);
    F2 := Y * SX + Z * CX;
    F1 := X * CY + SY * F2;
    YCX := Y * CX;
    ZSX := Z * SX;
    FData [0] := CZ * F1 + SZ * (ZSX - YCX);
    FData [1] := SZ * F1 + CZ * (YCX - ZSX);
    FData [2] := CY * F2 - X * SY;
  End;

Procedure T3DPoint.RotateVector (const NX, NY, NZ, Angle : TReal);
var S, C : Extended;
    X, Y, Z,
    F1 : TReal;
  Begin
    X := FData [0];
    Y := FData [1];
    Z := FData [2];
    SinCos (Angle, S, C);
    F1 := (1.0 - C) * (X * NX + Y * Y * NY + Z * Z * NZ);

    FData [0] := NX * F1 + C * X + S * (Y * NZ - Z * NY);
    FData [1] := NY * F1 + C * Y + S * (Z * NX - X * NZ);
    FData [2] := NZ * F1 + C * Z + S * (X * NY - Y * NX);
  End;

Procedure T3DPoint.Homogenize;
var W : TReal;
  Begin
    W := FData [3];
    if W = 0.0 then
      raise E3DPoint.Create ('Not a point');
    FData [0] := FData [0] / W;
    FData [1] := FData [1] / W;
    FData [2] := FData [2] / W;
    FData [3] := 1.0;
  End;

Procedure T3DPoint.CrossProduct (const P : T3DPoint);
var X, Y, Z,
    BX, BY, BZ : TReal;
  Begin
    X := FData [0];
    Y := FData [1];
    Z := FData [2];
    BX := P.FData [0];
    BY := P.FData [1];
    BZ := P.FData [2];
    FData [0] := Y * BZ - Z * BY;
    FData [1] := Z * BX - X * BZ;
    FData [2] := X * BY - Y * BX;
  End;

Procedure T3DPoint.Scale (const XScale, YScale, ZScale : TReal);
  Begin
    FData [0] := FData [0] * XScale;
    FData [1] := FData [1] * YScale;
    FData [2] := FData [2] * ZScale;
  End;

Procedure T3DPoint.Origin (const XOrigin, YOrigin, ZOrigin : TReal);
  Begin
    FData [0] := FData [0] + XOrigin;
    FData [1] := FData [1] + YOrigin;
    FData [2] := FData [2] + ZOrigin;
  End;

Procedure CavalierProjection (const Angle, X1, Y1, Z1 : TReal; var X, Y : TReal);
var S, C : Extended;
  Begin
    SinCos (Angle * OneDegree, S, C);
    X := X1 + Z1 * C;
    Y := Y1 + Z1 * S;
  End;

Procedure T3DPoint.CavalierProject (const Angle : TReal; var X, Y : TReal);
  Begin
    CavalierProjection (Angle, FData [0], FData [1], FData [2], X, Y);
  End;

Procedure T3DPoint.CabinetProject (const Angle : TReal; var X, Y : TReal);
  Begin
    CavalierProjection (Angle, FData [0], FData [1], 0.5 * FData [2], X, Y);
  End;

Function ClipPerspectiveProjection (const P, V : TReal) : TReal;
  Begin
    if V > 0 then
      Result := Min (P, V) else
      Result := Max (P, V);
  End;

Procedure T3DPoint.OnePointPerspectiveProject (const Angle, Zv : TReal; var X, Y : TReal);
var Z, ZF : TReal;
  Begin
    Z := ClipPerspectiveProjection (FData [2], Zv);
    ZF := (Zv - Z) / Zv;
    CavalierProjection (Angle, FData [0] * ZF, FData [1] * ZF, Z, X, Y);
  End;

Procedure T3DPoint.TwoPointPerspectiveProject (const Angle, Xv, Zv : TReal; var X, Y : TReal);
var XP, ZP, ZF, XF : TReal;
  Begin
    XP := ClipPerspectiveProjection (FData [0], Xv);
    ZP := ClipPerspectiveProjection (FData [2], Zv);
    XF := (Xv - XP) / Xv;
    ZF := (Zv - ZP) / Zv;
    CavalierProjection (Angle, XP * ZF, FData [1] * XF * ZF, ZP, X, Y);
  End;

{                                                                              }
{ Numerical solvers                                                            }
{                                                                              }
Function SecantSolver (const f : fx; const y, Guess1, Guess2 : TReal) : TReal;
var xn, xnm1, xnp1, fxn, fxnm1 : TReal;
  Begin
    xnm1 := Guess1;
    xn := Guess2;
    fxnm1 := f (xnm1) - y;
    Repeat
      fxn := f (xn) - y;
      xnp1 := xn - fxn * (xn - xnm1) / (fxn - fxnm1);
      fxnm1 := fxn;
      xnm1 := xn;
      xn := xnp1;
    Until (f (xn - 0.00000001) - y) * (f (xn + 0.00000001) - y) <= 0.0;
    Result := xn;
  End;

Function NewtonSolver (const f, df : fx; const y, Guess : TReal) : TReal;
var xn, xnp1 : TReal;
  Begin
    xnp1 := Guess;
    Repeat
      xn := xnp1;
      xnp1 := xn - f (xn) / df (xn);
    Until Abs (xnp1 - xn) < 0.000000000000001;
    Result := xn;
  End;

const h = 1e-15;

Function FirstDerivative (const f : fx; const x : TReal) : TReal;
  Begin
    Result := (-f (x + 2 * h) + 8 * f (x + h) - 8 * f (x - h) + f (x - 2 * h)) / (12 * h);
  End;

Function SecondDerivative (const f : fx; const x : TReal) : TReal;
  Begin
    Result := (-f (x + 2 * h) + 16 * f (x + h) - 30 * f (x) + 16 * f (x - h) - f (x - 2 * h)) / (12 * h * h);
  End;

Function ThirdDerivative (const f : fx; const x : TReal) : TReal;
  Begin
    Result := (f (x + 2 * h) - 2 * f (x + h) + 2 * f (x - h) - f (x - 2 * h)) / (2 * h * h * h);
  End;

Function FourthDerivative (const f : fx; const x : TReal) : TReal;
  Begin
    Result := (f (x + 2 * h) - 4 * f (x + h) + 6 * f (x) - 4 * f (x - h) + f (x - 2 * h)) / (h * h * h * h);
  End;

Function SimpsonIntegration (const f : fx; const a, b : TReal; N : TInteger) : TReal;
var h : TReal;
    I : Integer;
  Begin
    if N mod 2 = 1 then
     Inc (N); // N must be multiple of 2

    h := (b - a) / N;

    Result := 0.0;
    For I := 1 to N - 1 do
      Result := Result + ((I mod 2) * 2 + 2) * f (a + (I - 0.5) * h);
    Result := (Result + f (a) + f (b)) * h / 3.0;
  End;



{                                                                              }
{ TRange                                                                       }
{                                                                              }
Constructor TRange.Create (const Low, High : Integer);
  Begin
    inherited Create;
    FLow := Low;
    FHigh := High;
  End;

Function TRange.Overlap (const Low, High : Integer) : Boolean;
{  5..10                                            }
{  R       F     RL<L   RH>H   RH>=L   RL<=H        }
{ 0..4     0      1      0      0       1           }
{ 11..12   0      0      1      1       0           }
{ 6..11    1      0      1      1       1           }
{ 10..11   1      0      1      1       1           }
{ 0..5     1      1      0      1       1           }
{ 0..11    1      1      1      1       1           }
{ 5..10    1      0      0      1       1           }
{ 6..7     1      0      0      1       1           }
  Begin
    Result := (High >= FLow) xor (Low <= FHigh);
  End;

Function TRange.Touch (const Low, High : Integer) : Boolean;
  Begin
    Result := (High >= FLow - 1) xor (Low <= FHigh + 1);
  End;

Function TRange.Inside (const Low, High : Integer) : Boolean;
  Begin
    Result := (FLow >= Low) and (FHigh <= High);
  End;

Function TRange.HasElement (const I : Integer) : Boolean;
  Begin
    Result := (I >= FLow) and (I <= FHigh);
  End;

Procedure TRange.Merge (const Low, High : Integer);
  Begin
    if (High >= FLow - 1) xor (Low <= FHigh + 1) then // Touch
      begin
        if Low < FLow then
          FLow := Low;
        if High > FHigh then
          FHigh := High;
      end else
      raise ERange.Create ('Can not merge range');
  End;



{                                                                              }
{ TRangeSet                                                                    }
{                                                                              }
Constructor TRangeSet.Create;
  Begin
    inherited Create;
    FRanges := TList.Create;
  End;

Destructor TRangeSet.Destroy;
  Begin
    Clear;
    FRanges.Free;
    FRanges := nil;
    inherited Destroy;
  End;

Procedure TRangeSet.Clear;
var I : Integer;
  Begin
    For I := 0 to FRanges.Count - 1 do
      TRange (FRanges [I]).Free;
    FRanges.Clear;
  End;

Procedure TRangeSet.Invert;
var I      : Integer;
    R, Pre : TRange;
  Begin
    if (FRanges.Count > 0) and (TRange (FRanges [0]).FLow > 0) then
      Pre := TRange.Create (0, TRange (FRanges [0]).FLow - 1) else
      Pre := nil;
    For I := 0 to FRanges.Count - 2 do
      begin
        R := TRange (FRanges [I]);
        R.FLow := R.FHigh + 1;
        R.FHigh := TRange (FRanges [I + 1]).FLow - 1;
      end;
    if FRanges.Count > 0 then
      if TRange (FRanges [FRanges.Count - 1]).FHigh = MaxInt then
      begin
        TRange (FRanges [FRanges.Count - 1]).Free;
        FRanges.Delete (FRanges.Count - 1);
      end else
      begin
        TRange (FRanges [FRanges.Count - 1]).FLow := TRange (FRanges [FRanges.Count - 1]).FHigh + 1;
        TRange (FRanges [FRanges.Count - 1]).FHigh := MaxInt;
      end;
    if Assigned (Pre) then
      FRanges.Insert (0, Pre);
  End;

Procedure TRangeSet.SetRange (const Low, High : Integer; const Value : Boolean);
var I, J, K : Integer;
    R : TRange;
  Begin
    I := 0;
    K := FRanges.Count;
    While (I <= K - 1) and (Low > TRange (FRanges [I]).FHigh) do
      Inc (I);
    if (I = K) or  // append
       ((Low < TRange (FRanges [I]).FLow) and (High < TRange (FRanges [I]).FLow)) then // insert
      begin
        if Value then
          FRanges.Insert (I, TRange.Create (Low, High));
        exit;
      end;
    // I = first block that overlaps

    J := I;
    While (J < K - 1) and (TRange (FRanges [J + 1]).FLow <= High) do
      Inc (J);
    // J = last block that overlaps

    if not Value then // clear range (NOT IMPL)
      begin
        if I = J then
          begin
            R := FRanges [I];
            if R.Inside (Low, High) then
              begin
                R.Free;
                FRanges.Delete (I);
                exit;
              end;
          end else
          begin
            For K := I + 1 to J - 1 do
              begin
                TRange (FRanges [I + 1]).Free;
                FRanges.Delete (I + 1);
              end;
          end;
      end else // set range
      begin
        R := FRanges [I];
        if I = J then
          begin
            if Low < R.FLow then
              R.FLow := Low;
            if High > R.FHigh then
              R.FHigh := High;
          end else
          begin
            R.FLow := Low;
            R.FHigh := High;
            For K := I + 1 to J do
              begin
                TRange (FRanges [I + 1]).Free;
                FRanges.Delete (I + 1);
              end;
          end;
      end;
  End;

Function TRangeSet.GetBit (const Idx : Integer) : Boolean;
var I, J : Integer;
  Begin
    I := 0;
    J := FRanges.Count - 1;
    While (I <= J) and (Idx > TRange (FRanges [I]).FHigh) do
      Inc (I);
    Result := (I <= J) and (Idx >= TRange (FRanges [I]).FLow);
  End;

Function TRangeSet.GetRange (const Low, High : Integer; const Value : Boolean) : Boolean;
var I, J : Integer;
  Begin
    I := 0;
    J := FRanges.Count - 1;
    While (I <= J) and (Low > TRange (FRanges [I]).FHigh) do
      Inc (I);
    Result := (I <= J) and (Low >= TRange (FRanges [I]).FLow) and (High <= TRange (FRanges [I]).FHigh);
  End;

Procedure TRangeSet.SetBit (const Idx : Integer; const Value : Boolean);
  Begin
    SetRange (Idx, Idx, Value);
  End;



{                                                                              }
{ TFlatSet                                                                     }
{                                                                              }
Constructor TFlatSet.Create;
  Begin
    inherited Create;
    FBits := TBits.Create;
  End;

Destructor TFlatSet.Destroy;
  Begin
    FBits.Free;
    FBits := nil;
    inherited Destroy;
  End;

Procedure TFlatSet.Clear;
  Begin
    FBits.Size := 0;
  End;

Procedure TFlatSet.Invert;
var I : Integer;
  Begin
    For I := 0 to FBits.Size - 1 do
      FBits [I] := not FBits [I];
  End;

Procedure TFlatSet.SetRange (const Low, High : Integer; const Value : Boolean);
var I : Integer;
  Begin
    For I := High downto Low do
      FBits [I] := Value;
  End;

Function TFlatSet.GetBit (const Idx : Integer) : Boolean;
  Begin
    Result := FBits [Idx];
  End;

Function TFlatSet.GetRange (const Low, High : Integer; const Value : Boolean) : Boolean;
var I : Integer;
  Begin
    if not Value and (High >= FBits.Size) then
      begin
        Result := False;
        exit;
      end;
    For I := Low to Min (High, FBits.Size - 1) do
      if FBits [I] <> Value then
        begin
          Result := False;
          exit;
        end;
    Result := True;
  End;

Procedure TFlatSet.SetBit (const Idx : Integer; const Value : Boolean);
  Begin
    FBits [Idx] := Value;
  End;



{                                                                              }
{ TSparseFlatSet                                                               }
{                                                                              }
Destructor TSparseFlatSet.Destroy;
  Begin
    Clear;
    inherited Destroy;
  End;

Procedure TSparseFlatSet.Clear;
var F : Integer;
  Begin
    if Assigned (FSetList) then
      begin
        For F := 0 to FSetListEntries - 1 do
          if Assigned (FSetList^ [F]) then
            Release (PDelphiSet (FSetList^ [F]));
        FreeMem (FSetList, FSetListEntries * Sizeof (Pointer));
        FSetList := nil;
        FSetListEntries := 0;
      end;
  End;

Procedure TSparseFlatSet.Invert;
var F : Integer;
  Begin
    For F := 0 to FSetListEntries - 1 do
      if Assigned (FSetList^ [F]) then
        PDelphiSet (FSetList^ [F])^ := CompleteDelphiSet - PDelphiSet (FSetList^ [F])^;
  End;

Function TSparseFlatSet.GetBit (const Idx : Integer) : Boolean;
var SetIdx : Integer;
  Begin
    SetIdx := Idx shr 8;
    Result := (SetIdx < FSetListEntries) and
              Assigned (FSetList^ [SetIdx]) and
              (Byte (Idx and $FF) in PDelphiSet (FSetList^ [SetIdx])^);
  End;

Procedure TSparseFlatSet.SetBit (const Idx : Integer; const Value : Boolean);
var I,
    SetIdx : Integer;
    S      : PDelphiSet;
  Begin
    SetIdx := Idx shr 8;
    if SetIdx >= FSetListEntries then
      if Value then
        begin
          I := FSetListEntries;
          FSetListEntries := SetIdx + 1;
          ReallocMem (FSetList, FSetListEntries * Sizeof (Pointer));
          FillChar (FSetList^ [I], (FSetListEntries - I) * Sizeof (Pointer), #0);
        end else
        exit;
    S := FSetList^ [SetIdx];
    if not Assigned (S) then
      if Value then
        begin
          New (S);
          S^ := [];
          FSetList^ [SetIdx] := S;
        end else
        exit;
    Include (S^, Byte (Idx and $FF));
  End;

Procedure TSparseFlatSet.SetRange (const Low, High : Integer; const Value : Boolean);

  Procedure SetValue (const S : TDelphiSet; const SetIdx : Integer);
  var D : PDelphiSet;
    Begin
      D := FSetList^ [SetIdx];
      if not Assigned (D) then
        begin
          if Value then
            begin
              New (D);
              D^ := S;
              FSetList^ [SetIdx] := D;
            end;
        end else
      if Value then
        D^ := D^ + S else
        D^ := D^ - S;
    End;

var I, LowSet,
    HighSet : Integer;

  Begin
    LowSet := Low shr 8;
    HighSet := High shr 8;
    if HighSet >= FSetListEntries then
      begin
        I := FSetListEntries;
        FSetListEntries := HighSet + 1;
        ReallocMem (FSetList, FSetListEntries * Sizeof (Pointer));
        FillChar (FSetList^ [I], (FSetListEntries - I) * Sizeof (Pointer), #0);
      end;
    if LowSet = HighSet then
      SetValue ([Byte (Low and $FF)..Byte (High and $FF)], LowSet) else
      begin
        SetValue ([Byte (Low and $FF)..$FF], LowSet);
        SetValue ([0..Byte (High and $FF)], HighSet);
        For I := LowSet + 1 to HighSet - 1 do
          SetValue (CompleteDelphiSet, I);
      end;
  End;



{                                                                              }
{ Time Value of Money (TVM) functions                                          }
{                                                                              }
Function an (i, n : TReal) : TReal;
  Begin
    if n = 0.0 then
      an := 0.0 else
    if i = 0.0 then
      an := n else
      an := (1.0 - Power (1.0 + i, -n)) / i;
  End;

Function aDOTn (i, n : TReal) : TReal;
  Begin
    if n = 0.0 then
      aDOTn := 0.0 else
    if i = 0.0 then
      aDOTn := n else
      aDOTn := (1.0 + i - Power (1.0 + i, -n + 1.0)) / i;
  End;

Function sn (i, n : TReal) : TReal;
  Begin
    if n = 0.0 then
      sn := 0.0 else
    if i = 0.0 then
      sn := n else
      sn := (Power (1.0 + i, n) - 1.0) / i;
  End;

Function sDOTn (i, n : TReal) : TReal;
  Begin
    if n = 0.0 then
      sDOTn := 0.0 else
    if i = 0.0 then
      sDOTn := n else
      sDOTn := (Power (1.0 + i, n + 1.0) - 1.0 - i) / i;
  End;

Function aCONTn (i, n : TReal) : TReal;
  Begin
    aCONTn := (1.0 - Power (1.0 + i, -n)) / Ln (1.0 + i);
  End;

Function Ian (i, n : TReal) : TReal;
var P : TReal;
  Begin
    P := Power (1.0 + i, n);
    Ian := (P * (1.0 + i) - 1.0 - i * (1.0 + n)) / (P * i * i);
  End;

Function IaDOTn (i, n : TReal) : TReal;
  Begin
    IaDOTn := Ian (i, n) * (1.0 + i);
  End;

Function ForceAsI (d : TReal) : TReal;
  Begin
    ForceAsI := Exp (d) - 1.0;
  End;

Function DiscountAsI (d : TReal) : TReal;
  Begin
    DiscountAsI := 1.0 / (1.0 - d) - 1.0;
  End;

Function VAsI (v : TReal) : TReal;
  Begin
    VAsI := 1.0 / v - 1.0;
  End;

Function IAsDiscount (i : TReal) : TReal;
  Begin
    IAsDiscount := i / (1.0 + i);
  End;

Function IAsForce (i : TReal) : TReal;
  Begin
    IAsForce := Ln (1.0 + i);
  End;

Function IAsV (i : TReal) : TReal;
  Begin
    IAsV := 1.0 / (1.0 + i);
  End;

Function V (i, n : TReal) : TReal;
  Begin
    V := Power (IAsV (i), n);
  End;

Function IaCONTn (i, n : TReal) : TReal;
  Begin
    IaCONTn := (aDOTn (i, n) - n * Power (1.0 + i, -n)) / Ln (1.0 + i);
  End;

Function ICONTaCONTn (i, n : TReal) : TReal;
  Begin
    ICONTaCONTn := (aCONTn (i, n) - n * Power (1.0 + i, -n)) / Ln (1.0 + i);
  End;

Function ip (i, p : TReal) : TReal;
  Begin
    ip := p * (Power (1.0 + i, 1.0 / p) - 1.0);
  End;

Function dp (i, p : TReal) : TReal;
  Begin
    dp := p * (1.0 - Power (1.0 / (1.0 + i), 1.0 / p));
  End;

Function apn (i, p, n : TReal) : TReal;
  Begin
    apn := (1 - Power (1.0 + i, -n)) / ip (i, p);
  End;

Function aDOTpn (i, p, n : TReal) : TReal;
  Begin
    aDOTpn := (1 - Power (1.0 + i, -n)) / dp (i, p);
  End;

Function spn (i, p, n : TReal) : TReal;
  Begin
    spn := (Power (1.0 + i, n) - 1.0) / ip (i, p);
  End;

Function sDOTpn (i, p, n : TReal) : TReal;
  Begin
    sDOTpn := (Power (1.0 + i, n) - 1.0) / dp (i, p);
  End;



{                                                                              }
{ TLifeTable                                                                   }
{                                                                              }
Function TLifeTable.l (const x : Integer) : TReal;
  Begin
    Result := GetItem (x);
  End;

Function TLifeTable.d (const x, n : Integer) : TReal;
  Begin
    Result := l(x) - l(x+n);
  End;

Function TLifeTable.p (const x, n : Integer) : TReal;
  Begin
    Result := l(x+n) / l(x);
  End;

Function TLifeTable.q (const x, n, m : Integer) : TReal;
  Begin
    Result := (l(x+m) - l(x+m+n)) / l(x);
  End;

Function TLifeTable.Dx (const i : TReal; const x : Integer) : TReal;
  Begin
    Result := V (i, x) * l (x);
  End;

Function TLifeTable.Nx (const i : TReal; const x : Integer) : TReal;
var F : Integer;
  Begin
    Result := 0.0;
    For F := x to Count - 1 do
      Result := Result + Dx (i, F);
  End;

Function TLifeTable.Sx (const i : TReal; const x : Integer) : TReal;
var F : Integer;
  Begin
    Result := 0.0;
    For F := x to Count - 1 do
      Result := Result + Nx (i, F);
  End;

Function TLifeTable.Cx (const i : TReal; const x : Integer) : TReal;
  Begin
    Result := V (i, x + 1) * d (x, 1);
  End;

Function TLifeTable.Mx (const i : TReal; const x : Integer) : TReal;
var F : Integer;
  Begin
    Result := 0.0;
    For F := x to Count - 1 do
      Result := Result + Cx (i, F);
  End;

Function TLifeTable.Rx (const i : TReal; const x : Integer) : TReal;
var F : Integer;
  Begin
    Result := 0.0;
    For F := x to Count - 1 do
      Result := Result + Mx (i, F);
  End;



{                                                                              }
{ Computer maths                                                               }
{                                                                              }

{ Bit functions                                                                }
{   SwapBits, LSBit, MSBit and SwapEndian taken from the Delphi Encryption     }
{   Compendium 3.0 by Hagen Reddmann (HaReddmann@AOL.COM)                      }
Function SwapBits (const Value : LongWord) : LongWord; register;
  Asm
           BSWAP  EAX
           MOV    EDX,EAX
           AND    EAX,0AAAAAAAAh
           SHR    EAX,1
           AND    EDX,055555555h
           SHL    EDX,1
           OR     EAX,EDX
           MOV    EDX,EAX
           AND    EAX,0CCCCCCCCh
           SHR    EAX,2
           AND    EDX,033333333h
           SHL    EDX,2
           OR     EAX,EDX
           MOV    EDX,EAX
           AND    EAX,0F0F0F0F0h
           SHR    EAX,4
           AND    EDX,00F0F0F0Fh
           SHL    EDX,4
           OR     EAX,EDX
  End;

Function LSBit (const Value : Integer): Integer; assembler; register;
  Asm
       BSF   EAX,EAX
  End;

Function MSBit (const Value : Integer): Integer; assembler; register;
  Asm
       BSR   EAX,EAX
  End;

Function SwapEndian (const Value : LongWord) : LongWord; assembler; register;
  Asm
       XCHG  AH,AL
       ROL   EAX,16
       XCHG  AH,AL
  End;

{ Base conversion functions -------------------------------------------------- }
const ConversionAlphabeth : String [36] = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';

Function DecodeBase (const S : String; const Base : Byte) : TInteger;
var Tot, L : TInteger;
    P, F : Byte;
  Begin
    if Base > 36 then
      raise EInvalidArgument.Create ('Invalid base');
    L := 1;
    P := Length (S);
    Tot := 0;
    Repeat
      F := Pos (UpCase (S [P]), ConversionAlphabeth);
      if (F = 0) or (F > Base) then
        raise EInvalidArgument.Create ('Invalid character');
      Tot := Tot + L * (F - 1);
      L := L * Base;
      Dec (P);
    Until P = 0;
    DecodeBase := Tot;
  End;

Function EncodeBase (const I : TInteger; const Base : Byte) : String;
var D, J : TInteger;
  Begin
    if I = 0 then
      begin
        Result := '0';
        exit;
      end;
    D := Round (Power (Base, Trunc (Log10 (I) / Log10 (Base))));
    J := I;
    Result := '';
    While D > 0 do
      begin
        Result := Result + ConversionAlphabeth [J div D + 1];
        J := J mod D;
        D := D div Base;
      end;
  End;

Function DecodeBase64 (const S, Alphabet : String) : String;
var F : Integer;
    B : Byte;
    OutPos : Byte;
    OutB : Array [1..3] of Byte;
  Begin
    if Length (Alphabet) <> 64 then
      raise EInvalidArgument.Create ('Invalid base 64 alphabet');

    Result := '';
    OutPos := 0;
    For F := 1 to Length (S) do
      begin
        B := Pos (S [F], Alphabet);
        if B = 0 then
          raise EInvalidArgument.Create ('Invalid character (#' + IntToStr (B) + ')');

        Dec (B);
        Case OutPos of
            0 : OutB [1] := B shl 2;
            1 : begin
                  OutB [1] := OutB [1] or (B shr 4);
                  Result := Result + Char (OutB [1]);
                  OutB [2] := (B shl 4) and $FF;
                end;
            2 : begin
                  OutB [2] := OutB [2] or (B shr 2);
                  Result := Result + Char (OutB [2]);
                  OutB [3] := (B shl 6) and $FF;
                end;
            3 : begin
                  OutB [3] := OutB [3] or B;
                  Result := Result + Char (OutB [3]);
                end;
          end;
        OutPos := (OutPos + 1) mod 4;
      end;

    if OutPos > 0 then
      Result := Result + Char (OutB [OutPos]);
  End;

{                                                                              }
{ Hashing functions                                                            }
{                                                                              }
{   Some speed comparisons on a P166MMX, as a reference:                       }
{     CRC16         7.5m cps                                                   }
{     CRC32         8.5m cps                                                   }
{     Checksum32   19.7m cps                                                   }
{     XOR8        126.9m cps                                                   }
{     XOR16        11.5m cps                                                   }
{     XOR32        12.3m cps                                                   }
{     MD5           1.4m cps                                                   }
{   Note the 16 bit functions are slower than the 32 bit ones.                 }
{   XOR8 has been hand optimized. It averages about 1 character every 1.3      }
{   clock ticks on a P166MMX. The same technique can be applied to Checksum32  }
{   and XOR16.                                                                 }
{   The loops in MD5 have not been unrolled for neatness' sake.                }
{                                                                              }

{ XOR-8 ---------------------------------------------------------------------- }
Function CalcXOR8 (const Data : String) : Byte; register;
  Asm
    push ebx
    mov ebx, Data                 // ebx = Data [1]
    xor eax, eax                  // al = Result
    mov ecx, [ebx - 4]            // ecx = Length (Data)
    or ecx, ecx
    jz @Fin                       // Length (Data) = 0
    cmp ecx, 4
    jb @DoRest                    // Length (Data) < 4

    // The following is an optimization of:                                   //
    //  @l1:   xor al, [ebx + ecx - 1]                                        //
    //         loop @l1                                                       //
    push ecx                                                                  //
    shr ecx, 2                    { ecx = Length (Data) div 4 }               //
  @l1:                                                                        //
    xor eax, [ebx + ecx * 4 - 4]  { Data [ecx * 4 - 3] }                      //
    dec ecx                                                                   //
    jnz @l1                                                                   //
                                                                              //
    mov ecx, eax                  { XOR bytes in eax }                        //
    xor al, ch                                                                //
    shr ecx, 16                                                               //
    xor al, cl                                                                //
    xor al, ch                                                                //
                                                                              //
    pop ecx                                                                   //
    add ebx, ecx                                                              //
    and ecx, 3                    { ecx = bytes remaining (0-3) }             //
    sub ebx, ecx                                                              //
    jz @Fin                                                                   //
  @DoRest:                                                                    //
    xor al, [ebx + ecx - 1]       { Faster than increasing ebx }              //
    dec ecx                                                                   //
    jnz @DoRest                   { Faster than loop @DoReset }               //
  @Fin:
    pop ebx
  End;

{ XOR-16 --------------------------------------------------------------------- }
Function CalcXOR16 (const Data : String) : Word;
var I : Integer;
  Begin
    Result := 0;
    For I := 1 to Length (Data) do
      Result := Result xor (Byte (Data [I]) shl (((I - 1) mod 2) * 8));
  End;

{ XOR-32 --------------------------------------------------------------------- }
Function CalcXOR32 (const Data : String) : LongWord;
var I : Integer;
  Begin
    Result := 0;
    For I := 1 to Length (Data) do
      Result := Result xor (Byte (Data [I]) shl (((I - 1) mod 4) * 8));
  End;

{ Checksum-32 ---------------------------------------------------------------- }
Function CalcChecksum32 (const Data : String) : LongWord;
  Asm                          // eax = Data [1]
      mov ecx, [eax - 4]       // ecx = length (Data)
      or ecx, ecx
      jz @fin
      push esi
      mov esi, eax             // esi = Data [1]
      xor eax, eax             // eax = CheckSum
      xor edx, ebx
    @l1:
      mov dl, [esi + ecx - 1]  // edx = Data [ecx]
      add eax, edx
      loop @l1
      pop esi
    @fin:
  End;

{ CCITT CRC-16                                                                 }
{ The theory behind CCITT V.41 CRCs: (from CRCS.DOC by Guy Hirson)             }
{                                                                              }
{      1. Select the magnitude of the CRC to be used (typically 16 or 32       }
{         bits) and choose the polynomial to use. In the case of 16 bit        }
{         CRCs, the CCITT polynomial is recommended and is                     }
{                                                                              }
{                       16    12    5                                          }
{               G(x) = x   + x   + x  + 1                                      }
{                                                                              }
{         This polynomial traps 100% of 1 bit, 2 bit, odd numbers of bit       }
{         errors, 100% of <= 16 bit burst errors and over 99% of all           }
{         other errors.                                                        }
{                                                                              }
{                                                                              }
{      2. The CRC is calculated as                                             }
{                               r                                              }
{               D(x) = (M(x) * 2 )  mod G(x)                                   }
{                                                                              }
{         This may be better described as : Add r bits (0 content) to          }
{         the end of M(x). Divide this by G(x) and the remainder is the        }
{         CRC.                                                                 }
{                                                                              }
{      3. Tag the CRC onto the end of M(x).                                    }
{                                                                              }
{      4. To check it, calculate the CRC of the new message D(x), using        }
{         the same process as in 2. above. The newly calculated CRC            }
{         should be zero.                                                      }
{                                                                              }
{   This effectively means that using CRCs, it is possible to calculate a      }
{   series of bits to tag onto the data which makes the data an exact          }
{   multiple of the polynomial.                                                }
const
  CRC16Table : Array [0..255] of Word = (
    $0000, $1021, $2042, $3063, $4084, $50a5, $60c6, $70e7,
    $8108, $9129, $a14a, $b16b, $c18c, $d1ad, $e1ce, $f1ef,
    $1231, $0210, $3273, $2252, $52b5, $4294, $72f7, $62d6,
    $9339, $8318, $b37b, $a35a, $d3bd, $c39c, $f3ff, $e3de,
    $2462, $3443, $0420, $1401, $64e6, $74c7, $44a4, $5485,
    $a56a, $b54b, $8528, $9509, $e5ee, $f5cf, $c5ac, $d58d,
    $3653, $2672, $1611, $0630, $76d7, $66f6, $5695, $46b4,
    $b75b, $a77a, $9719, $8738, $f7df, $e7fe, $d79d, $c7bc,
    $48c4, $58e5, $6886, $78a7, $0840, $1861, $2802, $3823,
    $c9cc, $d9ed, $e98e, $f9af, $8948, $9969, $a90a, $b92b,
    $5af5, $4ad4, $7ab7, $6a96, $1a71, $0a50, $3a33, $2a12,
    $dbfd, $cbdc, $fbbf, $eb9e, $9b79, $8b58, $bb3b, $ab1a,
    $6ca6, $7c87, $4ce4, $5cc5, $2c22, $3c03, $0c60, $1c41,
    $edae, $fd8f, $cdec, $ddcd, $ad2a, $bd0b, $8d68, $9d49,
    $7e97, $6eb6, $5ed5, $4ef4, $3e13, $2e32, $1e51, $0e70,
    $ff9f, $efbe, $dfdd, $cffc, $bf1b, $af3a, $9f59, $8f78,
    $9188, $81a9, $b1ca, $a1eb, $d10c, $c12d, $f14e, $e16f,
    $1080, $00a1, $30c2, $20e3, $5004, $4025, $7046, $6067,
    $83b9, $9398, $a3fb, $b3da, $c33d, $d31c, $e37f, $f35e,
    $02b1, $1290, $22f3, $32d2, $4235, $5214, $6277, $7256,
    $b5ea, $a5cb, $95a8, $8589, $f56e, $e54f, $d52c, $c50d,
    $34e2, $24c3, $14a0, $0481, $7466, $6447, $5424, $4405,
    $a7db, $b7fa, $8799, $97b8, $e75f, $f77e, $c71d, $d73c,
    $26d3, $36f2, $0691, $16b0, $6657, $7676, $4615, $5634,
    $d94c, $c96d, $f90e, $e92f, $99c8, $89e9, $b98a, $a9ab,
    $5844, $4865, $7806, $6827, $18c0, $08e1, $3882, $28a3,
    $cb7d, $db5c, $eb3f, $fb1e, $8bf9, $9bd8, $abbb, $bb9a,
    $4a75, $5a54, $6a37, $7a16, $0af1, $1ad0, $2ab3, $3a92,
    $fd2e, $ed0f, $dd6c, $cd4d, $bdaa, $ad8b, $9de8, $8dc9,
    $7c26, $6c07, $5c64, $4c45, $3ca2, $2c83, $1ce0, $0cc1,
    $ef1f, $ff3e, $cf5d, $df7c, $af9b, $bfba, $8fd9, $9ff8,
    $6e17, $7e36, $4e55, $5e74, $2e93, $3eb2, $0ed1, $1ef0);

Function CalcCCITT_CRC16 (const Octet : Byte; const CRC16 : Word) : Word;
  Begin
    Result := CRC16Table [Hi (CRC16) xor Octet] xor (CRC16 * 256);
  End;

Function CalcCCITT_CRC16 (const Data : String) : Word;
var I : Integer;
  Begin
    Result := $FFFF;
    For I := 1 to Length (Data) do
      Result := CalcCCITT_CRC16 (Byte (Data [I]), Result);
  End;

{ CRC-32 --------------------------------------------------------------------- }
var
  CRC32TableInit : Boolean = False;
  CRC32Table     : Array [$00..$FF] of LongWord;

Procedure InitCRC32Table;
var I, J : Byte;
    R    : LongWord;
const CRCPoly = $EDB88320;
  Begin
    For I := $00 to $FF do
      begin
        R := I;
        For J := 8 downto 1 do
          if (R and 1) <> 0 then
            R := (R shr 1) xor CRCPoly else
            R := R shr 1;
        CRC32Table [I] := R;
      end;
  End;

Function CalcCRC32 (const Octet : Byte; const CRC32 : LongWord) : LongWord;
  Begin
    if not CRC32TableInit then // First call initializes the lookup table
      begin
        InitCRC32Table;
        CRC32TableInit := True;
      end;
    Result := CRC32Table [Byte (CRC32 xor LongWord (Octet))] xor ((CRC32 shr 8) and $00FFFFFF);
  End;

Function CalcCRC32 (const Data : String) : LongWord;
var I : Integer;
  Begin
    Result := $FFFFFFFF;
    For I := 1 to Length (Data) do
      CalcCRC32 (Byte (Data [I]), Result);
    Result := not Result;
  End;

{ MD5 ------------------------------------------------------------------------ }
const
  MD5Table_1 : Array [0..15] of LongWord = (
      $D76AA478, $E8C7B756, $242070DB, $C1BDCEEE,
      $F57C0FAF, $4787C62A, $A8304613, $FD469501,
      $698098D8, $8B44F7AF, $FFFF5BB1, $895CD7BE,
      $6B901122, $FD987193, $A679438E, $49B40821);
  MD5Table_2 : Array [0..15] of LongWord = (
      $F61E2562, $C040B340, $265E5A51, $E9B6C7AA,
      $D62F105D, $02441453, $D8A1E681, $E7D3FBC8,
      $21E1CDE6, $C33707D6, $F4D50D87, $455A14ED,
      $A9E3E905, $FCEFA3F8, $676F02D9, $8D2A4C8A);
  MD5Table_3 : Array [0..15] of LongWord = (
      $FFFA3942, $8771F681, $6D9D6122, $FDE5380C,
      $A4BEEA44, $4BDECFA9, $F6BB4B60, $BEBFBC70,
      $289B7EC6, $EAA127FA, $D4EF3085, $04881D05,
      $D9D4D039, $E6DB99E5, $1FA27CF8, $C4AC5665);
  MD5Table_4 : Array [0..15] of LongWord = (
      $F4292244, $432AFF97, $AB9423A7, $FC93A039,
      $655B59C3, $8F0CCC92, $FFEFF47D, $85845DD1,
      $6FA87E4F, $FE2CE6E0, $A3014314, $4E0811A1,
      $F7537E82, $BD3AF235, $2AD7D2BB, $EB86D391);

{ Calculates new MD5 Digest (Array [0..3] of LongWord) given a Buffer          }
{   (Array [0..15] of LongWord).                                               }
Procedure TransformMD5Buffer (var Digest : Array of LongWord; const Buffer : Array of LongWord);
var A, B, C, D : LongWord;
    I          : Integer;
    J          : Byte;
  Begin
    A := Digest [0];
    B := Digest [1];
    C := Digest [2];
    D := Digest [3];

    For I := 0 to 3 do
      begin
        J := I * 4;
        Inc (A, Buffer [J]     + MD5Table_1 [J]     + (D xor (B and (C xor D)))); A := A shl  7 or A shr 25 + B;
        Inc (D, Buffer [J + 1] + MD5Table_1 [J + 1] + (C xor (A and (B xor C)))); D := D shl 12 or D shr 20 + A;
        Inc (C, Buffer [J + 2] + MD5Table_1 [J + 2] + (B xor (D and (A xor B)))); C := C shl 17 or C shr 15 + D;
        Inc (B, Buffer [J + 3] + MD5Table_1 [J + 3] + (A xor (C and (D xor A)))); B := B shl 22 or B shr 10 + C;
      end;

    For I := 0 to 3 do
      begin
        J := I * 4;
        Inc (A, Buffer [J + 1]           + MD5Table_2 [J]     + (C xor (D and (B xor C)))); A := A shl  5 or A shr 27 + B;
        Inc (D, Buffer [(J + 6) mod 16]  + MD5Table_2 [J + 1] + (B xor (C and (A xor B)))); D := D shl  9 or D shr 23 + A;
        Inc (C, Buffer [(J + 11) mod 16] + MD5Table_2 [J + 2] + (A xor (B and (D xor A)))); C := C shl 14 or C shr 18 + D;
        Inc (B, Buffer [J]               + MD5Table_2 [J + 3] + (D xor (A and (C xor D)))); B := B shl 20 or B shr 12 + C;
      end;

    For I := 0 to 3 do
      begin
        J := 16 - (I * 4);
        Inc (A, Buffer [(J + 5) mod 16]  + MD5Table_3 [16 - J]     + (B xor C xor D)); A := A shl  4 or A shr 28 + B;
        Inc (D, Buffer [(J + 8) mod 16]  + MD5Table_3 [16 - J + 1] + (A xor B xor C)); D := D shl 11 or D shr 21 + A;
        Inc (C, Buffer [(J + 11) mod 16] + MD5Table_3 [16 - J + 2] + (D xor A xor B)); C := C shl 16 or C shr 16 + D;
        Inc (B, Buffer [(J + 14) mod 16] + MD5Table_3 [16 - J + 3] + (C xor D xor A)); B := B shl 23 or B shr  9 + C;
      end;

    For I := 0 to 3 do
      begin
        J := 16 - (I * 4);
        Inc (A, Buffer [J mod 16]        + MD5Table_4 [16 - J]     + (C xor (B or not D))); A := A shl  6 or A shr 26 + B;
        Inc (D, Buffer [(J + 7) mod 16]  + MD5Table_4 [16 - J + 1] + (B xor (A or not C))); D := D shl 10 or D shr 22 + A;
        Inc (C, Buffer [(J + 14) mod 16] + MD5Table_4 [16 - J + 2] + (A xor (D or not B))); C := C shl 15 or C shr 17 + D;
        Inc (B, Buffer [(J + 5) mod 16]  + MD5Table_4 [16 - J + 3] + (D xor (C or not A))); B := B shl 21 or B shr 11 + C;
      end;

    Inc (Digest [0], A);
    Inc (Digest [1], B);
    Inc (Digest [2], C);
    Inc (Digest [3], D);
  End;

Function MD5InitKey : LongWordArray;
  Begin
    SetLength (Result, 4);
    Result [0] := $67452301;        // fixed initialization key
    Result [1] := $EFCDAB89;
    Result [2] := $98BADCFE;
    Result [3] := $10325476;
  End;

Function CalcMD5 (const Data : AStream) : LongWordArray;
var S : String;
  Begin
    Result := MD5InitKey;

    Data.Reset;
    if Data.EOF then
      exit;

    While not Data.EOF do
      begin
        S := Data.Read (64);
        if Length (S) < 64 then
          break;
        TransformMD5Buffer (Result, LongWordArray (S));
      end;
    S := S + #$80;
    if Length (S) > 64 - Sizeof (Int64) then
      begin
        S := PadRight (S, #0, 64);
        TransformMD5Buffer (Result, LongWordArray (S));
        S := '';
      end;
    S := PadRight (S, #0, 64 - Sizeof (Int64));
    S := S + Pack (Data.Position * 8);
    TransformMD5Buffer (Result, LongWordArray (S));
  End;

Function CalcMD5 (const Data : String) : LongWordArray;
var S : TMemoryStream;
  Begin
    S := TMemoryStream.Create (Data);
    try
      Result := CalcMD5 (S);
    finally
      S.Free;
    end;
  End;

{ Hash ----------------------------------------------------------------------- }
Function Hash (const S : String; const Slots : Integer) : Integer;
  Begin
    Result := CalcCRC32 (S) mod Slots;
  End;


{                                                                              }
{ Fast factorial                                                               }
{                                                                              }
{   For small values of N, calculate normally using 2*3*..*N                   }
{   For larger values of N, calculate using Gamma (N+1)                        }
{   Where N is small the calculated values are cached to avoid recalculation.  }
{                                                                              }
const
  FactorialCacheLimit = 409;

var
  FactorialCacheInit : Boolean = False;
  FactorialCache : Array [0..FactorialCacheLimit] of TReal;

Function Factorial (const N : TInteger) : TReal;
const
  MaxLimit = 1754.0;
  SwitchLimit = 34;
var
  L : TReal;
  I : Integer;

  Begin
    if N > MaxLimit then
      raise EOverflow.Create ('') else
      if N < 0 then
        raise EInvalidArgument.Create ('') else
        if (N <= FactorialCacheLimit) and FactorialCacheInit and (FactorialCache [N] >= 1.0) then
          Result := FactorialCache [N] else
          begin
            if N < SwitchLimit then
              begin
                L := 1.0;
                I := 2;
                While I <= N do
                  begin
                    L := L * I;
                    Inc (I);
                  end;
                Result := L;
              end else
              Result := Exp (GammLn (N + 1));

            if N <= FactorialCacheLimit then
              begin
                if not FactorialCacheInit then
                  begin
                    FillChar (FactorialCache, FactorialCacheLimit * Sizeof (TReal), #0);
                    FactorialCacheInit := True;
                  end;
                FactorialCache [N] := Result;
              end;
          end;
  End;





{                                                                              }
{ Combinatorics                                                                }
{                                                                              }
Function Combinations (const N, C : TInteger) : TReal;
  Begin
    Result := Factorial (N) / (Factorial (C) * Factorial (N - C));
  End;

Function Permutations (const N, P : TInteger) : TReal;
  Begin
    Result := Factorial (N) / Factorial (N - P);
  End;



{                                                                              }
{ Trig                                                                         }
{                                                                              }
Function InverseTangentDeg (const X, Y : TReal) : TReal;
{ 0 <= Result <= 360 }
var Angle : TReal;
  Begin
    if RealZero (X) then
      Angle := Pi / 2.0 else
      Angle := ArcTan (Y / X);
    Angle := Angle * 180.0 / Pi;

    if (X <= 0.0) and (Y < 0.0) then
      Angle := Angle - 180.0 else
    if (X < 0.0) and (Y > 0.0) then
      Angle := Angle + 180.0;

    If Angle < 0.0 then
      Angle := Angle + 360.0;

    InverseTangentDeg := Angle;
  End;

Function InverseTangentRad (const X, Y : TReal) : TReal;
{ 0 <= result <= 2pi }
var Angle : TReal;
  Begin
    if RealZero (X) then
      Angle := Pi / 2.0 else
      Angle := ArcTan (Y / X);
    if (X <= 0.0) and (Y < 0) then
      Angle := Angle - Pi;
    if (X < 0.0) and (Y > 0) then
      Angle := Angle + Pi;
    If Angle < 0 then
      Angle := Angle + 2 * Pi;
    InverseTangentRad := Angle;
  End;

Function InverseSinDeg (const Y, R : TReal) : TReal;
{ -90 <= result <= 90 }
var X : TReal;
  Begin
    X := Sqrt (Sqr (R) - Sqr (Y));
    Result := InverseTangentDeg (X, Y);
    If Result > 90.0 then
      Result := Result - 360.0;
  End;

Function InverseSinRad (const Y, R : TReal) : TReal;
{ -90 <= result <= 90 }
var X : TReal;
  Begin
    X := Sqrt (Sqr (R)-Sqr (Y));
    Result := InverseTangentRad (X, Y);
    if Result > 90.0 then
      Result := Result - 360.0;
  End;

Function InverseCosDeg (const X, R : TReal) : TReal;
{ -90 <= result <= 90 }
var Y : TReal;
  Begin
    Y := Sqrt (Sqr (R)-Sqr (X));
    Result := InverseTangentDeg(X, Y);
    if Result > 90.0 then
      Result := Result - 360.0;
  End;

Function InverseCosRad (const X, R : TReal) : TReal;
{ -90 <= result <= 90 }
var Y : TReal;
  Begin
    Y := Sqrt (Sqr (R)-Sqr (X));
    Result := InverseTangentRad (X, Y);
    if Result > 90.0 then
      Result := Result - 360.0;
  End;

Function ATan360 (const X, Y : TReal) : TReal;
var Angle: TReal;
  Begin
    if RealZero (X) then
      Angle := Pi / 2.0 else
      Angle := ArcTan (Y / X);
    Angle := Angle * OneRad;
    if (X <= 0.0) and (Y < 0.0) then
      Angle := Angle - 180.0;
    if (X < 0.0) and (Y > 0.0) then
      Angle := Angle + 180.0;
    If Angle < 0.0 then
      Angle := Angle + 360.0;
    ATan360 := Angle;
  End;




{                                                                              }
{ Statistical functions                                                        }
{                                                                              }
Function RandomSeed : LongWord;
var I            : Int64;
    Ye, Mo, Da   : Word;
    H, Mi, S, S1 : Word;
  Begin
    DecodeDate (Date, Ye, Mo, Da);
    Result := Ye xor (Mo shl 16) xor (Da shl 24);
    Result := Result xor GetTickCount;
    if QueryPerformanceFrequency (I) then
      Result := Result xor LongWord (I) xor LongWord (I shr 32);
    if QueryPerformanceCounter (I) then
      Result := Result xor LongWord (I) xor LongWord (I shr 32);
    DecodeTime (Time, H, Mi, S, S1);
    Result := Result xor H xor (Mi shl 8) xor (S1 shl 16) xor (S shl 24);
  End;

Function BinomialCoeff (N, R : TInteger) : TReal;
var I, K : Integer;
  Begin
    if (N = 0) or (R > N) then
      raise EInvalidArgument.Create ('Invalid parameters to BinomialCoeff');
    if N > 1547 then
      raise EOverflow.Create ('BinomialCoeff overflow');

    Result := 1.0;
    if (R = 0) or (R = N) then
     exit;

    if R > N div 2 then
     R := N - R;

    K := 2;
    For I := N - R + 1 to N do
      begin
	Result := Result * I;
	if K <= R then
	  begin
	    Result := Result / K;
	    Inc (K);
          end;
      end;
    Result := Int (Result + 0.5);
  End;



{ Random number generator from ACM Transactions on Modeling and Computer       }
{ Simulation 8(1) 3-30, 1990.  Supposedly it has a period of -1 + 2^19937.     }
{ The original was in C; this translation returns the same values as the       }
{ original.  It is called the Mersenne Twister.                                }
{ The following code was written by Toby Ewing <ewing@iastate.edu>, slightly   }
{ modified by Frank Heckenbach <frank@pascal.gnu.de> and again slightly        }
{ modified by David Butler <david@e.co.za> for use in Delphi. It was inspired  }
{ by C code, released under the GNU Library General Public License, written by }
{ Makoto Matsumoto <matumoto@math.keio.ac.jp> and Takuji Nishimura,            }
{ considering the suggestions by Topher Cooper and Marc Rieffel in July-       }
{ Aug 97.                                                                      }
const
  N = 624; // Period parameters
  M = 397;

var
  mti : Integer;
  mt  : Array [0 .. N] of LongWord; // the array for the state vector
  RandomUniformInitialized : Boolean = False;

Procedure RandomUniformInit (const Seed : LongWord);
{ Set initial seeds to mt [N] using the generator Line 25 of Table 1 in        }
{ [KNUTH 1981, The Art of Computer Programming Vol. 2 (2nd Ed.), pp 102].      }
  Begin
    mt [0] := Seed;
    For mti := 1 to N do
      mt [mti] := (69069 * mt [mti - 1]) and $FFFFFFFF;
    mti := N;
    RandomUniformInitialized := True
  End;

Function RandomUniform : LongWord;
const
  Matrix_A = $9908B0DF; // constant vector a
  T_Mask_B = $9D2C5680; // Tempering parameters
  T_Mask_C = $EFC60000;
  Up_Mask  = $80000000; // most significant w-r bits
  Low_Mask = $7FFFFFFF; // least significant r bits
  mag01    : Array [0..1] of LongWord = (0, Matrix_A);

var
  y  : LongWord;
  kk : Integer;

  Begin
    if not RandomUniformInitialized then
      RandomUniformInit (RandomSeed);

    if mti >= N then { generate N words at one time }
      begin
        For kk := 0 to N - M do
          begin
            y := (mt [kk] and Up_Mask) or (mt [kk + 1] and Low_Mask);
            mt [kk] := mt [kk + M] xor (y shr 1) xor mag01 [y and 1]
          end;
        For kk := N - M to N - 1 do
          begin
            y := (mt [kk] and Up_Mask) or (mt [kk + 1] and Low_Mask);
            mt [kk] := mt [kk + M - N] xor (y shr 1) xor mag01 [y and 1]
          end;
        y := (mt [N - 1] and Up_Mask) or (mt [0] and Low_Mask);
        mt [N - 1] := mt [M - 1] xor (y shr 1) xor mag01 [y and 1];
        mti := 0
      end;
    y := mt [mti];
    Inc (mti);
    y := y xor (y shr 11);
    y := y xor ((y shl 7) and T_Mask_B);
    y := y xor ((y shl 15) and T_Mask_C);
    y := y xor (y shr 18);
    RandomUniform := y;
  End;

Function RandomUniformF : TReal;
  Begin
    Result := RandomUniform / High (LongWord);
  End;


{ Calculates polynomial of degree N:        }
{                                           }
{                      2          N         }
{  y  =  C  + C x + C x  +...+ C x          }
{         0    1     2          N           }
{  Coefficients are stored in reverse order }
Function PolEvl (X : TReal; var Coef : array of TReal; N : Integer) : TReal;
var Ans, P : TReal;
  Begin
    P := 1.0;
    Ans := 0.0;
    While N >= 0 do
      begin
        Ans := Ans + Coef [N] * P;
        P := P * X;
        Dec (N);
      end;
    PolEvl := Ans;
  End;



{ For arguments greater than 13, the logarithm of the gamma       }
{ function is approximated by the logarithmic version of          }
{ Stirling's formula using a polynomial approximation of          }
{ degree 4. Arguments between -33 and +33 are reduced by          }
{ recurrence to the interval [2,3] of a rational approximation.   }
{ The cosecant reflection formula is employed for arguments       }
{ less than -33.                                                  }
{                                                                 }
{ Arguments greater than MAXLGM return MAXNUM and an error        }
{ message.                                                        }
Function GammLn (X : TReal) : TReal;
const MaxLGM = 2.556348e305;
var P, Q, W, Z : TReal;
{ Stirling's formula expansion of log gamma }
const Stir : Array [0..4] of TReal = (
              8.11614167470508450300E-4,
             -5.95061904284301438324E-4,
              7.93650340457716943945E-4,
             -2.77777777730099687205E-3,
              8.33333333333331927722E-2);
{ B[], C[]: log gamma function between 2 and 3 }
      B    : Array [0..5] of TReal = (
             -1.37825152569120859100E3,
             -3.88016315134637840924E4,
             -3.31612992738871184744E5,
             -1.16237097492762307383E6,
             -1.72173700820839662146E6,
             -8.53555664245765465627E5);
      C    : Array [0..7] of TReal = (
              1.00000000000000000000E0,
             -3.51815701436523470549E2,
             -1.70642106651881159223E4,
             -2.20528590553854454839E5,
             -1.13933444367982507207E6,
             -2.53252307177582951285E6,
             -2.01889141433532773231E6,
              1.00000000000000000000E0);

  Begin
    if X < -34.0 then
      begin
	Q := -X;
	W := GammLn (Q);
	P := Trunc (Q);
        if P = Q then
          raise EOverflow.Create ('GammaLn') else
          begin
            Z := Q - P;
            if Z > 0.5 then
              begin
                P := P + 1.0;
                Z := P - Q;
              end;
            Z := Q * Sin (Pi * Z);
            if Z = 0.0 then
              raise EOverflow.Create ('GammaLn') else
              GammLn := LnPi - Ln (Z) - W;
          end;
      end else
    if X <= 13 then
      begin
	Z := 1.0;
        While X >= 3.0 do
          begin
	    X := X - 1.0;
            Z := Z * X;
          end;
        While (X < 2.0) and (X <> 0.0) do
          begin
            Z := Z / X;
            X := X + 1.0;
          end;
        if X = 0.0 then
          raise EOverflow.Create ('GammaLn') else
	if Z < 0.0 then
          Z := -Z;
        if X = 2.0 then
          GammLn := Ln (Z) else
          begin
            X := X - 2.0;
            P := X * PolEvl (X, B, 5) / PolEvl (X, C, 7);
            GammLn := Ln (Z) + P;
          end;
      end else
    if X > MAXLGM then
      raise EOverflow.Create ('GammaLn') else
      begin
        Q := (X - 0.5) * Ln (X) - X + LnSqrt2Pi;
        if X > 1.0e8 then
          GammLn := Q else
          begin
            P := 1.0 / (X * X);
            if X >= 1000.0 then
              GammLn := Q + ((7.9365079365079365079365e-4 * P
                            - 2.7777777777777777777778e-3)
                            * P + 0.0833333333333333333333) / X else
              GammLn := Q + PolEvl (P, Stir, 4) / X;
          end;
      end;
  End;

{ GSER  gamma function, incomplete, series evaluation }
Procedure gser (const a, x : TReal; var gamser, gln : TReal);
const itmax = 100;
      eps   = 3.0e-7;
var n : Integer;
    sum, del, ap : TReal;
  Begin
    gln := gammln (a);
    if RealZero (x) then
      GamSer := 0.0 else
      if X < 0.0 then
        raise EUnderflow.Create ('gser') else
        begin
          ap := a;
          sum := 1.0 / a;
          del := sum;
          for n := 1 to itmax do
            begin
              ap := ap + 1.0;
              del := del * x / ap;
              sum := sum + del;
              if abs (del) < abs (sum) * eps then
                begin
                  GamSer := sum * exp (-x + a * ln(x) - gln);
                  exit;
                end;
            end;
          raise EOverflow.Create ('gser: A too large, itmax too small');
        end;
  End;

{ GCF  gamma function, incomplete, continued fraction evaluation }
Procedure gcf (const a, x : TReal; var gammcf, gln : TReal);
const itmax = 100;
      eps   = 3.0e-7;
var n : integer;
    gold, g, fac, b1, b0, anf, ana, an, a1, a0 : TReal;
  Begin
    gln := gammln (a);
    gold := 0.0;
    g := 0.0;
    a0 := 1.0;
    a1 := x;
    b0 := 0.0;
    b1 := 1.0;
    fac := 1.0;
    For n := 1 to itmax do
      begin
        an := 1.0 * n;
        ana := an - a;
        a0 := (a1 + a0 * ana) * fac;
        b0 := (b1 + b0 * ana) * fac;
        anf := an * fac;
        a1 := x * a0 + anf * a1;
        b1 := x * b0 + anf * b1;
        if not RealZero (a1) then
          begin
            fac := 1.0 / a1;
            g := b1 * fac;
            if abs ((g - gold) / g) < eps then
              break;
            gold := g;
          end;
      end;
    Gammcf := exp (-x + a * ln (x) - gln) * g;
  End;

{ GAMMP  gamma function, incomplete }
Function GammP (const a,x : TReal) : TReal;
var gammcf, gln : TReal;
  Begin
    if (x < 0.0) or (a <= 0.0) then
      raise EInvalidArgument.Create ('GammP') else
      if x < a + 1.0 then
        begin
          gser (a, x, gammcf, gln);
          gammp := gammcf
        end else
        begin
          gcf (a, x, gammcf, gln);
          gammp := 1.0 - gammcf
        end;
  End;

{ GAMMQ  gamma function, incomplete, complementary }
Function gammq (const a, x : TReal) : TReal;
var gamser, gln : TReal;
  Begin
    if (x < 0.0) or (a <= 0.0) then
      raise EInvalidArgument.Create ('gammq');
    if x < a + 1.0 then
      begin
        gser (a, x, gamser, gln);
        Result := 1.0 - gamser;
      end else
      begin
        gcf (a, x, gamser, gln);
        Result := gamser
      end;
  End;

{ ERFC  error function, complementary }
Function erfc (const x : TReal) : TReal;
  Begin
    if x < 0.0 then
      erfc := 1.0 + gammp (0.5, sqr (x)) else
      erfc := gammq (0.5, sqr(x));
  End;

Function CummNormal (const u, s, X : TReal) : TReal;
  Begin
    CummNormal := ERFC (((X - u) / s) / Sqrt2) / 2.0;
  End;

Function CummNormal01 (const X : TReal) : TReal;
  Begin
    CummNormal01 := ERFC (X / Sqrt2) / 2.0;
  End;

Function CummChiSquare (const Chi, Df : TReal) : TReal;
  Begin
    CummChiSquare := 1.0 - gammq (0.5 * Df, 0.5 * Chi);
  End;

{ BETACF  beta function, incomplete, continued fraction evaluation }
Function betacf (const a, b, x : TReal) : TReal;
const itmax = 100;
      eps   = 3.0e-7;
var tem, qap, qam, qab, em, d : TReal;
    bz, bpp, bp, bm, az, app  : TReal;
    am, aold, ap              : TReal;
    m                         : Integer;
  Begin
    am := 1.0;
    bm := 1.0;
    az := 1.0;
    qab := a + b;
    qap := a + 1.0;
    qam := a - 1.0;
    bz := 1.0 - qab * x / qap;
    For m := 1 to itmax do
      begin
        em := m;
        tem := em + em;
        d := em * (b - m) * x / ((qam + tem) * (a + tem));
        ap := az + d * am;
        bp := bz + d * bm;
        d := -(a + em) * (qab + em) * x / ((a + tem) * (qap + tem));
        app := ap + d * az;
        bpp := bp + d * bz;
        aold := az;
        am := ap / bpp;
        bm := bp / bpp;
        az := app / bpp;
        bz := 1.0;
        if abs (az - aold) < eps * abs (az) then
          begin
            Result := az;
            exit;
          end;
      end;
    raise EOverflow.Create ('betacf: A or B too big or itmax too small');
  End;

{ BETAI  beta function, incomplete }
Function betai (const a, b, x : TReal) : TReal;
var bt : TReal;
  Begin
    if (x < 0.0) or (x > 1.0) then
      raise EInvalidArgument.Create ('betai');

    if RealZero (x) or RealEqual (x, 1.0) then
      bt := 0.0 else
      bt := exp (GammLn (a + b) - GammLn (a) - GammLn (b) + a * ln (x) + b * ln (1.0 - x));
    if x < (a + 1.0) / (a + b + 2.0) then
      Result := bt * betacf (a, b, x) / a else
      Result := 1.0 - bt * betacf (b, a, 1.0 - x) / b;
  End;

Function CumF (const f, Df1, Df2 : TReal) : TReal;
  Begin
    if F <= 0.0 then
      raise EInvalidArgument.Create ('') else
      CumF := 1.0 - (betai (0.5 * df2, 0.5 * df1, df2 / (df2 + df1 * f))
           + (1.0 - betai (0.5 * df1, 0.5 * df2, df1 / (df1 + df2 / f)))) / 2.0;
  End;



{                                                                   }
{ Returns the argument, x, for which the area under the             }
{ Gaussian probability density function (integrated from            }
{ minus infinity to x) is equal to y.                               }
{                                                                   }
{ For small arguments 0 < y < exp(-2), the program computes         }
{ z = sqrt( -2.0 * log(y) );  then the approximation is             }
{ x = z - log(z)/z  - (1/z) P(1/z) / Q(1/z).                        }
{ There are two rational functions P/Q, one for 0 < y < exp(-32)    }
{ and the other for y up to exp(-2).  For larger arguments,         }
{ w = y - 0.5, and  x/sqrt(2pi) = w + w**3 R(w**2)/S(w**2)).        }
{                                                                   }
Function InvCummNormal01 (Y0 : TReal) : TReal;
const P0 : Array [0..4] of TReal = (
           -5.99633501014107895267e1,
            9.80010754185999661536e1,
           -5.66762857469070293439e1,
            1.39312609387279679503e1,
           -1.23916583867381258016e0);
      Q0 : Array [0..8] of TReal = (
            1.00000000000000000000e0,
            1.95448858338141759834e0,
            4.67627912898881538453e0,
            8.63602421390890590575e1,
           -2.25462687854119370527e2,
            2.00260212380060660359e2,
           -8.20372256168333339912e1,
            1.59056225126211695515e1,
           -1.18331621121330003142e0);
      P1 : Array [0..8] of TReal = (
            4.05544892305962419923e0,
            3.15251094599893866154e1,
            5.71628192246421288162e1,
            4.40805073893200834700e1,
            1.46849561928858024014e1,
            2.18663306850790267539e0,
           -1.40256079171354495875e-1,
           -3.50424626827848203418e-2,
           -8.57456785154685413611e-4);
      Q1 : Array [0..8] of TReal = (
            1.00000000000000000000e0,
            1.57799883256466749731e1,
            4.53907635128879210584e1,
            4.13172038254672030440e1,
            1.50425385692907503408e1,
            2.50464946208309415979e0,
           -1.42182922854787788574e-1,
           -3.80806407691578277194e-2,
           -9.33259480895457427372e-4);
      P2 : Array [0..8] of TReal = (
            3.23774891776946035970e0,
            6.91522889068984211695e0,
            3.93881025292474443415e0,
            1.33303460815807542389e0,
            2.01485389549179081538e-1,
            1.23716634817820021358e-2,
            3.01581553508235416007e-4,
            2.65806974686737550832e-6,
            6.23974539184983293730e-9);
      Q2 : Array [0..8] of TReal = (
            1.00000000000000000000e0,
            6.02427039364742014255e0,
            3.67983563856160859403e0,
            1.37702099489081330271e0,
            2.16236993594496635890e-1,
            1.34204006088543189037e-2,
            3.28014464682127739104e-4,
            2.89247864745380683936e-6,
            6.79019408009981274425e-9);
var X, Z, Y2, X0, X1 : TReal;
    Code             : Boolean;
  Begin
    if Y0 <= 0.0 then
      raise EUnderflow.Create ('InvCummNormal01') else
      if Y0 >= 1.0 then
        raise EOverflow.Create ('InvCummNormal01') else
        begin
          Code := True;
          if Y0 > 1.0 - ExpM2 then
            begin
              Y0 := 1.0 - Y0;
              Code := False;
            end;
          if Y0 > ExpM2 then
            begin
              Y0 := Y0 - 0.5;
              Y2 := Y0 * Y0;
              X := Y0 + Y0 * (Y2 * PolEvl (Y2, P0, 4) / PolEvl (Y2, Q0, 8));
              InvCummNormal01 := X * Sqrt2Pi;
            end else
            begin
              X := Sqrt (-2.0 * Ln (Y0));
              X0 := X - Ln (X) / X;
              Z := 1.0 / X;
              if X < 8.0 then
                X1 := Z * PolEvl (Z, P1, 8) / PolEvl (Z, Q1, 8) else
                X1 := Z * PolEvl (Z, P2, 8) / PolEvl (Z, Q2, 8);
              X := X0 - X1;
              if Code then
                X := -X;
              InvCummNormal01 := X;
            end;
        end;
  End;

Function InvCummNormal (const u, s, Y0 : TReal) : TReal;
  Begin
    InvCummNormal := InvCummNormal01 (Y0) * s + u;
  End;

Function CummPoisson (const X : TInteger; const u : TReal) : TReal;
  Begin
    CummPoisson := GammQ (X + 1, u);
  End;





initialization
finalization
  PrimeSet.Free;
  PrimeSet := nil;
End. { Math }









(*================================================================================
================================================================================
================================================================================


{                                                                              }
{ HugeIntAdd                                                                   }
{   Handle arbitrary size integers.                                            }
{   Count1 and Count2 is the number of integers in X and Y. X must be big      }
{   enough to handle the result. At the moment both X and Y must be at least   }
{   max (count1, count2) big.                                                  }
{                                                                              }
Procedure HugeIntAdd (const X, Y : PIntegerList; var Count1, Count2 : Integer); assembler;
  Asm
    push ebx
    push esi
    push edi

    mov ecx, Count1
    mov ebx, Count2
    mov ecx, [ecx]
    cmp ecx, [ebx]
    jae @l1
    mov ecx, [ebx]
    @l1:

    mov edi, X
    mov esi, Y
    clc
    mov ebx, 0
    mov edx, ecx

    @l2:
    mov eax, [esi+ebx*4]
    adc [edi+ebx*4], eax
    inc ebx
    loop @l2

    adc edx, 0
    mov ebx, Count1
    mov [ebx], edx

    pop edi
    pop esi
    pop ebx
  End;

FROM CALCPI.PAS: (For the HugeInt Functions)

  PROCEDURE mullong(VAR a : long; mul : word; VAR answer : long);

    BEGIN { mullong   }
    inline($fc/                  { CLD                      }
           $1e/                  { PUSH    DS               }
           $8e/$5e/<a+2/         { MOV     DS,[BP+0C]       }
           $8b/$5e/<mul/         { MOV     BX,[BP+08]       }
           $c4/$7e/<answer/      { LES     DI,[BP+04]       }
           $8b/$0e/$00/$00/      { MOV     CX,[0000]        }
           $33/$d2/              { XOR     DX,DX            }
           $8b/$f2/              { MOV     SI,DX            }
           $47/                  { INC     DI               }
           $47/                  { INC     DI               }
           $f8/                  { CLC                      }
           $8b/$05/              { MOV     AX,[DI]          }
           $9c/                  { PUSHF                    }
           $f7/$e3/              { MUL     BX               }
           $9d/                  { POPF                     }
           $13/$c6/              { ADC     AX,SI            }
           $ab/                  { STOSW                    }
           $8b/$f2/              { MOV     SI,DX            }
           $e2/$f3/              { LOOP    0114             }
           $83/$d6/$00/          { ADC     SI,+00           }
           $26/$89/$35/          { MOV     ES:[DI],SI       }
           $1f);                 { POP     DS               }
    WITH answer DO
      IF dat[a.len+1] = 0 THEN len := a.len
      ELSE len := a.len + 1;
    END; { mullong   }

  { 1---------------1   }

{================================================================================
================================================================================
================================================================================
  PROCEDURE divlong(VAR a : long; del : word;
                    VAR answer : long; VAR remainder : word);

    BEGIN { divlong   }
    inline($fd/                  { STD                      }
           $1e/                  { PUSH    DS               }
           $8e/$5e/<a+2/         { MOV     DS,[BP+10]       }
           $8b/$5e/<del/         { MOV     BX,[BP+0C]       }
           $c4/$7e/<answer/      { LES     DI,[BP+08]       }
           $8b/$0e/$00/$00/      { MOV     CX,[0000]        }
           $03/$f9/              { ADD     DI,CX            }
           $03/$f9/              { ADD     DI,CX            }
           $33/$d2/              { XOR     DX,DX            }
           $8b/$05/              { MOV     AX,[DI]          }
           $f7/$f3/              { DIV     BX               }
           $ab/                  { STOSW                    }
           $e2/$f9/              { LOOP    0117             }
           $c5/$76/<remainder/   { LDS     SI,[BP+04]       }
           $89/$14/              { MOV     [SI],DX          }
           $1f);                 { POP     DS               }
    WITH answer DO BEGIN
      len := a.len;
      WHILE (dat[len] = 0) AND (len > 1) DO len := len - 1; END;
    END; { divlong   }

  { 1---------------1   }
{================================================================================
================================================================================
================================================================================

  PROCEDURE sublong(VAR answer, add : long);

    BEGIN { sublong   }
    inline($fc/                  { CLD                      }
           $1e/                  { PUSH    DS               }
           $8e/$5e/<answer+2/    { MOV     DS,[BP+08]       }
           $c4/$76/<add/         { LES     SI,[BP+06]       }
           $26/$8b/$0e/$00/$00/  { MOV     CX,ES:[0000]     }
           $83/$c6/$02/          { ADD     SI,2             }
           $f8/                  { CLC                      }
           $26/$ad/              { LODSW   ES:              }
           $19/$44/$fe/          { SBB     [SI-02],AX       }
           $e2/$f9/              { LOOP    012D             }
           $73/$07/              { JNB     013D             }
           $83/$2c/$01/          { SUB Word Ptr [SI],1      }
           $46/                  { INC     SI               }
           $46/                  { INC     SI               }
           $eb/$f7/              { JMP     0134             }
           $1f);                 { POP     DS               }
    WITH answer DO BEGIN
      WHILE (len > 1) AND (dat[len] = 0) DO len := len - 1; END;
    END; { sublong   }





(*--------------------------------------------------------------------------


================================================================================
================================================================================
================================================================================
================================================================================
More BigInt source (in asm)
================================================================================
================================================================================
================================================================================
================================================================================
================================================================================


.386
.model flat

public _bigadd, _bigsub, _bigmul, _bigdiv, _bigcmp, _bigshr

.code

_bigadd proc

    push ecx
    push esi
    push edi

    mov esi, [esp + 20]
    mov edi, [esp + 16]
    mov ecx, [esp + 24]

    clc

    jecxz bigaddcarry

    lea esi, [esi + 4 * ecx]
    lea edi, [edi + 4 * ecx]
    neg ecx

    clc

    bigaddloop:
    mov eax, [esi + 4 * ecx]
    adc [edi + 4 * ecx], eax
    inc ecx
    jnz bigaddloop

    bigaddcarry:

    jnc bigadd  End

    mov ecx, [esp + 28]

    jecxz bigadd  End

    lea edi, [edi + 4 * ecx]
    not ecx
    inc ecx

    bigaddcarryloop:
    adc dword ptr [edi + 4 * ecx], 0
    jnc bigadd  End
    inc ecx
    jnz bigaddcarryloop

    bigadd  End:

    mov eax, 0
    adc eax, 0

    pop edi
    pop esi
    pop ecx

    ret

_bigadd   Endp

_bigsub proc

    push ecx
    push esi
    push edi

    mov esi, [esp + 20]
    mov edi, [esp + 16]
    mov ecx, [esp + 24]

    clc

    jecxz bigsubcarry

    lea esi, [esi + 4 * ecx]
    lea edi, [edi + 4 * ecx]
    neg ecx

    clc

    bigsubloop:
    mov eax, [esi + 4 * ecx]
    sbb [edi + 4 * ecx], eax
    inc ecx
    jnz bigsubloop

    bigsubcarry:

    jnc bigsub  End

    mov ecx, [esp + 28]

    jecxz bigsub  End

    lea edi, [edi + 4 * ecx]
    not ecx
    inc ecx

    bigsubcarryloop:
    sbb dword ptr [edi + 4 * ecx], 0
    jnc bigsub  End
    inc ecx
    jnz bigsubcarryloop

    bigsub  End:

    mov eax, 0
    adc eax, 0

    pop edi
    pop esi
    pop ecx

    ret

_bigsub   Endp

_bigmul proc

    push ebx
    push ecx
    push edx
    push esi
    push edi

    xor ebx, ebx
    mov esi, [esp + 28]
    mov edi, [esp + 24]
    mov ecx, [esp + 36]

    xor edx, edx
    jecxz bigmul  End

    lea esi, [esi + 4 * ecx]
    lea edi, [edi + 4 * ecx]
    neg ecx

    bigmulloop:
    mov eax, [esi + 4 * ecx]
    mul dword ptr [esp + 32]
    add eax, ebx
    adc edx, 0
    mov [edi + 4 * ecx], eax
    mov ebx, edx
    inc ecx
    jnz bigmulloop

    bigmul  End:

    mov eax, edx

    pop edi
    pop esi
    pop edx
    pop ecx
    pop ebx

    ret

_bigmul   Endp

_bigdiv proc

    push ebx
    push ecx
    push edx
    push esi
    push edi

    mov ebx, [esp + 32]
    mov esi, [esp + 28]
    mov edi, [esp + 24]
    mov ecx, [esp + 36]

    xor edx, edx

    jecxz bigdiv  End

    mov eax, [esi + 4 * ecx - 4]
    cmp eax, ebx
    jae bigdivloop
    mov dword ptr [edi + 4 * ecx - 4], 0
    dec ecx
    mov edx, eax

    jecxz bigdiv  End

    bigdivloop:
    mov eax, [esi + 4 * ecx - 4]
    div ebx
    mov [edi + 4 * ecx - 4], eax
    dec ecx
    jnz bigdivloop

    bigdiv  End:

    mov eax, edx

    pop edi
    pop esi
    pop edx
    pop ecx
    pop ebx

    ret

_bigdiv   Endp

_bigshr proc

    push ecx
    push esi
    push edi

    mov esi, [esp + 20]
    mov edi, [esp + 16]
    mov ecx, [esp + 24]

    clc

    jecxz bigshr  End

    bigshrloop:
    mov eax, [esi + 4 * ecx - 4]
    rcr eax, 1
    mov [edi + 4 * ecx - 4], eax
    dec ecx
    jnz bigshrloop

    bigshr  End:

    mov eax, 0
    adc eax, 0

    pop edi
    pop esi
    pop ecx

    ret

_bigshr   Endp

_bigcmp proc

    push ecx
    push esi
    push edi

    mov esi, [esp + 20]
    mov edi, [esp + 16]
    mov ecx, [esp + 24]

    jecxz bigcmp  End

    bigcmploop:
    mov eax, [edi + 4 * ecx - 4]
    cmp eax, [esi + 4 * ecx - 4]
    jb bigcmpb
    ja bigcmpa
    dec ecx
    jnz bigcmploop

    bigcmp  End:
    xor eax, eax

    pop edi
    pop esi
    pop ecx

    ret

    bigcmpb:
    mov eax, -1

    pop edi
    pop esi
    pop ecx

    ret

    bigcmpa:
    mov eax, 1

    pop edi
    pop esi
    pop ecx

    ret

_bigcmp   Endp

  End





================================================================================
================================================================================
================================================================================
================================================================================
================================================================================
================================================================================
================================================================================
================================================================================
================================================================================
================================================================================
================================================================================
================================================================================


   XOR16     2 byte    16 bit   1x Word
   XOR32     4 byte    32 bit   1x Integer
   CRC32     4 byte    32 bit   1x Integer
   CRC16     2 byte    16 bit   1x Word
}





   forward direction, but computationally infeasible to invert.  The
   S/KEY system is based on the MD4 Message Digest algorithm designed by
   Ronald Rivest [2].  Since the S/KEY authentication system went into






================================================================================
================================================================================
================================================================================
================================================================================
================================================================================
================================================================================
================================================================================


Function RndXORBuffer(Seed: Integer; var Buffer; Size: Integer): Integer; assembler;
asm
      AND     EDX,EDX
      JZ      @@2
      AND     ECX,ECX
      JLE     @@2
      PUSH    EBX
@@1:  XOR     AL,[EDX]
      IMUL    EAX,EAX,134775813
      INC     EAX
      MOV     EBX,EAX
      SHR     EBX,24
      MOV     [EDX],BL
      INC     EDX
      DEC     ECX
      JNZ     @@1
      POP     EBX
@@2:
  End;







*)




/*
**********************************************************************
                                                                      
                                                                      
     (STANDARD-)  N O R M A L  DISTRIBUTION                           
                                                                      
                                                                      
**********************************************************************
**********************************************************************
                                                                      
     FOR DETAILS SEE:                                                 
                                                                      
               AHRENS, J.H. AND DIETER, U.
               EXTENSIONS OF FORSYTHE'S METHOD FOR RANDOM             
               SAMPLING FROM THE NORMAL DISTRIBUTION.                 
               MATH. COMPUT., 27,124 (OCT. 1973), 927 - 937.          
                                                                      
     ALL STATEMENT NUMBERS CORRESPOND TO THE STEPS OF ALGORITHM 'FL'  
     (M=5) IN THE ABOVE PAPER     (SLIGHTLY MODIFIED IMPLEMENTATION)  
                                                                      
     Modified by Barry W. Brown, Feb 3, 1988 to use RANF instead of   
     SUNIF.  The argument IR thus goes away.                          
                                                                      
**********************************************************************
     THE DEFINITIONS OF THE CONSTANTS A(K), D(K), T(K) AND
     H(K) ARE ACCORDING TO THE ABOVEMENTIONED ARTICLE
*/
{
static float a[32] = {
    0.0,3.917609E-2,7.841241E-2,0.11777,0.1573107,0.1970991,0.2372021,0.2776904,
    0.3186394,0.36013,0.4022501,0.4450965,0.4887764,0.5334097,0.5791322,
    0.626099,0.6744898,0.7245144,0.7764218,0.8305109,0.8871466,0.9467818,
    1.00999,1.077516,1.150349,1.229859,1.318011,1.417797,1.534121,1.67594,
    1.862732,2.153875
};
static float d[31] = {
    0.0,0.0,0.0,0.0,0.0,0.2636843,0.2425085,0.2255674,0.2116342,0.1999243,
    0.1899108,0.1812252,0.1736014,0.1668419,0.1607967,0.1553497,0.1504094,
    0.1459026,0.14177,0.1379632,0.1344418,0.1311722,0.128126,0.1252791,
    0.1226109,0.1201036,0.1177417,0.1155119,0.1134023,0.1114027,0.1095039
};
static float t[31] = {
    7.673828E-4,2.30687E-3,3.860618E-3,5.438454E-3,7.0507E-3,8.708396E-3,
    1.042357E-2,1.220953E-2,1.408125E-2,1.605579E-2,1.81529E-2,2.039573E-2,
    2.281177E-2,2.543407E-2,2.830296E-2,3.146822E-2,3.499233E-2,3.895483E-2,
    4.345878E-2,4.864035E-2,5.468334E-2,6.184222E-2,7.047983E-2,8.113195E-2,
    9.462444E-2,0.1123001,0.136498,0.1716886,0.2276241,0.330498,0.5847031
};
static float h[31] = {
    3.920617E-2,3.932705E-2,3.951E-2,3.975703E-2,4.007093E-2,4.045533E-2,
    4.091481E-2,4.145507E-2,4.208311E-2,4.280748E-2,4.363863E-2,4.458932E-2,
    4.567523E-2,4.691571E-2,4.833487E-2,4.996298E-2,5.183859E-2,5.401138E-2,
    5.654656E-2,5.95313E-2,6.308489E-2,6.737503E-2,7.264544E-2,7.926471E-2,
    8.781922E-2,9.930398E-2,0.11556,0.1404344,0.1836142,0.2790016,0.7010474
};
static long i;
static float snorm,u,s,ustar,aa,w,y,tt;
    u = ranf();
    s = 0.0;
    if(u > 0.5) s = 1.0;
    u += (u-s);
    u = 32.0*u;
    i = (long) (u);
    if(i == 32) i = 31;
    if(i == 0) goto S100;
/*
                                START CENTER
*/
    ustar = u-(float)i;
    aa = *(a+i-1);
S40:
    if(ustar <= *(t+i-1)) goto S60;
    w = (ustar-*(t+i-1))**(h+i-1);
S50:
/*
                                EXIT   (BOTH CASES)
*/
    y = aa+w;
    snorm = y;
    if(s == 1.0) snorm = -y;
    return snorm;
S60:
/*
                                CENTER CONTINUED
*/
    u = ranf();
    w = u*(*(a+i)-aa);
    tt = (0.5*w+aa)*w;
    goto S80;
S70:
    tt = u;
    ustar = ranf();
S80:
    if(ustar > tt) goto S50;
    u = ranf();
    if(ustar >= u) goto S70;
    ustar = ranf();
    goto S40;


S100:
/*
                                START TAIL
*/
    i := 6;
    aa = *(a+31);

    u := u + u;
    While u < 1.0 do
      begin
        aa += *(d+i-1);
        i := i + 1;
        u := u + u;
      end;
    u := u - 1.0;

S140:
    w = u**(d+i-1);
    tt = (0.5*w+aa)*w;
    goto S160;
S150:
    tt = u;
S160:
    ustar = ranf();
    if(ustar > tt) goto S50;
    u = ranf();
    if(ustar >= u) goto S150;
    u = ranf();
    goto S140;
}

