unit exdialc0;
{
TMagRas Dialling Complex Example
================================

Dial, monitor and hang-up a RAS connection, allowing the user to change 
the telephone number and logon details, and dialling location.  

Created by Angus Robertson, Magenta Systems Ltd, England
in 2000, delphi@magsys.co.uk, http://www.magsys.co.uk/delphi/
Last updated: 12th April 2000

To load this example, the TMagRas components need to have been previously
installed on the component palette.

}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, StdCtrls,
  magrascon, magrasapi;

type
  TForm1 = class(TForm)
    MagRasCon: TMagRasCon;
    doHangup: TButton;
    doExit: TButton;
    ConnLog: TMemo;
    Label2: TLabel;
    LabelConn: TLabel;
    LabelStat: TLabel;
    TimerStatus: TTimer;
    LabelOnline: TLabel;
    ConnList: TListBox;
    Label3: TLabel;
    doDial: TButton;
    Label1: TLabel;
    GroupBox1: TGroupBox;
    Label43: TLabel;
    Label45: TLabel;
    Label44: TLabel;
    Label11: TLabel;
    Label13: TLabel;
    Label12: TLabel;
    LabelNumberDisp: TLabel;
    LabelNumberDial: TLabel;
    entUserName: TEdit;
    entPassword: TEdit;
    entUseCountryandAreaCodes: TCheckBox;
    entCountryCode: TEdit;
    entAreaCode: TEdit;
    entLocalNumber: TEdit;
    entCanonNumber: TEdit;
    doPropDial: TButton;
    LabelLocation: TLabel;
    procedure doExitClick(Sender: TObject);
    procedure TimerStatusTimer(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure doHangupClick(Sender: TObject);
    procedure doDialClick(Sender: TObject);
    procedure MagRasConStateChanged(Sender: TObject);
    procedure ConnListClick(Sender: TObject);
    procedure entUseCountryandAreaCodesClick(Sender: TObject);
    procedure NumberChange(Sender: TObject);
    procedure entCanonNumberChange(Sender: TObject);
    procedure doPropDialClick(Sender: TObject);
  private
    { Private declarations }
	procedure GetDialProps ;
  public
    { Public declarations }
  end;

var
  Form1: TForm1;
  ConnHandle: HRasConn ;         	// handle for current connection
  ConnName: string ;                // name of current connection
  LastState: integer ;				// used to check if state has changed
  DialHandle: HRasConn ;         	// handle for dialled connection
  DialName: string ;                // name of dialled connection
  DialProps: TDialProps;       		// dialling properties
  DialLocation: TDialLocation;  	// default dialling location
  DialCard: TDialCard ;		 		// default dialling calling card
  DialCountry: TDialCountry ;		// default country dialling info

implementation

{$R *.DFM}

procedure TForm1.GetDialProps ;
begin
    MagRasCon.GetTransCaps (DialProps, DialLocation, DialCard, DialCountry) ;
    LabelLocation.Caption := 'Dialling from ' + DialLocation.LocationName ;
end ;

procedure TForm1.doExitClick(Sender: TObject);
begin
    Close ;
end;

// main event handler, while this is being processed RAS will wait
// called in response to CurrentStatusEx in the timer event, or
// while dialling a call (events much faster)

procedure TForm1.MagRasConStateChanged(Sender: TObject);
var
	info: string ;
begin

// check type of event
	info := '' ;
    case MagRasCon.StateEventSource of
    	SourceDial: info := 'Dial: ' ;
    	SourceStatus: info := 'Status: ' ;
    	SourceHangup: info := 'Hangup: ' ;
	end ;

// see if new event, else display it
	if LastState = MagRasCon.ConnectState then exit ;
    LastState := MagRasCon.ConnectState ;
    ConnLog.Lines.Add (info + MagRasCon.StatusStr
						      +	' (' + IntToStr (LastState) + ')') ;

// something has changed, talk to user
// ConnectState can be checked against literals in MagRasApi
//    to determine current state of connection
	LabelStat.Caption := MagRasCon.StatusStr ;
    if (MagRasCon.ConnectState < RASCS_Connected) then
    									LabelOnline.Caption := 'Dialling' ;
	if (MagRasCon.ConnectState = RASCS_Connected) then
    									LabelOnline.Caption := 'Online' ;
	if (MagRasCon.ConnectState = RASCS_DisConnected) then
    									LabelOnline.Caption := 'Hang-Up' ;

// if dialling need to see what's happened
	if DialHandle <> 0 then
    begin

	// online OK, get IP addresses and restart timer
		if (MagRasCon.ConnectState = RASCS_Connected) then
        begin
            ConnHandle := DialHandle ;
			DialHandle := 0 ;
			if MagRasCon.GetProtocolEx (ConnHandle) = 0 then
            begin
              	ConnLog.Lines.Add ('Protocol: ' + MagRasCon.ConnProtocol) ;
              	ConnLog.Lines.Add ('IP Address: ' +  MagRasCon.ClientIP +
				              					' > ' + MagRasCon.ServerIP) ;
        	  	ConnLog.Lines.Add ('PPP Reply Message : ' +
                                                   MagRasCon.PPPReplyMessage) ;
			end ;
    	    TimerStatus.Enabled := true ;
		end ;

	// dialling failed, either an error or disconnected
		if ((MagRasCon.ConnectState > RASBase) and
	            (MagRasCon.ConnectState < RASCS_Paused)) or
	             (MagRasCon.ConnectState = RASCS_Disconnected) then
		begin
	// disconnect, returns when done or after three seconds, no StateChanged
            ConnHandle := DialHandle ;
			DialHandle := 0 ;
			MagRasCon.DisconnectEx (ConnHandle, 0, 3000, false) ;
	        TimerStatus.Enabled := true ;
            // reset is done in timer event
		end ;
	end ;
end;

// to monitor a RAS connection, you only need a timer event to check
// if there's an active RAS connection and then check it's state
// this timer is set for once per second, and may miss some state
// messages during dialling and authentication that happen very fast
// the timer internval could be shorter, but on Win9x this may overload RAS

procedure TForm1.TimerStatusTimer(Sender: TObject);
begin

// check for active connections
// Win9x lists connection when it starts dialling
// WinNT/2K only list connection if it answers
	MagRasCon.GetConnections ;

// details of active connections are now available in Connections list
// no active connections, see if already closed down
	if MagRasCon.Connections.Count = 0 then
	begin
		if ConnHandle = 0 then exit ;
        ConnHandle := 0 ;
        DialName := '' ;
        doHangup.Enabled := false ;
        doDial.Enabled := true ;
        LabelStat.Caption := '' ;
        LabelOnline.Caption := 'Offline' ;
        ConnLog.Lines.Add ('Connection Offline') ;
        exit ;
    end ;

// connection list has changed, that means a new call
	if MagRasCon.ConnChangedFlag then
    begin
      // assume only a single connection (there may be more)
	    ConnHandle := MagRasCon.Connections.RasConn (0) ;
    	ConnName := MagRasCon.Connections.EntryName (0) ;
	    LabelConn.Caption := ConnName ;
        doHangup.Enabled := true ;
        if DialName <> ConnName then
	        ConnLog.Lines.Add ('New Connection Found: ' + ConnName) ;
	end ;

// get state of current connection
// calls StateChanged event where all checking is done
    MagRasCon.CurrentStatusEx (ConnHandle, 0) ;
end;

procedure TForm1.FormCreate(Sender: TObject);
begin
	LastState := 0 ;
	LabelConn.Caption := '' ;
    LabelStat.Caption := '' ;
    ConnLog.Lines.Clear ;
// see if RAS has been installed
	if MagRasCon.TestRAS then
    begin
    // get list of phonebook entries
		MagRasCon.GetPhoneBookEntries ;
        MagRasCon.PhoneBookEntries.Sort ;
		ConnList.Items.Assign (MagRasCon.PhoneBookEntries) ;	 // display it

	// get dialling properties, location, calling card, etc
		GetDialProps ;

	// clear fields
        entCanonNumber.Text := '' ;
	    entUserName.Text := '' ;
	    entPassword.Text := '' ;
    	entUseCountryandAreaCodes.Checked := false ;
	    entCountryCode.Text := '' ;
    	entAreaCode.Text := '' ;
	    entLocalNumber.Text := '' ;

	// start monitoring
	    TimerStatusTimer (self) ;  // avoid waiting one second until timer expires
        TimerStatus.Enabled := true ;
	end
    else
    begin
	 	ConnLog.Lines.Add ('RAS is not installed') ;
    end ;
end;

procedure TForm1.FormClose(Sender: TObject; var Action: TCloseAction);
begin
	// could check if still online and close connection
    Application.Terminate ;
end;

procedure TForm1.doHangupClick(Sender: TObject);
begin
    if (DialHandle = 0) and (ConnHandle = 0) then exit ;
	doHangup.Enabled := false ;
    doDial.Enabled := true ;

// disconnect, returns when done or after three seconds, calls StateChanged
	if ConnHandle = 0 then ConnHandle := DialHandle ;
	MagRasCon.DisconnectEx (ConnHandle, 0, 3000, true) ;
end;

procedure TForm1.doDialClick(Sender: TObject);
var
	dispnum, dialnum: string ;
begin
	if ConnList.ItemIndex < 0 then exit ;
    if DialName = '' then exit ;
  	ConnLog.Lines.Add ('Starting to Dial Connection: ' + DialName) ;
// note, already got all dial and entry stuff in ConnListClick

// stop timer since dialling creates events
    TimerStatus.Enabled := false ;
    doDial.Enabled := false ;

// dial params that may have been changed by user
// note the canonical number has already been assembled in the NumberChange event
	MagRasCon.TranslateAddr (0, entCanonNumber.Text, dispnum, dialnum) ;
   	with MagRasCon do
    begin
    	UserName := entUserName.Text ;
    	Password := entPassword.Text ;
		PhoneNumber := dialnum ;   // translated number
 	end ;

// start connection (sets handle)
	DialHandle := 0 ;
  	if MagRasCon.ConnectEx (DialHandle) <> 0 then
    begin
        // fails here is dialling did not even start
		ConnLog.Lines.Add  ('Dialling Failed - ' + MagRasCon.StatusStr) ;
		beep ;
        TimerStatus.Enabled := true ;
	    doDial.Enabled := true ;
        exit ;
	end ;

// dialling started OK
// dial connection or failure is checked in StateChanged event handler
	doHangup.Enabled := true ;
end;


procedure TForm1.ConnListClick(Sender: TObject);
var
	errcode: integer ;
begin
	if ConnList.ItemIndex = -1 then exit ;
	DialName  := ConnList.Items [ConnList.ItemIndex];
    LabelConn.Caption := DialName ;

// set phonebook entry to access
	MagRasCon.EntryName := DialName ;

// read entry and dial properties
	errcode := MagRasCon.GetDialParams ;
    if errcode = 0 then errcode := MagRasCon.GetEntryProperties ;
    if errcode <> 0 then
    begin
		ConnLog.Lines.Add (MagRasCon.StatusStr) ;
        beep ;
        exit ;
    end ;

// set-up edit boxes
	with MagRasCon do
	begin

    // if not going to edit number, could get PhoneCanonical property instead of bits
   	// entCanonNumber.Text := PhoneCanonical ;

	// Location and phone number, including alternates
    // event handler sets canonical number
		entUseCountryAndAreaCodes.Checked := UseCountryAndAreaCodes ;
        entCountryCode.Text := IntToStr (CountryCode) ;
		entAreaCode.Text := AreaCode ;
		entLocalNumber.Text := LocalPhoneNumber ;

	// dial params
      	entUserName.Text := UserName ;
    	entPassword.Text := Password ;

	end ;
	entUseCountryandAreaCodesClick (self) ;   // set a few fields
end;

procedure TForm1.entUseCountryandAreaCodesClick(Sender: TObject);
begin
   	entCountryCode.Enabled := entUseCountryAndAreaCodes.Checked ;
   	entAreaCode.Enabled := entUseCountryAndAreaCodes.Checked ;
   	entCanonNumberChange (self) ;
end;

procedure TForm1.NumberChange(Sender: TObject);
begin
	if entCountryCode.Text = '' then entCountryCode.Text := '0' ;
	entCanonNumber.Text := MagRasCon.GetCanonical
     	 (entUseCountryAndAreaCodes.Checked, StrToInt (entCountryCode.Text),
			            			   entAreaCode.Text, entLocalNumber.Text) ;
   	entCanonNumberChange (self) ;
end;

procedure TForm1.entCanonNumberChange(Sender: TObject);
var
	DispNum, DialNum: string ;
begin
	MagRasCon.TranslateAddr (0, entCanonNumber.Text, DispNum, DialNum) ;
   	LabelNumberDisp.Caption := 'Display Number: ' + DispNum ;
   	LabelNumberDial.Caption := 'Dialable Number: ' + DialNum ;
end;

procedure TForm1.doPropDialClick(Sender: TObject);
begin
	MagRasCon.TranslateDialog (Handle, 0, '') ;
	GetDialProps ;
   	entCanonNumberChange (self) ;
end;

end.
