unit Bart;

{  ******
   *
   * Module:    Bart
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    Implementation of our hero, Bart.
   *
   * Details:
   *
   *    This module defines most of the look and behavior of Bart, the hero of
   *    the game.  The TBart class is derived from TVisibleObject, which
   *    provides the core functionality that TBart requires for linking into
   *    the rest of the universe.
   *
   ******}

interface
uses Message, SoundDev, InputDev, InpCodes, Visible, Missle, State,
     Forms, WinTypes, Graphics, Classes, SysUtils;

type TBart = class(TVisibleObject)
public
   { *** Our constructor *** }
   constructor Create(lstUniverse: TList; iCollisionID: Integer);
   destructor Destroy; Override;

   { Method to handle keyboard messages. }
   procedure ProcessMessage(iCategory, iType: Integer; bValue0: Boolean; fValue1, fValue2: Real); Override;

   { Movement methods. }
   procedure Thrust;
   procedure Drag;
   procedure Move; Override;

   { Engine control. }
   procedure SetEngineFiringState(bFiring: Boolean);

   { Missle cannon control. }
   procedure FireMissle;

   { Handle Collisions. }
   procedure HandleCollision(objOtherObject: TVisibleObject); Override;

protected

   m_iHull, m_iFlame: Integer;  { Handles to the shapes comprising Bart. }
   m_fLastFireTime: Real;       { Time that Bart last fired a missle. }
   m_bEngineFiring: Boolean;    { TRUE when the engine is firing. }
   m_bFiringCannon: Boolean;    { TRUE when the cannon trigger is engaged. }
   m_inpDevice: TInputDevice;   { Input device that is controlling Bart's movements. }
end;

const
   FIRING_DELAY = 0.25;

implementation
uses General, Global;

constructor TBart.Create(lstUniverse: TList;
                         iCollisionID: Integer);
begin
    inherited Create(lstUniverse, iCollisionID);

    { Add us into the proper input notification lists.  This tells the input
      object to notify Bart directly as relatant input events occur.}
    m_inpDevice := g_envEnviron.devGetInputDevice;
    m_inpDevice.inpGetInput(BART_TURN_LEFT).NotifyOnChange(Self);
    m_inpDevice.inpGetInput(BART_TURN_RIGHT).NotifyOnChange(Self);
    m_inpDevice.inpGetInput(BART_THRUST).NotifyOnChange(Self);
    m_inpDevice.inpGetInput(BART_FIRE).NotifyOnChange(Self);
    m_inpDevice.inpGetInput(BART_HYPERSPACE).NotifyOnChange(Self);

    { Set initial orientation and direction. }
    mtrxTransform.fOrientation := 0.0;
    mtrxTransform.fDirection := 0;

    { Set up Bart's hull and engine flame shapes. }
    m_iHull := IncludeShape('Bart', 0, 0, clRed, True);
    m_iFlame := IncludeShape('Flame', 0, -5, clYellow, False);

    { Initially, the engine is turned off. }
    SetEngineFiringState(False);

    { Reset the cannon firing time. }
    m_fLastFireTime := fGetNow;

    { Bart will start under pilot control by default.  Notice that Bart does NOT
      appear on the screen until there is a safe clearing.  During this period
      of time he is not subject to collisions.  The following list of states is
      used to achieve this behavior for Bart.  Notice that ALL of these behaviors
      are implemented individually by TVisibleObject.  The state queue simply
      gives us a way to queue the behaviors in an arbitrary order. }
    AddToStateSequence(BecomeInvisible, 0, 0);
    AddToStateSequence(DisableCollisions, 0, 0);
    AddToStateSequence(WaitForClearRect, 0, 0);
    AddToStateSequence(BecomeVisible, 0, 0);
    AddToStateSequence(EnableCollisions, 0, 0);
    AddToStateSequence(UserControl, 0, 0);
end;

destructor TBart.Destroy;
begin
    // Ask the input device to stop sending us input messages, since we will
    // no longer be here to get them.
    m_inpDevice.inpGetInput(BART_TURN_LEFT).RemoveNotification(Self);
    m_inpDevice.inpGetInput(BART_TURN_RIGHT).RemoveNotification(Self);
    m_inpDevice.inpGetInput(BART_THRUST).RemoveNotification(Self);
    m_inpDevice.inpGetInput(BART_FIRE).RemoveNotification(Self);
    m_inpDevice.inpGetInput(BART_HYPERSPACE).RemoveNotification(Self);

    // Perform default procesing.
    inherited Destroy;
end;

procedure TBart.Thrust;
var
    fNewSpeed: Real;
begin
    // Set a new direction based upon which direction we are pointing.  Notice
    // that this logic is simply, and allows Bart to turn on a dime.  It could
    // be changed to make him a little harder to control.
    mtrxTransform.fDirection := mtrxTransform.fOrientation;

    { Get the current speed of the object, and increase the speed by a fixed
      amount to account for the momentum provided by the thrusters.  We will
      also cap off at a maximum allowed speed. }
    fNewSpeed := mtrxTransform.fSpeed + 0.5;
    mtrxTransform.fSpeed := fMin(fNewSpeed, 4.5);
end;

procedure TBart.Drag;
begin
   { Decrease the speed by a fixed percentage as if there were drag. }
   mtrxTransform.fSpeed := mtrxTransform.fSpeed * 0.95;
end;

procedure TBart.Move;
begin
   { Do the usual moving stuff. }
   inherited Move;

   { If the engine is firing then simulate thrust. }
   if m_bEngineFiring = True then
      Thrust;

   { If the cannon trigger is engaged then see if the cannon is ready to }
   { fire off another missle. }
   if m_bFiringCannon = True then
      FireMissle;
       
   { Simulate friction on the ship. }
   Drag;

   { Wrap around if we've gone off the screen. }
   Wrap;
end;

procedure TBart.SetEngineFiringState(bFiring: Boolean);
begin
    // Make the engine flames visible based upon the given firing state.
    SetShapeVisible(m_iFlame, bFiring);
    m_bEngineFiring := bFiring;
end;

procedure TBart.FireMissle;
var
   mslFire: TMissle;
   fMissleX, fMissleY: Real;
   fSecond: Real;
begin
   { Compute the current time in seconds.  We can only fire if the cannon is
     ready, and this takes a fixed interval of time. }
   fSecond := fGetNow;
   if ((fSecond - m_fLastFireTime) > FIRING_DELAY) then
   begin
      { Make the sound of the firing cannon. }
      g_envEnviron.devGetSoundDevice.PlaySound('Laser.wav', 0);

      { Create a new missle object.  This missle will have the same collision
        identifier as Bart, which means Bart cannot be killed by his own missle. }
      mslFire := TMissle.Create(lstUniverse, iCollisionID);

      { Find the initial coordinates of the missle.  They will start }
      { just at the front tip of the ship. }
      fMissleX := 0;
      fMissleY := 10;
      mtrxTransform.TransFormVertex(fMissleX, fMissleY, fMissleX, fMissleY);
      mtrxTransform.TranslateVertex(fMissleX, fMissleY, fMissleX, fMissleY);

      { Set initial position and orientation. }
      mslFire.mtrxTransform.fScale := 0.5;
      mslFire.mtrxTransform.SetTranslation(fMissleX, fMissleY);
      mslFire.mtrxTransform.fOrientation := mtrxTransform.fOrientation;
      mslFire.mtrxTransform.fDirection := mtrxTransform.fOrientation;

      { Add the missle object to the global object list. }
      lstUniverse.Add(mslFire);

      { Record the time the missle was squeezed off, so we can determine when
        the cannon is ready with another one. }
      m_fLastFireTime := fSecond;
   end;
end;

procedure TBart.HandleCollision;
begin
    { Create the sound of an explosion. }
    g_envEnviron.devGetSoundDevice.PlaySound('Explode.wav', 0);

    // Turn off the engine, since we've been killed ruthlessly!
    SetEngineFiringState(False);

    // Perform default death behavior by shattering Bart into a million pieces.
    KillObject(True, 70, 0.15, 0.25, 1.0);
end;

// This method gets triggered in response to input event from the current
// input device.  Bart uses these messages to respond to user input.
procedure TBart.ProcessMessage(iCategory, iType: Integer; bValue0: Boolean; fValue1, fValue2: Real);
begin
    { Process input messages ONLY if the user is in control.  This prevents
      the user from moving Bart while he is invisible, which can only cuase
      trouble. }
    if bIsInState(UserControl) = False then
       Exit;

    { Process the input message according to its type. }
    if iCategory = INPUT_DEVICE then
        { Notice that bValue0 is flag indicating whether the input was just
          fired, or just lifted. }
        case iType of
           BART_TURN_LEFT:
              if (bValue0 = True) then
                 mtrxTransform.fRotation := -0.15
              else
                 mtrxTransform.fRotation := 0;

           BART_TURN_RIGHT:
              if (bValue0 = True) then
                 mtrxTransform.fRotation := 0.15
              else
                 mtrxTransform.fRotation := 0;

           BART_THRUST:
              if (bValue0 = True) then
                 SetEngineFiringState(True)
               else
                 SetEngineFiringState(False);

           BART_FIRE:
              if (bValue0 = True) then
                  begin
                  FireMissle;

                  m_bFiringCannon := True;
                  end
              else
                  m_bFiringCannon := False;

           BART_HYPERSPACE:
              begin
                  { In hyperspace, we are NOT under the user's control! }
                 ExitCurrentState;

                 { Make sure the engine is off. }
                 SetEngineFiringState(False);

                 { Disappear, and reappear when and where an opening is found. }
                 AddToStateSequence(DisableCollisions, 0, 0);
                 AddToStateSequence(BecomeInvisible, 0, 0);
                 AddToStateSequence(StopMovement, 0, 0);
                 AddToStateSequence(Pause, 1.0, 0);
                 AddToStateSequence(FindClearRect, 0, 0);
                 AddToStateSequence(BecomeVisible, 0, 0);
                 AddToStateSequence(EnableCollisions, 0, 0);
                 AddToStateSequence(UserControl, 0, 0);
             end;
        else
        end;
end;

end.
