unit CanvsVid;

{  ******
   *
   * Module:    CanvsVid
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    This module defines the TCanvasVideo class, which is derived from the
   *    abstract TVideoDevice ancestor.  It provides the ability for the program
   *    to draw objects on a Delphi form by overriding abstract methods.
   *
   ****** }

interface

uses VideoDev, Classes, Graphics, Forms, General;

type TCanvasVideo = class(TVideoDevice)
    public
        { Public constructor. }
        constructor Create(frmOutput: TForm;
                           iInternalFrameWidth, iInternalFrameHeight: Integer);

        { Method to clear off the entire frame. }
        procedure ClearFrame; Override;

        { Methods to move the pen position and draw lines. }
        procedure MoveTo(iX, iY: Integer); Override;
        procedure LineTo(iX, iY: Integer); Override;

        { Method to set a single pixel. }
        procedure SetPixel(iX, iY: Integer; clrPixel: TColor); Override;

        { Method to set the current pen color. }
        procedure SetPenColor(clrNew: TColor); Override;

        { functions to return the current physical screen dimensions. }
        function iGetPhysicalScreenHeight: Integer; Override;
        function iGetPhysicalScreenWidth: Integer; Override;

   private

        { Reference to form and canvas objects being used for output. }
        m_frmOutput: TForm;
        m_cnvOutput: TCanvas;
end;

implementation

constructor TCanvasVideo.Create(frmOutput: TForm;
                                iInternalFrameWidth, iInternalFrameHeight: Integer);
begin
    { Perform default processing. }
    inherited Create(iInternalFrameWidth, iInternalFrameHeight);

    { Store a reference to the form and drawing canvas. }
    m_frmOutput := frmOutput;
    m_cnvOutput := frmOutput.Canvas;

    { Set the default stylus color. }
    SetPenColor(clWhite);
end;

procedure TCanvasVideo.SetPenColor(clrNew: TColor);
begin
    m_cnvOutput.Pen.Color := clrNew;
end;

procedure TCanvasVideo.MoveTo(iX, iY: Integer);
var
    iPhysicalX, iPhysicalY: Integer;
begin
    { Convert logical coordinates to physical ones. }
    iPhysicalX := Round(fToPhysicalX(iX));
    iPhysicalY := Round(fToPhysicalY(iY));

    { Move to the requested coordinates. }
    m_cnvOutput.MoveTo(iPhysicalX, iPhysicalY);
end;

procedure TCanvasVideo.LineTo(iX, iY: Integer);
var
    iPhysicalX, iPhysicalY: Integer;
begin
    { Convert logical coordinates to physical ones. }
    iPhysicalX := Round(fToPhysicalX(iX));
    iPhysicalY := Round(fToPhysicalY(iY));

    { Draw a line from the current stylus position to the requested spot. }
    m_cnvOutput.LineTo(iPhysicalX, iPhysicalY);
end;

procedure TCanvasVideo.SetPixel(iX, iY: Integer; clrPixel: TColor);
var
    iPhysicalX, iPhysicalY: Integer;
begin
    { Convert logical coordinates to physical ones. }
    iPhysicalX := Round(fToPhysicalX(iX));
    iPhysicalY := Round(fToPhysicalY(iY));

    { Turn on the requested pixel. }
    m_cnvOutput.Pixels[iPhysicalX, iPhysicalY] := clrPixel;
end;

function TCanvasVideo.iGetPhysicalScreenHeight: Integer;
begin
    Result := m_frmOutput.ClientHeight;
end;

function TCanvasVideo.iGetPhysicalScreenWidth: Integer;
begin
    Result := m_frmOutput.ClientWidth;
end;

procedure TCanvasVideo.ClearFrame;
begin
    { Blank out the entire frame. }
    m_cnvOutput.Pen.Color := clrGetBackgroundColor;
    m_cnvOutput.Brush.Color := clrGetBackgroundColor;
    m_cnvOutput.Rectangle(0, 0, iGetPhysicalScreenWidth - 1, iGetPhysicalScreenHeight - 1);
end;

end.
