unit General;

{  ******
   *
   * Module:    General
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    This module contains a bunch of unrelated general functions.
   *
   ****** }

interface
uses SoundDev, RocksKbd, About, Keys, Bart, Vertex, Edge, MainForm, Score, GameCtl,
     Forms, WinTypes, Graphics, Classes, SysUtils, StdCtrls;

{ Method to compute the distance between the given points. }
function fDistance(X1, Y1, X2, Y2: Real): Real;

{ Min and Max functions. }
function fMin(fValue1, fValue2: Real): Real;
function fMax(fValue1, fValue2: Real): Real;

{ Method to find the sign of the given value. }
function iSign(fValue: Real): Integer;

{ Rectagnel and polygon intersection algorithms. }
function bRectsIntersect(rect1, rect2: TRect): Boolean;
function bPolygonsIntersect(listVertex, listEdge: TList): Boolean;
function bIsVertexWithinPolygon(vtxPoint: TVertex; listEdge: TList): Boolean;

{ Methods to return logical screen atrributes. }
function iScreenWidth: Integer;
function iScreenHeight: Integer;
function iMidScreenX: Integer;
function iMidScreenY: Integer;

{ Method to determine if the given rectangle is off of the screen. }
function bIsRectOffScreen(rectCheck: TRect): Boolean;

{ Method to get the default background color of the screen. }
function clrGetBackgroundColor: TColor;

{ Methods to encode/decode the given string. }
function szEncode(szString: String): String;
function szDecode(szString: String): String;

{ Function to return the current Windows time, in seconds. }
function fGetNow: Real;

implementation
uses Constant, Global, Visible, Missle, Asteroid;

const
   SECONDS_IN_A_DAY = 86400;

function fDistance(X1, Y1, X2, Y2: Real): Real;
var
   fDiffX, fDiffY: Real;
begin
   fDiffX := X2 - X1;
   fDiffY := Y2 - Y1;

   fDistance := Sqrt((fDiffX * fDiffX) + (fDiffY * fDiffY));
end;

function fMin(fValue1, fValue2: Real): Real;
begin
   if fValue1 > fValue2 then
      Result := fValue2
   else
      Result := fValue1;
end;

function fMax(fValue1, fValue2: Real): Real;
begin
   if fValue1 > fValue2 then
      Result := fValue1
   else
      Result := fValue2;
end;

function iSign(fValue: Real): Integer;
begin
   if fValue >= 0 then
      Result := -1
   else
      Result := 1;
end;

function bRectsIntersect(rect1, rect2: TRect): Boolean;
begin
   if rect1.Top > rect2.Bottom then
   begin
      Result := False;
      Exit;
   end;

   if rect1.Bottom < rect2.Top then
   begin
      Result := False;
      Exit;
   end;

   if rect1.Left > rect2.Right then
   begin
      Result := False;
      Exit;
   end;

   if rect1.Right < rect2.Left then
   begin
      Result := False;
      Exit;
   end;

   Result := True;
end;

function bPolygonsIntersect(listVertex, listEdge: TList): Boolean;
var
   vtxObject: TVertex;
   iVertexIndex: Integer;
begin
   for iVertexIndex := 0 to (listVertex.Count - 1) do
   begin
      vtxObject := listVertex.Items[iVertexIndex];
      if bIsVertexWithinPolygon(vtxObject, listEdge) = True then
      begin
         Result := True;
         Exit;
      end;
   end;

   Result := False;
end;

function bIsVertexWithinPolygon(vtxPoint: TVertex; listEdge: TList): Boolean;
var
   edgeObject: TEdge;

   fRelativeX1, fRelativeX2: Real;
   fRelativeY1, fRelativeY2: Real;

   fXChangePerY: Real;
   fCoordCrossingX: Real;

   iEdgeIndex: Integer;
   iXOriginCrossings: Integer;
begin
   iXOriginCrossings := 0;

   { Look at each edge of the object. }
   for iEdgeIndex := 0 to (listEdge.Count - 1) do
   begin
      edgeObject := listEdge.Items[iEdgeIndex];

      { Process this edge only if it is solid. }
      if edgeObject.m_bIncludeInCollisions = True then
      begin
         { Correct line endpoint coordinates to be relative to the vertex's origin. }
         fRelativeY1 := edgeObject.m_vtxOne.m_fScreenY - vtxPoint.m_fScreenY;
         fRelativeY2 := edgeObject.m_vtxTwo.m_fScreenY - vtxPoint.m_fScreenY;

         { Look for all lines that cross the X-Axis. }
         if iSign(fRelativeY1) <> iSign(fRelativeY2) then
         begin
            fRelativeX1 := edgeObject.m_vtxOne.m_fScreenX - vtxPoint.m_fScreenX;
            fRelativeX2 := edgeObject.m_vtxTwo.m_fScreenX - vtxPoint.m_fScreenX;

            if (fRelativeX1 >= 0) and (fRelativeX2 >= 0) then
            begin
               fXChangePerY := (fRelativeX2 - fRelativeX1) / (fRelativeY2 - fRelativeY1);
               fCoordCrossingX := fRelativeX1 - fRelativeY1 * fXChangePerY;

               if fCoordCrossingX >= 0 then
                  iXOriginCrossings := iXOriginCrossings + 1;
            end;
         end;
      end;
   end;

   { If we've crossed the positive X axis an odd number of times then }
   { the point lies within the polygon. }
   if (iXOriginCrossings mod 2) = 1 then
      Result := True
   else
      Result := False;
end;

function iScreenWidth: Integer;
begin
   Result := INTERNAL_SCREEN_WIDTH;
end;

function iScreenHeight: Integer;
begin
   Result := INTERNAL_SCREEN_HEIGHT;
end;

function fGetNow: Real;
begin
   Result := Frac(Now) * SECONDS_IN_A_DAY;
end;

function bIsRectOffScreen(rectCheck: TRect): Boolean;
var
   rectScreen: TRect;
begin
   rectScreen.Top := 0;
   rectScreen.Left := 0;
   rectScreen.Right := iScreenWidth - 1;
   rectScreen.Bottom := iScreenHeight - 1;

   if bRectsIntersect(rectCheck, rectScreen) = False then
      Result := True
   else
      Result := False;
end;

function szEncode(szString: String): String;
var
   iIndex: Integer;
   cChar: Char;
   szOutput: String;
begin
   szOutput := '';

   for iIndex := 1 to (Length(szString)) do
   begin
      cChar := szString[iIndex];
      cChar := chr(not ord(cChar));

      szOutput := szOutput + cChar;
   end;

   Result := szOutput;
end;

function szDecode(szString: String): String;
var
   iIndex: Integer;
   cChar: Char;
   szOutput: String;
begin
   szOutput := '';

   for iIndex := 1 to (Length(szString)) do
   begin
      cChar := szString[iIndex];
      cChar := chr(not ord(cChar));

      szOutput := szOutput + cChar;
   end;

   Result := szOutput;
end;

function iMidScreenX: Integer;
begin
   Result := INTERNAL_SCREEN_WIDTH Div 2;
end;

function iMidScreenY: Integer;
begin
   Result := INTERNAL_SCREEN_HEIGHT Div 2;
end;

function clrGetBackgroundColor: TColor;
begin
   Result := Application.MainForm.Color;
end;

end.

