unit Keyboard;

{  ******
   *
   * Module:    Keyboard
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    This module defines the TWindowsKeyboard class, which provides
   *    functionality for converting keyboard input into logical game inputs.
   *
   ****** }

interface
uses InputDev, Input, WinTypes, WinProcs, Classes;

{ Record for associating keyboard keys with an input code. }
type TKeyAssociation = record
    wKeyCode: Word;         { Windows Virtual Key code. }
    iInputCode: Integer;    { Input code trigger by the key. }
end;
type PKeyAssociation = ^TKeyAssociation;

{ Definition of the TWindowsKeyboard class. }
type TWindowsKeyboard = class(TInputDevice)
    public
        { Class constructor and destructor. }
        constructor Create;
        destructor Destroy; Override;

        { Method to associate a key with an input. }
        procedure AssociateKey(wKeyCode: Word; iInputCode: Integer);

        { Interface for processing keyboard events directly. }
        procedure ProcessKeyDown(var wKeyCode: Word; Shift: TShiftState);
        procedure ProcessKeyUp(var wKeyCode: Word; Shift: TShiftState);

        { Functions to convert between key code and input codes. }
        function iGetInputCodeFromKey(wKeyCode: Word): Integer;
        function wGetKeyCodeFromInput(iInputCode: Integer): Word;

        { Functions to translate VK-codes and text. }
        function szGetTextFromKeyCode(wKeyCode: Word): String;
        function wGetKeyCodeFromText(szText: String): Word;

    private
        m_lstKeyAssociations: TList;    { List of key associations. }
end;

implementation

constructor TWindowsKeyboard.Create;
begin
    { Perform default actions. }
    inherited Create;

    { Create a list to manage key associations. }
    m_lstKeyAssociations := TList.Create;
end;

destructor TWindowsKeyboard.Destroy;
begin
    { Clean up after ourselves. }
    m_lstKeyAssociations.Free;

    { Perform default cleanup. }
    inherited Destroy;
end;

procedure TWindowsKeyboard.AssociateKey(wKeyCode: Word; iInputCode: Integer);
var
    kaCheck: PKeyAssociation;
    kaNew: PKeyAssociation;
    iIndex: Integer;
begin
    kaNew := nil;

    { Look for an existing key mapping to the given input. }
    for iIndex := 0 to (m_lstKeyAssociations.Count - 1) do
    begin
        kaCheck := m_lstKeyAssociations.Items[iIndex];
        if (kaCheck^.iInputCode = iInputCode) then
        begin
             kaNew := kaCheck;
             Break;
        end;
    end;

    { If required, create a new structure. }
    if kaNew = nil then
    begin
        New(kaNew);

        { Add the association to the current list. }
        m_lstKeyAssociations.Add(kaNew);
    end;

    { (Re)initialize the key association structure. }
    kaNew^.wKeyCode := wKeyCode;
    kaNew^.iInputCode := iInputCode;
end;

procedure TWindowsKeyboard.ProcessKeyDown(var wKeyCode: Word; Shift: TShiftState);
var
    iInputCode: Integer;
    inpKey: TInput;
begin
    { Get the input code corresponding to the given key code. }
    iInputCode := iGetInputCodeFromKey(wKeyCode);

    { Get a reference to the input object itself, and mark it as selected. }
    if iInputCode >= 0 then
    begin
        inpKey := inpGetInput(iInputCode);
        if inpKey <> nil then
            inpKey.SetSelectionState(True);
    end;
end;

procedure TWindowsKeyboard.ProcessKeyUp(var wKeyCode: Word; Shift: TShiftState);
var
    iInputCode: Integer;
    inpKey: TInput;
begin
    { Get the input code corresponding to the given key code. }
    iInputCode := iGetInputCodeFromKey(wKeyCode);

    { Get a reference to the input object itself, and mark it as deselected. }
    if iInputCode >= 0 then
    begin
        inpKey := inpGetInput(iInputCode);
        if inpKey <> nil then
            inpKey.SetSelectionState(False);
    end;
end;

function TWindowsKeyboard.iGetInputCodeFromKey(wKeyCode: Word): Integer;
var
    kaThis: PKeyAssociation;
    iIndex: Integer;
begin
    { Return -1 by default to indicate an unknown conversion. }
    Result := -1;

    { Look at each association in the list for a match. }
    for iIndex := 0 to (m_lstKeyAssociations.Count - 1) do
    begin
        kaThis := m_lstKeyAssociations.Items[iIndex];
        if kaThis^.wKeyCode = wKeyCode then
        begin
            Result := kaThis^.iInputCode;
            break;
        end;
    end;
end;

function TWindowsKeyboard.wGetKeyCodeFromInput(iInputCode: Integer): Word;
var
    kaThis: PKeyAssociation;
    iIndex: Integer;
begin
    { Return 0xFFFF by default to indicate an unknown conversion. }
    Result := $FFFF;

    { Look at each association in the list for a match. }
    for iIndex := 0 to (m_lstKeyAssociations.Count - 1) do
    begin
        kaThis := m_lstKeyAssociations.Items[iIndex];
        if kaThis^.iInputCode = iInputCode then
        begin
            Result := kaThis^.wKeyCode;
            Break;
        end;
    end;
end;

function TWindowsKeyboard.szGetTextFromKeyCode(wKeyCode: Word): String;
begin
    { Return text for the given key code. }
    case wKeyCode of
       VK_SPACE:   Result := 'SpaceBar';
       VK_RETURN:  Result := 'Enter';
       VK_TAB:     Result := 'Tab';
       VK_CAPITAL: Result := 'Caps Lock';
       VK_SHIFT:   Result := 'Shift';
       VK_LEFT:    Result := 'Left Arrow';
       VK_RIGHT:   Result := 'Right Arrow';
       VK_UP:      Result := 'Up Arrow';
       VK_DOWN:    Result := 'Down Arrow';
       VK_INSERT:  Result := 'Insert';
       VK_DELETE:  Result := 'Delete';
       VK_HOME:    Result := 'Home';
       VK_END:     Result := 'End';
       VK_PRIOR:   Result := 'Page Up';
       VK_NEXT:    Result := 'Page Down';
       VK_NUMLOCK: Result := 'Num Lock';
       VK_SCROLL:  Result := 'Scroll Lock';
       VK_PAUSE:   Result := 'Pause';
       $41:         Result := 'A';
       $42:         Result := 'B';
       $43:         Result := 'C';
       $44:         Result := 'D';
       $45:         Result := 'E';
       $46:         Result := 'F';
       $47:         Result := 'G';
       $48:         Result := 'H';
       $49:         Result := 'I';
       $4A:         Result := 'J';
       $4B:         Result := 'K';
       $4C:         Result := 'L';
       $4D:         Result := 'M';
       $4E:         Result := 'N';
       $4F:         Result := 'O';
       $50:         Result := 'P';
       $51:         Result := 'Q';
       $52:         Result := 'R';
       $53:         Result := 'S';
       $54:         Result := 'T';
       $55:         Result := 'U';
       $56:         Result := 'V';
       $57:         Result := 'W';
       $58:         Result := 'X';
       $59:         Result := 'Y';
       $5A:         Result := 'Z';
       $30:         Result := '0';
       $31:         Result := '1';
       $32:         Result := '2';
       $33:         Result := '3';
       $34:         Result := '4';
       $35:         Result := '5';
       $36:         Result := '6';
       $37:         Result := '7';
       $38:         Result := '8';
       $39:         Result := '9';
       VK_NUMPAD0:  Result := 'NumPad 0';
       VK_NUMPAD1:  Result := 'NumPad 1';
       VK_NUMPAD2:  Result := 'NumPad 2';
       VK_NUMPAD3:  Result := 'NumPad 3';
       VK_NUMPAD4:  Result := 'NumPad 4';
       VK_NUMPAD5:  Result := 'NumPad 5';
       VK_NUMPAD6:  Result := 'NumPad 6';
       VK_NUMPAD7:  Result := 'NumPad 7';
       VK_NUMPAD8:  Result := 'NumPad 8';
       VK_NUMPAD9:  Result := 'NumPad 9';
       VK_F1:       Result := 'F1';
       VK_F2:       Result := 'F2';
       VK_F3:       Result := 'F3';
       VK_F4:       Result := 'F4';
       VK_F5:       Result := 'F5';
       VK_F6:       Result := 'F6';
       VK_F7:       Result := 'F7';
       VK_F8:       Result := 'F8';
       VK_F9:       Result := 'F9';
       VK_F10:      Result := 'F10';
       VK_F11:      Result := 'F11';
       VK_F12:      Result := 'F12';
    else
        Result := '';
    end;
end;

function TWindowsKeyboard.wGetKeyCodeFromText(szText: String): Word;
begin
    { Return a default key code of 0 (which is invalid) if there is no match. }
    Result := 0;

    { Match text with the proper key code. }
    if szText = 'SpaceBar' then
        Result := VK_SPACE;

    if szText = 'Enter' then
        Result := VK_RETURN;

    if szText = 'Tab' then
        Result := VK_TAB;

    if szText = 'Caps Lock' then
        Result := VK_CAPITAL;

    if szText = 'Shift' then
        Result := VK_SHIFT;

    if szText = 'Left Arrow' then
        Result := VK_LEFT;

    if szText = 'Right Arrow' then
        Result := VK_RIGHT;

    if szText = 'Up Arrow' then
        Result := VK_UP;

    if szText = 'Down Arrow' then
        Result := VK_DOWN;

    if szText = 'Insert' then
        Result := VK_INSERT;

    if szText = 'Delete' then
        Result := VK_DELETE;

    if szText = 'Home' then
        Result := VK_HOME;

    if szText = 'End' then
        Result := VK_END;

    if szText = 'Page Up' then
        Result := VK_PRIOR;

    if szText = 'Page Down' then
        Result := VK_NEXT;

    if szText = 'Num Lock' then
        Result := VK_NUMLOCK;

    if szText = 'Scroll Lock' then
        Result := VK_SCROLL;

    if szText = 'Pause' then
        Result := VK_PAUSE;

    if szText = 'A' then
        Result := $41;

    if szText = 'B' then
        Result := $42;

    if szText = 'C' then
        Result := $43;

    if szText = 'D' then
        Result := $44;

    if szText = 'E' then
        Result := $45;

    if szText = 'F' then
        Result := $46;

    if szText = 'G' then
        Result := $47;

    if szText = 'H' then
        Result := $48;

    if szText = 'I' then
        Result := $49;

    if szText = 'J' then
        Result := $4A;

    if szText = 'K' then
        Result := $4B;

    if szText = 'L' then
        Result := $4C;

    if szText = 'M' then
        Result := $4D;

    if szText = 'N' then
        Result := $4E;

    if szText = 'O' then
        Result := $4F;

    if szText = 'P' then
        Result := $50;

    if szText = 'Q' then
        Result := $51;

    if szText = 'R' then
        Result := $52;

    if szText = 'S' then
        Result := $53;

    if szText = 'T' then
        Result := $54;

    if szText = 'U' then
        Result := $55;

    if szText = 'V' then
        Result := $56;

    if szText = 'W' then
        Result := $57;

    if szText = 'X' then
        Result := $58;

    if szText = 'Y' then
        Result := $59;

    if szText = 'Z' then
        Result := $5A;

    if szText = '0' then
        Result := $30;

    if szText = '1' then
        Result := $31;

    if szText = '2' then
        Result := $32;

    if szText = '3' then
        Result := $33;

    if szText = '4' then
        Result := $34;

    if szText = '5' then
        Result := $35;

    if szText = '6' then
        Result := $36;

    if szText = '7' then
        Result := $37;

    if szText = '8' then
        Result := $38;

    if szText = '9' then
        Result := $39;

    if szText = 'NumPad 0' then
        Result := VK_NUMPAD0;

    if szText = 'NumPad 1' then
        Result := VK_NUMPAD1;

    if szText = 'NumPad 2' then
        Result := VK_NUMPAD2;

    if szText = 'NumPad 3' then
        Result := VK_NUMPAD3;

    if szText = 'NumPad 4' then
        Result := VK_NUMPAD4;

    if szText = 'NumPad 5' then
        Result := VK_NUMPAD5;

    if szText = 'NumPad 6' then
        Result := VK_NUMPAD6;

    if szText = 'NumPad 7' then
        Result := VK_NUMPAD7;

    if szText = 'NumPad 8' then
        Result := VK_NUMPAD8;

    if szText = 'NumPad 9' then
        Result := VK_NUMPAD9;

    if szText = 'F1' then
        Result := VK_F1;

    if szText = 'F2' then
        Result := VK_F2;

    if szText = 'F3' then
        Result := VK_F3;

    if szText = 'F4' then
        Result := VK_F4;

    if szText = 'F5' then
        Result := VK_F5;

    if szText = 'F6' then
        Result := VK_F6;

    if szText = 'F7' then
        Result := VK_F7;

    if szText = 'F8' then
        Result := VK_F8;

    if szText = 'F9' then
        Result := VK_F9;

    if szText = 'F10' then
        Result := VK_F10;

    if szText = 'F11' then
        Result := VK_F11;

    if szText = 'F12' then
        Result := VK_F12;
end;

end.
