unit MainForm;

{  ******
   *
   * Module:    MainForm
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    Main user interface to Rocks.
   *
   * Details:
   *
   *    This form is the main application window, and contains the highest-level
   *    code required to drive the game.  The movement and drawing engines are
   *    invoked in response to Window timers, and user input via menus is
   *    passed down to the animation engine.
   *
   *    Notice that this form is nothing more than an interface layer between
   *    the player and the animation engine.
   *
   * Notes:
   *
   *    You may notice that the form is resizable.  For a cool effect, try
   *    resizing the screen during game play.
   *
   ****** }

interface

uses GameCtl, State, Keys, RocksKbd, KeyBoard,
     SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
     Forms, Dialogs, ExtCtrls, Menus, MPlayer;

type
  TRocksForm = class(TForm)
    m_mnuMainMenu: TMainMenu;
    m_mnuFile: TMenuItem;
    m_mniFileNew: TMenuItem;
    m_mniFileExit: TMenuItem;
    m_mniFileSep0: TMenuItem;
    m_mnuOptions: TMenuItem;
    m_mnuHelp: TMenuItem;
    m_mniHelpAbout: TMenuItem;
    m_tmrMovement: TTimer;
    m_mniFilePause: TMenuItem;
    m_mniOptionsCustomize: TMenuItem;
    m_mniOptionsSound: TMenuItem;
    m_mniOptionsSep0: TMenuItem;

    procedure m_tmrMovementTimer(Sender: TObject);
    procedure m_mniFileExitClick(Sender: TObject);
    procedure m_mniFileNewClick(Sender: TObject);
    procedure m_mniHelpAboutClick(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FormKeyUp(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FormCreate(Sender: TObject);
    procedure m_mniOptionsCustomizeClick(Sender: TObject);
    procedure m_mniFilePauseClick(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure m_mniOptionsSoundClick(Sender: TObject);

    private

        m_ctlGame: TGameControl;        { Game logic sequence controller. }

    public

        { Method to start the animation engine. }
        procedure StartGameEngine;
end;

implementation

{$R *.DFM}

uses Global, About;

{ Invoked by the operating system at regular intervals via timer. }
procedure TRocksForm.m_tmrMovementTimer(Sender: TObject);
begin
    { If the current game is "dead", restart it now.  Note that the game engine
       is designed to go through only one iteration of game play.  When a game
       is completed, the entire control engine is released and re-allocated. }
    if (m_ctlGame.bGameDead) then
        begin
        StartGameEngine;
        m_mniFileNew.Enabled := True;
        end;

    { Perform movements on all objects in the universe, and draw all objects
      in the universe in their current state. }
    if (m_ctlGame <> nil) then
    begin
        m_ctlGame.Move;
        m_ctlGame.DrawFrame;
    end;
end;

procedure TRocksForm.m_mniFileExitClick(Sender: TObject);
begin
    { Close out the application. }
    Close;
end;

procedure TRocksForm.m_mniFileNewClick(Sender: TObject);
begin
    { Tell the game control object to begin a new game. }
    m_ctlGame.StartNewGame;
    m_mniFileNew.Enabled := False;
end;

procedure TRocksForm.m_mniHelpAboutClick(Sender: TObject);
var
   frmAbout: TAboutBox;
begin
   { Pause the game while the dialog is showing. }
   m_ctlGame.bGamePaused := True;

   { Create and show the about form. }
   frmAbout := TAboutBox.Create(Self);
   frmAbout.ShowModal;

   { Free the about box. }
   frmAbout.Free;
end;

procedure TRocksForm.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
    { Pass the message down to the current input device object in the engine. }
    TWindowsKeyboard(g_envEnviron.devGetInputDevice).ProcessKeyDown(Key, Shift);
end;

procedure TRocksForm.FormKeyUp(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
    { Pass the message down to the current input device object in the engine. }
    TWindowsKeyboard(g_envEnviron.devGetInputDevice).ProcessKeyUp(Key, Shift);
end;

procedure TRocksForm.FormCreate(Sender: TObject);
var
   iDefaultLeft, iDefaultTop: Integer;
begin
    { Recall our size and shape from last time. }
    Width := g_envEnviron.iGetINIInteger('Screen', 'Width', 500);
    Height := g_envEnviron.iGetINIInteger('Screen', 'Height', 400);

    { The default screen position will be the center of the screen. }
    iDefaultLeft := (Screen.Width - Width) div 2;
    iDefaultTop := (Screen.Height - Height) div 2;
    Left := g_envEnviron.iGetINIInteger('Screen', 'Left', iDefaultLeft);
    Top := g_envEnviron.iGetINIInteger('Screen', 'Top', iDefaultTop);

    { Start a new game logic controller. }
    m_ctlGame := nil;
end;

procedure TRocksForm.m_mniOptionsCustomizeClick(Sender: TObject);
var
   frmKeySelect: TKeySelection;
begin
   { Pause the game while the dialog is showing. }
   m_ctlGame.bGamePaused := True;

   { Construct a control selection dialog object. }
   frmKeySelect := TKeySelection.Create(nil);

   { Tell the form which device to configure and show it! }
   frmKeySelect.UseKeyboard(TRocksKeyboard(g_envEnviron.devGetInputDevice));
   frmKeySelect.ShowModal;

   { Release the form object that was created. }
   frmKeySelect.Free;
end;

procedure TRocksForm.m_mniFilePauseClick(Sender: TObject);
begin
    { Pause the game engine. }
    m_ctlGame.bGamePaused := not m_ctlGame.bGamePaused;
end;

procedure TRocksForm.FormResize(Sender: TObject);
begin
    { Upon a resize, just clear of the current contents of the screen to avoid
      cluttering the new surface before the next frame is drawn. }
    if (m_ctlGame <> nil) then
        g_envEnviron.devGetVideoDevice.ClearFrame;
end;

procedure TRocksForm.FormDestroy(Sender: TObject);
begin
    { Store the current position of the form. }
    g_envEnviron.SetINIString('Screen', 'Left', IntToStr(Left));
    g_envEnviron.SetINIString('Screen', 'Top', IntToStr(Top));
    g_envEnviron.SetINIString('Screen', 'Width', IntToStr(Width));
    g_envEnviron.SetINIString('Screen', 'Height', IntToStr(Height));

    { Free the game control object. }
    m_ctlGame.Free;
end;

procedure TRocksForm.StartGameEngine;
begin
    { Create the game logic control object.  Free the previous one, if there
      was one.  Notice the Free method is safe for NULL objects. }
    m_ctlGame.Free;
    m_ctlGame := TGameControl.Create;

    { Start showing the game's splash screen. }
    m_ctlGame.DisplaySplashScreen;
end;

procedure TRocksForm.m_mniOptionsSoundClick(Sender: TObject);
begin
    { Toggle the current sound state. }
    m_mniOptionsSound.Checked := not m_mniOptionsSound.Checked;
    g_envEnviron.devGetSoundDevice.bSoundEnabled := m_mniOptionsSound.Checked;
end;

end.

