unit Shape;

{  ******
   *
   * Module:    Shape
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    This module describes constructs necessary to represent polygon shapes
   *    used by visible objects.  This include points, edge descriptors (which
   *    are different that edges), and TShapes to contain the all.  The ShapeLib
   *    unit defines a global container for all game shapes.
   *
   ****** }

interface
uses Classes, WinProcs, SysUtils;

{ This record defines a precise position in some arbitrary coordinate system. }
type TFinePoint = record
    x: Real;
    y: Real;
end;
type PFinePoint = ^TFinePoint;

{ This record describes a connection between two points in space. }
type TEdgeDescriptor = record
    { Vertex numbers that combine to create an edge. }
    iVertex1: Integer;
    iVertex2: Integer;

    { Actual endpoints of the edge. }
    pt1: PFinePoint;
    pt2: PFinePoint;
end;
type PEdgeDescriptor = ^TEdgeDescriptor;


{ The TShape class defines a sinple shape used in the game. } 
type TShape = class(TObject)
    private
        m_szName: String;       { Name assigned to the shape. }
        m_lstVertices: TList;   { List of vertices used in this shape. }
        m_lstEdges: TList;      { List of edges that connect the vertices together. }

    public
        { Our constructor and destructor. }
        constructor Create;
        destructor Destroy;

        { Method for defining a vertex or an edge. }
        procedure DefineVertex(pt: TFinePoint);
        procedure DefineEdge(iVertex1: Integer; iVertex2: Integer);

        { Public properties. }
        property szName: String     Read m_szName       Write m_szName;
        property lstVertices: TList Read m_lstVertices;
        property lstEdges: TList    Read m_lstEdges;
end;

implementation

constructor TShape.Create;
begin
    { Perform default processing. }
    inherited Create;

    { Create lists to manage vertices and edges. }
    m_lstVertices := TList.Create;
    m_lstEdges := TList.Create;
end;

destructor TShape.Destroy;
var
    iIndex: Integer;
begin
    { Release our vertex list. }
    for iIndex := 0 to (m_lstVertices.Count - 1) do
        Dispose(m_lstVertices.Items[iIndex]);
    m_lstVertices.Free;

    { Release our edge list. }
    for iIndex := 0 to (m_lstEdges.Count - 1) do
        Dispose(m_lstEdges.Items[iIndex]);
    m_lstEdges.Free;

    { Perform default processing. }
    inherited Destroy;
end;

procedure TShape.DefineVertex(pt: TFinePoint);
var
    pptCopy: PFinePoint;
begin
    { Create a duplicate of the given point. }
    New(pptCopy);
    pptCopy^ := pt;

    { Add the point to the vertex list. }
    m_lstVertices.Add(pptCopy);
end;

procedure TShape.DefineEdge(iVertex1: Integer; iVertex2: Integer);
var
    pedEdge: PEdgeDescriptor;
begin
    { Validate the vertex numbers that were given. }
    if (iVertex1 < 0) or (iVertex1 >= m_lstVertices.Count) or
       (iVertex2 < 0) or (iVertex2 >= m_lstVertices.Count) then
        raise Exception.Create('Invalid vertex indentifier');
        
    { Create a duplicate of the given point. }
    New(pedEdge);
    pedEdge^.pt1 := m_lstVertices.Items[iVertex1];
    pedEdge^.pt2 := m_lstVertices.Items[iVertex2];
    pedEdge^.iVertex1 := iVertex1;
    pedEdge^.iVertex2 := iVertex2;

    { Add the point to the vertex list. }
    m_lstEdges.Add(pedEdge);
end;

end.
