unit ShapeLib;

{  ******
   *
   * Module:    ShapeLib
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    This module manages all polygon shapes used by the game.  This includes
   *    Bart, Aliens, and all letters comprising the game font.  When an object
   *    wants to incorporate a shape into its look, the shape is retreived from
   *    the global TShapeLibrary object.
   *
   ****** }

interface
uses Shape, WinProcs, Classes, SysUtils;

type TShapeLibrary = class(TObject)
    public
        { Class constructor and destructor. }
        constructor Create;
        destructor Destroy;

        { Method to define a new shape as a series of vertices and edges. }
        procedure DefineShape(szShapeName: String;
                              aptVertices: Array of TFinePoint;
                              aeEdges: Array of LongInt);

        { Method to retrieve a defined shape from the ShapeLib. }
        function sGetShape(szShapeName: String): TShape;

    private

        m_lstShapes: TList;     { List of shapes that have been defined. }
end;

{ These global functions make it easier to define shapes. }
function Edge(iVertex1: Integer; iVertex2: Integer): LongInt;
function FPoint(fX: Real; fY: Real): TFinePoint;

{ Global shape library object. }
var
    g_slShapeLib: TShapeLibrary;

implementation

function Edge(iVertex1: Integer; iVertex2: Integer): LongInt;
begin
    { Pack the vertices into a single LongInt, shifting the first one into
      the high bytes of the fullword. }
    Result := (iVertex1 shl 16) + iVertex2;
end;

function FPoint(fX: Real; fY: Real): TFinePoint;
var
    fp: TFinePoint;
begin
    fp.x := fX;
    fp.y := fY;
    Result := fp;
end;

constructor TShapeLibrary.Create;
begin
    { Perform default processing. }
    inherited Create;

    { Create lists to manage shapes. }
    m_lstShapes := TList.Create;
end;

destructor TShapeLibrary.Destroy;
var
    iIndex: Integer;
begin
    { Release our shape list. }
    for iIndex := 0 to (m_lstShapes.Count - 1) do
        TShape(m_lstShapes.Items[iIndex]).Free;
    m_lstShapes.Free;

    { Perform default processing. }
    inherited Destroy;
end;

procedure TShapeLibrary.DefineShape(szShapeName: String;
                                    aptVertices: Array of TFinePoint;
                                    aeEdges: Array of LongInt);
var
    objShape: TShape;
    iIndex: Integer;
begin
    { Create and initialize a new shape object. }
    objShape := TShape.Create;
    objShape.szName := szShapeName;

    { Define vertices in the shape. }
    for iIndex := Low(aptVertices) to High(aptVertices) do
        objShape.DefineVertex(aptVertices[iIndex]);

    { Define the edges that connect the vertex points. }
    for iIndex := Low(aeEdges) to High(aeEdges) do
        objShape.DefineEdge(aeEdges[iIndex] shr 16, aeEdges[iIndex] and $ffff);

    { Add the shape to the list. }
    m_lstShapes.Add(objShape);
end;

function TShapeLibrary.sGetShape(szShapeName: String): TShape;
var
    iIndex: Integer;
begin
    { Search for a shape with the given name. }
    for iIndex := 0 to (m_lstShapes.Count - 1) do
    begin
        { Get a shape from the current list. }
        Result := m_lstShapes.Items[iIndex];

        { If the name matches, return a pointer to it. }
        if (CompareText(szShapeName, Result.szName) = 0) then
            Exit;
    end;

    { Return NULL if no shapes have the given name. }
    Result := nil;
end;

end.
