unit State;

{  ******
   *
   * Module:    State
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    The TStateAware class is the ancestor of all game objects.  It implements
   *    a "state queue" or "command queue" that allows that object to perform
   *    actions asynchronously and in order.
   *
   ****** }

interface
uses Message, Classes;

{ All state commands used by objects in Rocks. }
type enumState = (Undefined,
                  BecomeInvisible,
                  BecomeVisible,
                  BeginGame,
                  Credits,
                  Die,
                  DisableCollisions,
                  DisplayBonus,
                  DisplayLevel,
                  EnableCollisions,
                  EndChannelState,
                  EnterHyperSpace,
                  EnterName,
                  ExitHyperSpace,
                  FindClearRect,
                  FreeGroupedObjects,
                  GameDead,
                  GameOver,
                  GenerateAsteroids,
                  GenerateBart,
                  GetReady,
                  HighScore,
                  Pause,
                  ShowBart,
                  Splash,
                  StopMovement,
                  TakeDownHighScores,
                  UserControl,
                  WaitForClearRect);


{ Object state tracking structure. }
type TState = Record
    enumCommandState: enumState;
    fValue1, fValue2: Real;
end;
type PState = ^TState;

{ Class definition for a state-aware object. }
type TStateAware = class(TMessageAware)
    public
        { Class constructor and destructor. }
        constructor Create;
        destructor Destroy; Override;

        { Methods to respond to commands in the state queue. }
        procedure AddToStateSequence(enumNewState: enumState; fValue1, fValue2: Real);
        procedure ClearStateSequence;
        procedure ExitCurrentState;

        { Virtual method called to process pending states. }
        procedure Move; Virtual;

    protected

        { Routine to deactivate the currently active state. }
        procedure DeactivateCurrentState;

        { Virtual methods for customizing state behavior. }
        procedure EnterState(pstCurrent: PState); Virtual;
        procedure ExitState(pstCurrent: PState); Virtual;
        procedure ProcessState(pstCurrent: PState); Virtual;

        { Informational methods. }
        function bIsProcessingState: Boolean;
        function stGetCurrentState: enumState;
        function bIsInState(enumStateToTest: enumState): Boolean;

    private

        m_lstPendingStates: TList;      { Queue of pending commands. }
        m_stCurrent: PState;            { Command currently being processed. }
        m_bProcessingState: Boolean;    { TRUE when we're processing command. }
        m_bPendingStateExit: Boolean;   { TRUE when the current state has an exit pending. }
        m_bStatesPending: Boolean;      { TRUE when one or more states is pending. }
end;

implementation
uses General;

constructor TStateAware.Create;
begin
    inherited Create;

    { Initialize the pending state queue and current state flag. }
    m_lstPendingStates := TList.Create;
    m_bProcessingState := False;
    m_bPendingStateExit := False;
    m_bStatesPending := False;
end;

destructor TStateAware.Destroy;
begin
    { Release all pending state tracking structures. }
    ExitCurrentState;
    ClearStateSequence;

    { Release our state queue. }
    m_lstPendingStates.Free;

    inherited Destroy;
end;

procedure TStateAware.AddToStateSequence(enumNewState: enumState; fValue1, fValue2: Real);
var
    pstNew: PState;
begin
    { Create a new command structure and initialize it. }
    New(pstNew);
    pstNew^.enumCommandState := enumNewState;
    pstNew^.fValue1 := fValue1;
    pstNew^.fValue2 := fValue2;

    { Add the command to the current queue. }
    m_lstPendingStates.Add(pstNew);

    { Record that the object is now actively listening to states. }
    m_bStatesPending := True;
end;

procedure TStateAware.ClearStateSequence;
var
    iIndex: Integer;
begin
    { Exit the current state, if we are in one. }
    ExitCurrentState;

    { Iterate through and remove each state tracking structure in the queue. }
    for iIndex := (m_lstPendingStates.Count - 1) downto 0 do
        m_lstPendingStates.Delete(iIndex);
end;

procedure TStateAware.ExitCurrentState;
begin
    { Process the exit on the next movement. }
    if (m_bProcessingState = True) then
        m_bPendingStateExit := True;
end;

procedure TStateAware.DeactivateCurrentState;
var
    stNext: PState;
begin
    { If we're not processing a command, then exit. }
    if (m_bProcessingState = False) then
        Exit;

    { Cleanup from processing the current state. }
    ExitState(m_stCurrent);

    { Dispose of the current state tracking structure. }
    Dispose(m_stCurrent);

    { We are no longer processing the command. }
    m_bProcessingState := False;
    m_bPendingStateExit := False;
end;

{ Note: A maximum of one state change will occur during each iteration.
        Derived classes use this information to perform further processing. }
procedure TStateAware.Move;
begin
    { If no states have been queued at all then don't keep wasting time!!! }
    if m_bStatesPending = False then
        Exit;

    { Continue any state that has previously been entered. }
    if m_bPendingStateExit = True then
        DeactivateCurrentState
    else
        if (m_bProcessingState = True) then
            ProcessState(m_stCurrent)
        else
           { Enter the next pending state, if it is time. }
           if (m_bProcessingState = False) and (m_lstPendingStates.Count > 0) then
           begin
               { Get the next pending state. }
               { Remove the first node from the state list, since it is now current. }
               m_stCurrent := m_lstPendingStates.Items[0];
               m_lstPendingStates.Delete(0);

               { We are now processing a command. }
               m_bProcessingState := True;
               EnterState(m_stCurrent);
           end;
end;

procedure TStateAware.EnterState(pstCurrent: PState);
begin
    { The Pause command is inmplemented at this level. }
    case m_stCurrent^.enumCommandState of
        Pause:   { Pause state processing. }
            begin
            m_stCurrent^.fValue2 := fGetNow; { Record time of pause start. }
            end;
    else
    end;
end;

procedure TStateAware.ExitState(pstCurrent: PState);
begin
end;

function TStateAware.bIsProcessingState: Boolean;
begin
    Result := m_bProcessingState;
end;

function TStateAware.stGetCurrentState: enumState;
begin
    if m_bProcessingState = True then
        Result := m_stCurrent^.enumCommandState
    else
        Result := Undefined;
end;

procedure TStateAware.ProcessState(pstCurrent: PState);
var
    fInterval: Real;
begin
    case m_stCurrent^.enumCommandState of
        Pause:
            begin
            { Compute the interval since the pause started. }
            fInterval := fGetNow - m_stCurrent^.fValue2;

            { If the entire pause interval has passed, then end the pause. }
            if fInterval >= m_stCurrent^.fValue1 then
               ExitCurrentState;
            end;
    else
    end;
end;

function TStateAware.bIsInState(enumStateToTest: enumState): Boolean;
begin
    { Check if we're processing a command. }
    if (bIsProcessingState = True) and (stGetCurrentState = enumStateToTest) then
        Result := True
    else
        Result := False;
end;

end.
