unit VideoDev;

{  ******
   *
   * Module:    VideoDev
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    This module defines the base class for all video output objects.
   *    To allow Rocks to operate with a new type of video driver, a developer
   *    can simply derive a new class from this one, and implement all of the
   *    abstracted methods.
   *
   ****** }

interface

uses Graphics;

type TVideoDevice = class(TObject)
    public
        { Public constructor. }
        constructor Create(iInternalFrameWidth, iInternalFrameHeight: Integer);

        { Method to initialize prior to drawing a new frame. }
        procedure BeginNewFrame;
        procedure ClearFrame; Virtual; Abstract;

        { Methods to move the pen position and draw lines. }
        procedure MoveTo(iX, iY: Integer); Virtual; Abstract;
        procedure LineTo(iX, iY: Integer); Virtual; Abstract;

        { Method to set a single pixel. }
        procedure SetPixel(iX, iY: Integer; clrPixel: TColor); Virtual; Abstract;

        { Method to set the current pen color. }
        procedure SetPenColor(clrNew: TColor); Virtual; Abstract;

        { functions to return the current physical screen dimensions.  Notice
          that these can change dynamically at run time. }
        function iGetPhysicalScreenHeight: Integer; Virtual; Abstract;
        function iGetPhysicalScreenWidth: Integer; Virtual; Abstract;

        { Functions to return the logical screen dimensions.  Notice that these
          always stay the same in Rocks, providing an internal reference point.
          During drawing, images are scaled appropriately on the fly. }
        function iGetLogicalScreenHeight: Integer;
        function iGetLogicalScreenWidth: Integer;

        { Methods to convert from logical to physical screen coordinates. }
        function fToPhysicalX(iCoordX: Real): Real;
        function fToPhysicalY(iCoordY: Real): Real;

    private

        { Internal frame dimensions. }
        m_iInternalFrameWidth: Integer;
        m_iInternalFrameHeight: Integer;

        { Conversion factors for translating logical coordinates to physical ones. }
        m_fLogicalToPhysicalX: Real;
        m_fLogicalToPhysicalY: Real;
end;

implementation

constructor TVideoDevice.Create(iInternalFrameWidth, iInternalFrameHeight: Integer);
begin
    { Perform default initializations. }
    inherited Create;

    { Stash our logical screen dimensions.  These will not change. }
    m_iInternalFrameWidth := iInternalFrameWidth;
    m_iInternalFrameHeight := iInternalFrameHeight;
end;

function TVideoDevice.iGetLogicalScreenHeight: Integer;
begin
    Result := m_iInternalFrameHeight;
end;

function TVideoDevice.iGetLogicalScreenWidth: Integer;
begin
    Result := m_iInternalFrameWidth;
end;

function TVideoDevice.fToPhysicalX(iCoordX: Real): Real;
begin
    Result := iCoordX * m_fLogicalToPhysicalX;
end;

function TVideoDevice.fToPhysicalY(iCoordY: Real): Real;
begin
    Result := iCoordY * m_fLogicalToPhysicalY;
end;

procedure TVideoDevice.BeginNewFrame;
begin
    { Compute logical->physical scaling factors.  This is done only once per
      frame for speed, which means that resizing the screen during frame drawing
      can potentially mess up the display briefly. }
    m_fLogicalToPhysicalX := iGetPhysicalScreenWidth / iGetLogicalScreenWidth;
    m_fLogicalToPhysicalY := iGetPhysicalScreenHeight / iGetLogicalScreenHeight;
end;

end.

