unit Visible;

{  ******
   *
   * Module:    Visible
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    This module defines TVisibleObject, the base class for all objects that
   *    eppear on the screen.  This is one of the more complicated classes in
   *    the application, encapsulating all of the core behvaior required by
   *    objects in the universe.
   *
   ******}

interface

Uses SoundDev, State, MatCtrl, Classes, Vertex, Edge, Constant, VideoDev,
     WinTypes, WinProcs, Graphics, ShapeLib, Shape, SysUtils;

type TVisibleObject = class(TStateAware)
    private
        m_devVideo: TVideoDevice;   { Current video output object. }
        m_lstUniverse: TList;       { List of all objects in the universe. }
        m_iCollisionID: Integer;    { Collision ID, determines who can hit me. }

        m_lstVertices: TList;       { List of vertices that define the object's shape. }
        m_lstEdges: TList;          { List of edges between vertices. }
        m_iNextShapeID: Integer;    { ID to be associated with the next shape defined. }

        m_bVisible: Boolean;        { TRUE of the object is currently visible. }
        m_bDrawnBefore: Boolean;    { TRUE if the object has been drawn at least once. }
        m_bCurrentlyDrawn: Boolean; { TRUE if the object is currently on the screen. }
        m_bCanCollide: Boolean;     { TRUE if the object is eligible for collisions. }
        m_bObjectDead: Boolean;     { TRUE when the object can be released. }
        m_bShapeChanged: Boolean;   { TRUE if the object shape has changed recently. }

        m_mtrxTransform: TTemporalMatrix2D;   { Master tranformation matrix. }
        m_iGroupID: Integer;        { ID of control group this object belongs to. }

        { Values used to control color flashing. }
        m_bCurrentlyFlashing: Boolean;  { TRUE if the object is flashing wildly. }
        m_rgbCurrentFlashColor: TColor; { Current flash color from a random set. }
        m_fLastColorChange: Real;       { Time that the last color change occurred. }

    public

        { Constructor and destructor. }
        constructor Create(lstUniverse: TList;      { List of ALL objects. }
                           iCollisionID: Integer);  { Collision ID to be used. }
        destructor Destroy; Override;

        { Method to get a rectangle that bounds the entire object at run-time. }
        function rectGetBoundingRect: TRect;

        { Method to check if the given object is colliding with this one.  This
          method implements a polygon intersection algorithm to determine this. }
        function bIsObjectColliding(objObject: TVisibleObject): Boolean;

        { Methods to signal the death of an object.  The death can be made more
          dramatic by having the object shatter into a billion shards. }
        procedure KillObject(bShatter: Boolean; iDuration: Integer; fMaxScatter: Real; fMaxSpin: Real; fFragmentationRate: Real);
        procedure ShatterVertices(fDirection: Real;
                                  iDuration: Integer;
                                  fMaxScatter: Real;
                                  fMaxSpin: Real;
                                  fFragmentationRate: Real);

        { Method to draw and erase this polygon. }
        procedure Draw;
        procedure Erase;

        { Methods to begin and end random color flashing.  This looks great
          when used to highlight text items like bonuses, etc.. }
        procedure BeginFlashing;
        procedure EndFlashing;

        { Method providing default behavior in response to collisions. }
        procedure HandleCollision(objOtherObject: TVisibleObject); Virtual;

        { Method to check if the given screen rectangle is large enough for this
          object to appear in without colliding with object objects. }
        function bIsRectSafe(rectArea: TRect): Boolean;

        { Methods to get and set the polygon's visible state.  An object can be
          constructed from multiple polygons, each of which has its own state. }
        procedure SetShapeVisible(m_iShapeID: Integer; bNewState: Boolean);

        { Functions for defining new edges and vertices. }
        function vtxDefineNewVertex(fLocalX, fLocalY: Real): TVertex;
        function edgeDefineNewEdge(vtxOne, vtxTwo: TVertex; clrColor: TColor; bIncludeInCollisions: Boolean): TEdge;
        procedure ClearShape;

        { Default movement behavior for visible objects. }
        procedure Move; Override;
        procedure Wrap;

        { Method to declare a defined shape as a part of the object structure. }
        function IncludeShape(szShapeFile: String;
                              iRelativeX, iRelativeY: Integer;
                              clrEdge: TColor;
                              bIncludeInCollisions: Boolean): Integer;

        { Inherited methods that allow the visible object to respond to commands
          embedded in a state queue. }
        procedure EnterState(pCommand: PState); Override;
        procedure ProcessState(pstCurrent: PState); Override;

        { Exposed properties. }
        property bVisible: Boolean                  Read m_bVisible         Write m_bVisible;
        property bCanCollide: Boolean               Read m_bCanCollide      Write m_bCanCollide;
        property bObjectDead: Boolean               Read m_bObjectDead      Write m_bObjectDead;
        property lstVertices: TList                 Read m_lstVertices      Write m_lstVertices;
        property lstEdges: TList                    Read m_lstEdges         Write m_lstEdges;
        property iGroupID: Integer                  Read m_iGroupID         Write m_iGroupID;
        property iCollisionID: Integer              Read m_iCollisionID     Write m_iCollisionID;
        property devVideo: TVideoDevice             Read m_devVideo;
        property lstUniverse: TList                 Read m_lstUniverse;
        property mtrxTransform: TTemporalMatrix2D   Read m_mtrxTransform
                                                    Write m_mtrxTransform;

private

        { Method to process an object that is flashing wildly. }
        procedure ProgressFlashing;

        { Method to trim unused vertices from the object, in order to save
          execution time when transforming vertices. }
        procedure DumpUnusedVertices;

        { Method to compute the vertex positions based upon the object's current
          position and orientation. }
        procedure TransformAndTranslate;
end;

implementation

uses Scrap, General, Global;

constructor TVisibleObject.Create(lstUniverse: TList; iCollisionID: Integer);
begin
    { Perform default processing. }
    inherited Create;

    { Set up default values. }
    m_devVideo := g_envEnviron.devGetVideoDevice;
    m_lstUniverse := lstUniverse;
    m_iCollisionID := iCollisionID;
    m_bVisible := True;
    m_bDrawnBefore := False;
    m_bCurrentlyDrawn := False;
    m_bObjectDead := False;
    m_bCanCollide := True;
    m_bShapeChanged := True;
    m_iNextShapeID := 0;
    m_iGroupID := -1;

    { Create vertex and edge lists. }
    m_lstVertices := TList.Create;
    m_lstEdges := TList.Create;

    { Set up a transformation matrix for processing vertices in real-time. }
    m_mtrxTransform := TTemporalMatrix2D.Create;
end;

destructor TVisibleObject.Destroy;
begin
    { Delete our edge and vertex lists. }
    ClearShape;
    m_lstVertices.Free;
    m_lstEdges.Free;

    { Perform default cleanup processing. }
    inherited Destroy;
end;

{ This method gets called when the program wants to explicitly kill the object.
  The object can just disappear, or shatter violently.  These parameters control
  the varacity of the explosion. } 
procedure TVisibleObject.KillObject(bShatter: Boolean;
                                    iDuration: Integer;
                                    fMaxScatter: Real;
                                    fMaxSpin: Real;
                                    fFragmentationRate: Real);
begin
    { If requested, shatter the object into billion (roughly) pieces. }
    if bShatter = True then
        ShatterVertices(mtrxTransform.fDirection, iDuration, fMaxScatter, fMaxSpin, fFragmentationRate);

    { Erase the object and mark it dead.  The game control objet will
      eventually free the object at a convenient time. }
    Erase;
    m_bObjectDead := True;
end;

{ This method is called by inherited object types such as TAlien and TBart to
  define their shape.  The shape name that is passed in must be defined in the
  global shape library. }
function TVisibleObject.IncludeShape(szShapeFile: String;
                                     iRelativeX, iRelativeY: Integer;
                                     clrEdge: TColor;
                                     bIncludeInCollisions: Boolean): Integer;
var
    sShape: TShape;
    pedEdge: PEdgeDescriptor;
    iIndex: Integer;

    aiVertexIndex: array[0..255] of TVertex;
    edgeNew: TEdge;

    fLocalX, fLocalY: Real;
begin
    { Get the shape from the shape library. }
    sShape := g_slShapeLib.sGetShape(szShapeFile);
    if (sShape = nil) then
         Raise Exception.Create('Invalid shape name');

    { Read every vertex described in the file. }
    for iIndex := 0 to (sShape.lstVertices.Count - 1) do
    begin
        { Get local X and Y coordinates of the vertex. }
        fLocalX := PFinePoint(sShape.lstVertices.Items[iIndex])^.x;
        fLocalY := PFinePoint(sShape.lstVertices.Items[iIndex])^.y;

        { Create a vertex around the center of this object. }
        aiVertexIndex[iIndex] :=
           vtxDefineNewVertex(fLocalX + iRelativeX, fLocalY + iRelativeY);
   end;

   { Read every edge described in the file. }
   for iIndex := 0 to (sShape.lstEdges.Count - 1) do
   begin
       { Get a description of the edge. }
       pedEdge := sShape.lstEdges.Items[iIndex];

       { Create a new edge object, and assign vertices as described }
       { in the first part of the file... }
       edgeNew := edgeDefineNewEdge(aiVertexIndex[pedEdge.iVertex1],
                                    aiVertexIndex[pedEdge.iVertex2],
                                    clrEdge,
                                    bIncludeInCollisions);

       { Set up default coloring and visiblity information. }
       edgeNew.m_iShapeID := m_iNextShapeID;
    end;

    { For speed optimization, clear out unused vertices. }
    DumpUnusedVertices;
    m_bShapeChanged := True;

    { Return a unique shape ID and generate the next one. }
    Result := m_iNextShapeID;
    m_iNextShapeID := m_iNextShapeID + 1;
end;

{ This method will calculate the bounding screen rectangle that contains the
  object in its current state.  This logic works no matter how the object has
  been scaled or rotated. }
function TVisibleObject.rectGetBoundingRect: TRect;
var
   iIndex: Integer;
   iMinX, iMaxX: Integer;
   iMinY, iMaxY: Integer;
   vtxCurrent: TVertex;

   fOriginX, fOriginY: Real;
begin
   { Record the current translation origin of this object. }
   fOriginX := mtrxTransform.fTranslationX;
   fOriginY := mtrxTransform.fTranslationY;

   { Start out with some extreme values that match the object's translation origin. }
   iMinX := Round(fOriginX);
   iMaxX := Round(fOriginX);
   iMinY := Round(fOriginY);
   iMaxY := Round(fOriginY);

   { Find the extreme X and Y coordinates. }
   if (m_bDrawnBefore = True) then
      for iIndex := 0 to (m_lstVertices.Count - 1) do
      begin
         vtxCurrent := m_lstVertices.Items[iIndex];

         { Check if this vertex defines a new extreme. }
         if (vtxCurrent.m_iLastDrawnX < iMinX) then
            iMinX := vtxCurrent.m_iLastDrawnX;
         if (vtxCurrent.m_iLastDrawnX > iMaxX) then
            iMaxX := vtxCurrent.m_iLastDrawnX;
         if (vtxCurrent.m_iLastDrawnY < iMinY) then
            iMinY := vtxCurrent.m_iLastDrawnY;
         if (vtxCurrent.m_iLastDrawnY > iMaxY) then
            iMaxY := vtxCurrent.m_iLastDrawnY;
      end;

   { Store the resulting rectangle that the extremes form. }
   Result.Left := iMinX - 1;
   Result.Right := iMaxX + 1;
   Result.Top := iMinY - 1;
   Result.Bottom := iMaxY + 1;
end;

{ Since an object can consist of multiple combined shapes, this method allows
  an object to make an individual shape visible or invisible.  A perfect
  example is TBart's engine flame, which is visible ONLY when the player
  is firing the thrusters. }
procedure TVisibleObject.SetShapeVisible(m_iShapeID: Integer; bNewState: Boolean);
var
   iEdgeIndex: Integer;
   edgeShape: TEdge;
begin
    { Iterate through all edges comprising the object. }
    for iEdgeIndex := 0 to (m_lstEdges.Count - 1) do
    begin
        { If the edge is part of the given shape, set is visibility. }
        edgeShape := m_lstEdges.Items[iEdgeIndex];
        if edgeShape.m_iShapeID = m_iShapeID then
            edgeShape.m_bVisible := bNewState;
    end;
end;

procedure TVisibleObject.Erase;
var
   iIndex: Integer;
   edgeLine: TEdge;
begin
    { If we're not currently showing then don't erase! }
    if (m_bCurrentlyDrawn = False) then
        Exit;

    { Otherwise, erase all edges comprising the object. }
    m_devVideo.SetPenColor(clrGetBackgroundColor);

    { Move around to every edge in the shape. }
    for iIndex := 0 to (m_lstEdges.Count - 1) do
    begin
        { Get the two vertices that must be connected to form an edge. }
        edgeLine := m_lstEdges.Items[iIndex];

        { Connect the two edges using a line. }
        m_devVideo.MoveTo(edgeLine.m_vtxOne.m_iLastDrawnX, edgeLine.m_vtxOne.m_iLastDrawnY);
        m_devVideo.LineTo(edgeLine.m_vtxTwo.m_iLastDrawnX, edgeLine.m_vtxTwo.m_iLastDrawnY);
    end;

    m_bCurrentlyDrawn := False;
end;

procedure TVisibleObject.Draw;
var
   iIndex: Integer;
   vtxCurrent: TVertex;
   edgeLine: TEdge;
begin
    { Perform the default action, if we're still on the screen. }
    if (bIsRectOffScreen(rectGetBoundingRect) = True) then
        Exit;

    { If we're currently showing then erase us as we appear now. }
    if ((mtrxTransform.bAppearanceChanged = True) or (m_bShapeChanged = True)) and
       (m_bCurrentlyDrawn = True) then
        Erase;

    { If we're supposed to be invisible then don't draw us! }
    if (m_bVisible = False) or (bObjectDead = True) then
        Exit;

    { Perform matrix transformations and translations. }
    TransformAndTranslate;

    { Record the position at which we are drawing each vertex. }
    for iIndex := 0 to (m_lstVertices.Count - 1) do
    begin
        vtxCurrent := m_lstVertices.Items[iIndex];

        { Record the position of the vertices at drawing time. }
        vtxCurrent.m_iLastDrawnX := Trunc(vtxCurrent.m_fScreenX);
        vtxCurrent.m_iLastDrawnY := Trunc(vtxCurrent.m_fScreenY);
    end;

    { If we're flashing then get the current flash color. }
    if m_bCurrentlyFlashing = True then
        m_devVideo.SetPenColor(m_rgbCurrentFlashColor);

    { Now draw the edges. }
    for iIndex := 0 to (m_lstEdges.Count - 1) do
    begin
        { Get the two vertices that must be connected to form an edge. }
        edgeLine := m_lstEdges.Items[iIndex];

        { Only draw the edge if it's visible. }
        if edgeLine.m_bVisible = True then
        begin
            { Reconnect the two edges using a line. }
            if m_bCurrentlyFlashing = False then
                m_devVideo.SetPenColor(edgeLine.m_clrEdge);

            m_devVideo.MoveTo(edgeLine.m_vtxOne.m_iLastDrawnX, edgeLine.m_vtxOne.m_iLastDrawnY);
            m_devVideo.LineTo(edgeLine.m_vtxTwo.m_iLastDrawnX, edgeLine.m_vtxTwo.m_iLastDrawnY);
        end;
    end;

    { Record that we've been drawn before. }
    m_bDrawnBefore := True;
    m_bCurrentlyDrawn := True;
end;

procedure TVisibleObject.TransformAndTranslate;
var
   iVertexIndex: Integer;
   vtxLocal: TVertex;

   bTransformRequired: Boolean;
   bTranslateRequired: Boolean;
begin
    { Determine what actions must be performed. }
    bTransformRequired := (mtrxTransform.bTransformChanged = True) or (m_bShapeChanged = True);
    bTranslateRequired := (mtrxTransform.bTransformChanged = True) or
                          (mtrxTransform.bTranslateChanged = True) or
                          (m_bShapeChanged = True);

    { If the visual characteristics haven't changed then don't do anything here. }
    if (bTransformRequired = False) and (bTranslateRequired = False) then
        Exit;

    { Refresh the matrix before running vertices through. }
    if (bTransformRequired = True) then
    begin
        m_mtrxTransform.ConstructMatrix;

        { Transform all vertex coordinates by the master matrix. }
        for iVertexIndex := 0 to (m_lstVertices.Count - 1) do
        begin
            vtxLocal := m_lstVertices.items[iVertexIndex];

            { If the vertex is used in the shape then perform each of the }
            { transformations upon the vertex. }
            m_mtrxTransform.TransformVertex(vtxLocal.m_fLocalX,
                                            vtxLocal.m_fLocalY,
                                            vtxLocal.m_fTransformedX,
                                            vtxLocal.m_fTransformedY);
        end;
    end;

    { If the object's Transform all vertex coordinates by the master matrix. }
    if (bTranslateRequired = True) then
    begin
        for iVertexIndex := 0 to (m_lstVertices.Count - 1) do
        begin
            vtxLocal := m_lstVertices.items[iVertexIndex];

            { If the vertex is used in the shape then perform each of the }
            { transformations upon the vertex. }
            m_mtrxTransform.TranslateVertex(vtxLocal.m_fTransformedX,
                                            vtxLocal.m_fTransformedY,
                                            vtxLocal.m_fScreenX,
                                            vtxLocal.m_fScreenY);
        end;
    end;

    { Record the current state of the matrix. }
    m_mtrxTransform.RecordMatrixState;
    m_bShapeChanged := False;
end;

function TVisibleObject.vtxDefineNewVertex(fLocalX, fLocalY: Real): TVertex;
var
    iIndex: Integer;
begin
    { Look for an existing vertex with the same local coordinates
      that we can reuse. }
    Result := nil;
    for iIndex := 0 to (m_lstVertices.Count - 1) do
    begin
        { If we've found a match then return a reference to that vertex. }
        Result := m_lstVertices.Items[iIndex];
        if (Result.m_fLocalX = fLocalX) and (Result.m_fLocalY = fLocalY) then
            Break;
    end;

    { Create a new vertex if we need to. }
    if (iIndex >= m_lstVertices.Count) then
    begin
        Result := TVertex.Create;
        Result.m_fLocalX := fLocalX;
        Result.m_fLocalY := fLocalY;
        Result.m_bUsed := False;
    end;

    { Add the vertex to the current list. }
    m_lstVertices.Add(Result);
end;

function TVisibleObject.edgeDefineNewEdge(vtxOne, vtxTwo: TVertex; clrColor: TColor; bIncludeInCollisions: Boolean): TEdge;
begin
    { Define a new edge and assign its vertices. }
    Result := TEdge.Create;
    Result.m_vtxOne := vtxOne;
    Result.m_vtxTwo := vtxTwo;

    { Initialize the edge object. }
    Result.m_bIncludeInCollisions := bIncludeInCollisions;
    Result.m_clrEdge := clrColor;
    Result.m_bVisible := True;

    { Mark the vertices as having been used in the object's shape. }
    vtxOne.m_bUsed := True;
    vtxTwo.m_bUsed := True;

    { Add the edge to the current list, and return a reference to it. }
    m_lstEdges.Add(Result);
end;

procedure TVisibleObject.Move;
begin
    { If the object is flashing then progress the color wash. }
    if m_bCurrentlyFlashing = True then
        ProgressFlashing;

    { Transform and move the shapes comprising the object. }
    m_mtrxTransform.Move;

    { Perform default movement processing.  The base class uses this method to
      process the command wueue for the object. }
    inherited Move;
end;

{ This method is called to disintegrate the object into flying shards of
  twisted metal or rock.  Essentially, each edge of the object is converted to
  a separate TScrap object, and allowed to float independently in space. }
procedure TVisibleObject.ShatterVertices(fDirection: Real;
                                         iDuration: Integer;
                                         fMaxScatter: Real;
                                         fMaxSpin: Real;
                                         fFragmentationRate: Real);
var
    iEdgeIndex: Integer;
    edgeObject: TEdge;
    scrpObject: TVisibleObject;
    fNewDir: Real;
    fAngleDiff: Real;
    fAngleInc: Real;
    iEdgeCount: Integer;
    fSpeed: Real;
begin
    { Determine the deviation from the object's current path. }
    fAngleInc := random * (fMaxScatter - 0.05) + 0.05;
    fAngleDiff := m_lstEdges.Count * fAngleInc;
    fNewDir := fDirection - fAngleDiff / 2;

    { Iterate through each edge that comprises the object. }
    iEdgeCount := m_lstEdges.Count;
    for iEdgeIndex := 0 to (m_lstEdges.Count - 1) do
    begin
        { Shatter every even vertex.}
        if (Random < fFragmentationRate) then
        begin
            { Get information on the given edge. }
            edgeObject := m_lstEdges.Items[iEdgeIndex];

            { Convert the edge to a separate object. }
            if edgeObject.m_bVisible = True then
            begin
                { Split the edge into a separate object. }
                scrpObject := TScrap.Create(Self, edgeObject, m_mtrxTransform.fScale, iDuration,
                                            edgeObject.m_clrEdge, m_lstUniverse);
                lstUniverse.Add(scrpObject);

                { Establish the direction and speed of the scrap. }
                fSpeed := mtrxTransform.fSpeed;
                scrpObject.mtrxTransform.fSpeed := fMin(fSpeed, 1.5);
                scrpObject.mtrxTransform.fDirection := fNewDir;
                scrpObject.mtrxTransform.fRotation := (Random * fMaxSpin);

                { Compute the direction of movement of the next scrap. }
                fAngleDiff := m_lstEdges.Count * fAngleInc;
                fNewDir := fNewDir + fAngleDiff;
            end;
        end;
    end;
end;

procedure TVisibleObject.HandleCollision(objOtherObject: TVisibleObject);
begin
    { Upon collision, shatter the object violently. }
    KillObject(True, 20, 0.15, 0.5, 0.75);
end;

function TVisibleObject.bIsObjectColliding(objObject: TVisibleObject): Boolean;
begin
    { If the rectangles do NOT intersect, then no further }
    { processing is required to determine that the objects }
    { are NOT colliding. }
    if bRectsIntersect(rectGetBoundingRect, objObject.rectGetBoundingRect) = False then
    begin
        Result := False;
        Exit;
    end;

    { If the polygons no not overlap then the objects are not colliding. }
    if bPolygonsIntersect(lstVertices, objObject.lstEdges) = False then
    begin
        Result := False;
        Exit;
    end;

    { If all collision tests have been passed then we must be colliding. }
    Result := True;
end;

{ This method determines if the object is off the screen and, if so, wraps it
  around to the other side. }
procedure TVisibleObject.Wrap;
begin
   if mtrxTransform.fTranslationY >= iScreenHeight then
      mtrxTransform.SetTranslation(mtrxTransform.fTranslationX, 1);

   if mtrxTransform.fTranslationY <= 0 then
      mtrxTransform.SetTranslation(mtrxTransform.fTranslationX, iScreenHeight);

   if mtrxTransform.fTranslationX >= iScreenWidth then
      mtrxTransform.SetTranslation(1, mtrxTransform.fTranslationY);

   if mtrxTransform.fTranslationX <= 0 then
      mtrxTransform.SetTranslation(iScreenWidth, mtrxTransform.fTranslationY);
end;

procedure TVisibleObject.ClearShape;
var
    iIndex: Integer;
begin
    { Erase the object off the screen. }
    Erase;

    { Release our edge nodes. }
    for iIndex := 0 to (m_lstEdges.Count - 1) do
        Dispose(m_lstEdges.Items[iIndex]);

    { Release our vertex nodes. }
    for iIndex := 0 to (m_lstVertices.Count - 1) do
        Dispose(m_lstVertices.Items[iIndex]);

    { Clear our edge and vertex lists. }
    m_lstVertices.Clear;
    m_lstEdges.Clear;
end;

procedure TVisibleObject.BeginFlashing;
begin
    { Begin a color flashing sequence. }
    m_fLastColorChange := fGetNow;
    m_bCurrentlyFlashing := True;
    m_rgbCurrentFlashColor := clWhite;
end;

procedure TVisibleObject.EndFlashing;
begin
    { Stop the object from flashing. }
    m_bCurrentlyFlashing := False;
end;

procedure TVisibleObject.ProgressFlashing;
begin
    { If enough time has passed, change the flash-color of the text. }
    if (fGetNow - m_fLastColorChange) >= 0.005 then
    begin
        m_rgbCurrentFlashColor := RGB(Random(256), Random(256), Random(256));
        m_fLastColorChange := fGetNow;
    end;
end;

procedure TVisibleObject.DumpUnusedVertices;
var
   iVertexIndex: Integer;
   vtxObject: TVertex;
begin
    { Iterate through the vertices, and dump ones that are not in use. }
    for iVertexIndex := (m_lstVertices.Count - 1) downto 0 do
    begin
        vtxObject := m_lstVertices.Items[iVertexIndex];

        { If the vertex is unused then free it. }
        if vtxObject.m_bUsed = False then
        begin
            m_lstVertices.Delete(iVertexIndex);
            vtxObject.Free;
        end;
    end;
end;

procedure TVisibleObject.EnterState(pCommand: PState);
var
    bExitState: Boolean;
begin
    { Perform default processing. }
    inherited EnterState(pCommand);

    { Process the command. }
    bExitState := True;
    case pCommand^.enumCommandState of
       Die:
            KillObject(False, 0, 0, 0, 0);

       BecomeInvisible:
            bVisible := False;

       BecomeVisible:
            bVisible := True;

       EnableCollisions:
            bCanCollide := True;

       DisableCollisions:
            bCanCollide := False;

       StopMovement:
            mtrxTransform.fSpeed := 0;
    else
        { Don't exit states that we don't know how to handle. }
        bExitState := False;
    end;

    { If requested, exit the current state automatically. }
    if (bExitState = True) then
        ExitCurrentState;
end;

procedure TVisibleObject.ProcessState(pstCurrent: PState);
var
   iNewPosX, iNewPosY: Integer;
begin
   inherited ProcessState(pstCurrent);

   { Process the command. }
   case pstCurrent^.enumCommandState of
   WaitForClearRect:
      begin
         { If the object is in a clearing then exiting the wait state. }
         if bIsRectSafe(rectGetBoundingRect) then
            ExitCurrentState;
      end;

   FindClearRect:
      begin
         { Move to a random location on the screen. }
         iNewPosX := Round((Random(INTERNAL_SCREEN_WIDTH) * 0.60) + (INTERNAL_SCREEN_WIDTH * 0.20));
         iNewPosY := Round((Random(INTERNAL_SCREEN_HEIGHT) * 0.60) + (INTERNAL_SCREEN_HEIGHT * 0.20));

         mtrxTransform.SetTranslation(iNewPosX, iNewPosY);

         { If the object is in a clearing then exiting the wait state. }
         if bIsRectSafe(rectGetBoundingRect) then
            ExitCurrentState;
      end;
   else
   end;
end;

function TVisibleObject.bIsRectSafe(rectArea: TRect): Boolean;
var
    iObjectIndex: Integer;
    objThis: TVisibleObject;
    rectObject: TRect;
begin
    { Now, test each object for overlap with the given rectangle. }
    for iObjectIndex := 0 to (m_lstUniverse.Count - 1) do
    begin
        { Get reference to the object. }
        objThis := m_lstUniverse.Items[iObjectIndex];

        { Process only if the object is not this one and it is active. }
        if (objThis <> Self) and (objThis.bObjectDead = False) and (objThis.iCollisionID <> -1) then
        begin
            { Get the bounding rectangle for the object. }
            rectObject := objThis.rectGetBoundingRect;

            if bRectsIntersect(rectObject, rectArea) = True then
            begin
               Result := False;
               Exit;
            end;
        end;
    end;

    { If all objects are clear of the rectangle then return a value of TRUE. }
    Result := True;
end;

end.
