unit High;

{  ******
   *
   * Module:    High
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    This module impolements the THighScoreList class, which displays the
   *    best pilots while the game is idle.  The scores are stored in and INI
   *    file in an encoded form (to prevent tampering).
   *
   ****** }

interface
uses SoundDev, Visible, Text, Constant, WinProcs, WinTypes, Forms, Graphics,
     Classes, SysUtils;

{ Constants. }
const MAX_HIGH_SCORES = 10;                  { Total number of high scores tracked. }
const MAX_TEXT_FIELDS = MAX_HIGH_SCORES + 3; { Total number of text objects to be allocated. }

{ This record describes a single high score. }
type THighScoreRecord = record
    szName: String;
    lScore: LongInt;
end;

type THighScoreList = class(TObject)
    public
        { Class constructor and destructor. }
        constructor Create(lstObjects: TList; iGroupID: Integer);
        destructor Destroy; Override;

        { Method to get the score of the Nth pilot. }
        function lGetNthScore(iRank: Integer): LongInt;

        { Method to rank a score among the current list of pilots. }
        function iRankScore(lScore: LongInt): Integer;

        { Methods to insert a new high score and tag, and highlight the new
          score with flashing text. }
        function iSetNewHighScore(lScore: LongInt; szTag: String): Integer;
        procedure HighlightScore(iRank: Integer);

        { Method to display and hide the top pilot rankings. }
        procedure DisplayScores;
        procedure HideScores;

    protected

        { Methods to read and write high scores from the INI file. }
        procedure ReadHighScores;
        procedure WriteHighScores;

    protected

        m_szScoreINIPath: String;       { Path name of the score file. }
        m_vidSurface: TCanvas;          { Painting canvas. }
        m_lstObjects: TList;            { Reference to global object list. }

        { Current scoring array. }
        m_hsScores: Array [0 .. MAX_HIGH_SCORES - 1] of THighScoreRecord;

        { Text object for placing messages on the screen. }
        m_textMsg: Array [0 .. MAX_TEXT_FIELDS - 1] of TTextObject;
end;

implementation
uses General, Global;

constructor THighScoreList.Create(lstObjects: TList; iGroupID: Integer);
var
    iIndex: Integer;
    iResult: Integer;

    szKeyValuePtr: PChar;
    szTemp: String;
begin
    { Store the reference to the global object list. }
    m_lstObjects := lstObjects;

    { Create our array of text objects, and add them to the global object list. }
    for iIndex := 0 to (MAX_TEXT_FIELDS - 1) do
    begin
        m_textMsg[iIndex] := TTextObject.Create(m_lstObjects, '', clWhite);
        m_textMsg[iIndex].iGroupID := iGroupID;
        m_lstObjects.Add(m_textMsg[iIndex]);
    end;

    { Find the path name to the score INI file. }
    m_szScoreINIPath := g_envEnviron.szGetINIString('Defaults', 'HighScoreINIPath', '');

    { Read the current high score set. }
    ReadHighScores;
end;

destructor THighScoreList.Destroy;
var
    iIndex: Integer;
begin
    { Release our array of text objects. }
    for iIndex := 0 to (MAX_TEXT_FIELDS - 1) do
        if (m_lstObjects.IndexOf(m_textMsg[iIndex]) >= 0) then
            m_textMsg[iIndex].KillObject(False, 0, 0, 0, 0);

    { Perform default cleanup. }
    inherited Destroy;
end;

procedure THighScoreList.HideScores;
var
    iIndex: Integer;
begin
    { Iterate through our list of text objects. }
    for iIndex := 0 to (MAX_TEXT_FIELDS - 1) do
    begin
        m_textMsg[iIndex].Erase;
        m_textMsg[iIndex].bVisible := False;
    end;
end;

procedure THighScoreList.DisplayScores;
var
    iIndex: Integer;
    szText: String;
begin
    { Make sure we get the latest scores, in case a shared file is used. }
    ReadHighScores;

    { Create appropriate text from the current high score list. }
    m_textMsg[0].szText := 'Top Ten Pilots';
    m_textMsg[0].clrColor := clRed;
    m_textMsg[0].mtrxTransform.fScale := 3.0;
    m_textMsg[0].mtrxTransform.SetTranslation(iScreenWidth / 2, 60);
    m_textMsg[0].bVisible := True;

    { Create column headers. }
    m_textMsg[1].szText := 'Name:';
    m_textMsg[1].clrColor := clYellow;
    m_textMsg[1].mtrxTransform.fScale := 2.0;
    m_textMsg[1].mtrxTransform.SetTranslation(220, 100);
    m_textMsg[1].bVisible := True;

    m_textMsg[2].szText := 'Score:';
    m_textMsg[2].clrColor := clYellow;
    m_textMsg[2].mtrxTransform.fScale := 2.0;
    m_textMsg[2].mtrxTransform.SetTranslation(310, 100);
    m_textMsg[2].bVisible := True;

    { Show each high score entry. }
    for iIndex := 1 to MAX_HIGH_SCORES do
    begin
        szText := Format('%2d. %-3.3s  %7d', [iIndex, m_hsScores[iIndex - 1].szName, m_hsScores[iIndex - 1].lScore]);
        m_textMsg[iIndex + 2].szText := szText;
        m_textMsg[iIndex + 2].clrColor := clWhite;
        m_textMsg[iIndex + 2].mtrxTransform.fScale := 2.0;
        m_textMsg[iIndex + 2].mtrxTransform.SetTranslation(iScreenWidth / 2, 110 + iIndex * 15);
        m_textMsg[iIndex + 2].bVisible := True;
    end;
end;

{ Note that this routine was written for Delphi 1.0 and can be cleaned up. }
procedure THighScoreList.ReadHighScores;
var
    pcharINIFileName: PChar;

    iIndex: Integer;
    iResult: Integer;

    szKeyNamePtr: PChar;
    szKeyValuePtr: PChar;
    szKeyValueDefault: PChar;
    szTemp: String;

    szName: String;
    lScore: LongInt;

    szValue: String;
    szTempName: PChar;
begin
    { Allocate dynamic strings. }
    szKeyNamePtr := StrAlloc(50);
    szKeyValuePtr := StrAlloc(20);
    szKeyValueDefault := StrAlloc(20);
    pcharINIFileName := StrAlloc(256);

    { Concatentate any paths that were specified. }
    StrPCopy(pcharINIFileName, m_szScoreINIPath + g_envEnviron.szGetINIFilename);
    szTemp := StrPas(pcharINIFileName);

    { Show each high score entry. }
    for iIndex := 1 to MAX_HIGH_SCORES do
    begin
        { Get the name of the high scorer. }
        szTemp := 'HighScoreName' + IntToStr(iIndex);
        StrPCopy(szKeyNamePtr, szTemp);

        iResult := GetPrivateProfileString('Scores', szKeyNamePtr, '', szKeyValuePtr, 4, pcharINIFileName);
        szName := Format('%-3.3s', [szDecode(StrPas(szKeyValuePtr))]);

        { Get the associated score. }
        szTemp := 'HighScoreValue' + IntToStr(iIndex);
        StrPCopy(szKeyNamePtr, szTemp);

        { Establish the default value. }
        StrPCopy(szKeyValueDefault, szEncode('0'));

        { Read the score from the INI File. }
        iResult := GetPrivateProfileString('Scores', szKeyNamePtr, szKeyValueDefault, szKeyValuePtr, 7, pcharINIFileName);
        lScore := StrToInt(szDecode(StrPas(szKeyValuePtr)));

        { Store each of the high score record fields. }
        m_hsScores[iIndex - 1].szName := szName;
        m_hsScores[iIndex - 1].lScore := lScore;
    end;

    StrDispose(szKeyValueDefault);
    StrDispose(szKeyNamePtr);
    StrDispose(szKeyValuePtr);
    StrDispose(pcharINIFileName);
end;

{ Note that this routine was written for Delphi 1.0 and can be cleaned up. }
function THighScoreList.iRankScore(lScore: LongInt): Integer;
var
    iIndex: Integer;
    iRanking: Integer;
begin
    { Make sure we get the latest scores, in case a shared file is used. }
    ReadHighScores;

    { At first, assume that the user will NOT make the high score list. }
    iRanking := 11;

    { Show each high score entry. }
    for iIndex := 9 downto 0 do
        if lScore > m_hsScores[iIndex].lScore then
            iRanking := iIndex + 1;

    { Return the resulting ranking. }
    Result := iRanking;
end;

function THighScoreList.iSetNewHighScore(lScore: LongInt; szTag: String): Integer;
var
   iRanking: Integer;
   iIndex: Integer;
begin
   { Rank the given score, and pop out if it's not in the top ten. }
   iRanking := iRankScore(lScore);
   if iRanking > MAX_HIGH_SCORES then
      Exit;

   { Insert the score within the array.}
   for iIndex := 9 downto iRanking do
   begin
      m_hsScores[iIndex].szName := m_hsScores[iIndex - 1].szName;
      m_hsScores[iIndex].lScore := m_hsScores[iIndex - 1].lScore;
   end;

   { Fill in the new values. }
   m_hsScores[iRanking - 1].szName := szTag;
   m_hsScores[iRanking - 1].lScore := lScore;

   WriteHighScores;

   { Return the ranking for the user. }
   Result := iRanking;
end;

procedure THighScoreList.WriteHighScores;
var
    pcharINIFileName: PChar;

    iIndex: Integer;
    bResult: Boolean;

    szKeyNamePtr: PChar;
    szKeyValuePtr: PChar;
    szTemp: String;

    szName: String;
    lScore: LongInt;

    szValue: PChar;
begin
    szKeyNamePtr := StrAlloc(50);
    szKeyValuePtr := StrAlloc(20);
    szValue := StrAlloc(50);
    pcharINIFileName := StrAlloc(256);

    { Concatentate any paths that were specified. }
    StrPCopy(pcharINIFileName, m_szScoreINIPath + g_envEnviron.szGetINIFilename);
    szTemp := StrPas(pcharINIFileName);

    { Show each high score entry. }
    for iIndex := 1 to MAX_HIGH_SCORES do
    begin
        { Write the name of the high scorer. }
        szTemp := 'HighScoreName' + IntToStr(iIndex);
        StrPCopy(szKeyNamePtr, szTemp);

        StrPCopy(szValue, szEncode(m_hsScores[iIndex - 1].szName));
        bResult := WritePrivateProfileString('Scores', szKeyNamePtr, szValue, pcharINIFileName);

        { Write the associated score. }
        szTemp := 'HighScoreValue' + IntToStr(iIndex);
        StrPCopy(szKeyNamePtr, szTemp);

        StrPCopy(szValue, szEncode(IntToStr(m_hsScores[iIndex - 1].lScore)));
        bResult := WritePrivateProfileString('Scores', szKeyNamePtr, szValue, pcharINIFileName);
    end;

    StrDispose(szKeyNamePtr);
    StrDispose(szKeyValuePtr);
    StrDispose(szValue);
    StrDispose(pcharINIFileName);
end;

function THighScoreList.lGetNthScore(iRank: Integer): LongInt;
begin
    Result := m_hsScores[iRank - 1].lScore;
end;

procedure THighScoreList.HighlightScore(iRank: Integer);
begin
    m_textMsg[iRank + 2].BeginFlashing;
end;

end.
