/*======================================================================*/
/* Generic.cpp: Generic syntax handling library							*/
/*======================================================================*/
/* COPYRIGHT 1992-2002, Jean-Pierre MENICUCCI - All rights reserved		*/
/*======================================================================*/

#include <afxwin.h>						// mfc defines
#include "..\Ocx\Genfun.h"				// general functions
#include "..\Ocx\Syntax.h"				// syntax functions
#include "NumberRule.h"					// number rules
#include "Resource.h"					// resources

#define MAX_TOKEN_LENGTH 4095			// 12 bits

HINSTANCE hinstDLL;						// handle to DLL module 
static nCount = 0;						// reference count on DLL

#define KCC 3							// number of keywords categories

/*======================================================================*/
/* Syntax data															*/
/*======================================================================*/

#pragma warning (disable:4200)			// szKw[] is a non standard expression

typedef struct TagKEYWORD {				// keyword item data
	BYTE		bLen;					// item length
	BYTE		bCat;					// category
	TCHAR		szKw[];					// keyword
	} KEYWORD;

typedef struct TagSYNTAX {				// syntax item data
	char		cType[256];				// identifier character
	char		cEscape;				// escape character
	char		cString;				// string character
	char		cCharacter;				// character character
	LPTSTR		pszFontName;			// pointer to font name
	int			nFontSize;				// font size
	LPTSTR		pszFileTypes;			// pointer to file types
	LPTSTR		pszLevBeg;				// pointer to nesting level beginning
	LPTSTR		pszLevEnd;				// pointer to nesting level end
	DWORD		dwShading;				// shading delta for nesting levels
	LPTSTR		pszComBeg;				// pointer to comment beginning
	LPTSTR		pszComEnd;				// pointer to comment end
	LPTSTR		pszLinCom;				// pointer to line comments
	LPTSTR		pszLinBegCom;			// pointer to line beginning comments
	BOOL		bUppKey;				// uppercase keywords
	KEYWORD*	pKeywords[256];			// array of pointers to KeywordEntry lists
	NumberRule* numberRule;				// pointer to NumberRule class for this language
	} SYNTAX;

CArray<SYNTAX, SYNTAX&> SA;				// library syntax array

#pragma warning (disable:4786)			// identifier was truncated to '255' characters

#include <map>
using namespace std;

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*//////////////////////////////////////////////////////////////////////*/
/* strings handling functions											*/
/*//////////////////////////////////////////////////////////////////////*/
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

/*======================================================================*/
/* IsIdChar: is identifier character									*/
/* return:																*/
/*		TRUE							yes								*/
/*		FALSE							no								*/
/*======================================================================*/

int __inline IsIdChar (
	SYNTAX& S,							// syntax data
	TCHAR c								// character to check out
	)
{
	return (BOOL) S.cType[c];
}

/*======================================================================*/
/* DllMain: initialize syntax											*/
/* return:																*/
/*		void															*/
/*======================================================================*/

BOOL WINAPI DllMain (
    HINSTANCE hI,						// handle to DLL module 
    DWORD fdwReason,					// reason for calling function 
    LPVOID lpvReserved		 			// reserved 
	)
{
	if (nCount == 1 && fdwReason == DLL_PROCESS_DETACH) {

		nCount--;						// reference count on DLL

		for (int nSyn = 0; nSyn < SA.GetSize(); nSyn++) {

			SYNTAX S = SA.GetAt (nSyn);

			if (S.pszComBeg)
				free (S.pszComBeg);
			if (S.pszComEnd)
				free (S.pszComEnd);
			if (S.pszFileTypes)
				free (S.pszFileTypes);
			if (S.pszFontName)
				free (S.pszFontName);
			if (S.pszLevBeg)
				free (S.pszLevBeg);
			if (S.pszLevEnd)
				free (S.pszLevEnd);
			if (S.pszLinCom)
				free (S.pszLinCom);
			if (S.pszLinBegCom)
				free (S.pszLinBegCom);
			if (S.numberRule)
				delete S.numberRule;
			for (int n = 0; n < 256; n++) {
				if (S.pKeywords[n])
					delete[] (S.pKeywords[n]);
				}
			}
		}

	if (nCount == 0 && fdwReason == DLL_PROCESS_ATTACH) {

		nCount++;						// reference count on DLL

	    hinstDLL = hI;					// handle to DLL module
		}

	return TRUE;
}

/*======================================================================*/
/* InitSyntax: initialize library internal storage						*/
/* return:																*/
/*		nHan							index of syntax in library		*/
/*		GEN_ERROPEN						could not open file				*/
/*		GEN_ERRALLOC					could not allocate memory		*/
/*		GEN_ERRLOCK						could not lock memory			*/
/*		GEN_ERRREAD						could not read file				*/
/*		GEN_ERRSEC						section not found				*/
/*======================================================================*/

int __declspec(dllexport) InitSyntax (
	int nSyn,							// syntax index
	LPCTSTR	lpszName					// syntax name
	)
{
	SYNTAX S;							// syntax

	/*************************/
	/* set up .ini path name */
	/*************************/

	TCHAR szPath[MAX_PATH];				// .ini file path name
	int iPath, iRad, iSuf;				// pointers to file path name parts

	GetModuleFileName (hinstDLL, szPath, sizeof szPath);
	SplitPath (szPath, &iPath, &iRad, &iSuf);
	lstrcpy (szPath + iRad, lpszName);
	lstrcpy (szPath + iRad + lstrlen (lpszName), _T(".ini"));

	/*********************/
	/* read profile file */
	/*********************/

	HGLOBAL hBuf;						// handle to profile data
	LPTSTR pBuf;						// pointer to profile data

	int nError = FileRead (szPath, &hBuf, &pBuf);
	if (nError) return nError;

	/*************************/
	/* process rules section */
	/*************************/

	int nCharBeg;						// offset to current profile line
	int nNewLine;						// offset to next line
	int nLef;							// left part length
	LPTSTR pRig;						// pointer to right part
	int nRig;							// right part length
	int n;								// index

	TCHAR szRules[256];					// Rules
	TCHAR szIdAlpha[256];				// IdentifierAlphabet
	TCHAR szEscape[256];				// EscapeCharacter
	TCHAR szString[256];				// StringCharacter
	TCHAR szCharacter[256];				// CharacterCharacter
	TCHAR szFont[256];					// Font
	TCHAR szFileTypes[256];				// FileTypes
	TCHAR szNestingLevels[256];			// NestingLevels
	TCHAR szShading[256];				// Shading
	TCHAR szComments[256];				// Comments
	TCHAR szLineComments[256];			// LineComments
	TCHAR szLineBegComments[256];		// LineBegComments
	TCHAR szUppKey[256];				// UppercaseKeywords
	TCHAR szKeywords [KCC][256];		// Keywords1, Keywords2, Keywords3
	TCHAR szNumberFormats[256];			// NumberFormats

	LoadString (hinstDLL, IDS_RULES, szRules, sizeof szRules);
	LoadString (hinstDLL, IDS_IDENTIFIERALPHABET, szIdAlpha, sizeof szIdAlpha);
	LoadString (hinstDLL, IDS_ESCAPECHARACTER, szEscape, sizeof szEscape);
	LoadString (hinstDLL, IDS_STRINGCHARACTER, szString, sizeof szString);
	LoadString (hinstDLL, IDS_CHARACTERCHARACTER, szCharacter, sizeof szCharacter);
	LoadString (hinstDLL, IDS_FONT, szFont, sizeof szFont);
	LoadString (hinstDLL, IDS_FILETYPES, szFileTypes, sizeof szFileTypes);
	LoadString (hinstDLL, IDS_NESTINGLEVELS, szNestingLevels, sizeof szNestingLevels);
	LoadString (hinstDLL, IDS_SHADING, szShading, sizeof szShading);
	LoadString (hinstDLL, IDS_COMMENTS, szComments, sizeof szComments);
	LoadString (hinstDLL, IDS_LINECOMMENTS, szLineComments, sizeof szLineComments);
	LoadString (hinstDLL, IDS_LINEBEGCOMMENTS, szLineBegComments, sizeof szLineBegComments);
	LoadString (hinstDLL, IDS_UPPKEY, szUppKey, sizeof szUppKey);
	LoadString (hinstDLL, IDS_NUMBERFORMATS, szNumberFormats, sizeof szNumberFormats);

	for (BYTE bCat = 0; bCat < KCC; ++bCat)
		LoadString (hinstDLL, IDS_KEYWORDS1 + bCat,
			szKeywords [bCat], sizeof szKeywords [bCat]);

	nError = FindProfileSection (szRules, TRUE, pBuf, &nCharBeg);
	if (nError) return nError;

	for (n = 0; n < 256; n++)
		S.cType[n] = IsCharAlphaNumeric ((TCHAR) n) ? 1 : 0;

	S.cEscape = '\0';					// escape character
	S.cString = '\0';					// string character
	S.cCharacter = '\0';				// string character
	S.pszFontName = 0;					// pointer to font name
	S.nFontSize = 10;					// font size
	S.pszFileTypes = 0;					// pointer to file types
	S.pszLevBeg = 0;					// pointer to nesting level beginning
	S.pszLevEnd = 0;					// pointer to nesting level end
	S.dwShading = 0;					// shading delta for nesting levels
	S.pszComBeg = 0;					// pointer to comment beginning
	S.pszComEnd = 0;					// pointer to comment end
	S.pszLinCom = 0;					// pointer to line comments
	S.pszLinBegCom = 0;					// pointer to line beginning comments
	S.bUppKey = FALSE;					// uppercase keywords
	S.numberRule = 0;					// pointer to NumberRule class for this language

	ZeroMemory (&S.pKeywords, sizeof S.pKeywords);

	/*==================================================================*/
	/* guard against memory exceptions									*/
	/*==================================================================*/

	try {

		while ((nNewLine = GetProfileLine (pBuf + nCharBeg, &nLef, &pRig, &nRig)) != 0) {

			/***********************/
			/* identifier alphabet */
			/***********************/

			if (niCmp (pBuf + nCharBeg, szIdAlpha, lstrlen (szIdAlpha)) == 0) {

				ZeroMemory (&S.cType, sizeof (S.cType));
				for (n = 0; n < nRig; n++)
					S.cType[*(pRig+n)] = 1;
				}

			/********************/
			/* escape character */
			/********************/

			if (niCmp (pBuf + nCharBeg, szEscape, lstrlen (szEscape)) == 0)
				S.cEscape = *pRig;

			/********************/
			/* string character */
			/********************/

			if (niCmp (pBuf + nCharBeg, szString, lstrlen (szString)) == 0)
				S.cString = *pRig;

			/***********************/
			/* character character */
			/***********************/

			if (niCmp (pBuf + nCharBeg, szCharacter, lstrlen (szCharacter)) == 0)
				S.cCharacter = *pRig;

			/********/
			/* font */
			/********/

			if (niCmp (pBuf + nCharBeg, szFont, lstrlen (szFont)) == 0) {

				S.pszFontName = (LPTSTR) malloc (nRig+1);

				for (n = 0; n < nRig; n++)
					(*(S.pszFontName+n)) = (*(pRig+n));
				(*(S.pszFontName+n)) = '\0';

				for (n = 0; n < nRig; n++) {
					if (!IsIdChar (S, *(pRig+n)) && *(pRig+n) != ' ') {
						S.nFontSize = atoi (pRig+n+1);
						(*(S.pszFontName+n)) = '\0';
						break;
						}
					}
				}

			/**************/
			/* file types */
			/**************/

			if (niCmp (pBuf + nCharBeg, szFileTypes, lstrlen (szFileTypes)) == 0) {

				S.pszFileTypes = (LPTSTR) malloc (nRig+1);

				for (n = 0; n < nRig; n++)
					(*(S.pszFileTypes+n)) = (*(pRig+n));
				(*(S.pszFileTypes+n)) = '\0';
				}

			/******************/
			/* nesting levels */
			/******************/

			BOOL bEven;						// to distinguish beginnings and ends
			int nBeg;						// beginning index
			int nEnd;						// ending index

			if (niCmp (pBuf + nCharBeg, szNestingLevels, lstrlen (szNestingLevels)) == 0) {

				/*********************************************/
				/* count characters in begin and end strings */
				/*********************************************/

				for (n = 0, nBeg = 0, nEnd = 0, bEven = TRUE; n < nRig; n++) {
					if (bEven) {
						nBeg++;
						if (*(pRig+n) == ':')
							bEven = !bEven;
						}
					else {
						nEnd++;
						if (*(pRig+n) == '|')
							bEven = !bEven;
						}
					}

				/********************/
				/* allocate strings */
				/********************/

				S.pszLevBeg = (LPTSTR) malloc (nBeg+2);
				ZeroMemory (S.pszLevBeg, nBeg+2);

				S.pszLevEnd = (LPTSTR) malloc (nEnd+2);
				ZeroMemory (S.pszLevEnd, nEnd+2);

				/***************************************/
				/* set up beginning and ending strings */
				/***************************************/

				for (n = 0, nBeg = 0, nEnd = 0, bEven = TRUE; n < nRig; n++) {
					if (bEven) {
						if (*(pRig+n) == ':') {
							*(S.pszLevBeg+nBeg++) = '\0';
							bEven = !bEven;
							}
						else
							*(S.pszLevBeg+nBeg++) = *(pRig+n);
						}
					else {
						if (*(pRig+n) == '|') {
							*(S.pszLevEnd+nEnd++) = '\0';
							bEven = !bEven;
							}
						else
							*(S.pszLevEnd+nEnd++) = *(pRig+n);
						}
					}
				}

			/***********/
			/* shading */
			/***********/

			if (niCmp (pBuf + nCharBeg, szShading, lstrlen (szShading)) == 0) {
				if (nRig == 8 && *pRig == '0' && (*(pRig+1) & 0xDF) == 'X') {
					unsigned char szShading[4];
					hexbin ((unsigned char *) pRig+2, 6, szShading);
					szShading[3] = szShading[2];
					szShading[2] = szShading[0];
					szShading[0] = szShading[3];
					szShading[3] = '\x0';
					nCpy ((LPSTR) &S.dwShading, (LPCTSTR) szShading, 4);
					}
				}

			/************/
			/* comments */
			/************/

			if (niCmp (pBuf + nCharBeg, szComments, lstrlen (szComments)) == 0) {

				/*********************************************/
				/* count characters in begin and end strings */
				/*********************************************/

				for (n = 0, nBeg = 0, nEnd = 0, bEven = TRUE; n < nRig; n++) {
					if (bEven) {
						nBeg++;
						if (*(pRig+n) == ':')
							bEven = !bEven;
						if (*(pRig+n) == '\\')
							n++;
						}
					else {
						nEnd++;
						if (*(pRig+n) == '|')
							bEven = !bEven;
						if (*(pRig+n) == '\\')
							n++;
						}
					}

				/********************/
				/* allocate strings */
				/********************/

				S.pszComBeg = (LPTSTR) malloc (nBeg+2);
				ZeroMemory (S.pszComBeg, nBeg+2);

				S.pszComEnd = (LPTSTR) malloc (nEnd+2);
				ZeroMemory (S.pszComEnd, nEnd+2);

				/***************************************/
				/* set up beginning and ending strings */
				/***************************************/

				for (n = 0, nBeg = 0, nEnd = 0, bEven = TRUE; n < nRig; n++) {
					if (bEven) {
						if (*(pRig+n) == ':') {
							*(S.pszComBeg+nBeg++) = '\0';
							bEven = !bEven;
							}
						else {
							if (*(pRig+n) == '\\')
								n++;
							*(S.pszComBeg+nBeg++) = *(pRig+n);
							}
						}
					else {
						if (*(pRig+n) == '|') {
							*(S.pszComEnd+nEnd++) = '\0';
							bEven = !bEven;
							}
						else {
							if (*(pRig+n) == '\\')
								n++;
							*(S.pszComEnd+nEnd++) = *(pRig+n);
							}
						}
					}
				}

			/*****************/
			/* line comments */
			/*****************/

			if (niCmp (pBuf + nCharBeg, szLineComments, lstrlen (szLineComments)) == 0) {

				S.pszLinCom = (LPTSTR) malloc (nRig+2);

				nmCpy (S.pszLinCom, nRig+1, pRig, nRig);
				for (n = 0; n < nRig; n++)
					if (*(S.pszLinCom+n) == '|')
						*(S.pszLinCom+n) = '\0';
				*(S.pszLinCom+nRig+1) = '\0';
				}

			/***************************/
			/* line beginning comments */
			/***************************/

			if (niCmp (pBuf + nCharBeg, szLineBegComments, lstrlen (szLineBegComments)) == 0) {

				S.pszLinBegCom = (LPTSTR) malloc (nRig+2);

				nmCpy (S.pszLinBegCom, nRig+1, pRig, nRig);
				for (n = 0; n < nRig; n++)
					if (*(S.pszLinBegCom+n) == '|')
						*(S.pszLinBegCom+n) = '\0';
				*(S.pszLinBegCom+nRig+1) = '\0';
				}

			/**********************/
			/* uppercase keywords */
			/**********************/

			if (niCmp (pBuf + nCharBeg, szUppKey, lstrlen (szUppKey)) == 0) {
				if (niCmp (pRig, _T("yes"), 3) == 0)
					S.bUppKey = TRUE;
				}

			nCharBeg += nNewLine;
			}

		/**********************************/
		/* process NumberFormats sections */
		/**********************************/

		if (FindProfileSection (szNumberFormats, FALSE, pBuf, &nCharBeg) == 0) {

			LPTSTR allRules = NULL;
			int length = 0;

			while ((nNewLine = GetProfileLine (pBuf + nCharBeg, &nLef, &pRig, &nRig)) != 0) {

				if (nLef > 0) {
					if (length > 0)
						--length;

					allRules = (LPTSTR) realloc (allRules, length + nLef + nRig + 3  );

					nmCpy (allRules + length, nLef + 1, pBuf + nCharBeg, nLef);
					length += nLef + 1;
					nmCpy (allRules + length, nRig + 1, pRig, nRig);
					length += nRig + 1;
					allRules [length++] = 0;
					}

				nCharBeg += nNewLine;
				}

			S.numberRule = NumberRule::createRule (allRules);

			if (allRules)
				free (allRules);
			}

		/*****************************/
		/* process Keywords sections */
		/*****************************/

		typedef map <CString, WORD> KeywordsMap;
		KeywordsMap mKeywords;

		int nAQLen[256];
		ZeroMemory (&nAQLen, sizeof nAQLen);

		for (bCat = 0; bCat < KCC; ++bCat) {

			if (FindProfileSection (szKeywords [bCat], FALSE, pBuf, &nCharBeg) == 0) {

				while ((nNewLine = GetProfileLine (pBuf + nCharBeg, &nLef, &pRig, &nRig)) != 0) {

					if (nLef + nRig + 2 < nNewLine)		// keyword's last char is '='
						nLef++;

					if (nLef > 0) {

						int k = (int)(unsigned char) Lower (*(pBuf + nCharBeg));

						/************************/
						/* enter keyword in map */
						/************************/

						BYTE bLen = min (nLef, 255);
						WORD w = MAKEWORD (bLen, bCat);
						CString cs (pBuf + nCharBeg, bLen);
						mKeywords[cs] = w;

						/**********************/
						/* account for length */
						/**********************/

						nAQLen[*(pBuf + nCharBeg)] += bLen+2;
						}

					nCharBeg += nNewLine;
					}
				}
			}

		/***********************/
		/* setup keyword lists */
		/***********************/

		for (n = 0; n < 256; n++) {
			if (nAQLen[n]) {
				S.pKeywords[n] = (KEYWORD*) malloc (nAQLen[n] + 2);
				ZeroMemory ((LPSTR) S.pKeywords[n] + nAQLen[n], 2);
				nAQLen[n] = 0;
				}
			}

		for (map<CString,WORD>::iterator i = mKeywords.end(); i != mKeywords.begin();) {
			CString cs = (CString) (*--i).first;
			LPSTR lpszKw = cs.GetBuffer(0);
			BYTE bLen = LOBYTE ((WORD) (*i).second);
			BYTE bCat = HIBYTE ((WORD) (*i).second);
			LPSTR lp = (LPSTR) S.pKeywords[*lpszKw] + nAQLen[*lpszKw];
			*lp++ = bLen;
			*lp++ = bCat;
			nCpy (lp, lpszKw, bLen);
			nAQLen[*lpszKw] += bLen+2;
			}
		}

	/*==================================================================*/
	/* catch memory exceptions											*/
	/*==================================================================*/

	catch (CMemoryException* e) {

		e->Delete();

		if (S.pszComBeg)
			free (S.pszComBeg);
		if (S.pszComEnd)
			free (S.pszComEnd);
		if (S.pszFileTypes)
			free (S.pszFileTypes);
		if (S.pszFontName)
			free (S.pszFontName);
		if (S.pszLevBeg)
			free (S.pszLevBeg);
		if (S.pszLevEnd)
			free (S.pszLevEnd);
		if (S.pszLinCom)
			free (S.pszLinCom);
		if (S.pszLinBegCom)
			free (S.pszLinBegCom);
		if (S.numberRule)
			delete S.numberRule;
		for (int n = 0; n < 256; n++) {
			if (S.pKeywords[n])
				delete[] (S.pKeywords[n]);
			}

		GlobalUnlock (hBuf);
		GlobalFree (hBuf);

		return GEN_ERRALLOC;
		}

	GlobalUnlock (hBuf);
	GlobalFree (hBuf);

	/********************************/
	/* store syntax in syntax array */
	/********************************/

	nSyn = (nSyn == -1) ? SA.GetSize() : nSyn;
	SA.SetAtGrow (nSyn, S);
	return nSyn;
}

/*======================================================================*/
/* GetSyntax: return library syntax info								*/
/* return:																*/
/*		void															*/
/*======================================================================*/

void __declspec (dllexport) GetSyntax (
	int nSyn,							// syntax index
	LPCTSTR	lpszName,					// syntax name
	LPCTSTR lpszFile,					// file name
	HKEY hSyntKey						// registry key
	)
{
	SYNTAX S = SA.GetAt (nSyn);			// syntax data
	LSyntax LS;							// syntax object

	LS.o_csName = lpszName;				// syntax library name
	LS.o_csFile = lpszFile;				// syntax library file
	LS.o_csTypes = S.pszFileTypes;		// file types handled
	LS.o_dwShading = S.dwShading;		// shading delta for nesting levels

	LS.o_csFont = S.pszFontName;		// font name
	if (GetACP() == 1251)				// character set
		LS.o_lfCharSet = RUSSIAN_CHARSET;
	else
		LS.o_lfCharSet = DEFAULT_CHARSET;
	LS.o_nSize = S.nFontSize;			// size

	LS.o_csPFont = S.pszFontName;		// printer font name
	if (GetACP() == 1251)				// character set
		LS.o_lfPCharSet = RUSSIAN_CHARSET;
	else
		LS.o_lfPCharSet = DEFAULT_CHARSET;

	LS.o_nPSize = S.nFontSize;			// size
	LS.o_bMirror = TRUE;				// mirror screen font

	LSyntaxItem Selected (IDS_SELECTED,	0xFFFFFFFF, 0xFF800000, FW_NORMAL,	0x00);
	LSyntaxItem Space	 (IDS_SPACE,	0x00990000, 0xFFFFFFFF, FW_NORMAL,	0x00);
	LSyntaxItem Comment	 (IDS_COMMENT,	0x00009900, 0xFFFFFFFF, FW_BOLD,	0xFF);
	LSyntaxItem String	 (IDS_STRING,	0x00000099, 0x00CCFFFF, FW_BOLD,	0x00);

	LSyntaxItem Keyword1 (IDS_KEYWORD1,	0x00990000, 0xFFFFFFFF, FW_BOLD,	0x00);
	LSyntaxItem Keyword2 (IDS_KEYWORD2,	0x00000000, 0x00EEEEEE, FW_BOLD,	0x00);
	LSyntaxItem Keyword3 (IDS_KEYWORD3,	0x00996633, 0x00F7F7F7, FW_BOLD,	0x00);

	if (S.bUppKey)
	{
		Keyword1.o_bUppercase = TRUE;
		Keyword2.o_bUppercase = TRUE;
		Keyword3.o_bUppercase = TRUE;
	}

	LSyntaxItem Number   (IDS_NUMBER,   0x003300CC, 0x00FFF0F0, FW_BOLD,	0x00);
	LSyntaxItem Other	 (IDS_OTHER,	0xFF000000, 0xFFFFFFFF, FW_NORMAL,	0x00);

	LS.o_SIA.Add (Selected);
	LS.o_SIA.Add (Space);
	LS.o_SIA.Add (Comment);
	LS.o_SIA.Add (String);
	LS.o_SIA.Add (Keyword1);
	LS.o_SIA.Add (Keyword2);
	LS.o_SIA.Add (Keyword3);
	LS.o_SIA.Add (Number);
	LS.o_SIA.Add (Other);

	DWORD dwDisposition;				// REG_CREATED_NEW_KEY or REG_OPENED_EXISTING_KEY 
	HKEY hSyntax = 0;					// subtree key

	int rc = RegCreateKeyEx (hSyntKey, lpszName, NULL,
		TEXT(""), REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL,
		&hSyntax, &dwDisposition);

	if (hSyntax) {
		LS.Store (hSyntax);
		RegFlushKey (hSyntax);
		RegCloseKey (hSyntax);
		}
}

/*======================================================================*/
/* GLOBBUF		global buffer descriptor								*/
/*======================================================================*/

typedef struct TagGLOBBUF {
	int			nSiz;					// buffer size
	HANDLE		hHan;					// buffer handle
	LPTSTR		lpTex;					// buffer pointer
	int			nLen;					// bytes effective length
	} GLOBBUF, * P_GLOBBUF;

/*======================================================================*/
/* TOKINF		token info descriptor									*/
/*======================================================================*/

typedef struct TagTOKINF {				// token info item
    unsigned nSyn : 4;					// syntax index
    unsigned nVar : 4;					// variant index
    unsigned nInd : 4;					// current syntax
    unsigned nPrv : 4;					// previous syntax
    unsigned nLev : 4;					// nesting level
	unsigned nLen : 12;					// token length
	} TOKINF;

enum SYNVAR { Normal,					// normal
	InComBeg, InCom, InComEnd,			// comment parts
	InLinComBeg, InLinCom,				// line comment parts
	InStrBeg, InStr, InStrEnd,			// string parts
	InCharBeg, InChar, InCharEnd		// character parts
	};

enum SYNIND { Selected, Space, Comment, String, Keyword1, Keyword2, Keyword3, Number, Other};

/*======================================================================*/
/* GetToken: generic syntax processing									*/
/* return:																*/
/*		void															*/
/*======================================================================*/

void GetToken (
	int nSyn,							// syntax index
	GLOBBUF* pGB,						// global buffer
	int nOff,							// token offset
	TOKINF* pTI							// token info
	)
{
	SYNTAX S;							// syntax data
	LPTSTR lpTex;						// character pointer
	int nLen;							// line length
	int n;								// index
	int nInd;							// string delimiter index
	int nBeg;							// beginning of comment length
	LPTSTR lp;							// beginning of comment string
		
	nLen = pGB->nLen;					// line length
	if (nLen == 0)						// empty line
		return;							// return

	S = SA.GetAt (nSyn);				// syntax data

	lpTex = pGB->lpTex + nOff;			// pointer to token beginning

	/**********************************/
	/* set up previous syntax variant */
	/**********************************/

	switch (pTI->nVar) {
	case Normal:
	case InComEnd:
	case InStrEnd:
	case InCharEnd:
		pTI->nPrv = Other;				// other
		break;
	default:
		pTI->nPrv = pTI->nInd;			// previous syntax
		break;
		}

	/***************************************************************************************/
	/* reset variant to normal if there was a line comment at the end of the previous line */
	/***************************************************************************************/

	if (0 == nOff && (InLinComBeg == pTI->nVar || InLinCom == pTI->nVar))
		pTI->nVar = Normal;

	/********************/
	/* token is a space */
	/********************/

	if (*lpTex == ' ' || *lpTex == '\t') {

		pTI->nInd = Space;

		for (n = 1; n < nLen-nOff && n < MAX_TOKEN_LENGTH; n++)
			if (*(lpTex+n) != ' ' && *(lpTex+n) != '\t')
				break;

		pTI->nLen = n;

		switch (pTI->nVar) {
		case InComBeg:
			pTI->nVar = InCom;
			break;
		case InLinComBeg:
			pTI->nVar = InLinCom;
			break;
		case InStrBeg:
			pTI->nVar = InStr;
			break;
		case InCharBeg:
			pTI->nVar = InChar;
			break;
			}

		return;
		}

	/*======================================================================*/
	/* previous token was an end of comment or string or was normal			*/
	/*======================================================================*/

	switch (pTI->nVar) {
	case InComEnd:
	case InStrEnd:
	case InCharEnd:
	case Normal:

		/***************************/
		/* back to the normal case */
		/***************************/

		pTI->nVar = Normal;

		/***********************************/
		/* search for beginning of comment */
		/***********************************/

		if (S.pszComBeg) {
			for (lp = S.pszComBeg, nInd = 0; *lp; lp += nBeg+1, nInd++) {
				nBeg = lstrlen (lp);
				if (nBeg <= nLen-nOff && niCmp (lpTex, lp, nBeg) == 0) {
					pTI->nVar = InComBeg;
					pTI->nInd = Comment;
					pTI->nLen = nBeg;
					return;
					}
				}
			}

		/***************************/
		/* search for line comment */
		/***************************/

		if (S.pszLinCom) {
			for (lp = S.pszLinCom, nInd = 0; *lp; lp += nBeg+1, nInd++) {
				nBeg = lstrlen (lp);
				if (nBeg <= nLen-nOff && niCmp (lpTex, lp, nBeg) == 0) {
					pTI->nVar = InLinComBeg;
					pTI->nInd = Comment;
					pTI->nLen = nBeg;
					return;
					}
				}
			}

		/*************************************/
		/* search for line beginning comment */
		/*************************************/

		if (S.pszLinBegCom) {
			for (lp = S.pszLinBegCom, nInd = 0; *lp; lp += nBeg+1, nInd++) {
				nBeg = lstrlen (lp);
				if (nBeg <= nLen-nOff && niCmp (lpTex, lp, nBeg) == 0) {
					BOOL bFirst = TRUE;
					for (int n = nOff - 1; n >= 0; n--)
						if (*(pGB->lpTex+n) != '\t' && *(pGB->lpTex+n) != ' ')
							bFirst = FALSE;
					if (bFirst) {
						pTI->nVar = InLinComBeg;
						pTI->nInd = Comment;
						pTI->nLen = nBeg;
						return;
						}
					}
				}
			}

		/**********************/
		/* search for number  */
		/**********************/

		if (S.numberRule != NULL) {
			int len = S.numberRule->getTokenLength (lpTex, nLen-nOff);

			if (len > 0 && (len == nLen-nOff || !IsIdChar (S, lpTex [len]))) {
				pTI->nInd = Number;
				pTI->nLen = len;
				return;
				}
			}

		/**********************************/
		/* search for beginning of string */
		/**********************************/

		if (*lpTex == S.cString) {
			pTI->nVar = InStrBeg;
			pTI->nInd = String;
			pTI->nLen = 1;
			return;
			}

		/*************************************/
		/* search for beginning of character */
		/*************************************/

		if (*lpTex == S.cCharacter) {
			pTI->nVar = InCharBeg;
			pTI->nInd = String;
			pTI->nLen = 1;
			return;
			}

		/******************************/
		/* search for level increment */
		/******************************/

		if (S.pszLevBeg) {
			for (lp = S.pszLevBeg, nInd = 0; *lp; lp += nBeg+1, nInd++) {
				nBeg = lstrlen (lp);
				if (nBeg <= nLen-nOff && niCmp (lpTex, lp, nBeg) == 0) {
					if (!IsIdChar (S, *lpTex) || nBeg == nLen-nOff || !IsIdChar (S, *(lpTex+nBeg))) {
						pTI->nLen = nBeg;
						pTI->nLev++;
						break;
						}
					}
				}
			}

		/******************************/
		/* search for level decrement */
		/******************************/

		if (S.pszLevEnd) {
			for (lp = S.pszLevEnd, nInd = 0; *lp; lp += nBeg+1, nInd++) {
				nBeg = lstrlen (lp);
				if (nBeg <= nLen-nOff && niCmp (lpTex, lp, nBeg) == 0) {
					if (!IsIdChar (S, *lpTex) || nBeg == nLen-nOff || !IsIdChar (S, *(lpTex+nBeg))) {
						pTI->nLen = nBeg;
						pTI->nLev--;
						break;
						}
					}
				}
			}

		/**********************/
		/* search for keyword */
		/**********************/

		{
		int k = (int) (unsigned char) Lower (*lpTex);
		KEYWORD* pKeywords = S.pKeywords[k];
		if (pKeywords) {

			BYTE bLen = pKeywords->bLen;

			while (bLen) {

				if (bLen <= nLen-nOff && niCmp (pKeywords->szKw, lpTex, bLen) == 0) {
					if (bLen == nLen-nOff ||
						(bLen < nLen-nOff && !IsIdChar (S, *(lpTex+bLen)))) {
						pTI->nInd = Keyword1 + pKeywords->bCat;
						pTI->nLen = bLen;
						return;
						}
					}

				pKeywords = (KEYWORD*) ((LPSTR) pKeywords + bLen + 2);
				bLen = pKeywords->bLen;
				}
			}
		}

		/***************************************/
		/* other token - cut where appropriate */
		/***************************************/

		pTI->nInd = Other;

		if (IsIdChar (S, *lpTex)) {

			/**************************/
			/* token is an identifier */
			/**************************/

			for (n = 1; n < nLen-nOff && n < MAX_TOKEN_LENGTH; n++) {

				if (*(lpTex+n) == S.cEscape && n+1 < nLen-nOff) {
					n++;
					continue;			// skip escape character
					}

				pTI->nLen = n;			// set up length

				if (!IsIdChar (S, *(lpTex+n)) || *(lpTex+n) == ' ' || *(lpTex+n) == '\t')
					return;				// end of identifier

				if (S.pszComBeg) {
					for (lp = S.pszComBeg, nInd = 0; *lp; lp += nBeg+1, nInd++) {
						nBeg = lstrlen (lp);
						if (nBeg <= nLen-nOff-n && niCmp (lpTex+n, lp, nBeg) == 0)
							return;		// beginning of comment
						}
					}

				if (S.pszLinCom) {
					for (lp = S.pszLinCom, nInd = 0; *lp; lp += nBeg+1, nInd++) {
						nBeg = lstrlen (lp);
						if (nBeg <= nLen-nOff-n && niCmp (lpTex+n, lp, nBeg) == 0)
							return;		// beginning of line comment
						}
					}

				if (*(lpTex+n) == S.cString)
					return;				// beginning of string

				if (*(lpTex+n) == S.cCharacter)
					return;				// beginning of character

				if (S.pszLevBeg) {
					for (lp = S.pszLevBeg, nInd = 0; *lp; lp += nBeg+1, nInd++) {
						nBeg = lstrlen (lp);
						if (nBeg <= nLen-nOff-n && niCmp (lpTex+n, lp, nBeg) == 0)
							return;		// level increment
						}
					}

				if (S.pszLevEnd) {
					for (lp = S.pszLevEnd, nInd = 0; *lp; lp += nBeg+1, nInd++) {
						nBeg = lstrlen (lp);
						if (nBeg <= nLen-nOff-n && niCmp (lpTex+n, lp, nBeg) == 0)
							return;		// level decrement
						}
					}
				}
			}

		else {

			/******************************/
			/* token is not an identifier */
			/******************************/

			for (n = 1; n < nLen-nOff && n < MAX_TOKEN_LENGTH; n++) {

				if (*(lpTex+n) == S.cEscape && n+1 < nLen-nOff) {
					n++;
					continue;			// skip escape character
					}

				pTI->nLen = n;			// set up length

				if (IsIdChar (S, *(lpTex+n)) || *(lpTex+n) == ' ' || *(lpTex+n) == '\t')
					return;				// end of identifier

				if (S.pszComBeg) {
					for (lp = S.pszComBeg, nInd = 0; *lp; lp += nBeg+1, nInd++) {
						nBeg = lstrlen (lp);
						if (nBeg <= nLen-nOff-n && niCmp (lpTex+n, lp, nBeg) == 0)
							return;		// beginning of comment
						}
					}

				if (S.pszLinCom) {
					for (lp = S.pszLinCom, nInd = 0; *lp; lp += nBeg+1, nInd++) {
						nBeg = lstrlen (lp);
						if (nBeg <= nLen-nOff-n && niCmp (lpTex+n, lp, nBeg) == 0)
							return;		// beginning of line comment
						}
					}

				if (*(lpTex+n) == S.cString)
					return;				// beginning of string

				if (*(lpTex+n) == S.cCharacter)
					return;				// beginning of character

				if (S.pszLevBeg) {
					for (lp = S.pszLevBeg, nInd = 0; *lp; lp += nBeg+1, nInd++) {
						nBeg = lstrlen (lp);
						if (nBeg <= nLen-nOff-n && niCmp (lpTex+n, lp, nBeg) == 0)
							return;		// level increment
						}
					}

				if (S.pszLevEnd) {
					for (lp = S.pszLevEnd, nInd = 0; *lp; lp += nBeg+1, nInd++) {
						nBeg = lstrlen (lp);
						if (nBeg <= nLen-nOff-n && niCmp (lpTex+n, lp, nBeg) == 0)
							return;		// level decrement
						}
					}
				}
			}

		pTI->nLen = n;

		return;

	/*======================================================================*/
	/* previous token was a comment or a beginning of comment				*/
	/*======================================================================*/

	case InComBeg:
		pTI->nVar = InCom;
	case InCom:
		pTI->nInd = Comment;

		/*****************************/
		/* search for end of comment */
		/*****************************/

		for (n = 0; n < nLen-nOff && n < MAX_TOKEN_LENGTH; n++) {
			if (*(lpTex+n) == ' ' || *(lpTex+n) == '\t')
				break;					// end of non-space token
			for (lp = S.pszComEnd, nInd = 0; *lp; lp += nBeg + 1, nInd++) {
				nBeg = lstrlen (lp);
				if (nBeg <= nLen-nOff-n && niCmp (lpTex+n, lp, nBeg) == 0) {
					pTI->nVar = InComEnd;
					pTI->nLen = nBeg+n;
					return;				// end of comment
					}
				}
			}

		pTI->nLen = n;

		return;

	/*==========================================================================*/
	/* previous comment was an inline comment or beginning of inline comment	*/
	/*==========================================================================*/

	case InLinComBeg:
		pTI->nVar = InLinCom;
	case InLinCom:
		pTI->nInd = Comment;

		/*************************************/
		/* search for end of non-space token */
		/*************************************/

		for (n = 1; n < nLen-nOff && n < MAX_TOKEN_LENGTH; n++) {
			if (*(lpTex+n) == ' ' || *(lpTex+n) == '\t')
				break;					// end of non-space token
			}

		pTI->nLen = n;

		return;

	/*======================================================================*/
	/* previous token was a string or string beginning						*/
	/*======================================================================*/

	case InStrBeg:
		pTI->nVar = InStr;
	case InStr:
		pTI->nInd = String;

		/****************************/
		/* search for end of string */
		/****************************/

		for (n = 0; n < nLen-nOff && n < MAX_TOKEN_LENGTH; n++) {
			if (*(lpTex+n) == ' ' || *(lpTex+n) == '\t')
				break;					// end of non-space token
			if (*(lpTex+n) == S.cEscape && n+1 < nLen-nOff) {
				n++;
				continue;				// skip escape character
				}
			if (*(lpTex+n) == S.cString) {
				if (n+1 < nLen-nOff && *(lpTex+n+1) == S.cString) {
					n++;
					continue;			// skip double string character
					}
				pTI->nVar = InStrEnd;
				pTI->nLen = n+1;
				return;					// end of string
				}
			}

		pTI->nLen = n;

		return;

	/*======================================================================*/
	/* previous token was a character or beginning of character				*/
	/*======================================================================*/

	case InCharBeg:
		pTI->nVar = InChar;
	case InChar:
		pTI->nInd = String;

		/*******************************/
		/* search for end of character */
		/*******************************/

		for (n = 0; n < nLen-nOff && n < MAX_TOKEN_LENGTH; n++) {
			if (*(lpTex+n) == ' ' || *(lpTex+n) == '\t')
				break;					// end of non-space token
			if (*(lpTex+n) == S.cEscape && n+1 < nLen-nOff) {
				n++;
				continue;				// skip string character
				}

			if (*(lpTex+n) == S.cCharacter) {
				if (n+1 < nLen-nOff && *(lpTex+n+1) == S.cCharacter) {
					n++;
					continue;			// skip double string character
					}
				pTI->nVar = InCharEnd;
				pTI->nLen = n+1;
				return;					// end of character
				}
			}

		pTI->nLen = n;

		return;
		}
}
