/* ****************************************************************

 FT Fortran Tidy - To re-label, auto-indent, Fortran programs.

 N.B. The source code is assumed to be compiler error free. If any
 syntax errors (none matching brackets, pending IF.. ENDIF etc. errors
 are found, then translation will stop with an error message).

Call line:-

FT  input_file  <output_file> <options>

where:-

     input_file  = file to convert.

     output_file = converted file name. If omitted the default is
                   the output file takes the name of the input file, and
                   the input file is renamed with suffix of .BAK.


         The output file can also be with wild cards. E.G.
         "*.NEW" to take the first part of the input name and
         add on suffix ".NEW".

     options:-

          -C   Comments. Capitalize etc.

          -S Spell check comments or quotes with Words not in word list
             are capitalized. Needs either "Q" or "C" options or both!

          -R<nnnn>/<mm>/<ss> Relabel option.

               <nnnnn> is optional and is start number of new labels
                (default 1010).
               <mm> is increment (optional) 5 is default.
               <ss> is number of labels to leave unassigned between
               subroutines. Default is zero.

          -I<nn>/<mm>/<ss> for auto-indent.

             <nn>
              is number of columns for IF indent. default is 3. {optional}
             <mm>
              is number of columns for DO indent, default is 3. {optional}
             <ss>
              is start column for indentation, default is 7.    {optional}

          -H Convert Hollerith to standard character strings

          -3 Convert three label arithmetic IF to normal IF (.....) GOTO nnnn

          -U Remove unused labels.

          -K Keep all code in original case.
             Without this option all code is in upper case {except text}

          -P Remove text in columns 73 - 80

          -B Remove blank lines

          -M Convert multiple assigns to singles.

          -L Make all comments lower case.

          -G Convert SUBSTR calls to conventional format

          -JL Left justify labels in cols. 1-5
              Note:- This is default when renumbering labels.

          -JR Right justify labels in cols. 1-5

          -Q Spell check Quotes. (Used with "S" option)
             Errors are output in file with extension .spl
             format is input-line-num:' text' 'text'......
             All text is converted to lower case, except for words in
             error.

          -F Full processing, all Fortran code is reformatted.
             Without this option, the code is realigned in correct
             columns and the original spacing is kept as far as possible.

          -$ Converts error return labels from "$9999" format to the
             more conventional "*9999".

          -T Special testing mode.

          -X Make a cross-reference file. Only valid with -R or -U
             options. File name is same as input file with extension
             .XRF

          -E Place "END" at end of every subroutine/function, that needs
             one to make Unisys code compatible with other compilers.

          -A Replace Unisys string concatenation symbol "&" with
             Fortran standard "//".

******************************************************************* */

/* =====================================================================

  Outline of program workings.

    The original design was to build a simple design, one that would work
  anywhere on any machine with a C++ compiler. A kind of early Volkswagen
  or 2CV type program, easy to repair, works anywhere, even with limited
  resources. The idea was that it could be ported to mainframes, servers
  or wherever. (It works under DOS). In fact it is probably easier to
  bring the Fortran code to this program than to port the program to the
  host system. I have found that it is very rare for full tidy of an
  existing working program to take place. Most often the use would be to
  tidy up some new code (get the indentation right, get sequential labels)
  before testing.

    If you do tidy a large existing program, do the obvious things like
  comparing the absolutes, before and after. Except in the case of the
  three label IF changes, the absolutes should be identical. The only
  changes may be the compile dates, and some compilers re-order code to get
  pipeline efficiencies in a random way.

  Spell checking

  The spell-checking works like this. A dictionary of about 6,500 words
  {sorted, unique entries, lower-case, one word per line} has been
  compressed by a program called NEWDIC to produce tables in C format as
  a source element. This source element is used as a #include in
  CHKWORD.CPP The compression process is very simple. One letter words
  (such as "A" and "I" and are set as 1's flags in a 26 byte array. Two
  or more letter words are indexed by 26x26 word array. That is the
  first entry points at words beginning with "AA", the second entry
  those beginning with "AB" and so on until "ZZ". There is also an extra
  entry giving the total size in bytes of compressed data. This is there
  so that the size in bytes can be found by subtraction of the
  addresses. e.g. number of bytes in words beginning with "PB", is the
  pointer to "PC" entries minus pointer to "PB" entries. As the pointer
  implies which two letters a word starts with, the first two letters
  are not stored in the data. Each word begins with one byte that
  contains the final word size (-2) (5 bits) and number of initial
  characters (-2) that are the same as the previous word (3 bits). This
  data reduction means the 6,500 word file is reduced from about 59,500
  to 24,550 bytes.

  Label conversion

   If possible the conversion is done in a single pass of the source.
  This is not possible in the case of re-labeling, finding unused
  labels, and three label arithmetic IF conversions. In these cases two
  passes are used. The firt pass is to build up the table of labels.
  N.B. Three label arithmetic IFs can use ASSIGN variables, which have
  to be converted to GOTOs. The variables need a list of possible jumps
  to be output, hence the need for a label table.

  e.g.  IF (ABCD-12),XXX,XXX

        is translated to

        IF (ABCD .GE. 12) GOTO XXX,(6789,6543)

   The size of the label table is estimated from the size of the Fortran
  input program (after research into a few thousand Fortran programs). So
  that the initial allocation of memory can be right most of the time
  (memory expansions are expensive). The size is expanded if too small.

   All the labels in a program are kept, as a DELETE label can span more
  than one subroutine/ function. The different labels of each subroutine/
  function are separated by dummy zero labels.

   Special flags are kept in the label table for cases such as valid
  duplicate labels (they may be within different, mutually exclusive
  DELETE's).

  Comment lines

   The problem in Fortran is that you don't know when it is the end of a
  code line until a new code line comes up. In between there can be many
  comment lines, and often comment lines are mixed in between continuation
  code lines (especially with data initialization).

   After processing, the code line has be put back in its original
  position, with all embedded comments, and also inline comments -{Unisys
  special, started with an @ sign}.

   To make this as simple as possible all initial comments pass straight
  through to the output file until the first code line is reached. There
  are usually many comments at the beginning of a Fortran program. Once a
  code line is reached the comments up to the next code line are stored
  in memory, if room. If not, they are output to a temporary file. The
  size of this buffer can be adjusted to suit the computer (change size
  of MAX_COM).

   The maximum number of characters in a Fortran code line (after
  redundant spaces are removed is set at MAX_TXT = 1320. In the manuals it
  states the maximum number of lines in a "code line" is 20, that is from
  columns 7 to 72 = 66 columns, 1320/66 = 20. In practise more than 19
  continuation lines can be input and often are in data initialization
  lines. A table is needed to store in-line comments and change
  identification codes (cols. 73-80). The number of lines in a "code line"
  is set at MAX_CRDS = 58. This may be increased/decreased if more/less
  memory is available.

  In-line comments

  Unisys Fortran allows comments on the same line as code, with a
  separation character of '@'. When a code line is updated, by
  indentation, label changes or Hollerith conversions, the in-line
  comment may no longer fit on the line. If possible trailing spaces
  will be removed to allow it to fit, if that fails leading spaces are
  removed, and finally if all else fails it is output as a separate
  comment line.

  File names.

  Temporary file names are allocated with the standard "tmpnam"
  function. They have to be unique within their directory in case this
  program is being run by multiple users at the same time. The temporary
  files is (if needed) a file for the comments that overflow the memory
  buffer. If no specific file name is used for output, a temporary file
  name is used. The name is the base name (without any extension) of the
  input file with the extension of $$$. Only after successful
  completion, will the input file be renamed by giving the new extension
  of .BAK {any existing file with that name will first be deleted} and
  the temporary file will be given the input file's name. I have assumed
  that two conversions of the same program will not be done
  simultaneously.

  In case of program aborts, or other errors, any files created by this
  program with the extension $$$ may be safely deleted.

  Note:- The second (output) file name may be of the generic type,
  e.g. *.new - gives same name as input file but with extension 'new'.
  This is useful for batch runs, to save generating the output
  names in the batch run stream.

  Optional output files are *.SPL (with -Q) and *.XRF (with -X). The *
  is the base input file name (without any extension. The -Q option is
  for spell checking text within quotes or holleriths. The -X is a cross
  reference listing of old & new labels, used with -R renumbering
  option or -U unused labels.

==================================================================== */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include <ERRNO.H>
#define TRUE  1
#define FALSE 0
#define or ||
#define and &&
int tester = FALSE; // set to FALSE for off, TRUE for on.
FILE *in_f, *out_f, *com_ovf_f=NULL, *q_spl_f,* xref_f;
// *** Label Table ***
struct l_table { /* label table */
   long  int label_num; //  original label
   long  int new_num;   //  new label
   unsigned short int def_line;  //  line number where defined
   unsigned short int num_uses;  //  number of references
   unsigned short int link_pnt;  //  linkage for label sequence.
   unsigned char l_flags; // label flags
} * t_pnt = NULL;
// Values stored in l_flags
const int DL_FLAG = 1 << 0; // delete label flag.
const int UD_FLAG = 1 << 1;//delete defined in further subroutine/function.
const int DR_FLAG = 1 << 2;// defined within a DELETE RANGE.
const int FM_FLAG = 1 << 3;// defined as FORMAT label.
const int GO_FLAG = 1 << 4;// used in a goto or similar.
const int DO_FLAG = 1 << 5;// used as a DO ending label
short int numb_lab = 0, tot_lab = 0;
// *** Subroutine names ***
short int number_subs = 0; // number of subroutines found. (1st.pass)
short int sub_rel_pos = 0; // number of subroutines found. (2nd.pass)
const int SZ_sname = 7; // subroutine name list
struct sname_st {
   char sname_ent [SZ_sname];
} * s_nme_pnt = NULL;
// pending delete labels.
const int PEND_D_LABS = 10;
long int del_labs [PEND_D_LABS] = {0};
int del_count = 0; // count of delete labels pending.
int lbl_strt_pos = 0;
long int in_file_sz = 0; // input file size
int int_num; // initial number of labels to allocate in table
int last_link = 0;
int l_count;
const int MAX_TXT = 1320;// max.chars.on code line
const int MAX_CRDS=   58;// max.cards for a code line
const int ID_COLS =    8;// columns for id value (73-80)
const int MAX_XESS=    2;// amount of slip possible
char id_txt [MAX_CRDS] [ID_COLS+1]; // id text in cols 73-80
// ** ASSIGN label table (for arithmetic IF processing) **
struct if_3_value {
   short int level; // Subroutine - Function number.
   char variable [6] ; // variable name
   long int label; // 5 digit label number
} * assign_pnt = NULL;
const int ASS_SZ = 25; // initial number of entries.
int assign_act    = 0; // actual number of entries in use
int assign_avail  = 0; // number of entries available.
int curr_lvl      = 0; // current s/r-function level
int assg_strt     = 0; // start of current block
// ** in line comment text pointers **
struct in_line {
   unsigned char strt_pos; // start position of in-line comment in line.
   unsigned char numb_chrs; // number of characters in comment
   short int buff_pnt; // pointer into buffer position
   short int rel_line; // relative line number
   short int txt_posn; // text position for start of line.
} in_lin [MAX_CRDS];
char * in_buff = NULL; // buffer to hold in-line comments.
int num_code_lines = 0; // number of input lines code inputs as.
char cont_char [MAX_CRDS-1]; // continuation card characters
const int IN_INC = 10 ; // memory for in-line comments
int in_avail = 0;
int in_used  = 0;
int char_flg = 0; // CHARACTER define flag.
int call_flg = 0; // CALL flag.
int pprw_flg = 0; // Print, Punch, Read, Write flag.
// ** end of in line text variables **
#define MAX_LIN 150
#define LAB_INC 100
#define MIN_LAB_SZ 440
#define MAXPATH 150
int curr_blk;
/* Comment line Buffer defines */
// expand this if plenty of room
const int MAX_COM = 35; // internal (memory) comment buffer size.<lines>
//  allow for more than 80 character line
const int MAX_COM_LIN = 84; // maximum comment line size.
struct cb_line { // has line number with zero terminated text line.
   char cb_text [MAX_COM_LIN];
} act_cb [MAX_COM];
int cb_lines = 0;
int cb_lines_hd = 0; // number of lines buffered on disk file
int cb_file_open = 0; // set if file already opened.
char sav_line_file_name [L_tmpnam+2] = ""; // file name
int  sav_nam = 0; // flag for name found
/* Comment line Buffer defines- End */
int inc_flg  = 0; // flag - INCLUDE found in s/r or function.
int psp; // position in compressed string.
char comp_txt  [MAX_TXT+1]; // compressed text.
int n_chas; // number of characters in compressed text
char comp_typ  [MAX_TXT+1]; // type of text.
char * shad_txt = NULL;  // compressed text (before any changes)
char * comp_pos = NULL;  // character position
int flg_label   = FALSE; // labels within code line flag
int flg_3_label = FALSE; // arithmetic IF (3 labels) flag.
int flg_3_secondary = FALSE; // secondary 3 label IF (after primary one)
// maximum number of characters in a quote or hollerith string.
const int MAX_HOLLER = 511;
char * h_pnt = NULL; // quote string pointer
long int curr_lab = 0; // current line number.
int lab_strt_pnt = 0; // start column of label (can be zero)
int pass_num;  // pass number 1 or 2
char state = 0;
int id_curr, id_pend;
const int MAX_INDENT = 44; // maximum indent (based on zero)
/* 0= no line in buff, 1 = line in buffer, 2 = eof reached */
int str_it = 0; // store or output current line.
char temp_line [MAX_LIN]; // temporary line
long int line_number = 0, c_line_num = 0, oc_line_num = 0;
short int c_point = 0;
long int old_lbl;
long int lab_num [3] = {1010,5}; //default start & increment label number.
long int next_num; // next label number to be used.
int strt_indent;
//    Input options.
static char b_opt, c_opt, f_opt, g_opt, h_opt, i_opt, k_opt, l_opt, lj_opt,
m_opt, n3opt, p_opt, q_opt, r_opt, rj_opt, s_opt, u_opt, dl_opt, x_opt,
t_opt, e_opt, a_opt;
const int  K_dl_opt  =  1 <<  0;
const int  K_b_opt   =  1 <<  1;
const int  K_c_opt   =  1 <<  2;
const int  K_f_opt   =  1 <<  3;
const int  K_h_opt   =  1 <<  4;
const int  K_i_opt   =  1 <<  5;
const int  K_k_opt   =  1 <<  6;
const int  K_l_opt   =  1 <<  7;
const int  K_lj_opt  =  1 <<  8;
const int  K_n3opt   =  1 <<  9;
const int  K_p_opt   =  1 << 10;
const int  K_q_opt   =  1 << 11;
const int  K_r_opt   =  1 << 12;
const int  K_rj_opt  =  1 << 13;
const int  K_s_opt   =  1 << 14;
const int  K_u_opt   =  1 << 15;
// second options
const int  K_g_opt   =  1 <<  0;
const int  K_m_opt   =  1 <<  1;
const int  K_e_opt   =  1 <<  2;
const int  K_a_opt   =  1 <<  3;
char no_sec = FALSE;
long int options   = 0;
long int options_2 = 0;
// ** end of input options **
char lbl_need;
int change_flg; // if line has changed
// default values for indentation
int n_if = 3;  // IF
int n_do = 3;  // DO
int n_strt = 6;  // Start column (based on zero)
char e_first = 0; // first subroutine/function flag
char e_prev_end = 0; // previous code line was an END.
// maximum size of main program name.
#define MAXNME 12
char in_nm [MAXNME+1];
// Function calls.
void help_me(void);
int  wild_chk(char *new_nm, const char *base_nm, const char *wild_nm);
void chk_lbl (void);
void str_lab (long int labl, int type);//store label in table.
// values for type.
const int L_DEFN = 1; // Definition (in columns 1-5)
const int L_CODE = 2; // Used in code
const int L_SUBR = 3; // Dummy- start of Subroutine/Function.
const int L_FORM = 4; // Format label
const int L_DOLB = 5; // DO label
const int L_DELT = 6; // DELETE label
void chk_all(void);
void com_form_out (char * o_line, int str_sz);
int  new_sent = 1; // new sentence flag.
void out_line (char * tmp_lne);
int  chk_eq (int st_pnt);
int  next_line (void);
long int get_lbl (char * str, int * num_dig);
long int convert_label (long int orig_lab, int type);
int  bare_eq(void);
static inline int min (int a, int b){
   if (a < b) return a;
   return b;
}
static void rem_in_lin (void){
   // remove any in-line comments.
   int yy;
   for (yy = 0; yy < MAX_CRDS; yy++) in_lin[yy].numb_chrs = 0;
}
int spell_split (char * c_string, int str_len);
void save_line (char * com_line, int line_len);
void place_label (long int v_label);
void check_label (long int v_label);
const int MAX_DO_LBL = 15; // for do loop termiantion labels
long int do_table_label [MAX_DO_LBL];
short int do_table_count [MAX_DO_LBL];
int  lb_count = 0;
void mark_label (int strt_pos, int num_digits);
void replace_txt (int strt,int orgin,int new_v, char * new_text);
void update_txt(int s_pos);
void send_out (int type_out);
void out_in_line (int l_num);
void com_out (signed int num_line);
int  var_name(int i);
void set_pos (int id);
void label_print (void);
void mem_dmp (void * m_pnt);
void line_table (void);
void quote_find (void);
int  quote_flg = FALSE;
void close_down (void);
void full_format (void);
int  one_liner (int b_pnt, int e_pnt);
int  find_f (int c_pos);
int  stop_pause(int ppp);
int  print_punch(int st_pos);
int  at_label(int ps);
void check_lab_init (void); // check for initialization by labels
void conv_ss (int st_pnt); // SUBSTR conversion
void standard_error (void); // convert errno to text message
int  if_end (int stp); // find end of an IF statement
int  check_multi(int st_pos); // test if multi_assign
void multi_out (int n_items,int ident_flg); // output multiple lines.
void str_3_assign (int ppp); // store ASSIGN variables.
int  common_test(char * text_in); // check against common words
int  chkword(char * spword); // check spelling for word <external>
void assign_conv (void); // convert to real jump labels
int  check_flags (long int lab_num); // get flags for a label
int  multi_lab_ok (long int & con_lab); // check multi-label is OK
int  main(int argn, char * argv[]){
   int i, j, k,m, sz;
   int a,b;
   int strt, endp, n_len;
   char chs, chr;
   int split;
   int i_mode, i_temp, i_digit, t_chr;
   long int do_label;
   char * old_f = NULL; // temporary buffer
   int m_chas = 0; // number of characters
   char new_b [MAXPATH];
   s_opt= c_opt= r_opt= i_opt= l_opt= q_opt= f_opt= h_opt = t_opt = 0;
   n3opt= u_opt= k_opt= p_opt= b_opt= lj_opt= rj_opt= dl_opt = x_opt= 0;
   g_opt = m_opt = e_opt = a_opt = 0;
   lbl_need = 0; //label table needed flag.
   a = b = 0;
   if (argn == 1) { // nothing input
      help_me(); // display help screen
      return 1;
   }
   for (i = 1;i < argn; i++){
      chs = argv[i][0];
      if (chs != '-') {// must be file name
         if (a==0){
            a=i;
            continue;
         }
         else if (b == 0){
            b=i;
            continue;
         }
         else{
            printf ("Too many file names input\n");
            return 1;
         }
      }
      /* ******* option processing ****** */
      sz = strlen (argv[i]);
      if (sz < 2) {
inv_param:
         printf ("Invalid parameter %s\n", argv[i]);
         return 1;
      }
      m = 0;
multi_param:
      m++;
      chs = toupper(argv[i][m]);
      chr = toupper(argv[i][m+1]);
      switch (chs) {
         case ('C') : // comment processing
         {
            if (c_opt) goto inv_param;
            c_opt = 1;
            options += K_c_opt;
            if (sz > 2){  // multiple parameters behind "-"
               sz--;
               goto multi_param;
            }
            break;
         }
         case ('R'):  // renumber labels
         {
            if (r_opt) goto inv_param;
            r_opt = 1;
            options += K_r_opt;
            if (sz > 2 and (isdigit(chr) or chr == '/')){
               // check for start number etc.
               k = 0;
               // keep default until number comes in
               int first_flg = TRUE;
               for (j=m+1;;j++){
                  chr = argv[i][j];
                  if (chr == 0) break; // end of parameter
                  if (isalpha(chr)) break; // bext parameter found
                  if (chr == '/'){
                     k++;
                     sz--;
                     m++;
                     if (k > 2) goto inv_lab;
                     first_flg = TRUE;
                  }
                  else if (isdigit(chr)){
                     if (first_flg) {
                        first_flg = FALSE;
                        lab_num[k] = chr - '0';
                     }
                     else{
                        lab_num[k] = lab_num[k]*10 + (chr - '0');
                     }
                     sz--;
                     m++;
                  }
                  else{
inv_lab:
                     printf("Invalid label number\n");
                     return 1;
                  }
               }
               if (lab_num[0] == 0 or lab_num[1] == 0)goto inv_lab;
               for (j=0;j<3;j++) if (lab_num[j] >99999)goto inv_lab;
            }
            if (sz > 2){  // multiple parameters behind "-"
               sz--;
               goto multi_param;
            }
            break;
         }
         case ('I') :  // indent if & do
         {
            if (i_opt) goto inv_param;
            i_opt = 1;
            options += K_i_opt;
            if (sz > 2 and (isdigit(chr) or chr == '/')){
               // indent number for if
               i_mode = 0;
               i_temp = 0;
               i_digit = 0;
               for (k = m+1;; k ++){
                  t_chr = argv[i][k];
                  if (t_chr == 0 or isalpha(t_chr)) break;
                  m++;
                  sz--;
                  if (isdigit(t_chr)){
                     i_temp = i_temp*10 + t_chr - '0';
                     i_digit++;
                     continue;
                  }
                  else if (t_chr == '/'){
                     if (i_digit != 0) { // number entered
                        if (i_mode == 0) n_if = i_temp;
                        else if (i_mode == 1) n_do = i_temp;
                        else {
                           printf ("Error in format of indent\n");
                           exit (1);
                        }
                     }
                     i_mode++;
                     i_digit = 0;
                     i_temp  = 0;
                     continue;
                  }
                  else {
                     printf("Invalid character in indent "
                        "parameter\n");
                     exit (1);
                  }
               }
               if (i_digit != 0){
                  if (i_mode == 0) n_if = i_temp;
                  else if (i_mode == 1) n_do = i_temp;
                  else n_strt = i_temp - 1;
               }
               // sanity check
                  if (n_if < 0 or n_if > 30 or
                     n_do < 0 or n_do > 30 or
                     n_strt < 6 or n_strt > MAX_INDENT) {
                     printf ("Invalid indentation value\n");
                     exit (1);
                  }
            }
            if (sz > 2){  // multiple parameters behind "-"
               sz--;
               goto multi_param;
            }
            break;
         }
         case ('P') : // remove characters in cols. 73-80
         {
            if (p_opt) goto inv_param;
            p_opt = 1;
            options += K_p_opt;
            if (sz > 2){  // multiple parameters behind "-"
               sz--;
               goto multi_param;
            }
            break;
         }
         case ('B') :  // remove blank lines
         {
            if (b_opt) goto inv_param;
            b_opt = 1;
            options += K_b_opt;
            if (sz > 2){  // multiple parameters behind "-"
               sz--;
               goto multi_param;
            }
            break;
         }
         case ('L') : // make comments lower case
         {
            if (l_opt) goto inv_param;
            l_opt = 1;
            options += K_l_opt;
            if (sz > 2){  // multiple parameters behind "-"
               sz--;
               goto multi_param;
            }
            break;
         }
         case ('H') : // convert hollerith strings into normal strings.
         {
            if (h_opt) goto inv_param;
            h_opt = 1;
            options += K_h_opt;
            if (sz > 2){  // multiple parameters behind "-"
               sz--;
               goto multi_param;
            }
            break;
         }
         case ('G') : // convert SUBSTR calls to standard.
         {
            if (g_opt) goto inv_param;
            g_opt = 1;
            options_2 += K_g_opt;
            if (sz > 2){  // multiple parameters behind "-"
               sz--;
               goto multi_param;
            }
            break;
         }
         case ('3') : // convert arithmetic IFs to normal IFs
         {            // i.e. Three label IFs
            if (n3opt) goto inv_param;
            n3opt = 1;
            options += K_n3opt;
            if (sz > 2){  // multiple parameters behind "-"
               sz--;
               goto multi_param;
            }
            break;
         }
         case ('U') : // remove unused labels.
         {
            if (u_opt) goto inv_param;
            u_opt = 1;
            options += K_u_opt;
            if (sz > 2){  // multiple parameters behind "-"
               sz--;
               goto multi_param;
            }
            break;
         }
         case ('K') :   // Capitalize Fortran code
         {
            if (k_opt) goto inv_param;
            k_opt = 1;
            options += K_k_opt;
            if (sz > 2){  // multiple parameters behind "-"
               sz--;
               goto multi_param;
            }
            break;
         }
         case ('Q') : // Spell check strings in quotes & Hollerith
         {
            if (q_opt) goto inv_param;
            q_opt = 1;
            options += K_q_opt;
            if (sz > 2){  // multiple parameters behind "-"
               sz--;
               goto multi_param;
            }
            break;
         }
         case ('M') : // Spell check strings in quotes & Hollerith
         {
            if (m_opt) goto inv_param;
            m_opt = 1;
            options_2 += K_m_opt;
            if (sz > 2){  // multiple parameters behind "-"
               sz--;
               goto multi_param;
            }
            break;
         }
         case ('T') : // testing mode
         {
            if (t_opt) goto inv_param;
            t_opt = 1;
            if (sz > 2){  // multiple parameters behind "-"
               sz--;
               goto multi_param;
            }
            break;
         }
         case ('F') : // Full fortran re-format
         {
            if (f_opt) goto inv_param;
            f_opt = 1;
            options += K_f_opt;
            if (sz > 2){  // multiple parameters behind "-"
               sz--;
               goto multi_param;
            }
            break;
         }
         case ('X') : // Cross reference listing
         {
            if (x_opt) goto inv_param;
            x_opt = 1;
            if (sz > 2){  // multiple parameters behind "-"
               sz--;
               goto multi_param;
            }
            break;
         }
         case ('E') : // Place "END" at end of every subroutine/function
         {
            if (e_opt) goto inv_param;
            e_opt = 1;
            options_2 += K_e_opt;
            if (sz > 2){  // multiple parameters behind "-"
               sz--;
               goto multi_param;
            }
            break;
         }
         case ('A') : // Replace "&" by standard concatenation symbol.
         {
            if (a_opt) goto inv_param;
            a_opt = 1;
            options_2 += K_a_opt;
            if (sz > 2){  // multiple parameters behind "-"
               sz--;
               goto multi_param;
            }
            break;
         }
         case ('$') : // Convert dollar return addresses
         {
            if (dl_opt) goto inv_param;
            dl_opt = 1;
            options += K_dl_opt;
            if (sz > 2){  // multiple parameters behind "-"
               sz--;
               goto multi_param;
            }
            break;
         }
         case ('J') :  // Justify labels in columns 1-5
         {
            if (sz != 3) goto inv_param;
            if (chr == 'L'){ // Left justify <default>
               lj_opt = 1;
               options += K_lj_opt;
            }
            else if (chr == 'R'){ // Right justify
               rj_opt = 1;
               options += K_rj_opt;
            }
            else {
               goto inv_param;
            }
            if (lj_opt && rj_opt) goto inv_param;
            break;
         }
         case ('S') :  // Spell check
         {
            if (s_opt) goto inv_param;
            s_opt = 1;
            options += K_s_opt;
            if (sz > 2){  // multiple parameters behind "-"
               sz--;
               goto multi_param;
            }
            break;
         }
      default:
         {
            goto inv_param;
         }
      }
   }
   if (options == 0 and options_2 == 0){
      help_me();
   }
/*   if (r_opt) printf ("R opt, values %li/ %li /%li\n",
      lab_num[0],lab_num[1],lab_num[2]);
   if (i_opt) printf ("I opt, values %i,%i,%i\n",n_if,n_do,n_strt);
   if (a_opt) printf ("A opt\n");
   if (b_opt) printf ("B opt\n");
   if (c_opt) printf ("C opt\n");
   if (e_opt) printf ("E opt\n");
   if (f_opt) printf ("F opt\n");
   if (g_opt) printf ("G opt\n");
   if (h_opt) printf ("H opt\n");
   if (i_opt) printf ("I opt\n");
   if (k_opt) printf ("K opt\n");
   if (l_opt) printf ("L opt\n");
   if (m_opt) printf ("M opt\n");
   if (p_opt) printf ("P opt\n");
   if (q_opt) printf ("Q opt\n");
   if (r_opt) printf ("R opt\n");
   if (s_opt) printf ("S opt\n");
   if (u_opt) printf ("U opt\n");
   if (x_opt) printf ("X opt\n");     */
   // ================ Just for testing ========================
   if (t_opt) printf("Program = %s\n",argv[a]);
   // ==========================================================
   // any label table needed?
   if (r_opt or u_opt or n3opt or m_opt) lbl_need = 1;
   if (a==0){ // No input file
      printf ("No input file name given\n");
      return 1;
   }
   // if Quote spell checking
   if (q_opt and !s_opt){
      s_opt = 1;
      options += K_s_opt;
   }
   if (q_opt) {
      // if only "q" or "q" & "s" then no secondary "b" file needed.
      if (options - K_q_opt - K_s_opt == 0 and g_opt == 0){
         no_sec = TRUE;
         if (b){
            printf("Output file '%s' not required\n%",argv[b]);
            return 1;
         }
      }
   }
   if (x_opt and !(r_opt or u_opt)){
      printf ("Need R or U option with X option\n");
      exit (1);
   }
   // Is a first pass needed?
   pass_num = 2;
   if (lbl_need) pass_num = 1;
   if ((in_f = fopen (argv[a],"rt")) == NULL){
      printf("Unable to open input file %s\n",argv[a]);
      return 1;
   }
   // Open a quote spelling results file.
   // Use input name with extension ".spl"
   if (q_opt) {
      char * sp_name = NULL;
      sp_name = (char *) calloc(strlen(argv[a])+10,1);
      if (sp_name == NULL) {
         printf("No allocation of name string\n");
         return 1;
      }
      strcpy(sp_name, argv[a]);
      int st_sz = strlen(sp_name);
      // look for extension on name.
      int pnt = -1;
      int zz = st_sz - 1;

      for (; zz >= 0; zz --){
         char chaz = sp_name [zz];
         if (chaz == '.'){
            pnt = zz;
            break;
         }
         if (chaz == '/' or chaz == '\\' or chaz == ':') break;
      }
      if (pnt == -1) pnt = st_sz;
      strcpy(&sp_name[pnt],".spl");
      if ((q_spl_f = fopen(sp_name,"wt")) == NULL){
         printf ("Unable to open spell error file %s \n",sp_name);
         return 1;
      }
      free (sp_name);
      sp_name = NULL;
   }
   if (l_opt && c_opt) {
      printf("Invalid combination 'l' & 'c' options\n");
      return 1;
   }

   if (f_opt)i_opt = 1; // Full needs indentation.
   // parse input name.
   memset(in_nm,0,sizeof(in_nm)); // zero fill
   n_len = strlen(argv[a]);
   strt = 0, endp = n_len;
   for (i = 0; i < n_len; i ++){
      chs = argv[a][i];
      if (chs == ':' or chs == '/' or chs == '\\'){
         strt = i+1;
      }
      else if (chs == '.') {
         endp = i;
      }
   }
   if (strt > endp) endp = n_len;
   if ((endp - strt) > MAXNME) endp = strt + MAXNME;
   for (i=strt, j = 0; i < endp; i++){
      in_nm[j++] = toupper(argv[a][i]);
   }
   next_num = lab_num [0];
   if (lbl_need) { // label table needed
      // get file size
      if (fseek (in_f,0L,SEEK_END)){
         printf ("Error in read of input file %s\n",argv[a]);
         return 1;
      }
      in_file_sz = ftell (in_f);
      rewind (in_f);
   }
   // estimate number of labels from file size.
   int_num =  (int) (in_file_sz * 0.001505);
   if (int_num < MIN_LAB_SZ) int_num = MIN_LAB_SZ;
   if (! no_sec){// need a secondary output file
      if (b == 0) { //no output name,generate a dummy one
         i = strlen(argv[a]);
         if (i > (MAXPATH - 8)) {
            printf("Path length too long on input file name\n");
            return 1;
         }
         strcpy(new_b, argv[a]);
         // remove last part of name.
         split = 0;
         for (j=i-1; j >=0;j--){
            chs = new_b [j];
            if (chs == '.') { // replace suffix
               strcpy(new_b+j+1,"$$$");
               split = 1;
               break;
            }
            if (chs == '\\' or chs == ':' or chs == '/'){
               // no suffix found
               break;
            }
         }
         if (split == 0) {// no-suffix   -- add one on.
            strcat (new_b, ".$$$");
         }
      }
      else { // check for "wild" characters
         wild_chk(new_b, argv[a], argv[b]);
      }
      if ((out_f = fopen(new_b,"wt")) == NULL){
         printf ("Unable to open output file %s \n",new_b);
         return 1;
      }
   }
   if (pass_num == 1){
      if (lbl_need) str_lab (0,L_SUBR); // start label table off
      while ((n_chas = next_line()) != -1){
         if (q_opt and quote_flg) { // check for Quote spell checking.
            if (no_sec) continue;
         }
         if (lbl_need) {
            psp = 0;
            chk_lbl(); // check for label in line.
            // check if a labeled FORMAT line
            if (curr_lab != 0 and comp_txt[0] == 'F') {
               if (memcmp(comp_txt,"FORMAT(",7) == 0){
                  str_lab (curr_lab, L_FORM);
               }
            }
         }
      }
      chk_all();
   }
   if (tester) label_print();
   pass_num = 2;
   state = 0;
   curr_lvl = 0;
   rewind (in_f);
   line_number = c_line_num = oc_line_num = 0;
   id_curr = n_strt;
   id_pend = 0;
   if (h_opt) { // Hollerith to quotes. Make temporary space.
      h_pnt = (char *) malloc (MAX_HOLLER * 2 + 3);
      if (h_pnt == NULL){
         printf("Unable to allocate memory for Hollerith convert\n");
         exit (1);
      }
   }
   if (k_opt) { // keep original case for code
      if ((shad_txt = (char *) malloc (MAX_TXT+1)) == NULL){
         printf("Unable to allocate space for shadow text\n");
         exit (1);
      }
   }
   if ((comp_pos = (char *) malloc (MAX_TXT+1)) == NULL){
      printf("No memory for comp_pos array\n");
      exit (1);
   }
   if (x_opt) { // allocate memory for s/r names.
      s_nme_pnt = (sname_st *)calloc(number_subs, sizeof(sname_st));
      assert (s_nme_pnt != NULL);
   }
   while ((n_chas = next_line()) != -1){
      char_flg = call_flg = 0;
      change_flg = 0;
      if (a_opt) { // convert "&" to standard string concatenation,
         char * fnd_amp = NULL;
         int stp_pos = 0;
find_amp:
         ;
         fnd_amp = strchr(&comp_txt[stp_pos],'&');
         if (fnd_amp != NULL) {
            int amp_pos = fnd_amp - comp_txt;
            if (!(comp_typ[amp_pos] == 'Q' or
               comp_typ[amp_pos] == 'H')) { // replace
               replace_txt (amp_pos,1,2,"//");
               change_flg = 1;
            }
            stp_pos = amp_pos + 1;
            goto find_amp;
         }
      }
      if (g_opt) { // convert SUBSTR to standard format.
         char * ss_pnt;
         int st_pnt = 0;
next_str:
         ;
         ss_pnt = strstr(&comp_txt[st_pnt], "SUBSTR(");
         if (ss_pnt == NULL) goto no_substr;
         int char_in = ss_pnt - comp_txt;
         // is it a literal?
         if (comp_typ[char_in] == 'Q' or
            comp_typ[char_in] == 'H') {
            st_pnt = char_in + 1;
            goto next_str; // look further
         }
         conv_ss (char_in); // convert to standard form
         change_flg = 1;
         st_pnt = char_in + 1;
         goto next_str;
      }
no_substr:
      if (q_opt and quote_flg) { // check for Quote spell checking.
         quote_find();
         if (no_sec) continue;
      }
      // now check line in buffer.
      // Indentation?
      if (cb_lines_hd) {
         if (fclose (com_ovf_f)==EOF){
            printf("Error in close of temporary comment file\n");
            exit (1);
         }
         if ((com_ovf_f = fopen(sav_line_file_name,"rb")) == NULL){
            printf ("Unable to open temporary comment file (read)\n");
            exit (1);
         }
         cb_file_open = 1;
      }
      if (i_opt){
         id_curr = id_curr + id_pend;
         id_pend = 0;
         // (1) Is is an IF... THEN or ELSEIF or ELSE or ENDIF
         if (n_chas >= 9 and strncmp (comp_txt,"IF(",3) == 0 and
            strncmp(&comp_txt [n_chas-5],")THEN",5) == 0) {
            id_pend = n_if;
         }
         else if (n_chas == 4 and strncmp(comp_txt,"ELSE",4) == 0){
            id_curr = id_curr - n_if;
            id_pend = n_if;
         }
         else if (n_chas >= 13 and strncmp(comp_txt,"ELSEIF(",7) == 0
            and strncmp(&comp_txt [n_chas-5],")THEN",5)==0){
            id_curr = id_curr - n_if;
            id_pend = n_if;
         }
         else if (n_chas == 5 and strncmp(comp_txt,"ENDIF",5) == 0){
            id_curr = id_curr - n_if;
         }
         // DO loop indentation.
         else if (n_chas >= 8 and comp_txt[0] == 'D' and
            comp_txt [1] == 'O' and
            isdigit (comp_txt [2] )) { // possible do loop check further
            // pick up label
            do_label = 0;
            for (k = 2; k < 8; k++){
               chs = comp_txt [k];
               if (isdigit(chs)) {
                  do_label = do_label * 10 + chs - '0';
               }
               else if (isalpha(chs)) goto end_label;
               else if (chs == ','){
end_label:
                  place_label (do_label);
                  break;
               }
            }
         }
      }
      // if any label on statement check it.
      if (i_opt and curr_lab != 0) check_label(curr_lab);
      if (lbl_need or e_opt) {
         psp = 0;
         chk_lbl ();  // to identify labels
      }
      if (e_opt) { // test for an END
         if (n_chas == 3 and strcmp(comp_txt,"END") == 0){
            e_prev_end = 1;
         }
         else{
            e_prev_end = 0;
         }
      }
      if(r_opt or u_opt){
         if (flg_label){
            int digits;
            long int temp_label = 0, new_label;
            char out_string [12];
            // convert all label on line.
            int sp_pos = 0, sp_mode, st_point = 0;
next_label:
            sp_mode = 0;
            for (int kk=sp_pos; kk < n_chas; kk++){
               if (sp_mode == 0){
                  if (comp_typ [kk] == 'L') { //start of a label
                     assert (isdigit(comp_txt [kk]));
                     temp_label = comp_txt [kk] - '0';
                     sp_mode = 1;
                     st_point = kk;
                  }
                  continue;
               }
               else { // into label getting mode.
                  if (comp_typ[kk] != 'L') break; // end of label
                  else {
                     assert (isdigit(comp_txt [kk]));
                     temp_label = temp_label * 10 + comp_txt [kk] - '0';
                     sp_mode++;
                  }
               }
            }
            if (sp_mode != 0) { // label found.
               int typ_lab = 0;
               if (comp_txt[0] == 'D' and comp_txt[1] == 'E'){
                  // test for DELETE label
                  if (strncmp(comp_txt,"DELETE",6) == 0) typ_lab = 1;
               }
               new_label = convert_label (temp_label, typ_lab);
               // how many digits?
               digits = sprintf (out_string, "%ld", new_label);
               assert (digits < 6);
               replace_txt (st_point, sp_mode, digits, out_string);
               sp_pos = st_point + digits;
               goto next_label;

            }
         }
      }
once_more:
      if (h_opt) { // Hollerith to standard character strings
         int h_mode = 0, h_strt, h_in_sz, h_number, h_count;
         char h_chr, h_chas;
         for (int i = 0;i < n_chas; i++){
            h_chr = comp_typ [i];
            h_chas = comp_txt[i];
            if (h_mode == 0){
               if (h_chr == 'H') { // start of Hollerith
                  h_mode = 1;
                  h_strt = i;
                  h_in_sz = 1;
                  if (!isdigit(h_chas)){
                     printf ("i = %d \n",i);
                     exit (1);
                  }
                  h_number = h_chas - '0';
                  continue;
               }
            }
            else if(h_mode == 1){
               assert (h_chr == 'H');
               h_in_sz++;
               if (isdigit(h_chas)) {
                  h_number = h_number*10 + h_chas - '0';
               }
               else if (h_chas == 'H') {
                  h_mode = 2;
                  h_pnt[0] = '\'';
                  j = 1;
                  h_count = 0;
                  if (h_number <= 0 or h_number > MAX_HOLLER
                     or (n_chas - i - 1) < h_number){
                     printf("Maximum Hollerith error\n");
                     exit (1);
                  }
               }
               else {
                  printf("Hollerith system error\n");
                  exit (1);
               }
            }
            else if(h_mode == 2){
               h_in_sz++;
               assert (h_chr == 'H');
               h_count++;
               if (h_chas == '\'') {
                  h_pnt [j++] = '\'';
               }
               h_pnt [j++] = h_chas;
               if (h_count == h_number){
                  h_pnt [j++] = '\'';
                  h_pnt [j] = 0;
                  h_mode = 0;
                  // replace original Hollerith.
                  replace_txt (h_strt, h_in_sz, j, h_pnt);
                  // put in type as "Q" - quote
                  for (int k = h_strt; k < h_strt+j; k++){
                     comp_typ [k] = 'Q';
                  }
                  goto once_more;
               }
            }
         }
      }
      if (n3opt and flg_3_label) { // three label if.
         int origin_lines = 1;
         change_flg = 1;
         assert (strncmp(comp_txt,"IF(",3) == 0);
         // output compressed original as a comment.
         char temp_out [81];
         if (t_opt){
            sprintf (temp_out, "C *- %.62s -*",comp_txt);
            out_line (temp_out);
         }
         int multi_ln_f = 0; // multi-line flag
         long int label_after_f = 0; // label after flag.
         if (curr_lab) { // label on this statement.
            // check if end of a DO loop.
            int flgs = check_flags (curr_lab);
            if ((flgs & DO_FLAG) == 0 and
               // DELETE terminating label?
               (flgs & (DL_FLAG + UD_FLAG)) == 0)  goto no_do;
            // check if need multiple lines for output.
            if (flg_3_secondary)  { // Double IF case.
               multi_ln_f = 1;
            }
            else {
               // check number of labels used in arithmetic IF
                  char d_labs[3][12];
               memset (d_labs,0,sizeof(d_labs));
               for (int gg = n_chas-1; gg > 3; gg--){
                  char ccr = comp_txt[gg];
                  if (ccr == ')') { // start of 3 labels.
                     int chas_num = 0;
                     int field_n = 0;
                     for (int ppx = gg+1;ppx < n_chas; ppx++){
                        ccr = comp_txt[ppx];
                        if (ccr == ','){
                           field_n++;
                           if (field_n > 2) {
                              printf("System error 1345\n");
                              exit (1);
                           }
                           chas_num = 0;
                           continue;
                        }
                        d_labs [field_n] [chas_num++] = ccr;
                        if(chas_num >= sizeof(d_labs[0]) - 1){
                           printf("System error 1354\n");
                           exit (1);
                        }
                     }
                     break;
                  }
               }
               // now check- if two different labels, multiple lines
               for (int xx = 0; xx < 3; xx++){
                  if (strcmp(d_labs[xx],"") == 0) continue;
                  for (int yy = xx+1;yy < 3; yy++){
                     if (strcmp(d_labs[yy],"") == 0) continue;
                     if (strcmp(d_labs[yy],d_labs[xx]) == 0)continue;
                     multi_ln_f = 1;
                     goto multi_end;
                  }
               }
            }
multi_end:
            ;
            if (multi_ln_f) { // multi-lines needed
               if (flgs & GO_FLAG) { // DO & a GO
                  // This combination is impossible to convert
                  printf(
                     "Impossible to convert arithmetic IF at %li\n",
                     c_line_num);
                  // put warning in output file.
                  char warn_msg[]="C **IF needs manual conversion** ";
                  out_line (warn_msg);
                  goto skip_if;
               }
               // no "goto" on label- DO label on a seperate line.
               label_after_f = curr_lab;
               curr_lab = 0;
            }
         }
no_do:
         ;
         // find end of if
         int brak = 1, end_pnt = 0;
         char chs;
         for (int i = 3; i < n_chas; i++){
            if (comp_typ [i] == 'Q' or
               comp_typ [i] == 'H') continue;
            chs = comp_txt [i];
            if (chs == '(') brak++;
            else if (chs == ')') {
               brak--;
               if (brak == 0){
                  end_pnt = i;
                  break;
               }
            }
         }
         assert (end_pnt != 0);
         if (flg_3_secondary) { // very special case.
            // A 3 address IF is 2nd.part of a normal IF
            int temp_sz = n_chas - end_pnt - 1 ;
            int numb = 0;
            int bp_list [MAX_CRDS];
            assert (temp_sz > 0);
            char * temp_str = NULL;
            temp_str= (char *) malloc (temp_sz*4);
            if (temp_str == NULL) {
               printf("Unable to get temporary memory\n");
               exit (1);
            }
            memcpy (temp_str, &comp_txt[end_pnt+1],
               temp_sz);
            memcpy (&temp_str[temp_sz], &comp_typ[end_pnt+1],
               temp_sz);
            memcpy (&temp_str[temp_sz*2], &comp_pos[end_pnt+1],
               temp_sz);
            if (shad_txt != NULL) {
               memcpy (&temp_str[temp_sz*3], &shad_txt[end_pnt+1],
                  temp_sz);
            }
            // test if saved text spans any lines.
            if (!f_opt and num_code_lines > 1) {
               int stp = end_pnt + 1;
               int vv = 0;
               for (;vv < num_code_lines; vv++){
                  int brp = in_lin[vv].txt_posn;
                  if (brp > stp) {
                     assert (numb >= 0 and numb < MAX_CRDS);
                     bp_list[numb++] = brp - stp;
                  }
               }
            }
            int strt_posit = comp_pos[0]; // keep start position of line
            // replace arithmetic IF with THEN
            replace_txt (end_pnt+1, n_chas -end_pnt -1, 4,"THEN");
            for (i = end_pnt + 1; i <n_chas; i++){
               comp_typ[i] = 'K';  // set "THEN" as a keyword.
            }
            memset (&comp_pos[end_pnt+1],0,4);
            if (!(f_opt)) set_pos (end_pnt);
            // by removing 2nd. IF are number of lines reduced?
            if (!f_opt and num_code_lines > 1){
               int zw = 1;
               for (;zw < num_code_lines; zw ++){
                  if (n_chas - 1 < in_lin [zw].txt_posn){
                     num_code_lines = zw;
                     break;
                  }
               }
            }
            send_out(0); // send line out
            rem_in_lin(); // remove any in-line coment
            assert (temp_sz <= MAX_TXT);
            num_code_lines = 1;
            memcpy(comp_txt,temp_str,temp_sz);
            memcpy(comp_typ,&temp_str[temp_sz],temp_sz);
            memcpy(comp_pos,&temp_str[temp_sz*2],temp_sz);
            if (shad_txt != NULL){
               memcpy(shad_txt,&temp_str[temp_sz*3],temp_sz);
            }
            comp_txt [temp_sz] = 0;
            comp_typ [temp_sz] = 0;
            comp_pos [temp_sz] = 0;
            if (shad_txt != NULL) shad_txt [temp_sz] = 0;
            n_chas = temp_sz;
            // adjust relative positions if there was line span
            if (numb > 0) {
               int xx = 0;
               for (;xx < numb; xx++){
                  int end_p = temp_sz;
                  if (xx + 1 != numb) end_p = bp_list[xx+1];
                  int str_p = bp_list[xx];
                  int dd = str_p;
                  assert (dd>0);
                  // work out increment between fields should be.
                  int inc_p = 1;
                  // change of field?
                  if (comp_typ[dd-1] != comp_typ[dd]) inc_p++;
                  if (comp_txt[dd] == ')' or
                     comp_txt[dd-1] == '(') inc_p = 1;
                  int diff_x = comp_pos[dd] - (comp_pos[dd-1] + inc_p);
                  for (;dd < end_p;dd ++) {
                     comp_pos[dd] -= diff_x;
                  }
               }
            }
            // Normal reset
            int diff = comp_pos[0] - strt_posit - n_if;
            for (i = 0; i< n_chas; i++) { // reset positions
               if (comp_pos != 0) comp_pos[i] -= diff;
            }

            id_curr += n_if; // add indentation
            free (temp_str); // release temporary buffer.
            assert (strncmp(comp_txt,"IF(",3)==0);
            // find end of IF again.
            brak = 1;
            end_pnt = 0;
            for (int i = 3; i < n_chas; i++){
               if (comp_typ [i] == 'Q' or
                  comp_typ [i] == 'H') continue;
               chs = comp_txt [i];
               if (chs == '(') brak++;
               else if (chs == ')') {
                  brak--;
                  if (brak == 0){
                     end_pnt = i;
                     break;
                  }
               }
            }
            assert (end_pnt != 0);
         }
         // find three labels.
         int lab_size [3] = {0};
         char lab_value [3][13];
         memset (lab_value, 0 , sizeof(lab_value));
         int lab = 0, lab_cnt = 0;
         for (i = end_pnt+1;i < n_chas; i++){
            chs = comp_txt [i];
            if (chs == ','){
               lab_size [lab++] = lab_cnt;
               lab_cnt = 0;
               if (lab > 2) {
                  printf("Invalid three label IF\n");
                  exit (1);
               }
            }
            else if (isdigit(chs)) {
               lab_value[lab][lab_cnt++] = chs;
               if (lab_cnt > 5) {
bad_label:
                  printf ("Bad label in 3 value IF\n");
                  exit (1);
               }
            }
            else if (isalpha(chs) and lab_cnt == 0){
               // can have assigned goto variable
               // first character alphabetic.
               lab_value[lab][lab_cnt++] = chs;
               for (i++;i < n_chas;i++){ // pick up rest of variable name
                  chs = comp_txt[i];
                  if (isalnum(chs) or chs == '$'){
                     if (lab_cnt >= 6) goto bad_label;
                     lab_value[lab][lab_cnt++] = chs;
                  }
                  else if (chs == ',') {
                     i--;
                     goto loop_end;
                  }
                  else {
                     goto bad_character;
                  }
               }
            }
            else {
bad_character:
               printf ("Invalid character in 3 valued IF line %li\n",
                  c_line_num);
               exit (1);
            }
loop_end:;
         }
         if (lab_cnt != 0) lab_size [lab] = lab_cnt;
         // if any labels are alphabetic add a dummy label
         for (int gg = 0;gg < 3;gg++){
            if (isalpha(lab_value[gg][0])){
               strcat(&lab_value[gg][0],",(123)");
               lab_size [gg] += 6;
            }
         }
         // special case when no labels.
         if (lab_size[0] == 0 and lab_size [1] == 0 and
            lab_size[2] == 0){
            if(curr_lab == 0) goto end_game;
            replace_txt(0,n_chas,8,"CONTINUE");
            memset(comp_typ,'K',8); // type
            memset((comp_pos+1),0,7);
            if (!(f_opt))set_pos (0);
            send_out(1);
            goto end_game;
         }

         // special case when all 3 labels are the same.
         if ((lab_size[0] != 0) and
            (lab_size[0] == lab_size [1] and
            lab_size[1] == lab_size [2]) and
            (strcmp(lab_value[1],lab_value[2]) == 0 and
            strcmp(lab_value[2],lab_value[3]) == 0)){
            // replace with a GOTO
            replace_txt(0,n_chas,5,"GOTO0");
            replace_txt(n_chas-1, 1,
               lab_size[0], lab_value[0]);
            memset(&comp_pos[1],0,n_chas-1);
            memset(&comp_typ[4],'L',lab_size[0]);
            comp_typ[0] = 'K';
            comp_typ[1] = 'K';
            comp_typ[2] = 'K';
            comp_typ[3] = 'K';
            if (!(f_opt))set_pos (0);
            send_out(1);
            goto end_game;
         }
         // convert to generalized format.
         // check if number of lines has been reduced
         if (num_code_lines > 1){
            for(int zx = 1;zx < num_code_lines; zx++){
               if (in_lin[zx].txt_posn > end_pnt){
                  num_code_lines = zx;
                  break;
               }
            }
         }
         replace_txt (end_pnt+1, n_chas-end_pnt-1,11,".LT.0)GOTO0");
         memcpy (&comp_typ[end_pnt+1],"GGGGN)KKKKL",11);
         memset (&comp_pos[end_pnt+1],0,11);
         replace_txt(2,1,2,"((");
         end_pnt++;
         if (!f_opt) set_pos (end_pnt);
         origin_lines = num_code_lines;
         for (i=0;i<3;i++){
            num_code_lines = origin_lines;
            if (lab_size[i] > 0) {
               if (i == 1) {
                  comp_txt[end_pnt+2] = 'E';
                  comp_txt[end_pnt+3] = 'Q';
                  if (shad_txt != NULL) {
                     shad_txt[end_pnt+2] = 'E';
                     shad_txt[end_pnt+3] = 'Q';
                  }
               }
               else if (i == 2){
                  comp_txt[end_pnt+2] = 'G';
                  comp_txt[end_pnt+3] = 'T';
                  if (shad_txt != NULL) {
                     shad_txt[end_pnt+2] = 'G';
                     shad_txt[end_pnt+3] = 'T';
                  }
               }
               for (j=i+1;j < 3;j++){
                  if (strcmp(lab_value[i],lab_value[j])==0){
                     // change the test.
                     if (i==0 and j==1){
                        comp_txt[end_pnt+2] = 'L';
                        comp_txt[end_pnt+3] = 'E';
                        if (shad_txt != NULL) {
                           shad_txt[end_pnt+2] = 'L';
                           shad_txt[end_pnt+3] = 'E';
                        }
                     }
                     else if(i==0 and j==2){
                        comp_txt[end_pnt+2] = 'N';
                        comp_txt[end_pnt+3] = 'E';
                        if (shad_txt != NULL) {
                           shad_txt[end_pnt+2] = 'N';
                           shad_txt[end_pnt+3] = 'E';
                        }
                     }
                     else if (i==1 and j == 2){
                        comp_txt[end_pnt+2] = 'G';
                        comp_txt[end_pnt+3] = 'E';
                        if (shad_txt != NULL) {
                           shad_txt[end_pnt+2] = 'G';
                           shad_txt[end_pnt+3] = 'E';
                        }
                     }
                     lab_size[j] = -1;
                  }
               }
               // will a simple GOTO be enough?
               if (i > 0) { //test previous paths.
                  for (j=i-1; j >= 0; j--){
                     if (lab_size[j] == 0) goto normal_output;
                  }
               }
               if(i < 2){ // test future paths.
                  for (j = i+1;j < 3;j++){
                     if (lab_size [j] >= 0) goto normal_output;
                  }
               }
               // can output a GOTO
               replace_txt(0,n_chas,5,"GOTO0");
               replace_txt(4,1,lab_size[i],lab_value[i]);
               memset (&comp_pos[1], 0, n_chas-1);
               memset (comp_typ,0,n_chas);
               memset (comp_typ,'L',4);
               num_code_lines = 1;
               if (!f_opt) set_pos (0);
               // replace GOTO .... ,(123)
               if (isalpha(lab_value[i][0])){
                  assign_conv();
               }
               send_out(0);
               goto end_game;
normal_output:
               int far_end = end_pnt + 6;
               replace_txt (far_end + 1, n_chas - far_end - 1,
                  5, "GOTO0");
               replace_txt (n_chas-1,1,lab_size[i],lab_value[i]);
               assert (n_chas >= far_end +1);
               assert (n_chas - far_end -1 <= MAX_TXT);
               assert  (n_chas - far_end -1 > 0);
               memset (&comp_typ [far_end+1], 0, n_chas -far_end -1);
               memset (&comp_pos [far_end+1], 0, n_chas -far_end -1);
               memset (&comp_typ [far_end+1],'K',4);
               if (!f_opt) set_pos (end_pnt);
               // replace GOTO .... ,(123)
               if (isalpha(lab_value[i][0])){
                  assign_conv();
               }
               // if a simple IF ((A - B) .EQ. 0)
               // simplify to IF (A .EQ. B)
               old_f = NULL;
               m_chas = 0;
               assert (comp_txt[2] == '(');
               assert (comp_txt[3] == '(');
               int bra  = 1; // bracket counter
               int negv = 0; // negative sign flag.
               int oper = 0; // other operator flag
               int str_field_2 = 0; // start position of 2nd.field.
               int end_field_2 = 0; // end position of 2nd.field.
               for (int zx = 4; zx < n_chas - 10; zx++){
                  char typx = comp_typ [zx];
                  char valx = comp_txt [zx];
                  if (typx == 'Q' or typx == 'H') continue;
                  switch (valx) {
                  case '(':
                     bra++;
                     break;
                  case ')':
                     bra--;
                     if (bra == 0){
                        end_field_2 = zx - 1;
                        goto end_chk;
                     }
                     break;
                  case '-':
                  case '+':
                  case '*':
                  case '/':
                     // must be at level one.
                     if (bra != 1) continue;
                     if (valx == '-'){
                        negv++;
                        if (negv == 1){ // start of 2nd.field
                           if (zx == 4){
                              // precedes first field.
                              goto skip_simp;
                           }
                           str_field_2 = zx + 1;
                        }
                     }
                     else {
                        oper++;
                     }
                     continue;
                  }
               }
end_chk:
               ;   // just two fields ?
               if (oper == 0 and bra == 0 and negv == 1){
                  // need to keep old values?
                  if (i == 2) goto skip_keep;
                  if (i == 1 and lab_size[2] <= 0) goto skip_keep;
                  if (i == 0 and lab_size[1] <= 0 and
                     lab_size[2] <= 0 ) goto skip_keep;

                  m_chas = n_chas;
                  if ((old_f = (char *)malloc(4*n_chas))==NULL){
                     printf("Memory allocation failure at 1777\n");
                     exit (1);
                  }
                  // keep original values.
                  memcpy (old_f,           comp_txt,m_chas);
                  memcpy (&old_f[m_chas],  comp_pos,m_chas);
                  memcpy (&old_f[m_chas*2], comp_typ,m_chas);
                  if (shad_txt != NULL){
                     memcpy(&old_f[m_chas*3],shad_txt,m_chas);
                  }
skip_keep:
                  ;
                  char tmp_b[4];
                  // put ".EQ." etc. in tmp_b
                  memcpy (tmp_b, &comp_txt[end_field_2 +2],sizeof(tmp_b));
                  // remove ").EQ.0"
                  replace_txt (end_field_2+1,6,0,tmp_b);
                  // substitute ".EQ." for "-"
                  replace_txt (str_field_2 - 1, 1, sizeof(tmp_b), tmp_b);
                  // remove first "("
                  replace_txt (3,1,0,tmp_b);
               }
skip_simp:;
               send_out(0);
               // restore original values?
               if (m_chas != 0) {
                  // restore.
                  memcpy (comp_txt,old_f,           m_chas);
                  memcpy (comp_pos,&old_f[m_chas],  m_chas);
                  memcpy (comp_typ,&old_f[m_chas*2], m_chas);
                  if (shad_txt != NULL){
                     memcpy(shad_txt,&old_f[m_chas*3],m_chas);
                  }
                  free (old_f);
                  n_chas = m_chas;
                  m_chas = 0;
               }

               if (i==0) rem_in_lin(); // remove any in-line coment
            }
         }
end_game:
         // need a terminating "ENDIF" ?
         if (flg_3_secondary) {
            memcpy (comp_txt,"ENDIF",6);
            if (shad_txt != NULL) memcpy (shad_txt,"ENDIF",6);
            memset (comp_typ,'K',5);
            j = comp_pos[0];
            memset (comp_pos,0,6);
            comp_pos [0] = 6;
            n_chas = 5;
            if ((j - n_if) > 6) comp_pos [0] = j - n_if;
            id_curr -= n_if;
            num_code_lines = 1;
            if (!f_opt)set_pos (0);
            send_out(1);
         }
         // need a seperate line for label?
         if (label_after_f) {
            curr_lab = label_after_f;
            replace_txt(0,n_chas,8,"CONTINUE");
            memset(comp_typ,'K',8); // type
            memset((comp_pos+1),0,7);
            if (!(f_opt))set_pos (0);
            send_out(1);
         }
         continue;
      }
skip_if:
      ;
      if (m_opt) { // convert multiple assigns.
         long int kept_lab = 0;
         //multi assign at end of an IF?
         int num_items;
         if ( memcmp(comp_txt,"IF(",3) == 0){
            // find end bracket of IF
            int e_pnt;
            if ((e_pnt = if_end(0)) == -1){ // no ending bracket
               printf("No ending bracket for IF, line %li\n",
                  line_number);
               exit (1);
            }
            if (! (num_items = check_multi (e_pnt)) ) goto no_multi;
            if (curr_lab) { // check if OK to convert
               if (multi_lab_ok (kept_lab) == FALSE) goto no_multi;
            }
            // replace part after IF then "THEN"
            int x_part = n_chas - e_pnt; // characters to store
            char * t_store;
            int mem_siz = (x_part+1)*3;
            // keep original?
            if (shad_txt != NULL)mem_siz += (x_part +1);
            t_store = (char *) malloc (mem_siz);
            if (t_store == NULL) {
               printf("Unable to allocate memory for temp.store\n");
               exit (1);
            }
            char * txt_s = t_store;
            char * pos_s = txt_s + x_part +1;
            char * typ_s = pos_s + x_part +1;
            char * org_s = typ_s + x_part +1;
            memcpy (txt_s, &comp_txt[e_pnt], x_part+1);
            memcpy (pos_s, &comp_pos[e_pnt], x_part+1);
            memcpy (typ_s, &comp_typ[e_pnt], x_part+1);
            if (shad_txt != NULL){
               memcpy (org_s, &shad_txt[e_pnt], x_part+1);
            }
            int spos = comp_pos [e_pnt]; // start of THEN
            replace_txt(e_pnt,x_part,4,"THEN");
            int st_p = n_chas - 4; // start of "THEN"
            for (int i= st_p;i < n_chas; i++){
               comp_typ [i] = 'K';
               comp_pos [i] = spos++;
            }
            int orig_st = comp_pos[0]; // start of original line.
            send_out(0); // send line
            rem_in_lin(); // remove any in-line comments.
            if (i_opt) {
               id_pend = n_if;
            }
            // create new line from back part of IF.
            memcpy (comp_txt, txt_s, x_part+1);
            memcpy (comp_pos, pos_s, x_part+1);
            memcpy (comp_typ, typ_s, x_part+1);
            if (shad_txt != NULL){
               memcpy (shad_txt, org_s, x_part+1);
            }
            n_chas = x_part;
            free (t_store);
            // update positions.
            int diff = comp_pos[0] - (orig_st + n_if);
            int prev_p = 0, curr_p;
            for (int ix = 0;ix < n_chas; ix++){
               curr_p = comp_pos[ix] -= diff;
               if (curr_p<= prev_p) {
                  // text must have wrapped around a line.
                  int pos_1 = orig_st + n_if;
                  // set positions sequential
                  for (int yz=0;yz<n_chas;yz++) comp_pos[yz]=pos_1++;
                  break;
               }
               else {
                  prev_p = curr_p;
               }
            }
            multi_out (num_items, TRUE);
            // now output ENDIF
               memcpy(comp_txt,"ENDIF",6);
            if (shad_txt != NULL) memcpy(shad_txt, "ENDIF",6);
            memset (comp_typ,'K',5);
            int beg_p = orig_st;
            for (int iy = 0; iy < 5;iy++) comp_pos[iy] = beg_p++;
            n_chas = 5;
            if (i_opt) {
               id_curr -= n_if;
            }
            send_out(1);
         }
         else {
            if (!(num_items = check_multi(0))) goto no_multi;
            if (curr_lab) { // check if OK to convert
               if (multi_lab_ok (kept_lab) == FALSE) goto no_multi;
            }
            multi_out (num_items, FALSE);
         }
         if (kept_lab) {
            curr_lab = kept_lab;
            replace_txt(0,n_chas,8,"CONTINUE");
            memset(comp_typ,'K',8); // type
            int stp = comp_pos[0];
            for (int zz = 0;zz < n_chas; zz++) comp_pos[zz] = stp++;
            send_out(1);
         }
         continue;
      }
no_multi:
      ;
      send_out(1);
   }
   if (cb_file_open) { // remove temporary comments file.
      if (fclose (com_ovf_f)==EOF){
         printf("Error in close of temporary comment file\n");
         exit (1);
      }
      if (remove (sav_line_file_name)){
         printf("Unable to remove temp.file for comments %s\n",
            sav_line_file_name);
         standard_error();
      }
      sav_line_file_name[0] = 0;
   }
   if(! no_sec){
      if(fclose (out_f)){ // close output file
         printf ("Error in close of main output file\n");
         standard_error();
      }
   }
   if (fclose (in_f)){ // close main input file
      printf("Error in close of input file %s\n",argv[a]);
      standard_error();
   }
   // close quote errors file.
   if (q_opt){
      if(fclose (q_spl_f)){
         printf("Error in close of quote error file \n");
         standard_error();
      }
   }
   if (x_opt) { // output cross-reference listing
      int or_sz = strlen(argv[a]);
      char * in_nam = NULL;
      in_nam = (char *) malloc(or_sz+10);
      if (in_nam == NULL) {
         printf ("Allocation 1 name area error\n");
         exit (1);
      }
      strcpy (in_nam, argv[a]);
      // remove any old suffix.
      int suf_pnt = or_sz;
      for (int g = or_sz-1;g >= 0; g--){
         char cfs = in_nam[g];
         if (cfs == '.') {
            suf_pnt = g;
            break;
         }
         else if (cfs == '\\' or cfs == '/' or cfs == ':')break;
      }
      strcpy (&in_nam[suf_pnt],".XRF");
      if ((xref_f = fopen(in_nam,"wt")) == NULL){
         printf ("Unable to open x-ref file %s \n",in_nam);
         return 1;
      }
      free (in_nam);
      fprintf (xref_f," Cross-reference label file \n\n");
      fprintf (xref_f,"  Old   New  Line  Ref. Subroutine/Function\n");
      int link = t_pnt->link_pnt;
      char f_name [SZ_sname] = "";
      int name_pnt = 0;
      while (link != 0) {
         long int     original = (t_pnt+link)->label_num;
         long int     new_labl = (t_pnt+link)->new_num;
         unsigned int def_line = (t_pnt+link)->def_line;
         unsigned int num_user = (t_pnt+link)->num_uses;
         link                   = (t_pnt+link)->link_pnt;
         if (original == 0) { // subroutine name
            name_pnt ++;
            assert (name_pnt <= number_subs);
            memcpy (f_name,
               (s_nme_pnt + name_pnt)->sname_ent, SZ_sname);
            continue;
         }
         if (u_opt and !(r_opt) ) { // only undefined labels
            if (num_user != 0) continue;
         }
         fprintf (xref_f, "%5li %5li %5i %5i %s\n",
            original, new_labl, def_line, num_user, f_name);
      }
      fclose (xref_f);
   }
   // if no output name given, convert original one to ".BAK" and
   // rename new one.
   if (no_sec) return 0;
   if (b==0) {
      int orig_sz = strlen(argv[a]);
      char * old_nam = NULL;
      old_nam = (char *) malloc(orig_sz+10);
      if (old_nam == NULL) {
         printf ("Allocation name area error\n");
         exit (1);
      }
      strcpy (old_nam, argv[a]);
      // remove any old .BAK file.
      int bak_pnt = orig_sz;
      for (int g = orig_sz-1;g >= 0; g--){
         char cfs = old_nam[g];
         if (cfs == '.') {
            bak_pnt = g;
            break;
         }
         else if (cfs == '\\' or cfs == '/' or cfs == ':')break;
      }
      strcpy(&old_nam[bak_pnt],".BAK");
      int stats = remove (old_nam);
      if (stats == -1) { // error
         if (errno == EACCES) { // access denied to old .BAK
            printf("Unable to delete %s\n",old_nam);
            exit(1);
         }
      }
      // rename input file.
      if (rename(argv[a],old_nam)){
         printf("Error in rename of input file %s\n",argv[a]);
         standard_error();
      }
      // rename temporary output file.
      if(rename(new_b,argv[a]) != 0){
         printf("Error in rename of output file %s\n",new_b);
         exit (1);
      }
      return 0;
   }
   return 0;
}
//=======================
//   Display help screen
//=======================
static void help_me(void){
   printf(
      "FT - Fortran Beautifier Program Version 1.5 - Tues 3 June 2003\n"
      "Copyright Barton Systems Limited.\n"
      "Command line call 'FT input-file <output-file> <options>'\n"
      "Options:-\n"
      "\n"
      "-$  Convert CALL return labels from $999 to *999\n"
      "-3  convert three label IF to straight IF (....) GOTO nnn\n"
      "-A  convert Unisys '&' to // as standard string concatenation\n"
      "-B  Remove blank lines.\n"
      "-C  Capitalize etc. comments.\n"
      "-E  Insert END at end of any subroutine/function that needs it\n"
      "-F  Full reformatting of code.\n"
      "-G  Convert SUBSTR to standard Fortran character functions\n"
      "-H  convert Hollerith to standard character strings.\n"
      "-I  <if>/<do>/<st> Auto indent.\n"
      "-JL Left justify label in columns 1-5\n"
      "-JR Right justify labels.\n"
      "-K  Keep code in original case (upper or lower)\n"
      "-L  Make all comments lower case.\n"
      "-M  Multiple variable assigns converted to single lines\n"
      "-P  Remove text in columns 73-80\n"
      "-Q  Spell check character strings.\n"
      "-R  <sl>/<in>/<ss> Relabel sequentially\n"
      "-S  Use spell check dictionary.\n"
      "-U  Remove Unused labels\n"
      "-X  Create cross-reference file with suffix .XRF\n"
      "\n"
      "Enter option letter for more information (e.g.X), or other key for\n"
      "general information\n");
   int opt = toupper(getc(stdin));
   switch (opt){
   case 'C':
      printf(
         "C option. - Comment correction\n"
         "\n"
         "This will try and format comments using basic english grammar. For\n"
         "example starting each sentence off with a capitol letter, with the\n"
         "remainder of the sentence in lower case. The start of a new sentence\n"
         "is triggered by either intervening code lines or by a full stop.\n"
         "Some other basic english language formats are hard coded into the\n"
         "program, such as the 's and the capitol 'I' when used by itself.\n"
         "When the whole comment is just single characters separated by spaces\n"
         "e.g. P R O G R A M, then this is kept as capitol letters.\n"
         "It is mainly designed for very old programs where all the comments\n"
         "are in capitol letters, which are difficult to read.\n");
      break;
   case 'S':
      printf(
         " S option - Spelling \n"
         "\n"
         "If used with 'C' option any words not found in the dictionary are put\n"
         "into upper case. These are mainly program and variable names (and also\n"
         "spelling mistakes!).\n"
         "\n"
         "It is implicit with the 'Q' option. For details see 'Q' option.\n"
         "\n"
         "Example (used with 'C' option):-\n"
         "before:-\n"
         "\"PROGRAM CALLED IS XYZABC.\"\n"
         "after:-\n"
         "\"Program called is XYZABC.\"\n");
      break;
   case 'R':
      printf(
         "R option - relabel\n"
         "\n"
         "Can be followed by three optional parameters, separated by '/'.\n"
         "They are - start label - increment - jump between subroutines.\n"
         "the default values are 1010/5/0.\n"
         "The jump between subroutines (also applies to functions) is to allow\n"
         "space in the labels sequence for extra labels added later.\n"
         "By default all labels will be left justified in columns 1-5, unless\n"
         "the -JR option is used.\n"
         "Special cases:-\n"
         "\n"
         "-In the Unisys special \"DELETE nnnn,vvvv\"  the nnnn label can be\n"
         " in another subroutine/function from the DELETE.\n"
         "\n"
         "-Where two or more conditional DELETEs cover alternative versions of\n"
         " code, there can be quite legally the same label defined twice. This \n"
         " duplication will be kept by the relabelling.\n"
         "\n"
         "-Some code may not be present in the source, being added by an\n"
         " INCLUDE statement. Labels referenced in the INCLUDE code will be\n"
         " checked to make sure that after the relabeling duplicate numbers are\n"
         " not produced. In this case an error message will be output and the\n"
         " relabeling will not take place.\n"
         );
      break;
   case 'I':
      printf(
         "I option - Indentation\n"
         "\n"
         "Indentation can be done for IF-THEN-ELSE etc. & DO loops.\n"
         "There are three optional parameters separated by '/'. They are\n"
         "- if indentation - DO indentation - start indentation.\n"
         "\n"
         "The default values are 3/3/7. The start indentation value is useful\n"
         "in the case where only a slice of the code is being formatted.\n"
         "If only IF loops are to be indented then the DO indentation can be set\n"
         "to zero. e.g. -I3/0/7\n"
         "\n"
         "Indentation is only valid to a point and cannot be carried on until\n"
         "column 71. The cut off column is a program constant MAX_INDENT,\n"
         "currently set at 45. The number of pending DO labels also has a limit\n"
         "set by program constant MAX_DO_LBL, currently set at 15.\n"
         "\n"
         "If the maximum indent level has not been reached, secondary\n"
         "(continuation) lines of IF statements will also be indented by two\n"
         "columns.\n"
         "\n"
         "Note that long character strings spanning more than one line cannot be\n"
         "indented. They have to carry on from column 72 to column 6.\n"
         ) ;
      break;
   case 'H':
      printf(
         "H option - Convert Hollerith character strings\n"
         "\n"
         "Hollerith strings are a Unisys special. They are of the form\n"
         "nnH{string data} e.g. 6Hstring\n"
         "\n"
         "They are converted to a standard Fortran string enclosed in single\n"
         "quote characters.\n"
         "\n"
         "The conversion can result in larger or smaller items -larger if the\n"
         "Hollerith contained many ' , which have to be converted to double ''.\n"
         "Smaller if the Hollerith is more than 10 characters long. That is\n"
         "10Habcdefghij  translates to 'abcdefghij'.\n"
         );
      break;
   case '3':
      printf(
         "3 option\n"
         "Convert three labeled Arithmetic IF to straight IF(...) GOTO nnn\n"
         "\n"
         "The labels can be ASSIGNed variables e.g. IF(ABCD)20,JOCK,20\n"
         "In this case the resulting GOTO will have a list of possible\n"
         "ASSIGN to label's. If none can be found a dummy of 123 is used\n"
         "e.g. IF(ABC)HHH,HHH, => IF(ABC .LE.0) GOTO HHH,(123)\n"
         "A three labeled arithmetic IF can be second part of a normal IF.\n"
         "In this case the second IF is put in a THEN .. ENDIF.\n"
         "e.g. IF (AAA.EQ.BBB) IF(ABCD),123, becomes:-\n"
         "    IF (AAA .EQ. BBB) THEN\n"
         "      IF((ABC).EQ. 0) GOTO 123\n"
         "    ENDIF\n"
         "A label on the original statement can cause problems if the label\n"
         "is the terminator for a DO loop or a DELETE.\n"
         "If the result is a multi-line output a seperate CONTINUE with the\n"
         "label is appended. This does not work if the label is used in some\n"
         "kind of GOTO. See also notes on -M option\n"
         "\n"
         );
      break;
   case 'U':
      printf(
         "U option - Remove unused labels.\n"
         "\n"
         "Some labels are referenced from DELETEs in earlier\n"
         "Subroutines/Functions. The program checks for this (see also the text\n"
         "for the Relabel case.\n"
         );
      break;
   case 'G':
      printf(
         "G option - Convert SUBSTR to standard Fortran.\n"
         "\n"
         "SUBSTR is Unisys built-in function having 3 parameters:-\n"
         "\n"
         "Character-string, start-position, Number-of-characters\n"
         "\n"
         "It can be used both for storage and retrieval.It is equivalent\n"
         "to the standard Fortran character function which has the form of :-\n"
         "\n"
         " Character-string (Start-position:End position)\n"
         "\n"
         "To convert between the two:-\n"
         "\n"
         "SUBSTR(ABC,STP,NUC) is equivalent to ABC(STP:STP+NUC-1)\n"
         );
      break;
   case 'K':
      printf(
         "K option - Keep original code case.\n"
         "\n"
         "Without this option the Fortran code is stored as Upper case (except\n"
         "for character strings.\n"
         );
      break;
   case 'P':
      printf(
         "P option - Remove identity text in columns 73-80\n"
         "\n"
         "The correction identity number is kept in these columns. If\n"
         "you are making large corrections by renumbering etc. these columns\n"
         "can be cleared with this option.\n"
         );
      break;
   case 'B':
      printf(
         "B option - Remove blank lines.\n"
         "\n"
         "This includes comment lines that have only the comment character in \n"
         "column 1.\n"
         );
      break;

   case 'L':
      printf(
         "L option - make comments Lower case.\n"
         "\n"
         "Similar to 'C' option. It can make commented out code show up,\n"
         "in comparison to live code which is in upper case.\n"
         );
      break;
   case 'J':
      printf(
         "J option. Justification of labels.\n"
         "\n"
         "JL = Justify left  (start in column 1)\n"
         "JR = Justify right (end in column 5)\n"
         "\n"
         "If no option is input and renumbering is done (-R option) the\n"
         "default position for the renumbered labels is left justified.\n"
         );
      break;
   case 'Q':
      printf(
         "Q option. Quote spelling check.\n"
         "\n"
         "Highlights spelling mistakes in Quoted program character strings,\n"
         "both normal and Hollerith. This option does not change the standard\n"
         "output if used together with other options. All errors are output to\n"
         "a file with the same name as the input file, but with an extension of\n"
         "SPL. e.g. both MYFILE and MYFILE.FOR would be output to MYFILE.SPL\n"
         "\n"
         "If no 'S' option is input with the 'Q' option one is implied.\n"
         "If only the 'S' & 'Q' options are used with no others then it is an\n"
         "error to specify an output file as none is needed.\n"
         "\n"
         "The format of the error file:-\n"
         "\n"
         "nnnnnn:'a spelling MISTAK'\n"
         "\n"
         "where nnnnnn is input line number with string in error. The string\n"
         "is placed in lower case except for the word in error. There may be\n"
         "more than one string in error on the one code line.\n"
         "\n"
         "Note:- In most cases the errors found will be abbreviations, but it\n"
         "can highlight embarrassing errors before they get to any users screen.\n"
         );
      break;
   case 'F':
      printf(
         "F option. Full tidy mode.\n"
         "\n"
         "Without this option, the code is kept with same relative spacing as\n"
         "it had on input. Not using 'F' option may be best if just indenting,\n"
         "renumbering, converting Hollerith strings etc. are needed.\n"
         "\n"
         "The Full spacing may not be to everyone style, but you can change the\n"
         "rules to your own way of doing things,see the program code\n"
         "in function 'send_out'.\n"
         "\n"
         "If no indent 'I' option is input, one is implied with 'F' option.\n"
         );
      break;
   case '$':
      printf(
         "$ option. Replace $ by * in CALL error returns.\n"
         "\n"
         "In older versions of Fortran CALL line error returns were prefixed by\n"
         "either the '$ or '&'. This options converts either of these two cases\n"
         "to the more modern '*' form.\n"
         "\n"
         "e.g.\n"
         "CALL ABC($123,JIM)  becomes  CALL ABC(*123,JIM)\n"
         );
      break;
   case 'X':
      printf(
         "X option. Make a cross-reference file with extension .XRF\n"
         "\n"
         "The first part of the name is the same as the input file name.\n"
         "This option only is valid with one or both of the 'R' or 'U' options\n"
         );
      break;
   case 'E':
      printf(
         "E option. This inserts an END at the end of each subroutine or\n"
         "function that needs one to be compatible with standard compilers\n"
         "Normally in Unisys Fortran the internal subroutines follow each\n"
         "other without an explicit END\n"
         );
      break;
   case 'M':
      printf(
         "M option.\n"
         "Multiple variable assigns, e.g.  ABC,DEF,XYZ = 123\n"
         "Are placed as separate code lines. e.g.\n"
         "\n"
         "ABC = 123\n"
         "DEF = 123\n"
         "XYZ = 123\n"
         "\n"
         "There can be a problem if the original line has a label referenced\n"
         "as a DO loop terminator or as the terminator of a DELETE.\n"
         "In those cases a separate CONTINUE is placed after code lines with\n"
         "the label on the CONTINUE line. It can be impossible to convert if\n"
         "there is also a GOTO or ASSIGN to the same label. If this situation\n"
         "occurs no conversion takes place a Warning message is placed in the\n"
         "output source, and another warning message is sent to the screen.\n"
         "Processing carries on with the rest of the source. This is similar to\n"
         "the three label Arithmetic IF processing {-3}.\n"
         "\n"
         );
      break;
   case 'A':
      printf(
         "A option.\n"
         "\n"
         "Unisys string concatonation sign '&' is converted \n"
         "to Fortran standard '//'.\n"
         "\n"
         "N.B.It is not possible to use -& as an option with WIndows XP\n"
         );
      break;
   default:
      printf(
         "General notes on FT program.\n"
         "\n"
         "Input file names and options can be mixed in any order. The first\n"
         "field without a leading '-' is the input file name, the second is\n"
         "the output file.\n"
         "\n"
         "The input file name is mandatory, the output is not. If no output\n"
         "file name, a temporary file name is used. When the conversion has\n"
         "finished the input file has its name changed to have a suffix of .BAK\n"
         "and the temporary file is given the name of the original input file.\n"
         "\n"
         "The output file name can be of the form '*.new' in this case the '*'\n"
         "is replaced by the main part of the input file name (without the\n"
         "suffix). This can be useful in batch run streams as the input file\n"
         "name does not have to be copied over. It also saves typing.\n"
         "\n"
         "The input options can be placed together behind one '-', as long as\n"
         "the option does not have secondary fields (such as the dictionary name\n"
         "in the -S option. Options can be input in upper or lower case\n"
         "Examples:-\n"
         "FT -sc MYPROG.FOR {spell checks the comments}\n"
         "FT MYPROG.FOR -i -3 MYPROG.NEW {indents and replaces 3 label IFs}\n"
         "FT MYPROG.FOR -q {checks program character strings for spelling}\n"
         );
      return;
   }
   return;
}
// =============================
// set_pos - set the positions for 3 value processing
//         - parameter is last "good" position.
// =============================
static void set_pos (int end_pnt){
   assert(comp_pos[end_pnt] != 0);
   char old_typ, now_typ;
   int pos_bef;
   old_typ = comp_typ[end_pnt];
   pos_bef = comp_pos [end_pnt];
   int i;
   for (i=end_pnt+1; i < n_chas; i ++){
      now_typ = comp_typ [i];
      if (now_typ == old_typ){
         pos_bef++;
      }
      else{
         if (now_typ == ')'){
            pos_bef++;
         }
         else {
            pos_bef += 2;
         }
      }
      comp_pos [i] = pos_bef;
      old_typ = now_typ;
   }
   if (pos_bef > 71) change_flg = 1;
   return;
}
// =============================
// wild_chk- creates new file name from old one + plus wild cards.
// Input parameters are:-
//    New file name to be created,
//    base_template file name,
//       file    name     with    wild     characters.
//      Returns   zero   if   OK,non-zero  for  error.
// ==============================
static int  wild_chk(char *new_nm, const char *base_nm, const char *wild_nm){
   // only simplest case programmed here. i.e. if *.WAR gives name
   // plus new suffix of .WAR
   int len_b, len_a, chas, pnt;
   int i, j, wild_chs = 0;
   len_b = strlen (wild_nm);
   len_a = strlen (base_nm);
   if (len_a > MAXPATH -1) {
      printf("Initial   file  name  is  too  long\n");
      return 1;
   }
   if (len_b > MAXPATH -1) {
      printf("Secondary  file  name  is  too long\n");
      return 1;
   }
   if (wild_nm [0] == '*' and wild_nm [1]  ==  '.')  {
      wild_chs = 1;
   }
   //     any     wild    characters    to    process?
   if (wild_chs == 0) {
      strcpy(new_nm,wild_nm);
      return 0;
   }
   //   take   input   name    &    replace    suffix.
   pnt = len_a;
   for (i = len_a -1; i >= 0; i--){
      chas = base_nm[i];
      if (chas == '.') {
         pnt = i;
         break;
      }
   }
   strcpy(new_nm,base_nm);
   for  (i  =  pnt,   j   =   1;j   <   len_b;   j++){
      new_nm [i++] = wild_nm [j];
      if (i > MAXPATH-1) {
         printf("Combined secondary name too long\n");
         return 1;
      }
   }
   new_nm[i] = 0;    //   terminate   string
   return 0;
}
// ========================================================
//  next_line - produces compacted line
// returns number of characters in compacted line, or -1 if eof.
// ========================================================
static int  next_line (void) {
   flg_label   = FALSE;
   flg_3_label = FALSE;
   quote_flg   = FALSE;
   pprw_flg    = 0;
   int  con_flg  =  0,  t_len, posit = 0, h_count = 0;
   char com_line [MAX_LIN];
   long int lab_num;
   int i, j, k, brak = 0;
   int at_room, at_new_sz;
   int s_pnt, sc_pnt, bra, unit_flg;
   char chas, quote=0, hollar=0, ch2, cs, typx, chsx;
   memset (cont_char,0,sizeof(cont_char)); // clear continuation chars.
   if (state ==  2) { // eof reached
      return -1;
   }
   else  if  (state  ==  1)  {  //  line   in   buffer
   case_1:
      lab_num = 0;
      quote = hollar = 0;       // character string flags.
      c_line_num = line_number; // line number of first line
      t_len = strlen(temp_line);
      if (pass_num == 2){
         num_code_lines = 1;
         memset (in_lin, 0, sizeof(in_lin));
         memset (id_txt, 0, sizeof(id_txt));
         if (!p_opt and t_len > 72){
            memcpy (id_txt [0],
               &temp_line[72],ID_COLS+1);
         }
      }
      // pick up label
      if  (lab_num  ==  0){  //  only  if   no   label
         for   (i  =  0;  i  <  min(5,t_len);  i++)  {
            chas = temp_line [i];
            if     (chas    ==    ' ')    continue;
            if (isdigit(chas)) {
               if  (lab_num  ==  0)  lab_strt_pnt = i;
               lab_num = lab_num * 10 +  chas  -  '0';
            }
            else {
               //  no earlier numerics in cols 1 to 5.
                  printf ("Invalid character in cols.1-5 of line %d\n",
                     c_line_num);
               printf  ("    Conversion   Aborted\n");
               close_down();
               exit (1);
            }
         }
      }
      if (lab_num != 0) {
         assert (lab_num <= 99999);
         str_lab (lab_num, L_DEFN);
      }
      curr_lab  = lab_num;
unpack_more:
      for (i = 6; i < 72; i++) {
         chas = ch2 = temp_line[i];
         //  use capitol letters unless in quote mode.
         if (! (quote or hollar)) {
            if (chas == '@'){
               // remove trailing spaces.
               for   (j = 71;j > i;j--){
                  if  (temp_line[j]  !=  ' ')  break;
               }
               at_room = j - i + 1;
               if(pass_num == 2) {
                  // store in buffer.
                  if (num_code_lines == 1) in_used = 0;
                  if (in_buff == NULL) { // first time?
                     if ((in_buff = (char *) malloc(IN_INC)) == NULL){
bad_at_alloc:
                        printf("Memory for in-line comment error\n");
                        exit (1);
                     }
                     in_avail = IN_INC;
                     in_used  = 0;
                  }
                  // Check if enough room  in  buffer.
more_space:
                  if (at_room > in_avail - in_used)  {
                     // expand.
                     at_new_sz =  in_avail  +  IN_INC;
                     in_buff = (char *) realloc (in_buff, at_new_sz);
                     if (in_buff == NULL) goto bad_at_alloc;
                     in_avail        +=        IN_INC;
                     goto more_space;
                  }
                  // check spelling etc.
                  if (c_opt or l_opt)    {
                     new_sent = 1; // Always start with capitol letter
                     for (j = i+1,k = 0;k < at_room; j++, k++){
                        chas = tolower (temp_line[j]);
                        if (chas == '.' or chas == '?' or chas == '!'
                           or chas == ':') new_sent = 1;
                        if (isalpha(chas)){
                           if(c_opt  and  new_sent){
                              chas = toupper(chas);
                              new_sent = 0;
                           }
                        }
                        temp_line [j] = chas;
                     }
                     // Any spell check.
                     if (s_opt) {
                        spell_split( &temp_line[i], at_room);
                     }
                  }
                  //  move  into   temporary   buffer.
                  in_lin [num_code_lines-1].strt_pos = i;
                  in_lin [num_code_lines-1].numb_chrs = at_room;
                  in_lin [num_code_lines-1].buff_pnt = in_used;
                  memcpy (&in_buff[in_used], &temp_line[i], at_room);
                  in_used += at_room;
               }
               break;       //     in-line     comment
            }
            chas = toupper(chas);
         }
         comp_txt  [posit] = chas;
         if    (shad_txt != NULL    and    pass_num     ==     2)
            shad_txt  [posit] = ch2;  // original text.
         comp_typ [posit] = 0;
         if   (pass_num   ==   2)   //  Keep  position
            comp_pos [posit] = i;
         //  ignore  spaces  if  not  in  quote  mode.
         if (!(quote or hollar)) {
            if (chas == '(') {
               brak++;
            }
            else if (chas == ')') {
               brak--;
               if (brak < 0) {
mis_brak:
                  ;
                  printf ("Mismatched brackets at line %d\n", c_line_num);
                  close_down();
                  exit (1);
               }
            }
            else if (chas == ' ') {
               continue;     //ignore spaces
            }
         }
         if (quote or hollar) quote_flg = TRUE;
         if (quote) comp_typ [posit] = 'Q';
         else if (hollar)  comp_typ [posit]  =  'H';
         else  if  (ispunct(chas)  and  chas  !=  '$')
            comp_typ [posit] = chas;
         if (posit >= MAX_TXT){
            printf("Maximum characters in Compressed Text-Abort\n");
            close_down();
            exit (1);
         }
         if (hollar) {
            h_count --;
            if(h_count == 0) hollar   =   0;
         }
         else if (quote) {
            if (chas == '\''){
               quote = 0;
            }
         }
         else {
            if (chas == '\''){
               // check for special case of single quote in
               // WRITE,FIND & READ statements. They can be of the
               // form WRITE(u'r,.... where u=unit number &
               // r = record number.
               if (!(posit > 5 and
                  (isalnum(comp_txt[posit-1]) //variable or unit no.
                  or comp_txt[posit-1] == '$')))
                  goto is_quote;
               chsx  = comp_txt[0];
               s_pnt  = 0; // start point
               sc_pnt = 0; // scan point
switch_test:
               ;
               switch (chsx){
               case 'F':  // find
                  if (memcmp(&comp_txt[s_pnt],"FIND(",5) == 0){
                     pprw_flg = 1;
                     sc_pnt = s_pnt+5;
                     break;
                  }
                  goto is_quote;
               case 'R':  // read
                  if (memcmp(&comp_txt[s_pnt],"READ(",5) == 0){
                     sc_pnt = s_pnt+5;
                     break;
                  }
                  goto is_quote;
               case 'W':  // write
                  if (memcmp(&comp_txt[s_pnt],"WRITE(",6) == 0){
                     sc_pnt = s_pnt+6;
                     break;
                  }
                  goto is_quote;
               case 'I':  // maybe an IF
                  if (s_pnt != 0) goto is_quote;
                  if (memcmp(comp_txt,"IF(",3) != 0)
                     goto is_quote;
                  // find end of IF
                  bra = 0;
                  for (sc_pnt = 2;sc_pnt < posit;sc_pnt++){
                     typx = comp_typ [sc_pnt];
                     if (typx == 'Q' or typx == 'H') continue;
                     typx = comp_txt[sc_pnt];
                     if (typx == '('){
                        bra++;
                     }
                     else if (typx == ')'){
                        bra--;
                        if (bra == 0) { // end of IF reached.
                           s_pnt = sc_pnt + 1;
                           chsx = comp_txt[s_pnt];
                           goto switch_test;
                        }
                     }
                  }
                  goto is_quote;
               default:
                  goto is_quote;
               }
               // scan until ' is found.
               if (posit - sc_pnt > 6) goto is_quote;
               unit_flg = isdigit(comp_txt[sc_pnt]);
               for(;sc_pnt< posit;sc_pnt++){
                  cs = comp_txt[sc_pnt];
                  if(unit_flg){
                     if (! isdigit(cs)) goto is_quote;
                  }
                  else {
                     if (isalnum(cs) or cs == '$') continue;
                     else goto is_quote;
                  }
               }
               goto skip_quote;
is_quote:;
               quote = 1;
               comp_typ [posit] = 'Q';
skip_quote:;
            }
            else if(chas == 'H'){   //   possible  hollerith
               h_count = 0;
               k = 1;
               if (posit > 0) {
                  for (j = posit-1; j >= 0; j--){
                     ch2 = comp_txt [j];
                     if (isdigit(ch2)) {
                        h_count  =  (ch2 - '0') * k + h_count;
                        k = k * 10;
                        if  (h_count > 1000) goto quit_hollar;
                     }
                     else if (ch2 == '$' or isalpha(ch2) or ch2 == '*'){
                        goto quit_hollar;
                     }
                     else {
                        if (h_count != 0){
                           hollar = 1;
                           for   (j++;j   <=    posit+    1;    j++){
                              comp_typ [j] = 'H';
                           }
                        }
                        goto  quit_hollar;
                     }
                  }
               }
            }
quit_hollar:
            ;
         }
         posit = posit + 1;
      }
      con_flg = 1; // look for continuation line
      goto get_next;
   }
   else if (state ==  0) { // first time through
get_next:
      if (fgets(temp_line, sizeof(temp_line), in_f) == NULL){
         state = 2;
         if (posit == 0) {
            printf ("No valid code to process\n");
            close_down();
            exit (1);
         }
         else {
            goto end_it;
         }
      }
      else {
         line_number++;
         if (line_number > 65535) {
            // This limit is due to unsigned short in table l_table
            printf("Too many lines in source file > 65,535\n");
            exit (1);
         }
         state = 1;
      }
   }
   // remove LF character.
   t_len = strlen (temp_line) - 1;
   temp_line [t_len] = 0; // remove LF
   // remove any text in columns 73+
   if (p_opt) {
      if (t_len > 72) {
         memset  (&temp_line[72],   0,   sizeof(temp_line)   -   72);
         t_len = 72;
         // remove any trailing spaces.
         for (i = 71;i >= 0;i--){
            chas = temp_line [i];
            if (chas != ' ') {
               t_len = i + 1;
               temp_line [t_len] = 0;
               goto break_out;
            }
         }
         // case of all spaces
         t_len = 0;
         temp_line [0] = 0;
      }
break_out:;
   }
   if (t_len == 0 ) {
      if (pass_num == 2){
         // keep blank lines case
         if (str_it) {
            save_line("",0);
         }
         else {
            out_line("");
         }
      }
      goto get_next;
   }
   // look for comment
      chas = toupper(temp_line[0]);
   if (chas == 'C' or chas == '*'){
      if ((c_opt or l_opt) and pass_num == 2){
         // start new sentence if comment line not contigous
         if (oc_line_num + 1 != line_number) new_sent = 1;
         oc_line_num = line_number;
         temp_line[0] = chas;   //   make   "C"   upper   case
         com_form_out (&temp_line[1], min(71,t_len-1));
      }
      if (pass_num == 2){
blank_line:
         if (str_it){
            save_line(temp_line, t_len);
         }
         else{
            out_line(temp_line);
         }
      }
      goto get_next;
   }
   if (!(chas == ' ' or isdigit(chas))) {
      // invalid character in column 1, assume comment  but  give  an
      // error.
      printf("%5li: Invalid character col.1 - comment  assumed\n",
         line_number);
      printf ("Line = %s\n",temp_line);
      printf ("  Conversion Aborted\n");
      close_down();
      exit (1);
   }
   if (pass_num == 2) str_it = 1; // from now on store any comments.
   // first time & continuation?
   if (t_len <= 5) {
      goto get_next;
   }
   // ignore if cols 6-72 are blank.
   for (i = 5; i < min(t_len,72); i++){
      if (temp_line [i] != ' ') goto valid_line;
   }
   // line of all spaces - ignore it.
   // output blank line
   if (pass_num == 1) goto get_next;
   // treat as a comment
   goto blank_line;
valid_line:
   memset  (com_line,' ',sizeof(com_line)-2); // any in-line comments
   com_line [sizeof(com_line)-1] = 0;
   if (t_len > 72){

      memcpy  (&com_line[72], &temp_line[72], t_len - 72);
   }
   // expand line to 72 characters minimum.
   if (t_len < 72) {
      // this is for case of ABC = 4H
      memset (& temp_line [t_len],' ', 72-t_len);
      temp_line [72] = 0;
   }
   chas = temp_line [5];
   if  (chas  ==  ' '  or  chas  ==  '0'){  // not continuation line
      if (con_flg == 0) goto case_1;
      else
         goto end_it;
   }
   else { // continuation line
      if (con_flg == 0) {
         printf    ("Continuation    line    as    first    line\n");
         close_down();
         exit (1);
      }
      else {
         if (num_code_lines >= MAX_CRDS) {
            // too many continuation lines
            printf ("Too many continuation lines at line %u\n",
               c_line_num);
            close_down();
            exit (1);
         }
         if (pass_num == 2) { // update table
            cont_char [num_code_lines - 1] = chas;
            assert (line_number - c_line_num < 32768);
            in_lin [num_code_lines++].rel_line = (short int)
               (line_number  -  c_line_num );  //  relative  line   number.
            // position in compacted line.
            in_lin [num_code_lines - 1].txt_posn = posit;
            // store reference in columns 73-80
            if (t_len > 72)
               memcpy (id_txt [num_code_lines -1],
                  &temp_line[72], ID_COLS+1);
         }
         goto unpack_more;
      }
   }
end_it:
   if (brak != 0) goto mis_brak;
   if (quote != 0) {
      printf   ("Pending quote at line %d\n",c_line_num);
      close_down();
      exit (1);
   }
   if (hollar != 0) {
      printf   ("Pending Hollerith at line %d\n",c_line_num);
      close_down();
      exit (1);
   }
   comp_txt [posit] = 0;
   if (shad_txt != NULL) shad_txt [posit] = 0;
   return posit;
}
/* ==================================================================
  str_lab (label number in binary, label type)
      type can be:-
           L_DEFN = label in cols 1-5  {definition}
           L_CODE = label used in code line
           L_SUBR = start of new function/subroutine.
           L_FORM = Mark label as a FORMAT label.
  Returns relative subroutine position (2nd. pass only)

 ==================================================================== */
static void str_lab (long int labl, int type){
   int dc = 0,flags = 0,slot = 0,fnd = 0,fl_gs = 0;
   if (lbl_need == 0) return;
   if (pass_num == 2 and type == L_SUBR) {  //  start  of  new  subroutine
      for   (int   jj   =   lbl_strt_pos+1;   jj   <  numb_lab;jj++){
         if ((t_pnt+jj)->label_num == 0){
            lbl_strt_pos = jj;
            sub_rel_pos ++;
            return;
         }
      }
      // not found - system error
      printf("Next subroutine position not found-abort\n");
      close_down();
      exit (1);
   }
   if (pass_num == 2) {
      flg_label = TRUE; // label present flag.
      return;
   }
   //count on first pass
   if (pass_num == 1 and type == L_SUBR) number_subs++;
   int i;
   // any memory available?
   if (numb_lab >= tot_lab) {
      if (tot_lab == 0) { // first time case
         t_pnt = (struct l_table *)
            calloc (int_num, sizeof(struct l_table));
         if (tester) printf("Table pointer = %i\n",t_pnt);
         tot_lab = int_num;
      }
      else { // extension case
         t_pnt = (struct l_table *) realloc (t_pnt,
            sizeof(struct  l_table) * (tot_lab + LAB_INC));
         tot_lab += LAB_INC;
      }
      if (t_pnt == NULL) {
         printf("Unable to allocate memory\n");
         exit (2);
      }
   }
   switch (type){
   case L_DEFN:  // new label.
      // check if already defined.
      // test if in delete pending list.
      if (del_count and labl != 0){
         dc = 0;
         for (;dc < PEND_D_LABS;dc++){
            if (del_labs[dc] == labl){
               del_labs[dc] = 0;
               del_count--;
               break;
            }
         }
      }
      flags = 0;
      if (del_count) flags |= DR_FLAG;
      for (i = c_point; i < numb_lab; i ++){
         if ((t_pnt + i)->label_num == labl) {
            if ((t_pnt+i)->def_line == 0) {
               // used but not defined.
               (t_pnt+i)->def_line = (unsigned short) line_number;
               (t_pnt+i)->l_flags |=  flags;
               (t_pnt+last_link)->link_pnt = i;
               last_link = i;
               return;
            }
            else { //duplicate define.
               // this is OK if within a "delete" range &
                  // original was within a delete range.
                  if (del_count and
                     ((t_pnt+i)->l_flags & DR_FLAG)){
                     return;
                  }
               printf("Label %li duplicated at lines"
                  " %d and %d \n",
                  labl,(t_pnt+i)->def_line,line_number);
               exit (1);
            }
         }
      }
      // not found- add new entry.
      (t_pnt + numb_lab )-> label_num =  labl;
      (t_pnt + numb_lab )-> new_num   =  0;
      (t_pnt + numb_lab )-> def_line  =  (unsigned short) line_number;
      (t_pnt + numb_lab )-> num_uses  =  0;
      (t_pnt+last_link)->link_pnt = numb_lab;
      (t_pnt+numb_lab)->l_flags = flags;
      last_link = numb_lab;
      numb_lab = numb_lab + 1;
      return;
   case L_CODE: // used at case. (normal goto, assign etc.)
   case L_DELT: // used at case - DELETE
   case L_DOLB : // used at case- DO loop label.
      // check if already defined.
      for (i = c_point; i < numb_lab; i ++){
         if ((t_pnt + i)->label_num == labl) {
            (t_pnt + i)-> num_uses++;
            if (type == L_CODE)  (t_pnt + i)->l_flags |= GO_FLAG;
            if (type == L_DOLB)  (t_pnt + i)->l_flags |= DO_FLAG;
            if (type == L_DELT) { // DELETE
               (t_pnt + i)->l_flags |= DL_FLAG;
               // add to table.
add_delete:
               if (del_count >= PEND_D_LABS) {
                  // expand table or look for error.
                  printf("Pending delete table too small\n");
                  close_down();
                  exit (1);
               }
               // look for free slot or equal slot.
               slot = 0, fnd = 0;
               for (int z = 0;z < PEND_D_LABS; z ++){
                  if (del_labs [z] == labl){
                     fnd = 1;
                     break;
                  }
                  if (del_labs[z] == 0){
                     if (slot == 0) slot = z+1;
                  }
               }
               if (fnd == 0){
                  assert (slot != 0);
                  del_labs [slot-1] = labl;
                  del_count++; // count
               }
            }
            return;
         }
      }
      // not found- add new entry.
      (t_pnt + numb_lab )-> label_num =  labl;
      (t_pnt + numb_lab )-> new_num   =  0;
      (t_pnt + numb_lab )-> def_line  =  0;
      (t_pnt + numb_lab )-> num_uses  =  1;
      (t_pnt + numb_lab )-> link_pnt  =  0;
      fl_gs = 0;
      if      (type == L_DELT) fl_gs = DL_FLAG;
      else if (type == L_CODE) fl_gs = GO_FLAG;
      else if (type == L_DOLB) fl_gs = DO_FLAG;
      (t_pnt + numb_lab )-> l_flags  =  fl_gs;
      numb_lab = numb_lab + 1;
      if (type == L_DELT) goto add_delete;
      return;
   case L_SUBR: // start of new subroutine or function.
      (t_pnt + numb_lab)-> label_num =  0;
      (t_pnt + numb_lab)-> new_num   =  0;
      (t_pnt + numb_lab)-> def_line  =  (unsigned short)line_number;
      (t_pnt + numb_lab)-> num_uses  =  0;
      (t_pnt+last_link)->link_pnt = numb_lab;
      chk_all();   // check for missing labels etc.
      c_point = numb_lab;
      last_link = numb_lab;
      numb_lab = numb_lab + 1;
      return;
   case L_FORM: // mark label as a FORMAT label.
      for (i = c_point; i < numb_lab; i ++){
         if ((t_pnt + i)->label_num == labl) {
            (t_pnt +i )->l_flags |= FM_FLAG;
            break;
         }
      }
      return;
   }
}
// ==================================================================
//  chk_all - fills in replacement label numbers.
//     checks   for   unused    labels,    and    references
//     to labels that are not defined.
// ==================================================================
static void chk_all (void){
   int i, j, undef, labcnt, lnkpnt;
   long int newlab, ulab;
   if (numb_lab == 0) return; // first call
   undef = labcnt = 0;
   for (i= c_point+1; i < numb_lab; i++){
      // check through labels assigned in current subroutine/function.
      if((t_pnt+i)->label_num  ==   0)   {   //   something's   wrong
         printf("Error in label table %d \n",i);
         exit (1);
      }
      if  ((t_pnt+i)->def_line  ==  0)  {  //  label   not   defined.
         if  ((t_pnt+i)->l_flags & DL_FLAG) continue; // used in a delete
         if  (inc_flg  ==  0)  {  //  no   includes   in   this   s/r
            printf("Undefined  label  %ld  \n",(t_pnt+i)->label_num);
            exit (1);
         }
         undef++;
         continue;
      }
      if ((t_pnt+i)->num_uses == 0){
         // first check if reference by a DELETE in any earlier
         // subroutine-function.
         int bk, used = 0;
         long int test_label;
         int prev_sub = 0;
         test_label = (t_pnt+i)->label_num;
         for (bk = i - 1; bk >= 0; bk--){
            if (prev_sub == 0) { // dont look yet!
               if ((t_pnt+bk)->label_num == 0) {
                  prev_sub = 1;
               }
               else {
                  continue;
               }
            }
            if ((t_pnt+bk)->label_num == test_label and
               (t_pnt+bk)-> def_line == 0 and
               ((t_pnt+bk)-> l_flags & DL_FLAG)){
               (t_pnt+bk)->l_flags |= UD_FLAG;
               (t_pnt+i)->num_uses++;
               used = 1;
            }
            // N.B. Carry on searching there might be a few references
         }
         //   unused labels don't count.
         if (u_opt and used == 0) continue;
      }
      labcnt++;
   }
   // if any undefined labels,  check  that  new  labeling  will  not
   // match any of them.
   if (undef != 0 and r_opt) {
      for (i= c_point+1; i <= numb_lab; i++){
         if ((t_pnt+i)->def_line  ==  0)  {  //  label  not  defined.
            ulab =  (t_pnt+i)->label_num;
            for  (j  =  0,   newlab   =   next_num;   j   <   labcnt;
            j++, newlab += lab_num[1]){
               if (ulab == newlab) {
                  printf("Conflict with undefined label %ld \n",ulab);
                  close_down();
                  exit (1);
               }
            }
         }
      }
   }
   //  now  add  new  labels  in  correct  sequence  (by  the  link).
   if (r_opt){
      lnkpnt = (t_pnt + c_point)->link_pnt;
      for (;;lnkpnt=(t_pnt+lnkpnt)->link_pnt) {
         if  ((t_pnt  +  lnkpnt)->label_num == 0) break; // end of chain
         if (u_opt and
            (t_pnt+lnkpnt)->num_uses == 0) { // unused label - skip
            continue;
         }
         (t_pnt + lnkpnt)->new_num = next_num;
         next_num += lab_num [1];
      }
      next_num += lab_num [2];
   }
   return;
}
// ==================================================================
//   key_word  -  marks  position  of  key-word  in compressed string
// ==================================================================
static void key_word (int jock){
   int i;
   for (i = psp; i < psp + jock; i++){
      comp_typ [i] = 'K';
   }
   return;
}
// =================================================================
//   chk_lbl  -  build  table  with  label  numbers  (for re-labeling
// ==================================================================
static void chk_lbl (void){
   char chr, ch_2,fs;
   char * scn_pnt;
   char chf=0,cht=0;
   int i, k, pnt;
   char idx, mode;
   char nl, vl, ida, val, ch1, ch2, ch3, tchr;
   int brac = 0; // bracket count
   int num_par = 0; // number of parameters
   int param_pos [4] = {0}; //parameter positions
   char p_typ [4] = {0}; //parameter type (number/variable)
   int sr_nam_pnt = 0;
   long int lab_3 [3] ; // The three labels.
   int lab_3_posn [3] ; // position
   int lab_3_size [3] ; // size
   int ss = 0;
   int n_digs, pp, kk, num_dig, posn, b_lvl;
   int   switch_flg  =  0;  //  flag  for  switch  variable  checking
   int  bra, cm_count, eq_count, dig;
   // only store labels on pass 1.
   if (lbl_need == 0 and n3opt == 0 and e_opt == 0) return;
   if (psp != 0) goto not_func; // recursive call
   fs = comp_txt[0];
   switch (fs){
   case 'S':
      if(memcmp(comp_txt,"SUBROUTINE",10)  == 0 ) {
         key_word ( 10);
         goto new_func;
      }
      break;
   case 'F':
      if(memcmp(comp_txt,"FUNCTION",8) == 0)   {
         key_word ( 8);
         goto new_func;
      }
      break;
   case 'P':
      if(memcmp(comp_txt,"PROGRAM",7) == 0 )   {
         key_word ( 7);
         goto new_func;
      }
      break;
   case 'I':
      if(memcmp(comp_txt,"INTEGERFUNCTION",15) == 0)   {
         key_word ( 15);
         memset (&comp_typ[7],'k',8);
         goto new_func;
      }
      // Note:- This is a Unisys special variables stored in Intergers.
      else if(memcmp(comp_txt,"INTEGER",7) == 0){
         key_word (7);
         check_lab_init();
      }
      break;
   case 'L':
      if(memcmp(comp_txt,"LOGICALFUNCTION",15) == 0)   {
         key_word ( 15);
         memset (&comp_typ[7],'k',8);
         goto new_func;
      }
      break;
   case 'R':
      if(memcmp(comp_txt,"REALFUNCTION",12) == 0){
         key_word ( 12);
         memset (&comp_typ[4],'k',8);
         goto new_func;
      }
      break;
   case 'D':
      if(memcmp(comp_txt,"DOUBLEPRECISIONFUNCTION",23) == 0) {
         key_word ( 23);
         memset (&comp_typ[6],'k',9);
         goto new_func;
      }
      // Unisys special
      else if(memcmp(comp_txt,"DIMENSION",9) == 0){
         key_word(9);
         check_lab_init();
      }
      // Unisys special
      else if(memcmp(comp_txt,"DATA",4) == 0){
         key_word(4);
         check_lab_init();
      }
      break;
   case 'C':
      if(memcmp(comp_txt,"CHARACTERFUNCTION",17) == 0){
         key_word ( 17);
         memset (&comp_typ[9],'k',8);
         char_flg = 1;
         goto new_func;
      }
      if(memcmp(comp_txt,"COMPLEXFUNCTION",15) == 0){
         key_word ( 15);
         memset (&comp_typ[7],'k',8);
         goto new_func;
      }
      // Character functions can be defined as:-
      // CHARACTER*12 FUNCTION...   or
      // CHARACTER*(ABC) FUNCTION
      if(memcmp(comp_txt,"CHARACTER*",10) == 0) {
         chr = comp_txt[10];
         char_flg = 1;
         if  (!(chr == '(' or isdigit(chr))) goto not_func;
         //   look for matching end bracket or end of numerics.
         if (chr == '(') {
            bra = 1;
            for (i = 11; i < n_chas - 8; i ++){
               chr = comp_txt [i];
               if (chr == '(') bra++;
               else if(chr == ')') {
                  bra--;
                  pnt = i+1;
                  if (bra == 0) goto more_test;
               }
            }
            goto not_func;
         }
         else { // numeric case
            for (i = 11; i < n_chas -8; i ++){
               chr = comp_txt [i];
               if (!(isdigit(chr))) {
                  pnt = i;
                  goto more_test;
               }
            }
         }
         goto not_func;
more_test:
         if     (memcmp(&comp_txt[pnt],"FUNCTION",8) != 0)     {
            goto not_func;
         }
         else { //position of function name.
            sr_nam_pnt = pnt+8;
         }
new_func:
         curr_lvl ++; // subroutine/function number
         if (pass_num == 1 ) assg_strt = assign_act;
         if (pass_num == 2 and e_opt) {
            if (e_first == 0){
               e_first = 1;
            }
            else {
               if (e_prev_end == 0) {
                  // output an "END"
                  out_line("      END");
               }
            }
            if (lbl_need == 0 and n3opt == 0) return;
         }
         // new function - set zero in label table.
         str_lab (0,L_SUBR);
         inc_flg = 0;
         if (x_opt and pass_num == 2) {
            // store s/r function name.
            char s_name [SZ_sname];
            memset (s_name,0,sizeof(s_name));
            if (sr_nam_pnt != 0){
               pp = 0;
               for (int cx = sr_nam_pnt; cx < n_chas; cx++){
                  nl = comp_txt[cx];
                  if (isalnum(nl) or nl == '$'){
                     s_name[pp++] = nl;
                  }
                  else{
                     goto lab_fnd;
                  }
               }
            }
            else {
               for (int ab =0 ;ab < n_chas; ab++){
                  if (comp_typ[ab] == 'K' or
                     comp_typ[ab] == 'k') continue;
                  pnt = ab;
                  for (int xx= 0;pnt< ab + SZ_sname - 1; pnt++){
                     vl = comp_txt[pnt];
                     if (isalnum(vl) or vl == '$'){
                        s_name[xx++] = vl;
                     }
                     else {
                        goto lab_fnd;
                     }
                  }
                  break;
               }
            }
lab_fnd:
            ;
            memcpy ((s_nme_pnt + sub_rel_pos)->sname_ent,
               s_name, SZ_sname);
         }
         return;
      }
      break;
   }
not_func:
   if (lbl_need == 0 and n3opt == 0) return;
   // =======================================
   //   lab_chk  {label  checking  -  look  for label containing code
   // =======================================
   ida = comp_txt[psp];
   switch (ida){
   case 'A':
      goto a_key;
   case 'B':
      goto b_key;
   case 'C':
      goto c_key;
   case 'D':
      goto d_key;
   case 'E':
      goto e_key;
   case 'G':
      goto g_key;
   case 'I':
      goto i_key;
   case 'O':
      goto o_key;
   case 'P':
      goto p_key;
   case 'R':
      goto r_key;
   case 'W':
      goto w_key;
   }
   return;
a_key:
   if (memcmp(&comp_txt[psp],"ASSIGN",6) == 0){
      key_word (6);
      // find "TO"
      pnt = psp + 6;
      kk = pnt;
      n_digs = 0;
      for (;kk < n_chas; kk++){
         if (isdigit(comp_txt [kk])){
            n_digs++;
            if (n_digs > 5) break;
            continue;
         }
         else {
            if (comp_txt [kk]   == 'T' and
               comp_txt [kk+1] == 'O') {
               comp_typ[kk] = comp_typ[kk+1] = 'K';
               if (n3opt and pass_num == 1) {
                  // keep list of ASSIGNs for arithmetic IF
                  str_3_assign(pnt);
               }
            }
            break;
         }
      }
      goto common_entry;
   }
   else if (comp_txt[0] == 'A' and comp_txt[1] == 'T' and
      isdigit(comp_txt[2])) {// debug packet AT label
      if (at_label(2)) {
         comp_typ[0] = comp_typ[1] = 'K';
         if(pass_num == 2) mark_label( 2, n_chas - 2);
         old_lbl = get_lbl (&comp_txt [2], & num_dig);
         str_lab (old_lbl, L_CODE);
      }
   }
   return;
b_key:
   if (memcmp(&comp_txt[psp],"BACKSPACE(",10)   == 0){
      key_word (9);
      pnt = psp + 9;
      goto common_scan_err;
   }
   return;
c_key:
   if (memcmp(&comp_txt[psp],"CALL",4)   == 0  &&
      isalpha(comp_txt[psp+4])){
      key_word (4);
      pnt = psp+4;
      // check for any parameters.
      if (comp_txt[n_chas-1] != ')') return; // none
      // look for any labels in CALL Parameters.
      bra = 0;
      for (i = psp+5; i < n_chas; i ++) {
         chr = comp_txt [i];
         if (comp_typ [i] == 'Q' or comp_typ [i] == 'H') continue;
         if (bra == 0) { // look for first opening bracket
            if (chr == '('){
               bra = 1;
            }
            continue;
         }
         else if (chr == '(') {
            bra++;
            continue;
         }
         else if (chr == ')') {
            bra--;
            if (bra < 0) {
inv_call:
               printf("Invalid CALL at line %d\n",c_line_num);
               exit (1);
            }
         }
         // Labels can start with *, $ or &.
         else if((chr == '*' or chr == '$' or chr == '&') && bra > 0){
            ch_2 = comp_txt [i-1];
            if (! (ch_2 == '(' or ch_2 == ',') ) continue;
            ch_2 = comp_txt [i+1];
            if (isdigit (ch_2)){
               old_lbl = get_lbl (&comp_txt [i+1], & num_dig);
               if (pass_num == 2) mark_label(i+1, num_dig);
               str_lab (old_lbl, L_CODE);
               i = i + num_dig;
            }
         }
      }
      // end of loop
      if (bra != 0) goto inv_call;
   }
   else if (memcmp(&comp_txt[psp],"CLOSE(",6)   == 0){
      key_word (5);
      pnt = psp + 5;
      goto common_scan_err;
   }
   return;
d_key:
   if (memcmp(&comp_txt[psp],"DECODE(",7) == 0){
      key_word (6);
      pnt = psp + 6;
find_format:
      // this can be tricky. format label can be either first or second
      // parameter. Depends on how many parameters there are.
      // Don't count any "ERR="
      brac = 0; // bracket count
      num_par = 0; // number of parameters
      for (ss = pnt;ss < n_chas; ss++){
         chf = comp_txt[ss] ;
         cht = comp_typ[ss] ; // type
         if (cht == 'Q' or cht == 'H') continue;
         switch (chf){
         case '(':
            {
               if (brac == 0) {
                  // first parameter
                  param_pos [0] = ss+1;
                  if (isdigit(comp_txt[ss+1])){
                     p_typ[0] = '9';
                  }
                  else {
                     p_typ[0] = 'A';
                  }
                  num_par ++;
               }
               brac++;
               break;
            }
         case ')':
            {
               brac--;
               if (brac == 0) goto drop_out;
               break;
            }
         case ',':
            {
               if (brac == 1) { // only at first level
                  if (num_par > 2) { // check for ERR=
                     if (comp_txt[ss+1] == 'E'){
                        if(memcmp(&comp_txt[ss+1],"ERR=",4) == 0){
                           goto drop_out;
                        }
                     }
                  }
                  if (isdigit(comp_txt[ss+1])) {
                     p_typ [num_par] = '9';
                  }
                  else {
                     p_typ [num_par] = 'A';
                  }
                  param_pos[num_par] = ss+1;
                  num_par ++;
                  if (num_par >= 4) goto drop_out;
               }
               break;
            }
         }
      }
drop_out:; // now work out what is what.
      k = 0;
      if (num_par == 2) { // simple case, 1st. parameter is Format.
         if (p_typ[0] == '9') {
            k = param_pos [0];
         }
      }
      else if (num_par == 4) { // both extra optional parameters present
         // format is in 2nd. position
         if (p_typ[1] == '9'){
            k = param_pos [1];
         }
      }
      else if (num_par == 3) { // tricky, either 1st. or last optional
         // is present.
         if ((p_typ[0] == '9' and p_typ[1] == '9') or
            (p_typ[0] == 'A' and p_typ[1] == '9')) {
            // second entry is format label.
            k = param_pos[1];
         }
      }
      if (k != 0) {
         old_lbl = get_lbl (&comp_txt [k], & num_dig);
         if (pass_num == 2) mark_label(k, num_dig);
         str_lab (old_lbl, L_CODE);
      }
      goto common_scan_err;
common_scan:
      if (comp_txt[pnt] == '(') pprw_flg = 1;
      // look for FMT=
      scn_pnt = strstr(&comp_txt[pnt],"(FMT=");
      if (scn_pnt == NULL) {
         scn_pnt = strstr(&comp_txt[pnt],",FMT=");
      }
      if (scn_pnt != NULL) { // FMT found.
         scn_pnt = scn_pnt + 5;
         if (isdigit(*scn_pnt)) {
            old_lbl = get_lbl (scn_pnt, & num_dig);
            if (pass_num == 2) mark_label(scn_pnt - comp_txt, num_dig);
            str_lab (old_lbl, L_CODE);
         }
      }
      // look for END=
      scn_pnt = strstr(&comp_txt[pnt],"(END=");
      if (scn_pnt == NULL) {
         scn_pnt = strstr(&comp_txt[pnt],",END=");
      }
      if (scn_pnt != NULL) { // END found.
         scn_pnt = scn_pnt + 5;
         if (isdigit(*scn_pnt)) {
            old_lbl = get_lbl (scn_pnt, & num_dig);
            if (pass_num == 2) mark_label(scn_pnt - comp_txt, num_dig);
            str_lab (old_lbl, L_CODE);
         }
      }
common_scan_err:
      // look for ERR=
      scn_pnt = strstr(&comp_txt[pnt],"(ERR=");
      if (scn_pnt == NULL) {
         scn_pnt = strstr(&comp_txt[pnt],",ERR=");
      }
      if (scn_pnt != NULL) { // ERR found.
         scn_pnt = scn_pnt + 5;
         if (isdigit(*scn_pnt)) {
            old_lbl = get_lbl (scn_pnt, & num_dig);
            if (pass_num == 2) mark_label(scn_pnt - comp_txt, num_dig);
            str_lab (old_lbl, L_CODE);
         }
      }
   }
   else if (psp == 0 and memcmp(comp_txt,"DELETE",6) == 0 and
      isdigit(comp_txt[6])){
      key_word (6);
      pnt = psp + 6;
      goto common_entry;
   }
   else  if(psp == 0 &&  memcmp(comp_txt,"DO",2)   ==   0
      and isdigit (comp_txt[psp+2])){
      // note this  can  be  confusing  if  a  variable  has  a  name
      //  like  DO123A.  Check for "=" sign in code line, followed by
      // at least one and not more than 2 ','.
      dig = 0; // digit count.
      for (i = 2; i < n_chas; i ++){
         chr = comp_txt [i];
         if (isdigit(chr)){
            dig++;
            continue;
         }
         else   if   (isalpha(chr)){   //    start    of    variable.
            if (dig > 5) break; // too many digits
            goto do_check_2;
         }
         else if (chr == ',') { //
            if (!isdigit(comp_txt[i-1])) break;
            if (dig > 5) break; // too many digits
            continue;
         }
         else {  // unknown character
            break;
         }
      }
      return; // still on numbers.
do_check_2:  // There should  be  a  only  one  "="  and  at  least  one
      //   maximum   two   commas,   all  outside  of  any  brackets.
         eq_count = 0;
      cm_count = 0;
      bra      = 0;
      for (i=2;i < n_chas; i ++){
         chr = comp_typ[i];
         if (chr == '(') {
            bra++;
            continue;
         }
         if (chr == ')') {
            bra --;
            if (bra < 0) {
               // This has been checked before.
               printf("%s  %5hu:  System  error  975\n",in_nm,
                  line_number);
            }
            continue;
         }
         if (bra == 0 && chr == '='){
            if (eq_count != 0) return;
            eq_count = 1;
         }
         if  (bra  ==  0  &&  chr  ==  ','   &&   eq_count   ==   1){
            cm_count ++;
            if (cm_count > 2) return;
         }
      }
      // end of check - ensure at least one ','
      if (cm_count < 1) return;

      key_word (2);
      pnt = psp + 2;
      goto common_entry;
   }
   return;
e_key:
   if (memcmp(&comp_txt[psp],"ENCODE",6) == 0){
      key_word (6);
      pnt = psp + 6;
      goto find_format;
   }
   else if (memcmp(&comp_txt[psp],"ENDFILE(",8)   == 0){
      key_word (7);
      pnt = psp + 7;
      goto common_scan_err;
   }
   return;
g_key:
   if(memcmp(&comp_txt[psp],"GOTO",4) == 0){
      key_word (4);
      goto goto_chk;
   }
   return;
i_key:
   if (memcmp(&comp_txt[psp],"IF(",3) == 0){
      key_word (2);
      bra = 1;
      // find end of if
      pnt = if_end(psp);
      if (pnt != -1) goto end_if;
      // end of if search loop
      printf ("Invalid 'IF' at line %d \n", c_line_num);
      close_down();
      exit (1);
end_if: // find type of if.
      // check if variable assign.
      if (chk_eq(pnt) == 1) return;
      // it is an arithmetic if, if contains commas
      // & only alpha-numerics
      tchr = comp_txt[pnt];
      switch (tchr) {
         // N.B. Some jumps can be ASSIGN'ED variables.
      case 'T':
         if (strcmp(&comp_txt[n_chas-5],")THEN")== 0) goto not_arith;
         break;
      case 'R':
         if (n_chas - pnt == 6 and comp_txt[pnt] == 'R'){
            if (strcmp(&comp_txt[pnt],"RETURN") == 0) goto not_arith;
         }
         // can be an open READ e.g. READ 12,fred
         if (memcmp(&comp_txt[pnt],"READ",4) == 0) {
            // from there-on digits followed by a comma
            num_dig = 0;
            for (int cv = pnt+4;cv < pnt+4+6;cv++){
               val = comp_txt[cv];
               if (isdigit(val)){
                  num_dig ++;
                  if (num_dig > 2) break;
                  continue;
               }
               if (val == ',') goto not_arith;
               break;
            }
         }
         break;
      case 'G':
         // can also be confused by a GOTO with a one or two digit label.
         if (n_chas - pnt < 7 and
            memcmp(&comp_txt[pnt],"GOTO",4) == 0) {
            // check for 1 or two digits.
            for (i=pnt+4;i<n_chas;i++){
               if (! (isdigit(comp_txt[i]))) goto tst_3;
            }
            goto not_arith;
         }
         // can be Print or Punch, with a one digit format number
      case 'P':
         if (memcmp ( &comp_txt[pnt], "PRINT", 5) == 0 or
            memcmp (&comp_txt[pnt], "PUNCH", 5) == 0){
            //   test for one digit format label + a comma
            ch1 = comp_txt[pnt+5];
            ch2 = comp_txt[pnt+6];
            ch3 = comp_txt[pnt+7];
            if (isdigit(ch1) and
               ch2 == ',' and
               isalpha(ch3)) goto not_arith;
         }
         break;
      }
tst_3:
      mode = 0 ; // nothing mode. 1 = label, 2 = variable.
      idx  = 0 ; // first address/variable.
      old_lbl = 0;
      num_dig = 0;
      lab_3[0] = lab_3[1] = lab_3[2] = 0;
      for (i = pnt; i < n_chas; i++) {
         chr = comp_txt [i];
         if (isalnum(chr)) {
            if (isalpha(chr)) {
               if (mode == 0) {
                  mode = 2;
                  num_dig = 1;
                  continue;
               }
               else if (mode == 1){ // can't have alpha in a label.
                  goto not_arith;
               }
               else if (mode == 2) { // variable- check number of chars
                  if (num_dig == 6) goto not_arith;
                  num_dig++;
                  continue;
               }
               else {
                  abort();
               }
            }
            else if (isdigit(chr)){
               if (mode == 0) { // first time
                  mode = 1; // label
                  num_dig = 1;
                  old_lbl = chr - '0';
                  posn = i;
                  continue;
               }
               else if (mode == 1)  { // processing label
                  if (num_dig == 5) {
                     goto not_arith; // too many digits.
                  }
                  old_lbl = old_lbl * 10 + chr - '0';
                  num_dig ++;
                  continue;
               }
               else if (mode == 2) { // processing variable
                  if (num_dig == 6) {
                     goto not_arith; // only six in variable name
                  }
                  num_dig ++;
                  continue;
               }
               else {
                  abort();
               }
            }
         }
         else if (chr == ',') {
            if (idx == 2) { // already two commas
               goto not_arith;
            }
            lab_3 [idx] = 0;
            if (mode == 1) { // been label processing?
               if (old_lbl == 0) goto not_arith;
               lab_3 [idx] =  old_lbl;
               lab_3_size [idx] = num_dig;
               lab_3_posn [idx] = posn;
            }
            idx ++;
            mode = 0;
            old_lbl = 0;
            continue;
         }
         else {
            // only alpha numerics & commas allowed.
               goto not_arith;
         }
      } // end of loop - any last label?
      if (idx == 0) goto not_arith; // need at least one comma
      if (mode == 1) {
         if (old_lbl == 0) goto not_arith;
         lab_3 [2] = old_lbl;
         lab_3_posn [2] = posn;
         lab_3_size [2] = num_dig;
      }
      // now store in table.
      flg_3_label = TRUE;
      flg_3_secondary = FALSE;
      if (psp != 0) flg_3_secondary = TRUE;
      for (i = 0; i< 3; i++) {
         if (lab_3 [i] != 0) {
            str_lab (lab_3 [i], L_CODE);
            mark_label (lab_3_posn[i], lab_3_size [i]);
         }
      }
      return;
not_arith:
      // not an arithmetic - recursive call this routine
      // illegal as 2nd. part of IF
      if (psp != 0) {
         printf ("Invalid IF at line %d \n", c_line_num);
         close_down();
         exit (1);
      }
      psp = pnt;
      chk_lbl();
      return;
   }
   else if (memcmp(comp_txt,"INCLUDE",7) == 0){
      inc_flg = 1;
   }
   else if (memcmp(&comp_txt[psp],"INQUIRE",7) == 0){
      goto common_scan_err;
   }
   return;
o_key:
   if (memcmp(&comp_txt[psp],"OPEN(",5) == 0){
      key_word (4);
      pnt = psp + 4;
      goto common_scan_err;
   }
   return;
p_key:
   if (memcmp(&comp_txt[psp],"PRINT",5)  == 0){
      pnt = psp + 5;
      // check if PRINT is a variable.
      if (chk_eq(pnt) == 1) return;
      key_word (5);
      // without open bracket first item can be format label.
      if (isdigit(comp_txt [pnt])) goto form_first;
      goto common_scan;
   }
   else if (memcmp(&comp_txt[psp],"PUNCH",5)  == 0){
      pnt = psp + 5;
      // check if PUNCH is a variable.
      if (chk_eq(pnt) == 1) return;
      key_word (5);
      // without open bracket first item can be format label.
      if (isdigit(comp_txt [pnt])) goto form_first;
      goto common_scan;
   }
   return;
r_key:
   if (memcmp(&comp_txt[psp],"READ",4)   == 0){
      key_word (4);
      pnt = psp + 4;
      goto common_rw_a;
   }
   else if (memcmp(&comp_txt[psp],"REWIND",6)   == 0){
      key_word (6);
      pnt = psp + 6;
      return;
   }
   return;
w_key:
   if (memcmp(&comp_txt[psp],"WRITE",5)  == 0){
      key_word (5);
      pnt = psp + 5;
common_rw_a:
      // read & write because they are < 6 characters can be
      // confused with a variable name e.g. READ5 = 23.
      // To  handle these case check for an = at level zero,
      // that is outside any brackets.
      if (chk_eq(psp) == 1) return;
      // if no first "(", then format number.
      if (isdigit(comp_txt [pnt])){
form_first:
         old_lbl = get_lbl(&comp_txt [pnt],&num_dig);
         if (pass_num == 2) mark_label(pnt, num_dig);
         str_lab (old_lbl, L_CODE);
         return;
      }
      // look for 2nd. parameter which if a number is format label.
      if (comp_txt[pnt] == '(') {
         b_lvl = 1;  // bracket level.
         for (i = pnt+1; i < n_chas; i++){
            chr = comp_txt [i];
            if (comp_typ[i] == 'Q' or comp_typ[i] == 'H') continue;
            if (chr == '('){
               b_lvl++;
               continue;
            }
            if (chr == ')'){
               if (--b_lvl == 0) break; // end of search
            }
            if (chr == ',' and b_lvl == 1) {
               if ( isdigit (comp_txt [i+1])) {
                  old_lbl =  get_lbl (&comp_txt[i+1], &num_dig);
                  if (pass_num == 2) mark_label(i+1, num_dig);
                  if (old_lbl != 0) str_lab (old_lbl, L_CODE);
               }
               break;
            }

         }
         goto common_scan;
      }
   }
   return;
common_entry:
   if (isdigit(comp_txt[pnt])) {
      old_lbl = get_lbl (&comp_txt [pnt], & num_dig);
      if (pass_num == 2) mark_label(pnt, num_dig);
      int l_type = L_CODE; // label type
      if (comp_txt[0] == 'D' and comp_txt[1] == 'O' and
         isdigit(comp_txt[3])) l_type = L_DOLB ; // DO label
      else if (memcmp(comp_txt,"DELETE",6) == 0) l_type = L_DELT;
      str_lab (old_lbl, l_type);
   }
   return;
   // ================== GOTO ===================
goto_chk:
   pnt = psp + 4;
   chr = comp_txt[pnt] ;
   if ( isdigit(chr)) { // simple goto
      old_lbl = get_lbl (& comp_txt[pnt],& num_dig);
      if (pass_num == 2) mark_label(pnt, num_dig);
      str_lab (old_lbl, L_CODE);
      return;
   }
   else if (chr == '(' ){ // arithmetic goto.
      // for some reason arithmetic gotos can have
      // a switch variable in list.
      switch_flg = 1;
next_lab:
      pnt = pnt + 1;
list_found:
      if (switch_flg == 1) { // allow for switch variables in list
         if (isalpha(comp_txt[pnt])) { // move along to next item
            for (pnt++; pnt < n_chas; pnt++){
               if (comp_txt [pnt] == ')') return;
               if (comp_txt [pnt] == ',') goto next_lab;
            }
         }
      }
      // can be empty jump item.
      chr = comp_txt [pnt];
      if (chr == ',') goto next_lab;
      if (chr == ')') return;
      old_lbl = get_lbl (&comp_txt[pnt], & num_dig);
      if (pass_num == 2) mark_label(pnt, num_dig);
      str_lab (old_lbl, L_CODE);
      chr = comp_txt [pnt + num_dig];
      if (chr == ')') {
         return;
      }
      else if (chr == ',') {
         pnt = pnt + num_dig;
         goto next_lab;
      }
      else {
         abort ();
      }
   }
   else if (isalpha(chr)){ // assigned goto
      // search for list of labels (optional)
      for (i=pnt; i < n_chas; i ++){
         // N.B. comma is optional.
         if (comp_txt [i] == ',' && comp_txt [i+1] == '('){
            pnt = i+2;
            goto list_found;
         }
         if (comp_txt [i] == '('){
            pnt = i+1;
            goto list_found;
         }
      }
      return;
   }
}
/* **************************************************************
   check_lab_init - checks for label initialization in either
                    DATA, DIMENSION or INTEGER statments.
                    e.g. DATA BOX /&1234/ where 1234 is a label
                    can also be of form *1234, or $1234.
   ************************************************************** */
static void check_lab_init (void){
   int mode = 0;
   for (int ww = 0; ww < n_chas; ww++){
      char t_typ = comp_typ[ww];
      char t_txt = comp_txt[ww];
      if (t_typ == 'Q' or t_typ == 'H') continue;
      switch (t_txt)
      {
      case '/':
         if (mode == 0) {
            mode = 1;
         }
         else {
            mode = 0;
         }
         if (mode == 0) break;
      case ',':
         if (mode == 1){
            char next_c = comp_txt[ww+1];
            if (next_c == '*' or next_c == '$' or next_c == '&'){
               char next_n = comp_txt[ww+2];
               if (isdigit(next_n)) {
                  long int lab_set;
                  // possible label.
                  int num_d;
                  lab_set = get_lbl(&comp_txt[ww+2], & num_d);
                  if(pass_num == 2)
                     mark_label( ww+2, num_d);
                  str_lab (lab_set, L_CODE);
               }
            }
         }
         break;
      }
   }
   return;
}
/* **************************************************************
  get_lbl -  start position in character string for label.
          -  number of digits. (returned value)
   Returns label numeric value & number of digits.
   ************************************************************** */
static long int get_lbl (char * str, int * n_dig){
   int i;
   char chr;
   long int new_label = 0;
   for (i = 0; i < 6; i++) {
      chr = * (str+i);
      if (isdigit(chr)) {
         new_label = new_label * 10 + chr - '0';
      }
      else {
         goto lab_end;
      }
   }
   // should never be more than 5 digits.
bad_label:
      printf ("Invalid label at line %d \n", c_line_num);
   exit (1);
lab_end:
   if (new_label == 0) goto bad_label;
   * n_dig = i;
   return new_label;
}
// *******************************************************************
//  chk_eq - checks for an '=' outside of any brackets in the
//  compressed line of text. Returns 1 if any found, 0 otherwise.
//  Has one parameter, the start point in compressed text.
// *******************************************************************
static int  chk_eq (int st_pnt) {
   int i, bra, chs;
   bra = 0;
   for (i = st_pnt; i < n_chas; i++){
      chs = comp_typ [i];
      if (chs == '(') {
         bra++;
      }
      else if (chs == ')') {
         bra--;
         if (bra < 0) {
            printf (
               "%5hu: Invalid Text ABORT!!\n",
               c_line_num);
            printf("%s\n",comp_txt);
            close_down();
            exit (1);
         }
      }
      else if (bra == 0 and chs == '='){
         return 1;
      }
   }
   return 0;
}
/* ********************************************************************
    spell_split - splits a long of characters into words
    calling parameters:-
    (1) text string pointer.
    (2) number of characters.
    Returns 0 if no spelling errors found, otherwise number of words in
    error.
  ******************************************************************** */
static int  spell_split (char * c_string, int str_len){
   const int MAX_SPELL = 32; // max size of a word
   int  i, strt_pos, k;
   int  err_cnt = 0;
   char  num_alpha = 0; // numeric-alpha flag (for cases such as 1st.)
   char sp_word [MAX_SPELL+2] = {0};
   int  sp_count = 0;
   char chs;
   for (i = 0;i < str_len; i++){
      chs = tolower(c_string[i]);
      if (isalpha(chs)) {
as_alpha:
         if (sp_count >= MAX_SPELL) { // ignore
            continue;
         }
         else {
            if (sp_count == 0) strt_pos = i;
            sp_word [sp_count++] = chs;
            continue;
         }
      }
      else if (isdigit(chs)){
         if(sp_count > 0) {
            goto as_alpha;
         }
         else {
            num_alpha = chs;
         }
      }
      else if (chs == '$' and sp_count > 0) {
         // where a parameter name is input e.g. S$ABCD
         goto as_alpha;
      }
      else { // general separator
         if (sp_count > 0) {
sep_code:
            ;
            sp_word[sp_count] = 0;
            // test for special cases (1st. 2nd. 3rd. 4th. etc.)
            if (num_alpha != 0 and sp_count == 2){
               chs = num_alpha;
               switch (chs) {
               case '1':
                  if(sp_word[0] == 's' and sp_word[1] == 't'){
                     goto word_ok;
                  }
                  break;
               case '2':
                  if(sp_word[0] == 'n' and sp_word[1] == 'd'){
                     goto word_ok;
                  }
                  break;
               case '3':
                  if (sp_word[0] == 'r' and sp_word[1] == 'd'){
                     goto word_ok;
                  }
                  break;
               default:
                  if (sp_word[0] == 't' and sp_word[1] == 'h'){
                     goto word_ok;
                  }
                  break;
               }
            }
            // Special English Language test
            if (!(sp_count == 1 and
               (sp_word[0] == 's' or sp_word[0] == 't') and
               c_string[strt_pos - 1] == '\'')){
               if (!common_test(sp_word)){
                  if (!chkword(sp_word)) { // test for invalid spelling
                     err_cnt++;
                     for (k = strt_pos; k < strt_pos + sp_count; k++){
                        // to upper case
                        c_string [k] = toupper (c_string [k]);
                     }
                  }
               }
            }
         }
word_ok:
         ;
         sp_count = 0;
         num_alpha = 0;
      }
   }
   if (sp_count > 0) {
      goto sep_code;
   }
   return err_cnt;
}
/* ********************************************************************
  common_test - checks against most common words in English.
  returns TRUE is word OK, otherwise FALSE.
 ******************************************************************* */
static int  common_test(char * text_in) {
   // test if one of most common words.
   /* *** Most Common English Words *** */
   // The following words are 18% approx of all Fortran comments.
   const char * t_wrd[] = {"the","to"};
   const char * i_wrd[] = {"is","if","in"};
   const char * o_wrd   = "of";
   const char * f_wrd   = "for";
   const char * a_wrd[] = {"and","a"};
   const char * e_wrd   = "error";
   // number of entries.
   const int n_t_wrd = sizeof(t_wrd)/sizeof(t_wrd[0]);
   const int n_i_wrd = sizeof(i_wrd)/sizeof(i_wrd[0]);
   const int n_a_wrd = sizeof(a_wrd)/sizeof(a_wrd[0]);
   char f_chr = text_in[0];
   int jj;
   switch (f_chr) {
   case 't':
      {
         for (jj = 0;jj< n_t_wrd;jj++){
            if (strcmp(text_in,t_wrd[jj]) == 0) return TRUE;
         }
         break;
      }
   case 'i':
      {
         for (jj = 0;jj < n_i_wrd;jj++){
            if (strcmp(text_in,i_wrd[jj]) == 0) return TRUE;
         }
         break;
      }
   case 'a':
      {
         for (jj = 0;jj < n_a_wrd;jj++){
            if (strcmp(text_in,a_wrd[jj]) == 0) return TRUE;
         }
         break;
      }
   case 'o':
      {
         if (strcmp(text_in,o_wrd) == 0) return TRUE;
      }
   case 'f':
      {
         if (strcmp(text_in,f_wrd) == 0) return TRUE;
      }
   case 'e':
      {
         if (strcmp(text_in,e_wrd) == 0) return TRUE;
      }
   }
   return FALSE;
}
/* ****************************************************************
   com_form_out (o_line)- outputs comment line.
 *************************************************************** */
static void com_form_out (char * o_line, int str_sz){
   char chas;
   int j, alpha_flg = 0,words_flg = 0;
   // format and output comment.
   if (l_opt or c_opt) {
      for (j=0; j < str_sz; j++){
         chas = tolower(o_line[j]);
         if (chas == '.' or chas == '?' or chas == '!'
            or chas == ':'){
            new_sent = 1;
            // special case of "." within numbers e.g. "3.4.5"
            if (chas == '.' and isdigit(o_line[j+1])) new_sent = 0;
         }
         if (isalpha(chas)){
            if (words_flg == 0){ // check for words or just single letters
               if (j != 0 and isalpha(o_line[j-1])) words_flg = 1;
            }
            alpha_flg = 1; // shows some alphabetic characters on line.
            if (c_opt and new_sent){
               if (!(j != 0 and isdigit(o_line[j-1]))){
                  chas = toupper(chas);
               }
               new_sent = 0;
            }
         }
         o_line [j] = chas;
         if (chas == 0) break;
      }
      // whole line may be single letters seperated by spaces-
      // put them in upper case.
      if (words_flg == 0 and alpha_flg == 1){
         for (j=0; j< str_sz; j++) o_line [j] = toupper(o_line [j]);
      }
      else if (s_opt){
         spell_split (o_line, str_sz);
      }
      if (alpha_flg == 0) new_sent = 1;
      return;
   }
}
/* ****************************************************************
  out_line(char * tmp_lne) - sends a line to final output
 *************************************************************** */
static void out_line (char * tmp_lne){
   if (no_sec) return;
   if (b_opt) {// check for blank lines.
      int st_len = strlen(tmp_lne);
      for (int zz = st_len-1; zz>=0;zz--){
         if (tmp_lne[zz] != ' '){
            tmp_lne [zz+1] = 0;
            goto test_it;
         }
      }
      return;
      // also blank comment lines.
test_it:;
      if (tmp_lne[1] == 0) {
         char xv = tmp_lne[0];
         if (xv == 'C' or xv == 'c' or xv == '*') return;
      }
   }
   if (fprintf (out_f,"%s\n",tmp_lne)== EOF){
      printf("Error in sending output line\n");
      exit (1);
   }
   return;
}
/* ******************************************************************
   save_line - Saves a comment line in a buffer for later output.
 ***************************************************************** */
static void save_line (char * com_line, int l_len){
   int sz_ask = sizeof (cb_line);
   if (l_len > MAX_COM_LIN -1) { // check size
      printf ("Comment line too long\n");
      printf ("Length = %d Text = %s\n",l_len, com_line);
      close_down();
      exit (1);
   }
   if (cb_lines <  MAX_COM) {
      // store in array
      memcpy (act_cb[cb_lines].cb_text, com_line, MAX_COM_LIN);
      cb_lines++;
      return;
   }
   else if (cb_lines == MAX_COM) { // store on disk file
      if (cb_file_open) { // close temporary comments file.
         if (fclose (com_ovf_f)==EOF){
            printf("Error in close of temporary (read) comment file\n");
            exit (1);
         }
         cb_file_open = 0;
      }
      if (sav_nam == 0) {
         tmpnam (sav_line_file_name);
         sav_nam = 1;
      }
      if ((com_ovf_f = fopen(sav_line_file_name,"wb")) == NULL){
         printf ("Unable to open temporary comment file (write)\n");
         exit (1);
      }
   }
   if (fwrite (com_line, sz_ask, 1, com_ovf_f) !=  1) {
      printf ("Bad write of Comment overflow file\n");
      exit (1);
   }
   cb_lines_hd++;
   cb_lines++;
   return;
}
/* *********************************************************************
  place_label (label) put do loop label into a table -used for indenting
    ****************************************************************** */
static void place_label (long int v_label){
   int j;
   // check if already in table.
   for (j = 0;j < lb_count; j++){
      if ( do_table_label[j] == v_label) {
         do_table_count[j]++;
         id_pend += n_do;
         return;
      }
   }
   // add to table (if possible)
   if (lb_count >= MAX_DO_LBL) {
      printf ("Too many DO labels - expand table\n");
      close_down();
      exit (1);
   }
   do_table_label [lb_count] = v_label;
   do_table_count [lb_count++] = 1;
   id_pend += n_do;
   return;
}
/* *****************************************************************
   check_label (label) - checks if a label is a terminator to a
   do loop. If so it reduces the current indentation level.
 **************************************************************** */
static void check_label (long int v_label){
   int j,k;
   for (j = 0;j < lb_count; j++){
      if ( do_table_label[j] == v_label) {
         id_curr -= (n_do * do_table_count [j]);
         if (j +1 != lb_count) { // last one?
            for (k = j+1; k < lb_count; k++){
               // move all labels up by one
               do_table_label [k-1] = do_table_label [k];
               do_table_count [k-1] = do_table_count [k];
            }
         }
         lb_count --;
         return;
      }
   }
   return;
}
/* ********************************************************************
  mark_label - marks in type line "L" where label is found.
 ******************************************************************* */
static void mark_label (int strt_pos, int num_digits){
   int i;
   assert (num_digits > 0 and num_digits <= 5);
   for (i=strt_pos; i < (strt_pos + num_digits); i++){
      if (comp_typ [i] != 0){
         assert (comp_typ [i] == 0);
         printf( "Help\n");
         exit (1);
      }
      assert (comp_typ[i] == 0) ;
      comp_typ [i] = 'L';
   }
   return;
}
/* **********************************************************************
  replace_txt - Updates current compressed code text string.
       parameters are:-
       (1) start position of original string in code text.
       (2) original number of characters.
       (3) new number of characters.
       (4) address of string to insert.
 ********************************************************************** */
static void replace_txt (int strt, int origin_num,int new_num,char * new_text){
   int diff =  new_num - origin_num;
   int end_orig, end_newv,i, k, pos_diff;
   int case_typ = 0; // normal case
   int eol_pos; // end of line position.
   int beg_pos = comp_pos[strt]; // start position of original field
   assert (n_chas + diff <= MAX_TXT and n_chas + diff > 0);
   if (diff != 0) {
      change_flg = 1;
      end_orig = strt + origin_num - 1;
      end_newv = strt + new_num - 1;
      // does field straddle a line join?
      if (num_code_lines > 1 and diff < 0){
         int spl_bef = FALSE, spl_aft = FALSE;
         int s_pos = 0, e_pos1=0, e_pos2 = 0;
         // find lines where field starts and ends.
         for (int yy = 0;yy < num_code_lines; yy ++){
            int s_lin = in_lin[yy].txt_posn;
            if (strt >= s_lin) s_pos = yy;
            if (end_orig  >= s_lin) e_pos1 = yy; // before
            if (end_newv  >= s_lin) e_pos2 = yy; // after
         }
         if (s_pos != e_pos1 ) spl_bef = TRUE; // split before
         if (s_pos != e_pos2 ) spl_aft = TRUE; // split after
         if (spl_bef) {
            case_typ = 1;
            if (!spl_aft) {
               case_typ = 2;
            }
         }
      }
      pos_diff = comp_pos [end_orig+1] - comp_pos [end_orig]; // between fields
      int mov_num = n_chas - end_orig;
      memmove(&comp_txt[end_newv+1],
         &comp_txt[end_orig+1],mov_num); // upper case main code
      if (shad_txt != NULL) {
         memmove(&shad_txt[end_newv+1],
            &shad_txt[end_orig+1],mov_num); // original code
      }
      memmove((comp_pos+end_newv+1), (comp_pos+end_orig+1), mov_num); // Position
      memmove((comp_typ+end_newv+1),(comp_typ+end_orig+1), mov_num);  // Type
      if (diff > 0) { // update gaps
         char t_val = *(comp_typ+strt);
         // assume same type (change in caller if different)
         memset ((comp_typ+end_orig+1), t_val, diff); // type
         // increment position.
         int old_pos = *(comp_pos + end_orig);
         int x = 0;
         for (k=end_orig+1;x < diff;x++,k++){
            *(comp_pos+k) = ++old_pos;
         }
      }
   }
   // Exact same number of characters replaced.
   // Insert new field of text (assume it is a replacement field.)
   memcpy (&comp_txt [strt], new_text, new_num); // main compressed text.
   if (shad_txt != NULL) { // update shadow (if present.)
      memcpy((shad_txt+strt), new_text, new_num);
   }
   if (diff != 0){
      int case_typ_2 = case_typ;
      n_chas += diff;
      eol_pos = n_chas;
      // multiline positions
      if (num_code_lines > 1) {
         for (i = 1;i < num_code_lines; i++){
            if (case_typ != 2){ //normal case
               if (in_lin [i].txt_posn >
                  strt+origin_num -1) {
                  in_lin [i].txt_posn += diff;
               }
            }
            else if (case_typ == 2){
               if (in_lin [i].txt_posn >=
                  strt+origin_num -1) {
                  in_lin [i].txt_posn = strt+new_num;
                  case_typ = 0;
               }
            }

         }
      }
      case_typ = case_typ_2;
      // keep same distance between original and next field.
      if (num_code_lines > 1){
         for (k = num_code_lines - 2; k >= 0; k--){
            if (end_newv >= in_lin [k+1].txt_posn){
               break;
            }
            else {
               eol_pos = in_lin [k+1].txt_posn;
            }
         }
      }
      k= strt + new_num;
      int act_diff;
      if (k != 0) { //  previous position.
         act_diff = (*(comp_pos + k - 1) + pos_diff) - *(comp_pos + k);
      }
      else { // no previous position
         act_diff =  beg_pos -  comp_pos[strt];
      }
      for (;k < eol_pos; k++){
         *(comp_pos + k) += act_diff;
      }
      // check if number of lines, still valid.
         for (i = num_code_lines-1; i >= 0;i --){
         if (in_lin[i].txt_posn > n_chas -1) num_code_lines--;
      }
   }
   return;
}
/* ********************************************************************
  convert_label - parameter is original label number
  returned is converted label.
  Type = 0 for normal label
  Type = 1 when label is for a DELETE (which can span subroutines)
  Type = 2 when defined label (i.e. in columns 1 to 5).
 ******************************************************************* */
static long int convert_label (long int orig_lab, int type){
   assert (t_pnt != NULL);
   assert (type < 3);
   assert (orig_lab != 0);
   int k;
   long int cur_lbl, new_lab;
   for (k = lbl_strt_pos + 1; k< numb_lab; k++){
      cur_lbl =  (t_pnt+k)->label_num;
      if (orig_lab == cur_lbl) { // found
         if (u_opt) { // remove unused labels option
            if ((t_pnt+k)->num_uses == 0){
               return 0;
            }
         }
         // when only "u" option used.
         if (!r_opt) return orig_lab;
         new_lab = (t_pnt+k)->new_num;
         if (new_lab != 0) return new_lab;
         if (type == 0) {
            // case when label is in an "INCLUDE"
            return orig_lab;
         }
         else if (type == 1){ // DELETE label
            // search in further on subroutines.
            int srch = k + 1;
            int new_sub = 0;
            for (;srch <  numb_lab; srch++){
               if (new_sub == 0) { // test for start of new subroutine.
                  if ((t_pnt+srch)->new_num == 0){
                     new_sub = 1;
                  }
                  continue;
               }
               if (orig_lab == (t_pnt+srch)->label_num and
                  (t_pnt+srch)->def_line != 0) {
                  return (t_pnt+srch)->new_num;
               }
            }
         }
         printf ("System error- no alternative label number\n");
         exit (1);
      }
      if (cur_lbl == 0) { // end of this subroutine's labels
         break;
      }
   }
   // Label not found, but should have been in table from 1st. pass.
   printf("System error- Label %li not found on line %li\n",
      orig_lab, line_number);
   if (tester) label_print();
   exit (1);
   return 0;
}
/* ********************************************************************
 update_txt - redoes position & type for 3 label changes
 ******************************************************************* */
static void update_txt(int s_pos){
   int i;
   char chs;
   for (i = s_pos;i < n_chas;i++){
      if (i != 0) comp_pos [i] = 0;
      chs = comp_txt [i];
      if (isdigit(chs)) {
         comp_typ [i] = 'L'; // Label
      }
      else if (isalpha(chs)) {
         comp_typ [i] = 'K'; // keyword
      }
      else {
         comp_typ [i] = chs;
      }
   }
}
/* ********************************************************************
  send_out - sends text in comp_txt to output stream.
  Has one parameter:- 0 = stop after basic text is output.
                      1 = continue with any following comment lines.
********************************************************************** */
static void send_out (int type_out){
   int t_flags = 0;
   int full_flag = 0; // full reformatting needed
   char  tmp [91];
   static const char * two_codes [] = {".GT.",".GE.",".LT.",
         ".LE.",".EQ.",".NE.",".OR."};
      char  lbl [6];
   int i, z, no_sp;
   long int new_label = 0;
   // real processing here.
   if (dl_opt and pass_num == 2) change_flg = 1; // spot the CALLs
   memset(tmp,' ',sizeof(tmp));
   comp_pos[n_chas] = 0;
   tmp [sizeof(tmp)-1] = 0; // terminator
   if (curr_lab != 0) { // label in cols 1-5
      if (u_opt or r_opt) {
         new_label = convert_label(curr_lab,2);
      }
      else new_label = curr_lab;
   }
   if (new_label != 0) {
      if (rj_opt) {
         sprintf(lbl,"%5ld",new_label);
      }
      else {
         sprintf(lbl,"%-5ld",new_label);
      }
      if (rj_opt == 0 and lj_opt == 0 and r_opt == 0){
         // try and put in original position (if room)
         for (z = 0;z < 5;z ++){
            if (lbl[z] == ' ')break;
         }
         char chr;
         if (5 - lab_strt_pnt >= z) {
            int j = 0;
            int g=lab_strt_pnt;
            for (;j < 5; j++){
               chr = lbl[j];
               if (chr == ' ') break;
               tmp[g++] = chr;
            }
            goto label_complete;
         }
      }
      memcpy (tmp,lbl,5);
label_complete:;
   }
   curr_lab = 0; // only use once.
   strt_indent = id_curr;
   if (i_opt){
      if (id_curr < n_strt) strt_indent = n_strt;
      if (id_curr > MAX_INDENT) strt_indent = MAX_INDENT;
   }
   if (i_opt and ! f_opt) { // check indentation is correct.
      assert (strt_indent >= 6 and strt_indent < 72);
      // check for character strings that span lines.
      if (num_code_lines > 1) {
         for (int ff = 1;ff < num_code_lines; ff++){
            int stp = in_lin[ff].txt_posn;
            if (!(stp != 0)){
               printf("Error stp !=0, line %li\n",c_line_num);
               printf("Txt=%s\n",comp_txt);
               exit (1);
            }
            assert (stp != 0);
            if (comp_typ[stp] == comp_typ[stp-1]){
               if (comp_typ[stp] == 'Q' or comp_typ[stp] == 'H'){
                  full_flag = 1;
                  break;
               }
            }
         }
      }
      int diff = comp_pos[0] - strt_indent;
      if (diff != 0) {
         change_flg = 1;
         for(int jy = 0; jy < num_code_lines; jy++){
            int e_pnt = n_chas;
            int s_pnt = in_lin[jy].txt_posn;
            if (jy+1 != num_code_lines) {
               e_pnt = in_lin[jy+1].txt_posn;
            }
            int act_diff = diff;
            if (diff > 0 ) { // moving text <-
               if (comp_pos[s_pnt] - diff < 6) {
                  act_diff = comp_pos[s_pnt] - 6;
               }
            }
            else { // moving text ->
               if (comp_pos [s_pnt] - diff > MAX_INDENT + 3) {
                  act_diff = comp_pos[s_pnt] - (MAX_INDENT+3);
               }
            }
            for (int ff = s_pnt;ff < e_pnt; ff++){
               comp_pos[ff] -= act_diff;
            }
         }
      }
   }
   // if no change in line forget following code.
   if (change_flg != 0 or f_opt or full_flag) {
      t_flags |= 8;
      // Quick check
      int need_tests = 0;
      if (!f_opt and !dl_opt) {
         for (int zzz = 0;zzz < num_code_lines; zzz++){
            int last_pnt = n_chas - 1;
            if (zzz+1 < num_code_lines)
               last_pnt = in_lin[zzz+1].txt_posn - 1;
            int lin_pos = comp_pos [last_pnt];
            if (lin_pos > 71) {
               need_tests = 1;
               break;
            }
            if (in_lin[zzz].strt_pos != 0){
               if (lin_pos >= in_lin[zzz].strt_pos ){
                  need_tests = 1;
                  break;
               }
            }
         }
         if (need_tests == 0) goto skip_test;
      }
      // check code for type change.
      comp_txt [n_chas] = 0;
      for (i = 0;i < n_chas; i++){
         if (comp_txt[i] == '.'){
            // can also be start of a number
            if (isdigit(comp_txt [i+1])) continue;
            if (n_chas - i >= 6 and comp_txt [i+6] == '.'){
               if (strncmp(&comp_txt[i],".FALSE.",7) == 0){
                  memset(&comp_typ[i],'G',7);
                  i += 7;
                  continue;
               }
            }
            if (n_chas -i >= 5 and comp_txt[i+5] == '.') {
               if (strncmp(&comp_txt[i],".TRUE.",6) == 0 or
                  strncmp(&comp_txt[i],".NEQV.",6) == 0){
                  memset(&comp_typ[i],'G',6);
                  i += 6;
                  continue;
               }
            }
            if (n_chas -i >= 4 and comp_txt[i+4] == '.') {
               if (strncmp(&comp_txt[i],".NOT.",5) == 0 or
                  strncmp(&comp_txt[i],".AND.",5) == 0 or
                  strncmp(&comp_txt[i],".EQV.",5) == 0){
                  memset(&comp_typ[i],'G',5);
                  i += 5;
                  continue;
               }
            }
            if (n_chas -i >= 3  and comp_txt[i+3] == '.') {
               for (int j=0;
               j < sizeof(two_codes)/sizeof(two_codes[0]);j++){
                  if (strncmp(&comp_txt[i],two_codes[j],4) == 0){
                     memset(&comp_typ[i],'G',4);
                     i += 4;
                     break;
                  }
               }
               if (comp_typ[i-1] == 'G') continue;
            }
         }
      }
      // find GOTO'S.
      if (comp_txt[0] == 'G' and comp_typ[0] == 0
         and n_chas > 4){
         if (strncmp(comp_txt,"GOTO",4) == 0) {
            if (isalpha(comp_txt[4])){
               // This can be an assigned GOTO. Should be name of
               // variable followed by brackets containing labels
               int mode = 0, cnt = 1;
               int ix = 5;
               char chs,typ;
               for (;ix < n_chas; ix ++){
                  chs = comp_txt [ix];
                  typ = comp_typ [ix];
                  if (mode == 0) { // variable name
                     if (isalnum(chs) or chs == '$'){
                        cnt ++;
                        if (cnt > 6) goto no_goto;
                        continue;
                     }
                     else if (chs == '(' and typ == '('){
                        mode = 1;
                        continue;
                     }
                     else if(chs == ','){ // optional after a variable
                        continue;
                     }
                     goto no_goto;
                  }
                  else if (mode == 1){ // labels, commas
                     if (isdigit(chs) or
                        chs == ',' or chs == ')'){
                        if (chs == ')'){
                           if (ix+1 == n_chas) memset(comp_typ,'K',4);
                           goto no_goto;
                        }
                        continue;
                     }
                     else goto no_goto;
                  }
                  goto no_goto;
               }
            }
            else if (n_chas < 10){
               // check all numeric until end of line
               int jj = 4;
               for (;jj < n_chas; jj ++){
                  if (isdigit(comp_txt[jj])) continue;
                  goto no_goto;
               }
               memset (comp_typ,'K',4);
            }
         }
no_goto:;
      }
      // now look for GOTO at end of IF
      if (comp_txt[0] == 'I' and comp_txt[1] == 'F' and
         comp_txt[2] == '('){
         if(isdigit(comp_txt[n_chas-1]) and
            comp_typ[n_chas-1] == 0){
            int bwd = n_chas - 1;
            int num_f = 0;
            for (;bwd>=3; bwd--){
               if (comp_typ[bwd] != 0) break;
               if (isdigit(comp_txt[bwd])){
                  num_f++;
                  if (num_f > 5) break;
                  continue;
               }
               // not a digit - check for GOTO
               if (bwd < 7) break;
               if (strncmp(&comp_txt[bwd-4],")GOTO",5) == 0){
                  memset(&comp_typ[bwd-3],'K',4);
               }
               break;
            }
         }
         else if (comp_txt[n_chas-1] == ')') {
            // might be assigned goto at end of if
            int ty = 3;
            int cnt = 1;
            char chs,typ;
            for (;ty<n_chas;ty++){
               chs = comp_txt[ty];
               typ = comp_typ[ty];
               if (chs == '(' and typ == '(')cnt++;
               if (chs == ')' and typ == ')'){
                  cnt--;
                  if (cnt == 0) {
                     if(strncmp(&comp_txt[ty+1],"GOTO",4) == 0
                        and isalpha(comp_txt[ty+5]) ) {
                        int np = ty+6;
                        int mode = 0;
                        int cht = 1;
                        for (;np < n_chas; np ++){
                           chs = comp_txt [np];
                           if (mode == 0){
                              if (isalnum(chs) or chs == '$'){
                                 cht++;
                                 if (cht > 6) goto not_valid_gt;
                                 continue;
                              }
                              else if (chs == '('){
                                 mode = 1;
                                 continue;
                              }
                              else if (chs == ','){
                                 continue;
                              }
                           }
                           else if (mode == 1){
                              //looking for list of labels
                              if (chs == ',' or isdigit(chs) or
                                 chs == ')'){
                                 if (chs == ')' and
                                    np+1 == n_chas){ // end found
                                    memset(&comp_typ[ty+1],'K',4);
                                    break;
                                 }
                              }
                              else {
                                 goto not_valid_gt;
                              }
                           }
                        }
                        break;
                     }
                     else {
                        break;
                     }
                  }
               }
            }
         }
      }
not_valid_gt:
      // Pick up CALLs
         if (comp_txt[0] == 'C' and
            (comp_typ[0] == 0 or comp_typ[0] == 'K')) {
            // at beginning of line
            if (strncmp(comp_txt,"CALL",4) == 0 and
               isalpha(comp_txt[4])){
               // look for naked "="
               int lp = 4;
               int eqls = 0;
               for (;lp<n_chas;lp++){
                  if (comp_txt[lp] == '=' and
                     comp_typ[lp] == '=') {
                     eqls = 1;
                     break;
                  }
               }
               if (eqls == 0){
                  call_flg = 4;
                  memset(comp_typ,'K',4);
               }
            }
         }
      // now look for end of IF CALL
      if (comp_txt[0] == 'I' and comp_txt[1] == 'F' and
         comp_txt[2] == '(') {
         int lp = 3;
         int bra_cnt = 1;
         char chas;
         for (;lp < n_chas;lp++){
            chas = comp_txt[lp];
            if (chas == '(' and comp_typ[lp] == '('){
               bra_cnt++;
               continue;
            }
            else if(chas == ')' and comp_typ[lp] == ')'){
               bra_cnt--;
               if (bra_cnt == 0) { // at the end
                  if (comp_txt[lp+1] != 'C') break;
                  if (strncmp(&comp_txt[lp+1],"CALL",4)==0 and
                     // start of subroutine name (alpha)
                     isalpha(comp_txt[lp+5])) {
                     int fn = lp+6;
                     int eql = 0;
                     // check no "=" on line.
                     for(;fn<n_chas;fn++){
                        if(comp_txt[fn] == '='
                           and comp_typ[fn] == '='){
                           eql = 1;
                           break;
                        }
                     }
                     if (eql == 0) {
                        call_flg = lp+4;
                        memset(&comp_typ[lp+1],'K',4);
                     }
                  }
                  break;
               }
            }
         }
      }
      // DELETE
      if (comp_txt[0] == 'D' and isdigit(comp_txt[6])) {
         if (strncmp(comp_txt,"DELETE",6) == 0){
            memset(comp_typ,'K',6);
         }
      }
      // DATA
      if (comp_txt[0] == 'D' and comp_txt[n_chas - 1] == '/'
         and comp_typ[0] == 0){
         if (strncmp(comp_txt,"DATA",4) == 0 and
            isalpha(comp_txt[4])){ // check for 1-6 character variable
            int ip = 5,cnt = 1;
            char chs;
            for (;ip < n_chas; ip ++){
               chs = comp_txt [ip];
               if (isalnum(chs) or chs == '$'){
                  cnt ++;
                  if (cnt > 6) goto not_data;
                  continue;
               }
               else if (chs == '/' or chs == '(' or chs == ',') {
                  memset(comp_typ,'K',4);
                  break;
               }
               goto not_data;
            }
         }
      }
not_data:
      // A tricky case "DO" identification
      if (comp_txt[0] == 'D' and comp_txt [1] == 'O' and
         isdigit(comp_txt[2]) ){
         // format is DO nnn[,] vvvv = xxx,yyy [,zzz]
         int pt = 3, stg = 0;
         char chs,typ;
         for (;pt<n_chas;pt++){
            chs = comp_txt[pt];
            typ = comp_typ[pt];
            if (typ == 'Q' or typ == 'H') continue;
            switch (stg){
            case 0: // looking for start of variable
               if (isdigit(chs) or chs == ',') continue;
               if (isalpha(chs)){
                  stg = 1; // variable.
                  break;
               }
               goto not_do;
            case 1: // looking for "="
               if (chs == '=') {
                  stg = 2; // first expersion.
                  break;
               }
               break;
            case 2: // first expression
               if (isdigit(chs) or isalpha(chs)) {
                  stg = 3; // looking for first ','
                  break;
               }
               break;
            case 3: // looking for ','
               if (chs == ',') {
                  stg = 4;
                  break;
               }
               break;
            case 4: // second expression
               if (isdigit(chs) or isalpha(chs)) {
                  // it a do loop!!!
                  comp_typ[0] = 'K';
                  comp_typ[1] = 'K';
                  goto not_do;
               }
               break;
            }
         }
not_do:;
      }
      // now that logical values found look for numbers & variable names.
      for (i = 0; i < n_chas; i++){
         int txt = comp_txt [i];
         int typ = comp_typ [i];
         if ((typ == 0 and isdigit(txt)) or
            (typ == '.' and isdigit(comp_txt[i+1]))) {
            // start of a number
            comp_typ [i] = '9';
            int point_flg = 0, exp_flg = 0;
            if (typ == '.') point_flg = 1;
            for (int nc = i+1;
            nc < n_chas;
            comp_typ[nc++] = '9'){
               int txt2 = comp_txt [nc];
               int typ2 = comp_typ [nc];
               if (isdigit(txt2)) continue;
               if (typ2 == '.') {
                  if (point_flg or exp_flg){
roll_back:
                     i = nc -1;
                     goto loop_end;
                  }
                  point_flg = 1;
                  continue;
               }
               if (txt2 == 'E' or txt2 == 'D'){
                  // n.b. "D" is Unisys special to indicate double
                  // precision.
                  if (exp_flg) goto roll_back;
                  exp_flg = 1;
                  continue;
               }
               if (txt2 == '+' or txt2 == '-'){
                  // can only follow exponential
                  if (! exp_flg) goto roll_back;
                  txt2 = comp_txt [nc-1];
                  if (txt2 == 'E' or txt2 == 'D') continue;
                  goto roll_back;
               }
               goto roll_back;
            }
         }
         else if (typ == 0 and isalpha(txt)){
            // start of a variable name or subroutine name
            i = var_name(i);
         }
loop_end:;
      }
      if (char_flg) { // character declaration, put the stars together.
         int xy = 0,zz;
         char typx, typ2;
         for (;xy < n_chas; xy++){
            if (comp_typ [xy] == '*'){
               typx = comp_typ [xy - 1];
               comp_typ [xy] = typx;
               typ2 = comp_typ [xy+1];
               for (zz = xy + 1; zz < n_chas; zz++){
                  if (comp_typ[zz] == typ2){
                     comp_typ [zz] = typx;
                     continue;
                  }
                  xy = zz - 1;
                  break;
               }
            }
         }
      }
      if (call_flg) { // correct return labels
         int sp = call_flg;
         for (;sp < n_chas; sp++){
            if (comp_typ [sp] == '*' or
               (!(comp_typ [sp] == 'Q' or comp_typ [sp] == 'H')
               and (comp_txt [sp] == '$' or comp_txt [sp] == '&'))) {
               if (isdigit(comp_txt[sp+1])  and
                  (comp_typ[sp-1] == ',' or
                  comp_typ[sp-1] == '(' )){
                  comp_typ [sp] = comp_typ [sp+1];
                  if(dl_opt) { // convert "$" & "&" to "*"
                     if (comp_txt[sp] == '$' or
                        comp_txt[sp] == '&') comp_txt[sp] = '*';
                     if (shad_txt != NULL) shad_txt[sp] = '*';
                  }
               }
            }
         }
      }
      if (f_opt or full_flag) {
         full_format ();
      }
      else { // case when indenting etc. has caused overflow.
         t_flags |= 1;
         // try to fit all onto a line, by taking out spaces.
            // first check for split-fields.
            if (num_code_lines > 1) {
               for (int jk = 1;jk < num_code_lines; jk ++){
                  int beg_p = in_lin[jk].txt_posn;
                  if (comp_typ[beg_p] == comp_typ[beg_p-1]){
                     full_format(); // over two lines.
                     goto skip_test;
                  }
               }
            }
         // keep original positions
         char * old_pos = (char *)malloc(n_chas+1);
         assert (old_pos != NULL);
         memcpy(old_pos,comp_pos,n_chas);
         old_pos[n_chas] = 0;
         int done = TRUE;
         for (int aa = 0; aa < num_code_lines; aa++){
            int e_p = n_chas;
            int s_p = in_lin[aa]. txt_posn;
            if (aa+1 < num_code_lines){ // not last line?
               e_p =  in_lin[aa+1]. txt_posn;
            }
            if (!one_liner (s_p, e_p)){
               // restore original positions
               memcpy(comp_pos,old_pos,n_chas);
               done = FALSE;
               break;
            }
         }
         free (old_pos);
         if (done == TRUE) goto skip_test;
         // squeeze of lines failed
         // move excess text onto next line.
         if (num_code_lines == 1){
            // find overflow field.
            int o_pos = 0;
            for (int vv = n_chas-1; vv >= 0;vv--){
               int ppos = comp_pos[vv];
               if (ppos <= 71) {
                  o_pos = vv + 1;
                  break;
               }
            }
            if (o_pos == 0) {
               full_format();
               t_flags |= 4;
               goto skip_test;
            }
            // find start of field
            char f_typ = comp_typ[o_pos];
            int st_pos = o_pos;
            for (int xx = o_pos; xx >= 0; xx--){
               if (comp_typ[xx] == f_typ){
                  st_pos = xx;
               }
               else {
                  break;
               }
            }
            // find size of field.
               int fd_size = comp_pos[n_chas-1] - comp_pos[st_pos] + 1;
            int room_avail = 72 - comp_pos[0];
            if (fd_size > room_avail) {
               full_format();
               t_flags |= 2;
               goto skip_test;
            }
            int new_strt = comp_pos[0];
            // room for a slight indent?
            if (comp_pos[0] < MAX_INDENT -2){
               if (fd_size+MAX_XESS <= room_avail){
                  new_strt += MAX_XESS;
               }
            }
            int n_diff = comp_pos[st_pos] - new_strt;
            for (int kk = st_pos; kk < n_chas; kk++){
               comp_pos [kk] -= n_diff;
            }
            // excess onto new line.
            num_code_lines = 2;
            in_lin[1].txt_posn = st_pos;
         }
         else {
            full_format();
         }
      }
   }
skip_test:
   ;
   for (int ln = 0; ln < num_code_lines; ln ++){
      if (ln != 0) { // initialise line
         memset (tmp,' ',sizeof(tmp));
         tmp [5] = '-'; // continuation character
         if (!f_opt and cont_char[ln-1] != 0)
            // original continuation character.
            tmp [5] = cont_char [ln-1];
         tmp [sizeof(tmp)-1] = 0; // terminator
      }
      int str_ch = in_lin[ln].txt_posn;
      int end_ch = n_chas;
      if (ln+1 < num_code_lines) {
         end_ch = in_lin[ln+1].txt_posn;
      }
      int pos,old_pos = 0;
      for (int lp = str_ch;lp < end_ch; lp++){
         pos = comp_pos[lp];
         //   assert (pos > old_pos);
         if (!(pos > old_pos)){
            printf( "Help line = %li\n",c_line_num);
            printf ("Txt = %s\n",comp_txt);
            if (tester) line_table();
            exit (1);
         }
         if (pos > 71) {
            printf("Overflow of program line at %li\n",c_line_num);
            printf("Positions=");
            for (int bb = 0; bb < n_chas; bb++){
               int jj = comp_pos[bb];
               printf ("%2i,",jj);
            }
            printf("\n");
            printf ("Flags = %i\n",t_flags);
            printf("Text = %s\n",comp_txt);
            exit (1);
         }
         // original or Capitols?
         char c_val;
         if (shad_txt != NULL){  //original code
            c_val = shad_txt [lp];
         }
         else {  // upper case
            c_val = comp_txt [lp];
         }
         tmp [pos] = c_val;
         old_pos = pos;
      }
      // test for any in-line comment.
         int str_p = in_lin [ln].strt_pos;
      int num_c = in_lin [ln].numb_chrs;
      int buf_p = in_lin [ln].buff_pnt;
      if (str_p != 0) {
         assert (in_buff != NULL);
         int spc = str_p - pos;
         if (spc <= 0) { // needs to be on a seperate line.
            // check for trailing or leading blanks to make it fit.
            int back_p = 72 - (str_p + num_c);
            int trail = 0;
            for (int tt = buf_p + num_c - 1;
            tt >= buf_p;
            tt --){
               char csp = *(in_buff + tt);
               if (csp == ' '){
                  trail ++;
               }
               else {
                  break;
               }
            }
            // enough space now?
               int new_s = spc + trail + back_p;
            int spp = pos+1;
            if (new_s > 0) {
               if (new_s > 1) spp++;
               memcpy (&tmp[spp],
                  &in_buff[buf_p],
                  num_c - trail);
               goto add_cols;
            }
            // check for any leading spaces (except first one).
            int lead_sp = 0;
            for (int sz = buf_p + 1; sz < buf_p+num_c; sz++){
               char lsp = *(in_buff + sz);
               if (lsp == ' '){
                  lead_sp++;
               }
               else {
                  break;
               }
            }
            // allow one leading space.
               if (lead_sp > 0) lead_sp --;
            new_s = spc + trail + back_p + lead_sp;
            spp = pos+1;
            if (new_s > 0) { // enough room
               // room for separator space?
               if (new_s > 1) spp++;
               tmp[spp] = '@';
               memcpy (&tmp[spp+2],
                  &in_buff[buf_p + 2 + lead_sp],
                  num_c - (trail + 2 +lead_sp));
               goto add_cols;
            }
            else {// output in-line comment on a separate line.
               out_in_line (ln);
            }
         }
         else {
            memcpy(&tmp[str_p], &in_buff[buf_p], num_c);
            assert (str_p + num_c <= 72);
         }
      }
add_cols:;
      // add in columns 73 to 80
         memcpy (&tmp[72], id_txt[ln], ID_COLS+1);
      // remove trailing spaces.
      for (no_sp = sizeof(tmp)-1;no_sp >= 0; no_sp--){
         if (tmp[no_sp] == ' ' or tmp[no_sp] == 0) continue;
         break;
      }
      tmp [no_sp+1] = 0;
      out_line (tmp);
      if (ln+1 < num_code_lines){
         int inter_line = in_lin [ln+1].rel_line - in_lin [ln].rel_line;
         if (inter_line > 1) { //output some comment lines.
            com_out (inter_line - 1);
         }
      }
      else { // last line
         if (type_out != 0) com_out (-1);
      }
   }
   return;
}
/* *********************************************************************
    out_in_line (line number) - removes in-line comment, sends it out
    as a comment. Called when not enough room on line (for indentation)
 ******************************************************************** */
static void out_in_line (signed int l_num){
   char char_line [73];
   int num_chs = in_lin [l_num].numb_chrs;
   int strt_ps = in_lin [l_num].strt_pos;
   int buff_ps = in_lin [l_num].buff_pnt;
   assert (num_chs != 0);
   assert (strt_ps + num_chs < sizeof(char_line));
   memset (char_line, ' ', sizeof(char_line));
   char_line [sizeof(char_line) - 1] = 0;
   char_line [0] = 'C';
   memcpy (&char_line [strt_ps], &in_buff [buff_ps],num_chs);
   out_line (char_line);
   in_lin [l_num - 1].numb_chrs = 0;
   return;
}
/* *********************************************************************
   com_out - outputs a number of comment lines. If parameter = -1 then
   flush the buffer.
   ****************************************************************** */
static void com_out (signed int num_line){
   int counter = num_line;
   // All lines case.
   if (num_line == -1) counter = cb_lines;
   while (counter != 0) {
      out_line(act_cb[0].cb_text);
      int num_move = MAX_COM - 1;
      if (cb_lines < MAX_COM) num_move = cb_lines - 1;
      memmove (&act_cb[0],&act_cb[1], num_move * sizeof(act_cb[0]));
      cb_lines --;
      counter --;
      if (cb_lines_hd != 0){
         if (fread (&act_cb[MAX_COM-1],
            sizeof (cb_line), 1, com_ovf_f) != 1) {
            printf("Error in read of Comments file\n");
            exit (1);
         }
         cb_lines_hd--;
      }
   }
   return;
}
/* ********************************************************************
   bare_eq - checks line for a stand-alone "=".
 ******************************************************************* */
static int  bare_eq(void){
   int i;
   for (i=0;i<n_chas;i++){
      if (comp_txt[i] == '=' and
         comp_typ [i] == '='){
         return 1;
      }
   }
   return 0;
}
/* ********************************************************************
 var_name - Checks variable names
            Input is start of alpha character defining name.
            Output is start position for field checking.
 ******************************************************************** */
static int  var_name(int i){
   // 7 Character.....
   static const char  code_7 [][8] =
   {"COMPLEX", "DISPLAY", "ENDFILE", "INCLUDE", "INTEGER",
      "LOGICAL", "PROGRAM", "VIRTUAL"};
   enum
   {e_COMPLEX, e_DISPLAY, e_ENDFILE, e_INCLUDE, e_INTEGER,
      e_LOGICAL, e_PROGRAM, e_VIRTUAL};
   const int code_7_sz = sizeof(code_7)/sizeof(code_7[0]);
   // 8 Character.....
   static const char  code_8 [][9] =
   {"EXTERNAL", "FUNCTION", "IMPLICIT",
      "NAMELIST", "STOPEDIT"};
   enum
   {e_EXTERNAL, e_FUNCTION, e_IMPLICIT,
      e_NAMELIST, e_STOPEDIT};
   const int code_8_sz = sizeof(code_8)/sizeof(code_8[0]);
   // 9 Character.....
   static const char  code_9 [][10] =
   {"BLOCKDATA", "CHARACTER", "DIMENSION","INTRINSIC",
      "PARAMETER", "STARTEDIT"};
   enum
   {e_BLOCKDATA, e_CHARACTER, e_DIMENSION, e_INTRINSIC,
      e_PARAMETER, e_STARTEDIT};
   const int code_9_sz = sizeof(code_9)/sizeof(code_9[0]);
   // .............
   int vn = i+1;
   int oct_flg = 0; // octal number flag.
   int num_alfa = 1;
   int num_chas = 1;
   comp_typ [i] = 'V';
   for (; vn < n_chas; comp_typ[vn++]='V'){
      int txt3 = comp_txt [vn];
      if (comp_typ[vn] != 0) return vn;
      if (! (isalnum(txt3) or txt3 == '$')) return vn;
      num_chas ++;
      if (num_chas > 6 and isdigit(txt3)){
         if (oct_flg) continue;
         // check for octal number.
         if (num_chas == 7 and num_alfa == 1){
            if (comp_txt [vn-6] == 'O' and
               (comp_txt  [vn-7] == '/' or
               comp_txt  [vn-7] == ','  or
               comp_txt  [vn-7] == '*')) {
               oct_flg = 1;
               continue;
            }
         }
         return vn - 1;
      }
      if (isalpha(txt3)) num_alfa++;
      if (num_chas == num_alfa){
         if (i == 0){ // must start in first column.
            switch (num_chas){
            case 2:
               {
                  if (comp_txt[0] == 'A' and
                     comp_txt[1] == 'T' and
                     isdigit (comp_txt[2]) ){
                     if (at_label(2)){
                        comp_typ[0] =
                        comp_typ[1] = 'K';
                        return 1;
                     }
                  }
                  break;
               }
            case 4:
               {
                  if (comp_txt[0] == 'R' or comp_txt[0] == 'S'){
                     if(bare_eq()) continue;
                     if(strncmp(comp_txt,"REAL",4) == 0 or
                        strncmp(comp_txt,"SAVE",4) == 0) {
                        memset (comp_typ,'K',4);
                        if (comp_txt[0] == 'R' and
                           comp_txt[4] == 'F' ){
                           // check for "REAL FUNCTION"
                           if(find_f(4))return 3+8;
                        }
                        return 3;
                     }
                  }
                  break;
               }
            case 5:
               {
                  if((comp_txt[0] == 'E' and comp_txt[4] == 'Y') or
                     (comp_txt[0] == 'D' and comp_txt[4] == 'G')) {
                     if(strncmp(comp_txt,"ENTRY",5) == 0 or
                        strncmp(comp_txt,"DEBUG",5) == 0){
                        if(bare_eq()) continue;
                        memset (comp_typ,'K',5);
                        return 4;
                     }
                  }
                  break;
               }
            case 6:
               {
                  if (comp_txt[0] == 'D' and
                     comp_txt[5] == 'E'){
                     if(strncmp(comp_txt,"DEFINE",6) == 0){
                        memset(comp_typ,'K',6);
                        if (strncmp(&comp_txt[6],"FILE",4) == 0){
                           memset(&comp_typ[6],'k',4);
                           return 9;
                        }
                        else {
                           return 5;
                        }
                     }
                  }
                  if(comp_txt[0] == 'C' and
                     comp_txt[5] == 'N') {
                     if (strncmp(comp_txt,"COMMON",6) == 0 and
                        (isalpha(comp_txt[6]) or
                        comp_txt[6] == '/')){
                        memset(comp_typ,'K',6);
                        return 5;
                     }
                  }
                  break;
               }
            case 7:
               {
                  int bb = 0;
                  for(;bb < code_7_sz; bb ++){
                     if(strncmp(comp_txt, code_7[bb], 7) == 0){
                        memset(comp_typ,'K',7);
                        // Is it "ENDFILE" ?
                        if (bb == e_ENDFILE) {
                           memset (&comp_typ[3],'k',4);
                        }
                        // special case of "INCLUDE"
                        if (bb == e_INCLUDE){
                           // make all until end (or comma) into a field
                           int ee = 7;
                           for (;ee < n_chas; comp_typ[ee++] = 'V'){
                              if (comp_txt[ee] == ','){
                                 return ee;
                              }
                           }
                           return n_chas;
                        }
                        // for INTEGER, LOGICAL & COMPLEX
                        // check for FUNCTION.
                        if(bb == e_INTEGER or
                           bb == e_LOGICAL or
                           bb == e_COMPLEX ){
                           if (comp_txt[7] == 'F'){
                              if(find_f(7))return 6+8;
                           }
                        }
                        return 6;
                     }
                  }
                  break;
               }
            case 8:
               {
                  int bb = 0;
                  for(;bb<code_8_sz;bb++){
                     if(strncmp(comp_txt, code_8[bb], 8) == 0){
                        memset(comp_typ,'K',8);
                        // was it "STOPEDIT" ?
                        if (bb == e_STOPEDIT){
                           memset (&comp_typ[4],'k',4);
                        }
                        else if (bb == e_IMPLICIT){
                           if (comp_txt[8] == 'C' and
                              memcmp(&comp_txt[8], "CHARACTER",9) == 0){
                              char_flg = 1;
                           }
                           else if (comp_txt[8] == 'D' and
                              memcmp(&comp_txt[8],"DOUBLEPRECISION",15)
                              == 0){
                              memset(&comp_typ[8],'k',6);
                           }
                        }
                        return 7;
                     }
                  }
                  break;
               }
            case 9:
               {
                  int bb=0;
                  for(;bb < code_9_sz; bb++){
                     if(strncmp (comp_txt, code_9 [bb], 9) == 0){
                        memset(comp_typ,'K',9);
                        // was it "CHARACTER" ?
                        if (bb == e_CHARACTER){
                           char_flg = 1;
                           if(comp_txt[9] == 'F'){
                              if(find_f(9))return 8+8;
                           }
                           else if (comp_txt[9] == '*'){
                              // look for first alpha outside
                              // of any brakets.
                              int bra = 0;
                              for (int az=10;az<n_chas;az++){
                                 char asc = comp_txt[az];
                                 if (asc == '('){
                                    bra++;
                                 }
                                 else if (asc == ')') {
                                    bra--;
                                 }
                                 else if (bra == 0 and
                                    isalpha(asc)){ // start var.name.
                                    if (asc == 'F'){
                                       if (find_f(az)) {
                                          return az+7;
                                       }
                                    }
                                    return az-1;
                                 }
                              }
                           }
                        }
                        // was it "STARTEDIT" or "BLOCKDATA" ?
                        else if(bb == e_STARTEDIT or
                           bb == e_BLOCKDATA){
                           memset (&comp_typ[5],'k',4);
                        }
                        return 8;
                     }
                  }
                  break;
               }
            case 10:
               {
                  if(strncmp(comp_txt,"SUBROUTINE",10) == 0){
                     memset(comp_typ,'K',10);
                     return 9;
                  }
                  break;
               }
            case 15:
               {
                  if(strncmp(comp_txt,"DOUBLEPRECISION",15) == 0){
                     memset (comp_typ,'k',6);
                     memset (&comp_typ[6],'K',9);
                     // check for "FUNCTION"
                     if (comp_txt[15] == 'F'){
                        if(find_f(15))return 3+8;
                     }
                     return 14;
                  }
                  break;
               }
            }
         }
         // *** can start in any column. ***
         if (num_chas == 4){
            if (comp_txt[i] == 'S' and
               memcmp(&comp_txt[i],"STOP",4) == 0){
               if(stop_pause(i+4)) {
                  memset(&comp_typ[i],'K',4);
                  return i+3;
               }
            }
         }
         else if (num_chas == 5){
            if(comp_txt[i] == 'P') {
               if (memcmp(&comp_txt[i],"PAUSE",5) == 0){
                  if(stop_pause(i+5)){
mark_it:
                     ;
                     memset(&comp_typ[i],'K',5);
                     return i+4;
                  }
               }
               if(memcmp(&comp_txt[i],"PRINT",5) == 0 or
                  memcmp(&comp_txt[i],"PUNCH",5) == 0){
                  if (print_punch(i+5)){
                     goto mark_it;
                  }
               }
            }
         }
         else if (num_chas == 6){
            if(comp_txt[i] == 'A' and
               strncmp(&comp_txt[i],"ASSIGN",6) == 0){
               memset (&comp_typ[i],'K',6);
               // special case look for "TO"
               int sr = i+6;
               for (;sr < n_chas; sr++){
                  if (isdigit(comp_txt[sr])) continue;
                  if (comp_txt [sr]   == 'T' and
                     comp_txt [sr+1] == 'O') {
                     comp_typ [sr] = comp_typ[sr+1] = 'K';
                     return sr+1;
                  }
               }
               return i+5;
            }
         }
         else if ((num_chas == 7 or num_chas == 8) and
            comp_txt[i] == 'T') {
            if (memcmp(&comp_txt[i],"TRACEON",7) == 0) {
               memset (&comp_typ[i],'K',5);
               memset (&comp_typ[i+5],'k',2);
               return i+6;
            }
            else if (memcmp(&comp_txt[i],"TRACEOFF",8) == 0){
               memset (&comp_typ[i],'K',5);
               memset (&comp_typ[i+5],'k',3);
               return i+7;
            }
         }
      }
   }
   return n_chas;
}

/* *******************************************************************
   find_f - Finds "FUNCTION" & sets type to 'k'
        Parameter - position in comp_txt string.
        Returns TRUE or FALSE
********************************************************************* */
static int  find_f (int c_pos){
   if (memcmp(&comp_txt[c_pos],"FUNCTION", 8) == 0) {
      memset (&comp_typ[c_pos],'k',8);
      return TRUE;
   }
   return FALSE;
}
/* *******************************************************************
   label_print - Testing tool to print out label table.
********************************************************************* */
static void label_print (void){
   if (! tester) return;
   printf("Number of labels = %i \n",numb_lab );
   printf("Total labels available = %i\n", tot_lab);
   int i;
   for (i = 0;i < numb_lab;i ++){
      printf("org=%5li new=%5li def=%5i use=%3i link=%5i flags= %03o \n",
         (t_pnt+i)->label_num,
         (t_pnt+i)->new_num,
         (t_pnt+i)->def_line,
         (t_pnt+i)->num_uses ,
         (t_pnt+i)->link_pnt ,
         (t_pnt+i)->l_flags);
   }
   return;
}
/* **********************************************************************
   mem_dmp - dumps off 256 bytes of memory in hex and characters
   input parameter is void pointer to memory location
********************************************************************* */
static void mem_dmp (void * m_pnt){
   const int D_SIZE = 256;  // total
   const int N_LINE = 16;   // number on each line.
   char * temp_pnt = (char *) m_pnt;
   int x = 0;
   static char line_x [N_LINE+1];
   if (! tester) return;
   printf ("Dump of memory at location %p \n",m_pnt);
   while (x+1 <= D_SIZE){
      // do it in lines of N_LINE bytes.
      memcpy (line_x,(temp_pnt+x),N_LINE);
      // first in hex.
      printf("%3i(%2x) ",x,x);
      for (int i=0;i < N_LINE;i ++){
         unsigned int xxx = line_x [i];
         xxx = xxx & 0xFF;
         printf ("%2x ",xxx);
         // change if not printable
         if (!isprint(line_x[i])) line_x[i] = ' ';
      }
      // now as characters
      line_x [N_LINE] = 0;
      printf ("=%s=\n",line_x);
      x += N_LINE;
   }
   return;
}
/* *******************************************************************
   line_table - dumps off line table
 ******************************************************************** */
static void line_table (void){
   if (! tester) return;
   printf ("Number of lines = %i\n",num_code_lines);
   for (int x = 0;x < num_code_lines; x++){
      printf("Line= %02i Start= %02i Num.Chs= %02i Buff.pnt= %04i"
         " Rel.Line= %04i Txt.pos = %04i\n", x,
         in_lin[x].strt_pos,
         in_lin[x].numb_chrs,
         in_lin[x].buff_pnt,
         in_lin[x].rel_line,
         in_lin[x].txt_posn);
   }
   return;
}
/* ********************************************************************
   quote_find - find quote word in code text.
 ******************************************************************* */
static void quote_find (void) {
   char temp_string [MAX_HOLLER + 1];
   int qi = 0, end_p, s_err;
   int st_len = strlen(comp_txt);
   int first_flg = TRUE;
   for (;qi < st_len;qi++){
      char q_typ = comp_typ [qi];
      if (q_typ == 'Q' or q_typ == 'H'){
         int stp = qi;
         if (q_typ == 'H'){ // move past count + "H"
            int hs = qi;
            stp = -1;
            for(; hs < st_len; hs++){
               if (isdigit(comp_txt[hs])) continue;
               if (comp_txt[hs] == 'H'){
                  stp = hs+1;
                  break;
               }
               else {
                  printf("System error 71\n");
                  exit (1);
               }
            }
            if (stp == -1){
               printf("System error 722\n");
               exit (1);
            }
         }
         // now find end of quote
         int jj=stp;
         end_p = st_len - 1;
         for (;jj < st_len;jj ++){
            if (comp_typ[jj] != q_typ){
               end_p = jj - 1;
               break;
            }
         }

         if (q_typ == 'Q') { // when a normal quote stream
            assert (comp_txt[stp] == '\'' and
               comp_txt[end_p] == '\'');
            stp  += 1;
            end_p -= 1;
         }
         int st_siz = end_p - stp + 1;
         assert (st_siz > 0 and st_siz < MAX_HOLLER);
         if (!(st_siz > 0 and st_siz < MAX_HOLLER)){
            printf ("error at line %li\n",c_line_num);
            exit(1);
         }
         if (pass_num == 2){
            // put string into temporary buffer.
            strncpy (temp_string, &comp_txt[stp], st_siz);
            temp_string [st_siz] = 0; // terminate string
            s_err = spell_split (temp_string, st_siz);
            if (s_err) { // convert string to lower
               for (int xx = 0; xx < st_siz; xx++){
                  temp_string[xx] = tolower(temp_string[xx]);
               }
               spell_split (temp_string, st_siz);
               if (first_flg) {
                  fprintf (q_spl_f, "%6li:", c_line_num);
                  first_flg = FALSE;
               }
               fprintf (q_spl_f,"'%s'", temp_string);
            }
         }
         qi = end_p;
         // allow for last apos.
         if (q_typ == 'Q')qi += 1;
      }
   }
   if (! first_flg){
      fprintf(q_spl_f, "\n");
   }
   return;
}
/* ********************************************************************
  Close-Down closes any temporary files and deletes them.
********************************************************************* */
static void close_down (void) {
   if (com_ovf_f != NULL) { // comment overflow file
      fclose(com_ovf_f);
   }
   if (sav_line_file_name[0] != 0){
      remove(sav_line_file_name);
   }
   return;
}
/* *******************************************************************
   one_liner - if overflow is only by a small amount try to make room
               by closing excess spaces.
               Returns TRUE if able to fit on a line
               otherwise FALSE.
 ****************************************************************** */
static int  one_liner (int b_pnt, int e_pnt){
   int a_end = e_pnt - 1;
   if (comp_pos[a_end] <= 71)  return TRUE;
   if (b_pnt + 3 >= e_pnt) return FALSE;
   const int For_ward  = +1;
   const int Back_ward = -1;
   int dir_tion = For_ward;
   int pass = 1;  // switch between beginning & end for gap search
   int st_pnt = b_pnt;
   while (comp_pos[a_end] > 71) {
      if (pass == 1) {
         st_pnt = b_pnt;
         dir_tion = For_ward;
         pass = 2;
      }
      else{
         st_pnt =  e_pnt-2;
         dir_tion = Back_ward;
         pass = 1;
      }
      int gap_fnd = 0;
      // find two fields with gap & close gap.
      for (int sd= st_pnt; sd < e_pnt-1 and sd >= b_pnt; sd +=dir_tion){
         int gap = comp_pos [sd+1] - comp_pos [sd];
         int gap_typ = 0;
         if (gap > 1) {
            if (comp_typ[sd] != comp_typ[sd+1]) gap_typ = 1;
            if (gap_typ == 1 and  // change of field.
               isalnum(comp_txt[sd]) and isalnum(comp_txt[sd+1])) {
               // need a space between key-words.
               gap --;
            }
            if (gap > 1) {
               gap --;
               for (int xy = sd+1;xy < e_pnt; xy++){
                  comp_pos[xy] -= gap;
               }
               gap_fnd = 1;
               break;
            }
         }
      }
      if (gap_fnd == 0) break; // no more gaps
   }
   int x_tra = comp_pos[a_end];
   if (x_tra <= 71) return TRUE;
   int spc = x_tra - 71;
   // if only 1 or two chars. move whole line <-
   if (spc <= 2) {
      if (comp_pos[b_pnt] - spc < 6) return FALSE;
      for (int sb = b_pnt; sb < e_pnt; sb++){
         comp_pos [sb] -= spc;
      }
      return TRUE;
   }
   else {
      return FALSE;
   }
}
/* ********************************************************************
   full_format - Finds correct positions in case of full formatting
      Also used in other cases when re-formattimg is too difficult.
 ******************************************************************* */
static void full_format (void){
   comp_pos[0] = strt_indent;
   // if full restructure - remove all previous positioning
      memset (&comp_pos[1],0,n_chas-1);
   int pnt = 1;
   int lin = 0;
   int overf_flg = FALSE;
   int new_fld = 0;
   int new_field = 0;
   char chas,pre_chas;
   int p_pos = strt_indent;
   int ln_limit = 71; // maximum position.
   if (pprw_flg) { // compact fields in READ WRITE etc. parameters
      int sp = 0;
      int bra = 0;
      if (comp_txt[0] == 'I') { // look for end of IF
         for(int gg = 2; gg < n_chas; gg++){
            char tst = comp_txt[gg];
            char typ = comp_typ[gg];
            if (typ == 'Q' or typ == 'H') continue;
            if (tst == '('){
               bra++;
            }
            else if (tst == ')'){
               bra--;
               if (bra == 0) {
                  sp = gg+1;
                  break;
               }
            }
         }
         assert(sp != 0);
      }
      // all parameters at level 1, make into single field
      int b_lvl = 0;
      for (int dd=sp;dd<n_chas;dd++){
         char chs = comp_txt[dd];
         char cty = comp_typ[dd];
         if (b_lvl == 0) {
            if (cty == 'Q' or cty == 'H') continue;
            if (chs == '(') b_lvl = 1;
            continue;
         }
         // inside brackets
         if (chs == ','){
            continue;
         }
         else if (chs == '(') {
            b_lvl ++;
            continue;
         }
         else if (chs == ')'){
            b_lvl --;
            continue;
         }
         else {
            comp_typ[dd] = 'F';
         }
      }
   }
   char pre_pre_chas = 0;
   for (; pnt < n_chas; pnt++,pre_pre_chas = pre_chas){
      if (comp_typ[pnt] == comp_typ[pnt-1]){
         p_pos = p_pos + 1;
         new_field = 0;
      }
      else { //new field
         new_field = 1;
         overf_flg = FALSE;
         chas     = comp_txt [pnt];
         pre_chas = comp_txt [pnt - 1];
         if (chas == '('){
            if (pre_chas == '(' ){
               p_pos = p_pos + 1;
            }
            else {
               p_pos = p_pos + 2;
            }
         }
         else if (chas == ')'){
            p_pos += 1;
         }
         else if (chas == ','){
            p_pos += 1;
         }
         else if (chas == ':'){
            p_pos += 1;
         }
         else if (pre_chas == '('){
            p_pos += 1;
         }
         else if (pre_chas == '\''){
            p_pos += 1;
         }
         else if (pre_chas == ',' ){
            p_pos += 1;
         }
         else if (pre_chas == ':' ){
            p_pos += 1;
         }
         else if (pre_chas == '-' ){
            if (pre_pre_chas == ',' or
               pre_pre_chas == '(' or
               pre_pre_chas == '=') { // unary.
               p_pos += 1;
               // make into a single field
               comp_typ[pnt-1] = comp_typ[pnt];
            }
            else { // normal case
               p_pos = p_pos + 2;
            }
         }
         else { // normal case
            p_pos = p_pos + 2;
         }
      }
      // gone past end of line?
         if (p_pos > ln_limit) { // past end of line?
            if (overf_flg) { // just start on line line.
               p_pos = 6;
               comp_pos[pnt] = p_pos;
               lin++;
               assert (lin <= MAX_CRDS);
               in_lin[lin].txt_posn = pnt;
               if (in_lin[lin].rel_line == 0){
                  in_lin[lin].rel_line =
                  in_lin[lin-1].rel_line +1;
               }
               goto carry_on;

            }

            // return to start of earlier field
            int old_pnt = pnt;
            if (!new_field) pnt = new_fld;
            if (new_field) new_fld = pnt;
            lin ++; // increment line
            if (lin >= MAX_CRDS) {
               printf ("Impossible to reformat\n");
               close_down();
               exit (1);
            }
            if (in_lin[lin].rel_line == 0){
               in_lin[lin].rel_line = in_lin[lin-1].rel_line +1;
            }
            if (new_fld == in_lin[lin-1].txt_posn){
               // cannot fit field onto line.
               int f_size = 0;
               char t_typ = comp_typ [pnt];
               int cnt = new_fld;
               for (;cnt <n_chas; cnt ++){
                  if (t_typ == comp_typ [cnt]){
                     f_size++;
                     continue;
                  }
                  break;
               }
               if (f_size > (73 -7)){
                  // should only occur for character string.
                  if (!(t_typ == 'Q' or t_typ == 'H')){
                     printf("No room on line %li\n", c_line_num);
                     exit(1);
                  }
                  pnt = old_pnt;
                  in_lin[lin].txt_posn = pnt;
                  comp_pos[pnt] = p_pos = 6;//start at line beginning
                  overf_flg = TRUE; // special condition
                  goto carry_on;
               }
               lin--;
               comp_pos [new_fld] = p_pos = 72 - f_size;
            }
            else {
               in_lin[lin].txt_posn = new_fld;
               int new_spot = strt_indent;
               if (!(new_spot == MAX_INDENT) and
                  strncmp(comp_txt,"IF(",3) == 0){
                  new_spot = strt_indent + 2;
               }
               comp_pos[new_fld] = new_spot;
               p_pos = new_spot;
            }
         }
         else { // normal case
            comp_pos[pnt] = p_pos;
            if (new_field) new_fld = pnt;
         }

carry_on:
      ;
   }
   num_code_lines = lin + 1;
}
/* *********************************************************************
  stop_pause - single parameter - postion in line to start searching.
  returns either TRUE or FALSE.
  Looks for valid data after STOP or PAUSE. Can be nothing, a number or
  a text string.
 ******************************************************************** */
static int  stop_pause(int ppp){
   char fst = comp_txt[ppp];
   if (fst == 0) return TRUE; // EOL case- end of line reached.
   if (fst == '\'') return TRUE; // text string case
   if (! isdigit(fst)) return FALSE; // can only be a digit or txt.str.
   for (int yy = ppp+1; yy < n_chas; yy++) {
      if(! isdigit(comp_txt[yy])) return FALSE;
   }
   return TRUE;
}
/* *******************************************************************
   print_punch - checks data following a PRINT or PUNCH
                 returns TRUE or FALSE
                 input parameter is start position in COMP_TXT.
  ******************************************************************* */
static int  print_punch(int st_pos){
   /* should be followed by 1-5 digit format label, or an '*' or an assign
  variable. If anything follows (optional) it must be a comma and start
  of variable name (begining with an alpha) */
   char chs = comp_txt[st_pos];
   int zz = 0;
   if (chs == '(' ) return TRUE;
   if (chs == '*' and comp_txt[st_pos+1] == ',') return TRUE;
   if (isdigit(chs)){ // 1-5 digit format label.
      int num_dig = 1;
      for (zz = st_pos+1; zz < n_chas;zz++){
         chs = comp_txt [zz];
         if (isdigit(chs)){
            num_dig ++;
            if (num_dig > 5) return FALSE;
            continue;
         }
         else if (chs == ','){
            goto after_comma;
         }
         else {
            return FALSE;
         }
      }
      return TRUE;
   }
   else if (isalpha(chs)){ // variable ASSIGNED to format.
      int num_var = 1;
      for (zz = st_pos+1; zz < n_chas; zz++){
         chs = comp_txt [zz];
         if (isalnum(chs) or chs == '$'){
            num_var ++;
            if (num_var > 6) return FALSE;
            continue;
         }
         else if (chs == ','){
            goto after_comma;
         }
         else {
            return FALSE;
         }
      }
      return TRUE;
   }
   else {
      return FALSE;
   }
after_comma:;
   // check for an alpha- start of variable name.
      if (isalpha(comp_txt[zz+1])){
         return TRUE;
      }
      else {
         return FALSE;
      }
}
/* ********************************************************************
   at_label - input parameter is start position of label.
              output is TRUE or FALSE.
              Testing of DEBUG "AT 999" where 999 = label.
 ****************************************************************** */
static int  at_label(int ps){
   char chs = comp_txt [ps];
   if (! isdigit(chs))  return FALSE;
   if (n_chas - ps > 5) return FALSE;
   int num_dig = 1;
   for (int zz= ps+1; zz < n_chas; zz++){
      chs = comp_txt[zz];
      if (isdigit(chs)){
         num_dig++;
         if (num_dig > 5) return FALSE;
         continue;
      }
      else {
         return FALSE;
      }
   }
   return TRUE;
}
/* *****************************************************************
   conv_ss - convert SUBSTR to standard format
             one parameter, start of "SUBSTR(" in comp_txt
 **************************************************************** */
static void conv_ss (int st_pnt){
   const int MAX_FLD = 60; // max. size
   char field_1 [MAX_FLD+1]; // name or literal
   char field_t [MAX_FLD+1]; // type of field.
   char field_2 [MAX_FLD+1]; // start position
   char field_3 [MAX_FLD+1]; // number of characters.
   char combined [MAX_FLD*4+1];
   int in_pnt, lp, lv, reduc, xz, i;
   int fz1, fz2, fs1, fs2, rd1, rd2;
   char ct, sign1, sign2;
   int num_num, temp_num, total_num;
   const int MAX_NUF = 15; // Maximum number of split fields
   char a_fld [MAX_FLD+1]; // analysis fields
   int fld_st [MAX_NUF] = {0}; // field start positions
   int fld_sz [MAX_NUF] = {0}; // field sizes (without (+ or -)
   char num_fld [MAX_NUF] = {0}; // number field indicator
   int fld_nm = 0; // number of fields found
   int stp, stz, zf;
   assert (comp_txt[st_pnt] == 'S');
   memset (field_1,0,sizeof(field_1));
   memset (field_t,0,sizeof(field_t));
   memset (field_2,0,sizeof(field_2));
   memset (field_3,0,sizeof(field_3));
   memset (combined,0,sizeof(combined));
   // pick up the three fields (terminated by ',' at level 1.
      int field = 1; // field number
   int posit = 0; // next position
   int levl  = 1; // level of extraction
   char chs, cht;
   for (i = st_pnt+7; i < n_chas; i++){
      chs = comp_txt[i];
      cht = comp_typ[i];
      // original or capitol letters?
      if (shad_txt != NULL) chs = shad_txt[i];
      if (cht == 'Q' or cht == 'H') goto store_next;
      switch (chs) {
      case '(':
         {
            levl++;
            break;
         }
      case ')':
         {
            levl--;
            if (levl == 0) goto end_loop;
            break;
         }
      case ',':
         {
            if (levl != 1) break;
            if (field >= 3) {
sub_s_error:;
               printf("Invalid SUBSTR at line %li \n",c_line_num);
               exit (1);
            }
            field++;
            posit = 0;
            continue;
         }
      }
store_next:
      ;
      if (field == 1){
         field_1 [posit] = chs;
         field_t [posit] = cht;
      }
      else if (field == 2){
         field_2 [posit] = chs;
      }
      else { // field == 3
         field_3 [posit] = chs;
      }
      if (posit >= MAX_FLD) goto sub_s_error;
      posit++;
   }
   // end-of-loop & not found terminating ")"
   goto sub_s_error;
end_loop:;
   int orig_num_chs = i - st_pnt + 1; //original number characters
   // Just for testing show original as a comment.
   if (t_opt){
      sprintf(combined,"C *s %.62s *s",&comp_txt[st_pnt]);
      out_line (combined);
   }
   memset (combined,0,sizeof(combined));
   // create new string.
   strcpy (combined,field_1);
   strcat (combined,"(");
   strcat (combined, field_2);
   strcat (combined, ":");
   int rep_pos = strlen(combined);
   strcat (combined,field_2);
   strcat (combined,"+");
   strcat (combined,field_3);
   strcat (combined,"-1)");
   int new_num_chs = strlen(combined);
   // try & simplify second field.
   int sim_siz = new_num_chs - rep_pos;
   if (sim_siz > MAX_FLD) goto no_sim;
   in_pnt = rep_pos;
   char old_c, c_tst, num_del, bra_fnd;
   old_c = 0;
   num_del = 0;
   bra_fnd = 0;
   for (xz = 0; xz < sim_siz; xz++){
      c_tst = a_fld [xz] = toupper (combined[in_pnt++]);
      if (c_tst == '(') { // possible un-needed brackets?
         bra_fnd = 1;
      }
   }
   if (bra_fnd) {
      for (xz = 0; xz < sim_siz; xz++,old_c = c_tst){
         c_tst = a_fld [xz];
         if (c_tst == '('){ // look for match
            if (old_c == 0 or
               old_c == '+' or old_c == '-' or old_c == '('){
               int xyz, match = 0;
               for (xyz = xz+1; xyz < sim_siz - 1; xyz++){
                  // look for matching ')'
                  char tmc = a_fld [xyz];
                  if (tmc == '(') {
                     match ++;
                     continue;
                  }
                  else if (tmc == ')'){
                     if (match == 0) { // matching bracket found
                        char nxt_c = a_fld [xyz+1];
                        if (nxt_c == ')' or
                           nxt_c == '+' or nxt_c == '-') {
                           // eliminate two brackets
                           a_fld [xyz] = 0;
                           a_fld [xz]  = 0;
                           num_del ++;
                           break;
                        }
                     }
                     else {
                        match--;
                        continue;
                     }
                  }
               }
            }
         }
      }
      if (num_del) {
         // some brackets removed- compact line.
         int to_c, frm_c;
         to_c = frm_c = 0;
         for (;frm_c < sim_siz;frm_c++){
            char t_c;
            t_c = a_fld [to_c++] = a_fld [frm_c];
            if (t_c == 0) to_c--;
         }
         sim_siz -= num_del*2;
      }
   }
   a_fld [sim_siz] = 0; // string terminator
   // break string into fields
   lv = 0; // bracket level
   for (lp = 0; lp < sim_siz; lp++){
      ct = a_fld [lp];
      switch (ct) {
      case '(':
         lv++;
         break;
      case ')':
         lv--;
         if (lv == -1) { // last bracket found
            fld_sz[fld_nm] = lp - fld_st [fld_nm];
            fld_nm++;
            goto terminal;
         }
         break;
      case '+': // these are only de-limiters
      case '-':
         if (lv != 0) break; // only use outside any brackets
         if (fld_nm == 0){
            fld_sz[0] = lp;
         }
         else {
            fld_sz[fld_nm] = lp - fld_st [fld_nm];
         }
         fld_nm++;
         if (fld_nm >= MAX_NUF) goto no_sim;
         fld_st[fld_nm] = lp + 1;
      }
   }
   // should never get to loop end
   printf("System error 6160\n");
   exit (1);
terminal:
   ;
   reduc = 0; // number of reductions.
   for (rd1 = 0; rd1 < fld_nm; rd1++){
      fz1 = fld_sz [rd1];
      fs1 = fld_st [rd1];
      if (fz1 == 0) continue;
      for(rd2 = rd1+1; rd2 < fld_nm; rd2++){
         fz2 = fld_sz[rd2];
         fs2 = fld_st[rd2];
         if (fz2 != fz1) continue;  // no match
         // test for opposite signs
         if (fld_st[rd1] == 0){
            sign1 = '+';
         }
         else {
            sign1 = a_fld [fs1 -1];
         }
         sign2 = a_fld [fs2 - 1];
         if((sign1 == '+' and sign2 == '-') or
            (sign2 == '+' and sign1 == '-')){
            if (memcmp( &a_fld [fs1],
               &a_fld [fs2], fz1) == 0){
               // a match - remove second entry.
               stp = fs2 - 1;
               stz = fz2 + 1;
               memmove (&a_fld [stp],
                  &a_fld [stp + stz],((sim_siz+1) - (stp+stz)));
               // update sizes and table.
               sim_siz -= stz;
               fld_sz [rd2] = 0;
               for (zf = rd2+1; zf < fld_nm; zf++){
                  fld_st [zf] -= stz;
               }
               // remove primary entry.
               stp = fs1;
               stz = fz1;
               if (stp != 0) {
                  stp = stp - 1;
                  stz = stz + 1;
               }
               memmove (&a_fld [stp],
                  &a_fld [stp + stz],((sim_siz+1) - (stp+stz)));
               sim_siz = sim_siz - stz;
               fld_sz [rd1] = 0;
               for (zf = rd1+1; zf < fld_nm; zf++){
                  fld_st [zf] -= stz;
               }
               reduc++;
               goto outer_loop; // try next field
            }
         }
      }
outer_loop:
      ;
   }
   // make a second pass through to consolidate number values.
   num_num = 0; // number of pure numeric fields
   total_num = 0; // sum of all numerics
   for (xz = 0; xz < fld_nm; xz++){
      fz1 = fld_sz [xz];
      fs1 = fld_st [xz];
      if (fz1 == 0) continue;
      if (isdigit (a_fld [fs1])) { // posible number field
         temp_num = 0;
         lp = fs1;
         for (lv = 0; lv < fz1; lv++){
            char nv = a_fld [lp++];
            if (isdigit(nv)) {
               temp_num = temp_num * 10 + nv - '0';
            }
            else { // not a straight number- try next field
               goto tnf;
            }
         }
         // all numeric
            if (fs1 == 0 or a_fld[fs1 -1] == '+') {
               total_num += temp_num;
            }
            else if (a_fld [fs1 - 1] == '-') {
               total_num -= temp_num;
            }
            else {
               printf ("System error 6265\n");
               exit (1);
            }
         num_num ++;
         num_fld [xz] = 1;
      }
tnf:
      ;
   }
   // if more than one numeric field - delete them all and
      // add consolidate total at end.
      if (num_num > 1) {
         for (xz = 0; xz < fld_nm; xz++){
            if (num_fld [xz] == 1) { // remove field
               stp = fld_st[xz];
               stz = fld_sz[xz];
               assert (stz != 0);
               if (stp != 0) {
                  stp = stp - 1;
                  stz = stz + 1;
               }
               memmove (&a_fld [stp],
                  &a_fld [stp + stz],((sim_siz+1) - (stp+stz)));
               sim_siz = sim_siz - stz;
               fld_sz [xz] = 0;
               for (zf = xz+1; zf < fld_nm; zf++){
                  fld_st [zf] -= stz;
               }
               reduc++;
            }
         }
         // put in consolidated total value
         sprintf(&a_fld [sim_siz-1],"%+i)",total_num);
      }
   if (reduc != 0) { // some changes made
      // skip over any leading '+'
      int lz = 0;
      if (a_fld[0] == '+') lz = 1;
      strcpy (&combined[rep_pos],&a_fld[lz]);
      new_num_chs = strlen(combined);
   }
no_sim:
   ;
   // final check- if first field is a quote or Hollerith.
   if (combined[0] == '\'' or isdigit(combined[0])){
      // start position should be '1'.
      if (field_2[0] == '1' and field_2[1] == 0){
         new_num_chs = strlen(field_1);
      }
   }
   // replace original.
   replace_txt(st_pnt,orig_num_chs,new_num_chs,combined);
   // put in character type
   int ct_pos = st_pnt;
   int zz;
   int zz_sz = strlen(field_t);
   for (zz = 0; zz < zz_sz;zz++){
      comp_typ[ct_pos++] = field_t[zz];
   }
   for (zz = st_pnt+strlen(field_1); zz < st_pnt+new_num_chs; zz++){
      char xyz = comp_txt[zz];
      if (isalnum(xyz) or xyz == '$') {
         comp_typ[zz] = 0;
      }
      else {
         comp_typ[zz] = xyz;
      }
   }
   // make sure that all characters are contiguous.
      if (num_code_lines == 1) { // only if on one line.
         int last_p = st_pnt + new_num_chs -1;
         int c_diff = comp_pos[last_p+1] - comp_pos [last_p];
         int post = comp_pos [st_pnt];
         for (zz = st_pnt; zz < st_pnt + new_num_chs; zz++){
            comp_pos [zz] = post ++;
         }
         if(n_chas != last_p + 1){ // only if more on line
            assert (c_diff >= 1);
            // now keep distance (if changed)
            int new_diff = comp_pos[last_p+1] - comp_pos [last_p];
            int chng = new_diff - c_diff;
            if (chng){
               for (zz = last_p+1;zz < n_chas; zz++){
                  comp_pos [zz] -= chng;
               }
            }
         }
      }
   return;
}
/* *******************************************************************
   standard_error
   Produces a message based on errno.
********************************************************************* */
static void standard_error (void){
   printf ("Error number %i, Message=%s\n",errno,sys_errlist[errno]);
   exit (1);
}
/* ********************************************************************
  end_if - to find ending position of an if
    input parameter is start of "IF(", output is position of character
    after final bracket ")". If no final ")" found -1 is returned.
 ******************************************************************* */
static int  if_end (int stp){
   assert (memcmp(&comp_txt[stp],"IF(",3) == 0);
   int bra, i;
   char chr;
   bra = 1;
   // find end of if
   for (i = stp+3; i < n_chas; i++){
      chr = comp_txt [i];
      if (comp_typ[i] == 'H' or  comp_typ[i] == 'Q') continue;
      if (chr == '(') {
         bra ++;
         continue;
      }
      if (chr == ')') {
         bra--;
         if (bra == 0){
            return  i +1;
         }
      }
   }
   return -1;
}
/* *******************************************************************
   check_multi - checks if a multi assign, Tests for at least one (level
        zero) comma before any "=" and zero afterwards (to test DO case)
        returns either number of assign items or FALSE.
        Input parameter is start position in comp_txt to start searching
        from.
********************************************************************* */
static int  check_multi (int st_pos){
   int i, bra = 0;
   int eql = 0; // number of equals found
   int com_count = 0; // number of commas before equals.
   char chs, cht;
   for (i= st_pos; i < n_chas; i++){
      chs = comp_txt[i];
      cht = comp_typ[i];
      // Hollerith or character string.
      if (cht == 'H' or cht == 'Q') continue;
      switch (chs) {
      case '(':
         bra++;
         break;
      case ')':
         bra--;
         break;
      case ',':
         if (bra == 0) {
            if (eql == 0){
               com_count++;
            }
            else{
               // comma after the "=" , assume a DO loop.
               com_count = 0;
               return FALSE;
            }
         }
         break;
      case '=':
         if (bra == 0){
            eql++;
         }
         break;
      }
   }
   // more than one comma before "="
   if (com_count > 0 and eql == 1) return com_count+1;
   return FALSE;
}
/* *********************************************************************
 multi_out - processes multi-assigns, producing multiple single-line
             outputs (assumes all text in comp_txt etc.
             input parameter(1) = number of multi-assign items.
                            (2) = call from THEN-ENDIF (TRUE/FALSE)
********************************************************************* */
static void multi_out (int num_assign, int ident_flg){
   struct pnts {
      int pos_asg; // position of assign variable
      int siz_asg; // size of assign variable
   } * v_arry = NULL ;
   char  * alt_comp_txt = NULL; // upper case text
   char  * alt_comp_pos = NULL; // position
   char  * alt_comp_typ = NULL; // type
   char  * alt_shad_txt = NULL; // original text {optional}
   char  * comb_string = NULL;
   char temp[80];
   int pos, siz, old_pos, diff;
   int j, i, pos_2, siz_2;
   int srcsz = 0; // size of source
   const int MAX_MIN = 6; // definition of small size varaible size.
   char m_txt [MAX_MIN+1]; // Upper-case text
   char m_org [MAX_MIN+1]; // original text
   char m_pos [MAX_MIN+1]; // position of text
   char m_typ [MAX_MIN+1]; // type of text
   int sim_src = FALSE; // simple source variable flag.
   if (t_opt) { // just for testing.
      sprintf (temp,"C *m* %.60s *m*",comp_txt);
      out_line (temp);
   }
   // keep original in one long string.
   int size_s = 3*(n_chas+1);
   // N.B. Use size + 1 to include terminating NUL.
   if (shad_txt != NULL) size_s += (n_chas+1);
   comb_string = (char *) malloc(size_s);
   if (comb_string == NULL){
      printf ("Allocation of memory failure alt_strings\n");
      exit (1);
   }
   memcpy(comb_string,comp_txt,n_chas+1);
   alt_comp_txt = comb_string;
   memcpy(&comb_string[n_chas+1],comp_pos,n_chas+1);
   alt_comp_pos = alt_comp_txt + (n_chas + 1);
   memcpy(&comb_string[(n_chas+1)*2],comp_typ,n_chas+1);
   alt_comp_typ = alt_comp_pos + (n_chas + 1);
   if (shad_txt != NULL){
      memcpy(&comb_string[(n_chas+1)*3],shad_txt,n_chas+1);
      alt_shad_txt = alt_comp_typ + (n_chas + 1);
   }
   // assign area for indexing
   v_arry = (struct pnts *) calloc(sizeof(struct pnts) , num_assign);
   if (v_arry == NULL) {
      printf("Allocation of memory fail for v_arry\n");
      exit (1);
   }
   // find out where variables are in string.
   int bra = 0, act_len = 0, max_len = 0;
   int field_num = 0;
   int prev_pos  = 0;
   int c_count   = 0;
   for (i = 0; i < n_chas; i++){
      char cht = comp_txt[i];
      switch (cht){
      case '(':
         bra++;
         break;
      case ')':
         bra--;
         break;
      case ',':
      case '=':
         if (bra == 0){
            if (field_num == 0) { // first field
               v_arry->siz_asg = i;
               c_count = i+1;
               act_len = comp_pos[i-1] - comp_pos[0] + 1;
               if (act_len < c_count) act_len = c_count;
               max_len = act_len;
            }
            else { // 2+
               (v_arry+field_num)->siz_asg = c_count = i  - prev_pos;
               act_len = comp_pos[i-1] - comp_pos[prev_pos] + 1;
               // can be less when field straddles a line.
                  if (act_len < c_count) act_len = c_count;
               if (act_len > max_len) max_len = act_len;
            }
            // if not last item update next values.
            field_num++;
            assert (field_num <= num_assign);
            if (cht == '=') {
               // size of source variable.
               srcsz = n_chas - (i+1);
               if (srcsz <= MAX_MIN){
                  memcpy(m_txt,&comp_txt[i+1],srcsz+1);
                  memcpy(m_pos,&comp_pos[i+1],srcsz+1);
                  memcpy(m_typ,&comp_typ[i+1],srcsz+1);
                  if (shad_txt != NULL) {
                     memcpy(m_org,&shad_txt[i+1],srcsz+1);
                  }
                  // make positions relative to zero
                  int stf = m_pos[0];
                  int newp = 0, oldp = -1;
                  for (int bn = 0;bn < srcsz; bn++){
                     newp = m_pos[bn] - stf;
                     if (newp <= oldp){
                        // renumber
                        for (int j = 0;j < srcsz;j++) m_pos[j] = j;
                        break;
                     }
                     m_pos[bn] = oldp = newp;
                  }
                  sim_src = TRUE;
               }
               goto end_loop;
            }
            (v_arry+field_num)->pos_asg = i+1;
            prev_pos = i+1;
            break;
         }
      }
   }
end_loop:
   ;
   // test for simple case of initial value.
   // output fields.
   int strt_p = comp_pos[0];
   // indent if from IF-THEN, ENDIF case.
   if (ident_flg and i_opt) id_curr += n_if;
   // simple source field <= MAX_MIN characters
   if (sim_src){
      // put in standard position of "=" sign.
      int eq_pos = max_len + strt_p + 1;// position of equals
      int vr_pos = max_len + strt_p + 3;//variable start position.
      // update position information
      for (j = 0;j < srcsz; j++) m_pos[j] += vr_pos;
      for (i = 0;i < num_assign; i++){
         pos = (v_arry+i)->pos_asg;
         siz = (v_arry+i)->siz_asg;
         //** Upper case text. **
         memcpy (comp_txt, &alt_comp_txt[pos], siz);
         // add on "="
         comp_txt[siz] = '=';
         memcpy (&comp_txt[siz+1], m_txt, srcsz+1);
         n_chas = siz + 1 + srcsz;
         memcpy(comp_pos,&alt_comp_pos[pos],siz);
         //** update positions.**
         old_pos = comp_pos[0];
         diff = old_pos - strt_p;
         int pp_pos = 0; // continuity check
         for (j = 0; j < siz; j++){
            if(comp_pos[j] <= pp_pos) {
               // skipped a line - just give sequential positioning
               int new_p = strt_p;
               for (int xa = 0; xa < siz; xa++){
                  comp_pos[xa] = new_p++;
               }
               break;
            }
            pp_pos = comp_pos[j];
            comp_pos[j] -= diff;
         }
         comp_pos[siz] = eq_pos; // position of "="
         memcpy(&comp_pos[siz+1], m_pos, srcsz);
         //** move over type.
         memcpy (comp_typ,&alt_comp_typ[pos],siz);
         comp_typ[siz] = '=';
         memcpy (&comp_typ[siz+1], m_typ, srcsz);
         if (shad_txt != NULL){ // optional original text.
            memcpy (shad_txt,&alt_shad_txt[pos],siz);
            // add on "="
            shad_txt[siz] = '=';
            memcpy (&shad_txt[siz+1],m_org, srcsz);
         }
         // type.
         num_code_lines = 1;
         if (i != num_assign-1) {
            send_out (0);
         }
         else {
            send_out(1);
         }
         if (i==0) { // first time
            // remove any in-line comments.
            rem_in_lin();
         }
      }
   }
   else { // none simple case, variable is too big.
      // output last assign first
         int b_start = (v_arry+num_assign-1)->pos_asg;
      //      int begin_p = comp_pos[0];
      replace_txt (0, b_start, 0, comb_string);
      // is first line empty?
      if (num_code_lines > 1){
         if (in_lin[1].txt_posn == 0) {
            // split on "=" sign
            for (int gg = 0; gg < n_chas; gg++){
               char ch = comp_txt[gg];
               if (ch == '=') {
                  in_lin[1].txt_posn = gg+1;
                  // re-align second line.
                  int new_diff = comp_pos[gg+1] - comp_pos[0] - 3;
                  int end_spot = n_chas;
                  if (num_code_lines > 2) {
                     end_spot = in_lin[2].txt_posn;
                  }
                  for (int xt = gg+1; xt < end_spot; xt++){
                     comp_pos[xt] -= new_diff;
                  }
                  break;
               }
            }
         }
      }
      send_out(0);
      // remove any in-line comments.
      rem_in_lin();
      int ij;
      num_code_lines = 1;
      for (ij = num_assign-2; ij >= 0; ij --) {
         // work backwards.
         pos = (v_arry+ij)->pos_asg;
         siz = (v_arry+ij)->siz_asg;
         memcpy(comp_txt,&alt_comp_txt[pos],siz);
         // add on "="
         comp_txt[siz] = '=';
         memcpy(comp_pos,&alt_comp_pos[pos],siz);
         // update positions.
         old_pos = comp_pos[0];
         diff = old_pos - strt_p;
         for (j=0;j < siz;j++) comp_pos[j] -= diff;
         memcpy(comp_typ,&alt_comp_typ[pos],siz);
         if(shad_txt != NULL){
            memcpy(shad_txt,&alt_shad_txt[pos],siz);
            // add on "="
            shad_txt[siz] = '=';
         }
         // put in standard position of "=" sign.
         comp_pos [siz] = max_len + strt_p + 1;
         // type.
         comp_typ [siz] = '=';
         // now put in previous assign variable.
         pos_2 = (v_arry+ij+1)->pos_asg;
         siz_2 = (v_arry+ij+1)->siz_asg;
         memcpy (&comp_txt[siz+1], &alt_comp_txt[pos_2], siz_2);
         memcpy (&comp_pos[siz+1], &alt_comp_pos[pos_2], siz_2);
         // update positions.
         old_pos = comp_pos [siz+1];
         diff = old_pos - (max_len + strt_p + 3);
         for (j = siz+1;j < siz + siz_2 + 1; j++) comp_pos[j] -= diff;
         memcpy (&comp_typ [siz+1], &alt_comp_typ[pos_2], siz_2);
         if (shad_txt != NULL){ // original code
            memcpy (&shad_txt [siz+1], &alt_shad_txt[pos_2],siz_2);
         }
         n_chas = siz + 1 + siz_2;
         comp_txt [n_chas] = 0;
         if (ident_flg or i != num_assign-2) {
            send_out (0);
         }
         else {
            send_out(1);
         }
      }
   }
   // indent if from IF-THEN, ENDIF case.
      if (ident_flg and i_opt) id_curr -= n_if;
   // release memory
   free (v_arry);
   free (comb_string);
   return;
}
/* *******************************************************************
  str_3_assign - store labels used for ASSIGN
               - only used when 3 label arithmetic IF processing.
               - input parameter is start of label.
                 e.g. ASSIGN 1234 TO ABCD
 ****************************************************************** */
static void str_3_assign (int ppp){
   if_3_value tmp;
   int i,k;
   char chas;
   tmp.level = curr_lvl; // subroutine level
   tmp.label = get_lbl(&comp_txt[ppp],&i);
   // pick up variable name.
   memset (tmp.variable,' ',sizeof(tmp.variable));
   k = ppp + i + 2 ;
   for (int x = 0; x < sizeof(tmp.variable); x++,k++){
      chas = comp_txt[k];
      if (isalnum(chas) or chas == '$'){
         tmp.variable [x] = chas;
      }
      else {
         break;
      }
   }
   // check if entry already in table.
      for (i = assg_strt; i < assign_act; i ++){
      if (tmp.label == (assign_pnt+i)->label and
         memcmp (tmp.variable,
         &(assign_pnt+i)->variable,
         sizeof(tmp.variable)) == 0){
         return;
      }
   }
   // not in table - add new entry.
   if (assign_pnt == NULL) { // first time
      assign_pnt = (struct if_3_value *)
         malloc (sizeof(if_3_value)*ASS_SZ);
      if (assign_pnt == NULL) {
no_mem:
         ;
         printf("No memory for Assign table\n");
         exit (1);
      }
      assign_avail = ASS_SZ;
   }
   // any room?
   if (assign_act+1 > assign_avail) { // extend area.
      assign_pnt = (struct if_3_value *) realloc (assign_pnt,
         (assign_avail+ASS_SZ)*sizeof(if_3_value));
      if (assign_pnt == NULL) goto no_mem;
      assign_avail += ASS_SZ;
   }
   *(assign_pnt + assign_act) = tmp;
   assign_act++;
   return;
}
/* ********************************************************************
   assign_conv - converts "GOTO abc ,(123)" to real labels assigned
                 to variable "abc" - if possible.
 ******************************************************************* */
static void assign_conv (void) {
   const int MAX_LABS = 20; // maximum labels
   long int labs [MAX_LABS];
   const long int BIG_NUMBER = 999999;
   int i,j,k;
   int num_labs = 0; // number of labels
   char v_name [6]; // variable name to look for.
   char a_label[8]; // used to output labels.
   char lab_str [6*MAX_LABS + 3]; //  replacement string.
   // find variable name.
   memset (v_name,' ',sizeof(v_name));
   int st_pos = n_chas - 16;
   if (st_pos < 0) st_pos = 0;
   char * spnt = strstr(&comp_txt[st_pos],"GOTO");
   if (spnt == NULL) {
      printf ("System error 7054\n");
      exit (1);
   }
   // how many characters?
   int num_chas = n_chas - (spnt - comp_txt);
   // allow for GOTO & ,(123)
   num_chas -= 10;
   if (num_chas > 6 or num_chas < 1){
      printf ("System error 7062\n");
      exit (1);
   }
   spnt += 4;
   for (k = 0;k<num_chas;k++){
      v_name[k] = spnt[0];
      spnt += 1;
   }
   // now find labels in table.
   for (j = 0;j < assign_act; j++){
      if (curr_lvl == (assign_pnt + j)->level) {
         if (memcmp(v_name,(assign_pnt+j)->variable,sizeof(v_name)) == 0){
            if (num_labs == MAX_LABS){
               printf("MAX_LABS too small\n");
               exit (1);
            }
            labs[num_labs++] = (assign_pnt+j)->label;
         }
      }
      else {
         if(curr_lvl < (assign_pnt +j)->level) break;
      }
   }
   // now table is filled.
   if (num_labs == 0) return; // no entries-perhaps they are in an INCLUDE?
   // form up list
   lab_str[0] = 0;
   // allow for renumbering.
   if (r_opt){
      for (i = 0;i < num_labs; i++){
         labs[i] = convert_label (labs[i], 0);
      }
   }
   for (i = 0;i < num_labs; i++){
      if (r_opt) { // output in ascending order if re-numbered
         long int t_lab = BIG_NUMBER;
         int t_lab_pos = 99999;
         for (j = 0;j < num_labs; j++){
            if(labs[j] == 0) continue; // already selected.
            if (labs[j] < t_lab){
               t_lab = labs[j];
               t_lab_pos = j;
            }
         }
         assert (t_lab != BIG_NUMBER);
         labs[t_lab_pos] = 0;
         sprintf(a_label, "%li,",t_lab);
      }
      else {
         sprintf(a_label, "%li,",labs[i]);
      }
      strcat (lab_str,a_label);
   }
   int lab_len = strlen(lab_str);
   lab_str [lab_len - 1] = 0; // remove last comma
   // replace 123
      int old_strt = n_chas - 4;  // start of replace
   int pp = comp_pos[old_strt]; // position.
   replace_txt (old_strt, 3 , --lab_len, lab_str);
   // update position & type data.
   for (i = old_strt;i < n_chas; i++){
      char cc = comp_txt[i];
      comp_pos[i] = pp++;
      if (isdigit(cc)) {
         comp_typ [i] = 'L';
      }
      else {
         comp_typ [i] = cc;
      }
   }
   return;
}
/* *******************************************************************
   check_flags - input is a label, return flag bits for label
 ****************************************************************** */
static int  check_flags (long int lab_num){
   int i;
   int flgs = 0;
   long int tmp_lab;
   assert (lab_num != 0);
   for (i= lbl_strt_pos+1;i < numb_lab; i++){
      tmp_lab = (t_pnt+i)->label_num;
      if (tmp_lab == 0){
         printf("System error 7188\n");
         // should have been found.
         exit (1);
      }
      if (tmp_lab == lab_num){
         flgs = (t_pnt+i)->l_flags;
         break;
      }
   }
   return flgs;
}
/* *******************************************************************
  multi_lab_ok - If a label on a multiple assign, checks if label is
                 valid. e.g. If label is terminator for a DO loop then
                 multiple assigns cannot replace a single assign.
                 Returns TRUE if it is OK, FALSE otherwise. If an
                 additional CONTINUE with the label on is required then
                 the input parameter will be set to the label, otherwise
                 it will be set to zero.
 ****************************************************************** */
static int  multi_lab_ok (long int & con_lab){
   con_lab = 0;
   assert(curr_lab != 0);  // statement has a label in cols 1-5
   int flgs = check_flags(curr_lab);
   // test for end of DO loop or DELETE target.
   if ((flgs & (DL_FLAG + UD_FLAG)) == 0 and
      (flgs & DO_FLAG) == 0) goto ok_lab;
   // any GOTO type jump?
   if (flgs & GO_FLAG){ // unable to process
      printf("Unable to process multiple assign varaiables "
         " at line %li\n",c_line_num);
      char warn_m[]= "C** Unable to process multiple assigned "
      " variables **  ";
      out_line(warn_m);
      return FALSE;
   }
   con_lab = curr_lab;
   curr_lab = 0;
ok_lab:;
   return TRUE;
}
