// HexEditMacro.h : header file for keystroke macro structure

// Copyright (c) 1999 by Andrew W. Phillips.
//
// No restrictions are placed on the noncommercial use of this code,
// as long as this text (from the above copyright notice to the
// disclaimer below) is preserved.
//
// This code may be redistributed as long as it remains unmodified
// and is not sold for profit without the author's written consent.
//
// This code, or any part of it, may not be used in any software that
// is sold for profit, without the author's written consent.
//
// DISCLAIMER: This file is provided "as is" with no expressed or
// implied warranty. The author accepts no liability for any damage
// or loss of business that this product may cause.
//

enum km_type
{
    km_unknown = 0,             // Help detect errors
    km_new,                     // Create new file
    km_open,                    // Open existing file
    km_print_setup,             // Print setup
    km_exit,                    // Exit program

    km_find_text,               // Set find text (in Find Tool)
    km_toolbar,                 // Toggle toolbar
    km_editbar,                 // Toggle edit bar
    km_bar,                     // Toggle other bar (inc. status bar)
//    km_options,               // Options dlg no longer invoked in macros
    km_prop,                    // Properties dialog
    km_prop_file,               // Change to the file page
    km_prop_char,               // Change to the char page
    km_prop_dec,                // Change to the decimal values page
    km_prop_float,              // Change to the floating point page
    km_prop_ibmfp,              // Change to the IBM floating point page
    km_prop_checksum,           // Change to the checksum page (not implemented)
    km_prop_close,              // Close properties dialog
    km_calc_dlg,                // Invoke the goto/calculator dialog (see also km_goto)
    km_calc_close,              // Close the calculator
    km_help,                    // F1 pressed (help)
    km_topics,                  // Help Topics
    km_tips,                    // Tip of the day
    km_about,                   // About dialog (not implemented)

    km_focus,                   // Switch view (mouse click or windows menu)
    km_mainsys,                 // System command (mainframe control menu)
    km_childsys,                // System command (child frame control menu)

    //---------------------------------------------------------
    // The following are generated by the calculator.  The gap was left
    // so that new commands can be added later and not intermixed,
    // without changing any existing enum values.  This means existing
    // macro files (when we have them) won't be invalidated.
    // There is also a 2nd group of calculator operations (km_go etc)
    // that require an active view (hence they are past km_view).
    km_calc = 50,               // First calc value (not used)

    // Calculator misc operations
    km_result = 50,             // Used to signal that the value is not to be stored, which
                                // happens if the value is the result of a binary or unary
                                // operation or it was entered before recording started.
                                // Note: this value is never actually stored in a macro.
    km_user,                    // User entered value (stored in v64)
    km_clear,                   // Clear calculator operands and operator
    km_clear_entry,             // Clear current operand
    km_equals,                  // Evaluate result (equals key)

    // Change modes
    km_change_bits,             // Change the number of bits displayed (and used for calcs)
    km_change_base,             // Change the radix for display (also affects results since decimal means signed operations)
    km_endian,                  // Toggle endianness (only affects operations that read/write file)

    // Operations (the actual op number is stored in vv)
    km_binop,                   // Binary operation
    km_unaryop,                 // Unary operation

    // Calculator memory operations (store etc)
    km_memget,                  // Get current value of calculator memory
    km_memstore,                // Store into calc memory
    km_memclear,                // Clear calc memory
    km_memadd,                  // Add to calc memory
    km_memsubtract,             // Subtract from calc memory


    //-------------------------------------------------
    km_view = 100,              // All past this point require a view
    km_close,                   // File Close
    km_save,                    // File Save
    km_saveas,                  // File Save As
    km_print,                   // Print - bring up print dlg
    km_preview,                 // Print preview

    km_undo,                    // Undo
    km_redo,                    // Redo (not implemented)
    km_cut,                     // Cut
    km_copy,                    // Copy
    km_paste,                   // Paste
    km_del,                     // Delete character or selection
    km_copy_hex,                // Copy as hex text
    km_copy_decimal,            // Copy as decimal text (not implemented)
    km_copy_binary,             // Copy as binary text (not implemented)
    km_copy_cchar,              // Copy as C source (bytes)
    km_copy_cstring,            // Copy as a C string (not implemented)
    km_copy_cword,              // Copy as C words (not implemented)
    km_copy_cdword,             // Copy as C double words (not implemented)
    km_copy_cqword,             // Copy as C quad words (not implemented)
    km_copy_cfloat,             // Copy as C floats (not implemented)
    km_copy_cdouble,            // Copy as C doubles (not implemented)
    km_paste_ascii,             // Paste as ASCII
    km_paste_unicode,           // Paste as Unicode
    km_paste_ebcdic,            // Paste as EBCDIC
    km_sel_all,                 // Select whole file

    km_write_file,              // Write selection to file
    km_append_file,             // Append selection to a file (not implemented)
    km_read_file,               // Read file & leave selected
    km_find_dlg,                // Invoke the Find dialog
    km_find_close,              // Close the Find dialog
    km_find_back,               // Repeat find backwards (SF3)
    km_find_forw,               // Repeat find forwards (F3)
    km_find_sel,                // Find current selection (^F3)

    km_goto,                    // Invoke calculator in goto mode
    km_ro_rw,                   // Toggle RO/RW
    km_ovr_ins,                 // Toggle OVR/INS
    km_mark_pos,                // Mark position
    km_goto_mark,               // Goto mark
    km_extendto_mark,           // Extend selection to mark
    km_swap_mark,               // Swap caret with mark
    km_highlight,               // Highlighter (not implemented)

    km_aerial,                  // Show aerial view (not implemented)
    km_autofit,                 // Toggle autofit
    km_font,                    // Font dialog
    km_inc_font,                // Increase font
    km_dec_font,                // Decrease font
    km_addr,                    // Toggle hex/decimal addresses
    km_char_area,               // Toggle char area
    km_ebcdic,                  // Toggle ASCII/EBCDIC
    km_control,                 // Change display of control chars (3 ways)
    km_control2,                // Toggle display of control chars (2 ways)
    km_graphic,                 // Toggle display of graphic chars
    km_oem,                     // Toggle between ANSI/OEM graphic chars

    km_rowsize,                 // Change number of columns
    km_group_by,                // Change column grouping
    km_offset,                  // Change column of first byte

    km_win_new,                 // Open new window on active file
    km_win_next,                // Move to next non-minimized window
    km_win_cmd,                 // Handles window menu commands: cascade, tile, arrange

    km_redraw,                  // Redraw and scroll to centre
    km_scroll_up,               // Scroll up
    km_scroll_down,             // Scroll down
    km_swap_areas,              // Swap between hex and character areas
    km_start_line,              // Change offset so that cursor is at start of line

    km_key,                     // Key (from OnKeyDown event)
    km_char,                    // Normal char + control chars
    km_mouse,                   // Mouse click/drag
    km_shift_mouse,             // Mouse click/drag shift key down
    km_hscroll,                 // Scroll window horizontally
    km_vscroll,                 // Scroll window vertically

    km_address_tool,            // Hex/decimal address tool used

    // The following are tools.  This is likely to be added to extensively.
    km_compare,                 // Compare windows
    km_inc8,                    // Increment byte
    km_inc16,                   // Increment word
    km_inc32,                   // Increment double word
    km_inc64,                   // Increment quad word
    km_dec8,                    // Decrement byte
    km_dec16,                   // Decrement word
    km_dec32,                   // Decrement double word
    km_dec64,                   // Decrement quad word
    km_flip16,                  // Swap bytes of word
    km_flip32,                  // Swap bytes of double word
    km_flip64,                  // Swap bytes of quad word

    km_rs_forw,                 // Encom custom format handling
    km_rs_back,                 // Encom custom format handling

    // Calculator file operations
    km_go,                      // Move caret and move focus from calc to edit view
    km_markstore,               // Set the mark using the current calc value
    km_markclear,               // Set mark to beginning of file
    km_markadd,                 // Add calc value to mark address
    km_marksubtract,            // Subtract cal value from mark address
    km_markatstore,             // Store calc value to file at the current mark position
    km_selstore,                // Move current caret position to current calc value
    km_selatstore,              // Store calc value to file at the current caret position
    km_selendstore,             // Set end of current selection (must be >= sel start && <= eof)
    km_sellenstore,             // Set length of selection (must be >= 0 && <= dist to EOF)

    // Where is a calculator value obtained from?
    // Note: km_user should also be in this group but as it does not require the presence
    // of an active view it appears with the calculator group before km_view (above).
    km_markget,                 // Get current address of mark
    km_markat,                  // Get value from file at mark
    km_bookmark,                // Get address of named bookmark (bookmark name is stored in *pss)
    km_bookmarkat,              // Get value at named bookmark (bookmark name is stored in *pss)
    km_selget,                  // Get current address of caret/start of selection
    km_selat,                   // Get current value from file at caret
    km_selend,                  // Get current address of end of selection
    km_sellen,                  // Get length of selection
    km_eofget,                  // Get length of file

    km_last = 255
};

struct mouse_sel
{
    CPoint dev_down;            // Mouse down (device coordinates)
    CSize doc_dist;             // Distance to mouse up (doc coords)
};

#pragma pack(1)                 // Make sure macros fit into as little memory as possible
struct key_macro                // Stores one keystroke of the macro
{
//    km_type ktype;              // Type of keystroke
    unsigned char ktype;        // Type of "key" stored in a byte
    union
    {
        unsigned __int64 v64;   // Address or calc. value
        long vv;                // Character/key/flags etc
        CString *pss;           // Search string, file name etc
        LOGFONT *plf;           // Font attributes when font changed
        mouse_sel *pms;         // Mouse coords (km_mouse & km_shift_mouse)
    };

    // General constructor + default constructor (required by vector)
    key_macro(enum km_type k = km_unknown, unsigned __int64 v = 0)
        { ktype = k; v64 = v; }
// Note to avoid ambiguous calls we had to get rid of the constructor below.  But this may cause
// portability problems since some keys values are now saved via v64 but replayed using vv.
// Because of the way unions (and little-endianness) works for this compiler this works OK for now.
//    key_macro(enum km_type k, long v = 0L)
//        { ktype = k; vv = v; }
    key_macro(enum km_type k, CString &ss)
        { ktype = k; pss = new CString(ss); }
    key_macro(enum km_type k, LOGFONT *p)
        { ktype = k; plf = new LOGFONT; *plf = *p; }
    key_macro(enum km_type k, mouse_sel *p)
        { ktype = k; pms = new mouse_sel; *pms = *p; }

    // Destructor
    ~key_macro()
    {
        ASSERT(ktype != km_unknown);

        // Delete any allocated memory for specific km_type here
        if (ktype == km_find_text || ktype == km_open || ktype == km_read_file || ktype == km_focus)
            delete pss;
        else if (ktype == km_font)
            delete plf;
        else if (ktype == km_mouse || ktype == km_shift_mouse)
            delete pms;
    }

    // Copy constructor
    key_macro(const key_macro &from)
    {
        ASSERT(from.ktype != km_unknown);
        ktype = from.ktype;
        if (ktype == km_find_text || ktype == km_open || ktype == km_read_file || ktype == km_focus)
            pss = new CString(*from.pss);
        else if (ktype == km_font)
        {
            plf = new LOGFONT; *plf = *from.plf;
        }
        else if (ktype == km_mouse || ktype == km_shift_mouse)
        {
            pms = new mouse_sel; *pms = *from.pms;
        }
        else
            v64 = from.v64;               // Copy long by default (most used)
        // Dealloc/realloc memory for specific km_type here
    }
    // Assignment copy operator
    key_macro &operator=(const key_macro &from)
    {
        if (&from != this)
        {
            ASSERT(from.ktype != km_unknown);

            // Dealloc/realloc memory for existing km_type
            if (ktype == km_find_text || ktype == km_open || ktype == km_read_file || ktype == km_focus)
                delete pss;
            else if (ktype == km_font)
                delete plf;
            else if (ktype == km_mouse || ktype == km_shift_mouse)
                delete pms;

            // Allocate memory for new object and copy over
            ktype = from.ktype;
            if (ktype == km_find_text || ktype == km_open || ktype == km_read_file || ktype == km_focus)
                pss = new CString(*from.pss);
            else if (ktype == km_font)
            {
                plf = new LOGFONT; *plf = *from.plf;
            }
            else if (ktype == km_mouse || ktype == km_shift_mouse)
            {
                pms = new mouse_sel; *pms = *from.pms;
            }
            else
                v64 = from.v64;           // Copy long by default (most used)
        }
        return *this;
    }
    // These are required by vector<> under VC++ 5 even though not used
    operator==(const key_macro &) const { return false; }
    operator<(const key_macro &) const { return false; }
};
#pragma pack()
