/* grep.c - main driver file for grep.
   Copyright (C) 1992 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   Written July 1992 by Mike Haertel.  

Modified for Windows interface Donald Munro 1996 */

#pragma warning(disable : 4018) //signed/unsigned mismatch

#define HAVE_WORKING_MMAP  // Undefine this to not use Win32 Memory Mapping
#include "stdafx.h"
//#include <windows.h>
#include <errno.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <io.h>
#ifndef _WIN32
#include "getpagesize.h"
#endif

#include "grep.h"
#include "GrepView.h"
#include "kwset.h"

#undef MAX
#define MAX(A,B) ((A) > (B) ? (A) : (B))

typedef char * caddr_t;

extern int lastexact;

#define VOID void
    
/* Define flags declared in grep.h. */
char *matcher;
int match_icase;
int match_words;
int match_lines;
int count_matches;
int status;
int list_files;
int suppress_errors;

int initialized =FALSE;

GrepDisplayCallback g_GrepCallback;
CGrepView *g_pView =NULL;

/* Functions we'll use to search. */
static void (*compile)(char *, size_t);
static char *(*execute)(char *, size_t, char **);

void RegFree(); // In search.cpp

/* For error messages. */
static char *prog;
static int errseen;

extern kwset_t kwset;

#ifdef _WIN32
HFILE hFile;
OFSTRUCT ReOpenBuff;
HANDLE hMapAddr;
caddr_t pMappedAddress;

long getpagesize()
//-----------------
{	SYSTEM_INFO sysInfo;
	GetSystemInfo(&sysInfo);
	return sysInfo.dwPageSize;
}
#endif

/* Print a message and possibly an error string.  Remember
   that something awful happened. */
static void error(const char *mesg, int errnum =0)
//-------------------------------------------------
{	CString strErr;
	if (errnum)					 
		strErr.Format("%s: %s: (%d)", prog, mesg, errnum);
	else
		strErr.Format("%s: %s: (%d)", prog, mesg, GetLastError());
	errseen = 1;
	(*g_GrepCallback)(g_pView, strErr);
}

extern "C"
	{	BOOL CreatePrivateHeap();
		BOOL DestroyPrivateHeap();
		char *xmalloc(size_t size);
		char *xrealloc(char *ptr, size_t size);
		void xfree(char *ptr);
		void fatal(char *szErr, int nErrNo);
	}

#define malloc xmalloc
#define realloc xrealloc
#define valloc xmalloc
#define free xfree

/* Hairy buffering mechanism for grep.  The intent is to keep
   all reads aligned on a page boundary and multiples of the
   page size. */

static char *buffer;		/* Base of buffer. */
static size_t bufsalloc;	/* Allocated size of buffer save region. */
static size_t bufalloc;		/* Total buffer size. */
static int bufdesc;		/* File descriptor. */
static char *bufbeg;		/* Beginning of user-visible stuff. */
static char *buflim;		/* Limit of user-visible stuff. */

#if defined(HAVE_WORKING_MMAP) 
static int bufmapped;		/* True for ordinary files. */
static struct stat bufstat;	/* From fstat(). */
static off_t bufoffset;		/* What read() normally remembers. */
#endif

/* Reset the buffer for a new file.  Initialize
   on the first time through. */
void reset(int fd, CString strPath)
//---------------------------------
{	if (!initialized)
		{	initialized = TRUE;
#ifndef BUFSALLOC
			bufsalloc = MAX(8192, getpagesize());
#else
			bufsalloc = BUFSALLOC;
#endif
			bufalloc = 5 * bufsalloc;
/* The 1 byte of overflow is a kludge for dfaexec(), which
inserts a sentinel newline at the end of the buffer
being searched.  There's gotta be a better way... */
			buffer = (char *) valloc(bufalloc + 1);
			if (!buffer)
				fatal("memory exhausted", 0);
			bufbeg = buffer;
			buflim = buffer;
		}
	bufdesc = fd; 
#if defined(HAVE_WORKING_MMAP)
#ifdef _WIN32
#define S_ISREG(mode)  ((mode&0XF000) == 0X8000)
#endif
	if ( (fstat(fd, &bufstat) < 0) || (!S_ISREG(bufstat.st_mode)) )
		bufmapped = 0;
	else
		{ bufmapped = 1;
			bufoffset = lseek(fd, 0, 1);
		}
#ifdef _WIN32
	if (pMappedAddress != NULL)
		{	UnmapViewOfFile(pMappedAddress);
			pMappedAddress = NULL;
		}
	if (hMapAddr != NULL)
		{	CloseHandle(hMapAddr);
			hMapAddr = NULL;
		}
	hMapAddr = CreateFileMapping((HANDLE)hFile, NULL, PAGE_READONLY,0,0,NULL);
	if (hMapAddr != NULL)
		pMappedAddress = (caddr_t)MapViewOfFile(hMapAddr,FILE_MAP_READ,0,0,0);
	else
		TRACE2("Error %d opening file %s for mapping\n",GetLastError(),strPath);
#endif
#endif
}

/* Read new stuff into the buffer, saving the specified
   amount of old stuff.  When we're done, 'bufbeg' points
   to the beginning of the buffer contents, and 'buflim'
   points just after the end.  Return count of new stuff. */
static int fillbuf(size_t save)
//-----------------------------
{	char *nbuffer, *dp, *sp;
	int cc;
#if defined(HAVE_WORKING_MMAP)
	caddr_t maddr;
#ifdef _WIN32
	size_t sizeCopy;
#endif
#endif
	static int pagesize;

	if (pagesize == 0 && (pagesize = getpagesize()) == 0)
		abort();

	if (save > bufsalloc)
		{	while (save > bufsalloc)
			bufsalloc *= 2;
			bufalloc = 5 * bufsalloc;
			nbuffer = (char *) valloc(bufalloc + 1);
			if (!nbuffer)
				fatal("memory exhausted", 0);
		}
	else
		nbuffer = buffer;

	sp = buflim - save;
	dp = nbuffer + bufsalloc - save;
	bufbeg = dp;
	while (save--)
		*dp++ = *sp++;

/* We may have allocated a new, larger buffer.  Since
there is no portable vfree(), we just have to forget
about the old one.  Sorry. */
	buffer = nbuffer;

#if defined(HAVE_WORKING_MMAP)
#ifdef _WIN32
	if (hMapAddr != NULL && pMappedAddress != NULL)
		{	sizeCopy = bufalloc - bufsalloc;
			if ( ((long)(bufoffset + sizeCopy)) >= bufstat.st_size)
				sizeCopy = bufstat.st_size - bufoffset;
			memcpy(buffer + bufsalloc,pMappedAddress + bufoffset, sizeCopy);
			cc = sizeCopy;
			bufoffset += cc;
		}
//else will do tryread below
#else
	if (bufmapped && bufoffset % pagesize == 0
			&& bufstat.st_size - bufoffset >= bufalloc - bufsalloc)
		{	maddr = buffer + bufsalloc;
			maddr = mmap(maddr, bufalloc - bufsalloc, PROT_READ | PROT_WRITE,
										MAP_PRIVATE | MAP_FIXED, bufdesc, bufoffset);
			if (maddr == (caddr_t) -1)
				{	strerror(errno));
					goto tryread;
				}

#if 0
/* You might thing this (or MADV_WILLNEED) would help,
but it doesn't, at least not on a Sun running 4.1.
In fact, it actually slows us down about 30%! */
			madvise(maddr, bufalloc - bufsalloc, MADV_SEQUENTIAL);
#endif
			cc = bufalloc - bufsalloc;
			bufoffset += cc;
		}
#endif // _WIN32
else
	{
tryread:
/* We come here when we're not going to use mmap() any more.
Note that we need to synchronize the file offset the
first time through. */
		if (bufmapped)
			{	bufmapped = 0;
				lseek(bufdesc, bufoffset, 0);
			}
		cc = read(bufdesc, buffer + bufsalloc, bufalloc - bufsalloc);
	}
#else
	cc = read(bufdesc, buffer + bufsalloc, bufalloc - bufsalloc);
#endif
	if (cc > 0)
		buflim = buffer + bufsalloc + cc;
	else
		buflim = buffer + bufsalloc;
	return cc;
}

/* Flags controlling the style of output. */
static int out_quiet;		/* Suppress all normal output. */
static int out_invert;		/* Print nonmatching stuff. */
static int out_file;		/* Print filenames. */
static int out_line;		/* Print line numbers. */
static int out_byte;		/* Print byte offsets. */
static int out_before;		/* Lines of leading context. */
static int out_after;		/* Lines of trailing context. */

/* Internal variables to keep track of byte count, context, etc. */
static size_t totalcc;		/* Total character count before bufbeg. */
static char *lastnl;		/* Pointer after last newline counted. */
static char *lastout;		/* Pointer after last character output;
				   NULL if no character has been output
				   or if it's conceptually before bufbeg. */
static size_t totalnl;		/* Total newline count before lastnl. */
static int pending;		/* Pending lines of output. */

static void nlscan(char *lim)
//---------------------------
{ char *beg;

  for (beg = lastnl; beg < lim; ++beg)
    if (*beg == '\n')
      ++totalnl;
  lastnl = beg;
}

static void prline(char *beg, char *lim, char sep, CString strPath ="", 
									 CString strLine ="")
//---------------------------------------------------------------------
{ nlscan(beg);

	strLine.Format("%s%c%d%c", strPath, sep, ++totalnl, sep);
	lastnl = lim;
	//if (out_byte)
		{	CString strByte;
			strByte.Format("%lu%c", totalcc + (beg - bufbeg), sep);
			strLine += strByte;
		}
	int i = strLine.GetLength();
	char *pch = strLine.GetBuffer(strLine.GetLength()+(int)(lim - beg) + 10);
	char *p;
	for (p=beg; p<lim; p++)
		pch[i++] = *p;
	pch[i] = 0;
	strLine.ReleaseBuffer();
	(*g_GrepCallback)(g_pView, strLine);
	lastout = lim;
}

/* Print pending lines of trailing context prior to LIM. */
static void prpending(char *lim, CString strPath ="")
//-----------------------------------------------
{ char *nl;

	if (!lastout)
		lastout = bufbeg;
	while (pending > 0 && lastout < lim)
		{	--pending;
			if ((nl = (char *)memchr(lastout, '\n', lim - lastout)) != 0)
				++nl;
			else
				nl = lim;
			prline(lastout, nl, '|',strPath); //-
		}
}

/* Print the lines between BEG and LIM.  Deal with context crap.
   If NLINESP is non-null, store a count of lines between BEG and LIM. */
static void prtext(char *beg, char *lim, int *nlinesp, CString strPath)
//---------------------------------------------------------------------
{	static int used;		/* avoid printing "--" before any output */
	char *bp, *p, *nl;
	int i, n;

	if (!out_quiet && pending > 0)
		prpending(beg, strPath);

	p = beg;

	if (!out_quiet)
		{	/* Deal with leading context crap. */
			CString strLine ="";
			bp = lastout ? lastout : bufbeg;
			for (i = 0; i < out_before; ++i)
				if (p > bp)
					do
						--p;
					while (p > bp && p[-1] != '\n');

			/* We only print the "--" separator if our output is
				discontiguous from the last output in the file. */
			if ((out_before || out_after) && used && p != lastout)
				strLine = "--";

			while (p < beg)
				{	nl = (char *)memchr(p, '\n', beg - p);
					prline(p, nl + 1, '|',strPath,strLine); //-
					p = nl + 1;
				}
		}

	if (nlinesp)
		{
	/* Caller wants a line count. */
			for (n = 0; p < lim; ++n)
				{	if ((nl = (char *)memchr(p, '\n', lim - p)) != 0)
						++nl;
					else
						nl = lim;
					if (!out_quiet)
						prline(p, nl, '|',strPath);
					p = nl;
				}
			*nlinesp = n;
		}
	else
		if (!out_quiet)
			prline(beg, lim, '|',strPath);

	pending = out_after;
	used = 1;
}

/* Scan the specified portion of the buffer, matching lines (or
   between matching lines if OUT_INVERT is true).  Return a count of
   lines printed. */
static int grepbuf(char *beg, char *lim, CString strPath)
//-------------------------------------------------------
{	int nlines, n;
	register char *p, *b;
	char *endp;

	nlines = 0;
	p = beg;
	while ((b = (*execute)(p, lim - p, &endp)) != 0)
		{
			/* Avoid matching the empty line at the end of the buffer. */
			if (b == lim && ((b > beg && b[-1] == '\n') || b == beg))
				break;
			if (!out_invert)
				{	prtext(b, endp, (int *) 0, strPath);
					nlines += 1;
				}
			else 
				if (p < b)
					{	prtext(p, b, &n, strPath);
						nlines += n;
					}
			p = endp;
		}
	if (out_invert && p < lim)
		{	prtext(p, lim, &n, strPath);
			nlines += n;
		}
	return nlines;
}

/* Search a given file.  Return a count of lines printed. */
static int grep(int fd, CString strPath)
//--------------------------------------
{	int nlines, i;
	size_t residue, save;
	char *beg, *lim;

	reset(fd, strPath);

	totalcc = 0;
	lastout = 0;
	totalnl = 0;
	pending = 0;

	nlines = 0;
	residue = 0;
	save = 0;

	for (;;)
		{	if (fillbuf(save) < 0)
				{	error(strPath, errno);
					return nlines;
				}
			lastnl = bufbeg;
			if (lastout)
				lastout = bufbeg;
			if (buflim - bufbeg == save)
				break;
			beg = bufbeg + save - residue;
			for (lim = buflim; lim > beg && lim[-1] != '\n'; --lim)
				;
			residue = buflim - lim;
			if (beg < lim)
				{	nlines += grepbuf(beg, lim, strPath);
					if (pending)
						prpending(lim, strPath);
				}
			i = 0;
			beg = lim;
			while (i < out_before && beg > bufbeg && beg != lastout)
				{	++i;
					do
						--beg;
					while (beg > bufbeg && beg[-1] != '\n');
				}
			if (beg != lastout)
				lastout = 0;
			save = residue + lim - beg;
			totalcc += buflim - bufbeg - save;
			//if (out_line)
			nlscan(beg);
		}
	if (residue)
		{	nlines += grepbuf(bufbeg + save - residue, buflim, strPath);
			if (pending)
				prpending(buflim, strPath);
		}
	return nlines;
}


/* Go through the matchers vector and look for the specified matcher.
   If we find it, install it in compile and execute, and return 1.  */
int	setmatcher(char *name)
//------------------------
{	int i;

	for (i = 0; matchers[i].name; ++i)
		if (strcmp(name, matchers[i].name) == 0)
			{	compile = matchers[i].compile;
				execute = matchers[i].execute;
				return 1;
			}
	return 0;
}  

int Grep(CGrepView *pView, GrepDisplayCallback GrepCallback)
//----------------------------------------------------------
{ char *keys;
	size_t keycc;
	int keyfound, no_filenames;
	
	// Added by D Munro.  There seem to be lots of memory leaks somewhere in the C
	// code so we create a Private heap to allocate from and then destroy it
	// when finished.  All the code allocating and using the heap is in regex.c
	// malloc, realloc and free are #defined to use xmalloc etc in all the C files.
	if (! CreatePrivateHeap())
		{ AfxMessageBox("ERROR : Alocating Heap");
			return FALSE;
		}
	g_GrepCallback = GrepCallback;
	g_pView = pView;
	pMappedAddress = NULL;
	hMapAddr = NULL;
	hFile = 0;

	keys = NULL;
	keycc = 0;
	keyfound = 0;
	count_matches = 0;
	no_filenames = 0;
	list_files = 0;
	suppress_errors = 0;
	matcher = NULL;
	initialized = FALSE; //Added DM
	lastexact = 0;  // Added DM

	out_after = atoi(pView->m_strLinesAfter);
	out_before = atoi(pView->m_strLinesBefore);
	if (pView->m_strGrep == "GNU E-Grep")
		matcher = "grep";
	else
		if (pView->m_strGrep == "POSIX E-Grep")
			matcher = "posix-egrep";
		else
			matcher = "fgrep";		
	TRACE1("Matcher = %s\n",matcher);
	out_byte = pView->m_bOutputByte;
	if (pView->m_bCountMatchesOnly)
		{	out_quiet = 1;
			count_matches = 1;
		}
	match_icase = (! pView->m_bMatchCase);
	if (pView->m_bNamesOnly)
		{	out_quiet = 1;
			list_files = 1;
		}
	if (pView->m_bNoMatchFiles)
		{	out_quiet = 1;
			list_files = -1;
		}
	suppress_errors = pView->m_bSuppressErrors;
	out_invert = pView->m_bNonMatching;
	match_words = pView->m_bMatchWord;
  match_lines = pView->m_bMatchLine;

	// Remove all \r for FGREP 
	CString strPattern;
	strPattern = "";
	if (strcmp(matcher,"fgrep") == 0)	
		for (int i=0; i<pView->m_strPattern.GetLength(); i++)
			if (pView->m_strPattern[i] != '\r')
				strPattern += pView->m_strPattern[i];
			else;
	else
		for (int i=0; i<pView->m_strPattern.GetLength(); i++)
			if ( (pView->m_strPattern[i] == '\r') || (pView->m_strPattern[i] == '\n') )
				break;
			else
				strPattern += pView->m_strPattern[i];
			

	keys = (char *) ((const char *) strPattern);
	keycc = strlen(keys);

  if (!setmatcher(matcher) && !setmatcher("default"))
		abort();
	
	(*compile)(keys, keycc);

	status = 1;
	
	pView->m_bStopGrep = FALSE;
	pView->m_bGrepping = TRUE;
	CString strDir;
	for (int nDirIndex=0; nDirIndex<pView->m_listDirectories.GetCount(); nDirIndex++)
		{	pView->m_listDirectories.GetText(nDirIndex,strDir);
			DoGrep(strDir, pView, GrepCallback);
		}

	// DM 19/07/1996
	if (kwset != NULL)
		{ kwsfree(kwset);
			kwset = NULL;
		}

	DestroyPrivateHeap();
	return(errseen ? 2 : status);
}

void DoGrep(CString strDir, CGrepView *pView, GrepDisplayCallback GrepCallback)
//-----------------------------------------------------------------------------
{	strDir.TrimLeft(); strDir.TrimRight();
	SetCurrentDirectory(strDir);
	CFileFind DirData;
	BOOL bStop = FALSE;
	int desc;
	
	MSG msg;
	for (int nSpec=0; nSpec < pView->m_strarrFileSpecs.GetSize(); nSpec++)
		{	CString strSpec = pView->m_strarrFileSpecs[nSpec];
			if (! DirData.FindFile(strSpec))
				{	DirData.Close();
					continue;
				}
			DirData.FindNextFile();
			while (1)
				{	while (PeekMessage(&msg,NULL,NULL,NULL,PM_REMOVE))
						DispatchMessage(&msg);
					if (pView->m_bStopGrep) return;
					if (DirData.MatchesMask(FILE_ATTRIBUTE_ARCHIVE | FILE_ATTRIBUTE_COMPRESSED | FILE_ATTRIBUTE_NORMAL | FILE_ATTRIBUTE_READONLY | FILE_ATTRIBUTE_HIDDEN | FILE_ATTRIBUTE_SYSTEM | FILE_ATTRIBUTE_TEMPORARY))
						{	desc = _open(DirData.GetFilePath(), _O_RDONLY);
							if (desc < 0)
								{	if (!suppress_errors)
										error(DirData.GetFilePath(), errno);
								}
							else
								{	int count = grep(desc, DirData.GetFilePath());
									CString strMatches;
									if (count_matches)
										{	strMatches.Format("%s|%d", DirData.GetFilePath(),count);
											(*GrepCallback)(g_pView, strMatches);
										}
									if (count)
										{	status = 0;
											if (list_files == 1)
												{	strMatches.Format("%s", DirData.GetFilePath());
													(*GrepCallback)(g_pView, strMatches);
												}
										}
									else 
										if (list_files == -1)
											{	strMatches.Format("%s", DirData.GetFilePath());
												(*GrepCallback)(g_pView, strMatches);
											}
								}
							if (desc != 0)
								_close(desc);									
						}
					if (! DirData.FindNextFile())
						{	if (bStop) break;
							bStop = TRUE;
						}
				}
		}

	DirData.Close();
	if (! DirData.FindFile())
		{	DirData.Close();
			return;
		}
	bStop = FALSE;
	DirData.FindNextFile();
	while (1)
		{	while (PeekMessage(&msg,NULL,NULL,NULL,PM_REMOVE))
				DispatchMessage(&msg);

			if (! DirData.IsDots())
				if (DirData.IsDirectory())
					if (! pView->m_bStopGrep)
						if (pView->m_bRecurseDirectories)
							{	DoGrep(DirData.GetFilePath(), pView, GrepCallback);
								SetCurrentDirectory(strDir);
							}
						else;
					else;
				else;
			else;
			if (! DirData.FindNextFile())
				{	if (bStop) break;
					bStop = TRUE;
				}
		}
	DirData.Close();
	return;
}
