unit MainForm;

{
WinTidy - program to clear out unwanted files

Revision history:

1995 Apr 23  1.0.0  First version derived from DOS-based TIDY
                    and WIN_TIDY programs.  Takes command line
                    parameter for root directory.
1995 Apr 25  1.0.2  Use proper Screen.Cursor for hourglass
                    Make caption reflect current scan location
1995 Nov 25  1.0.4  Add .RWS as unwanted file, Borland Resource Workshop binary
                    Replace two list boxes by one with size integral
                    Order files by size
                    Add Select All button
1995 Dec 02  1.0.6  Add .WBK as unwanted file, WinWord 7 backup
1996 Jan 06  1.0.8  Only enable Delete button when relevant
                    Make Find button have a Cancel function
                    Use Delphi's own LowerCase function
1996 Mar 30  2.0.0  Delphi 2.0 32-bit version
                    Allow limited form re-sizing
                    Add .APS as unwanted file, Vis C++ binary saved resources
                    Remove hour-glass cursor
                    Add indication of bytes found for deleting
                    Note that .MOZ and other cache files might be candidates
1996 Apr 16  2.0.2  Add severity button, find .FTS, .GID
                    Add .DMP files to normal list
                    Add status bar
1996 Sep 19  2.0.4  Update btnSelectAllClick to use more efficient method
1997 Mar 16  2.1.0  Suggestions from John Weersing (temlib@iinet.com)
                    Option to ignore zero-length, ReadOnly, Hidden & System files
                    Multiple drive search capability,
                    still uses directory to filter on each individual drive
                    Add -auto_delete command-line option - exact case required
                    Display file attributes
                    Allow for -keep= and -delete= parameters
                    Remove case formatting for file and directory names
                    Store unwanted file name lists in the registry
                    Normally use the Recycle Bin for delete
                    Check for Win95 or NT 4 minimum OS
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, FileCtrl, ExtCtrls, Buttons, ComCtrls, Menus, Registry;

type
  TForm1 = class(TForm)
    Panel1: TPanel;
    Panel2: TPanel;
    DriveComboBox1: TDriveComboBox;
    btnFind: TButton;
    ListBox1: TListBox;
    btnSelectAll: TButton;
    btnDelete: TButton;
    btnExit: TButton;
    Label1: TLabel;
    Label2: TLabel;
    lblSize: TLabel;
    StatusBar1: TStatusBar;
    MainMenu1: TMainMenu;
    File1: TMenuItem;
    Exit1: TMenuItem;
    N1: TMenuItem;
    Find1: TMenuItem;
    lbxDrives: TListBox;
    Timer1: TTimer;
    Label3: TLabel;
    edtDirectory: TEdit;
    Help1: TMenuItem;
    About1: TMenuItem;
    Tools1: TMenuItem;
    Options1: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure btnExitClick(Sender: TObject);
    procedure Exit1Click(Sender: TObject);
    procedure btnFindClick(Sender: TObject);
    procedure About1Click(Sender: TObject);
    procedure Options1Click(Sender: TObject);
    procedure Find1Click(Sender: TObject);
    procedure btnSelectAllClick(Sender: TObject);
    procedure btnDeleteClick(Sender: TObject);
    procedure ListBox1Click(Sender: TObject);
    procedure Panel1Resize(Sender: TObject);
    procedure Panel2Resize(Sender: TObject);
    procedure Timer1Timer(Sender: TObject);
    procedure lbxDrivesDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure lbxDrivesExit(Sender: TObject);
    procedure FormActivate(Sender: TObject);
  private
    { Private declarations }
    initialised_ok: boolean;
    searching: boolean;
    stop_requested: boolean;
    files_found: integer;
    KB_found: integer;
    suggested_min_x: integer;
    suggested_min_y: integer;
    registry_data: TRegIniFile;
    unwanted_attributes: integer;
    find_extra_files: boolean;
    ignore_hidden_files: boolean;
    ignore_readonly_files: boolean;
    ignore_system_files: boolean;
    ignore_zero_length_files: boolean;
    auto_delete: boolean;
    use_recycle_bin: boolean;
    keep_list: TStringList;      // read from command-line, don't delete these
    delete_list: TStringList;    // read from command-line, _do_ delete these
    normal_names: TStringList;   // what to find on normal search
    extra_names: TStringList;    // what to find on extra search
    procedure scan_tree (root: string;  unwanted_list: TStringList);
    procedure fill_list_from_wildcard (list: TStringList;  wildcard: string);
  protected
    procedure GetMinMaxInfo (var info: TWMGetMinMaxInfo);  message WM_GETMINMAXINFO;
  public
    { Public declarations }
  end;

var
  Form1: TForm1;

implementation

uses ShellAPI, AboutFrm, Options;

{$R *.DFM}
{$R version.res}

// these for the About... box
const
  product_name = 'David''s WinTidy';
  product_version = 'Version 2.1.0';
  product_copyright = 'Copyright '#169' David J Taylor, Edinburgh, 1995-1997';
  product_comments = 'Author: david.taylor@gecm.com';

// used to pre-load the search lists (if not already defined in the registry)
const
  normal_list = '*.tmp,~*.*,*.~*,*.?~?,*.aps,*.bak,*.bk?,*.bsc,*.dmp,' +
                '*.dsm,*.ilk,*.pch,*.rws,*.sbr,backup*.wbk,*.$$$';
  extra_list = '*.fts,*.gid';

// registry locations in HKEY_CURRENT_USER
const
  registry_path = '\Software\David J Taylor\';
  prof_section = 'Windows Tidy';
  dir_mask_key = 'DirectoryMask';
  ignore_zero_length_key = 'IgnoreZeroLengthFiles';
  ignore_hidden_key = 'IgnoreHiddenFiles';
  ignore_readonly_key = 'IgnoreReadOnlyFiles';
  ignore_system_key = 'IgnoreSystemFiles';
  normal_names_key = 'FilesToFind';
  extra_names_key = 'ExtraFilesToFind';
  use_recycle_bin_key = 'UseRecycleBin';

procedure TForm1.FormCreate(Sender: TObject);
var
  params: string;
  param: integer;
  equals: integer;
  v: TOSversionInfo;     // to determine Windows version
begin
  initialised_ok := False;

  // set initial values for state variables
  searching := False;
  stop_requested := False;

  // and for the user choice variables
  find_extra_files := False;
  auto_delete := False;
  btnDelete.Enabled := False;
  btnSelectAll.Enabled := False;

  // get initial settings from the registry
  registry_data := TRegIniFile.Create (registry_path);
  with registry_data do
    begin
    edtDirectory.Text := ReadString (prof_section, dir_mask_key, '\');
    ignore_hidden_files := ReadBool (prof_section, ignore_hidden_key, True);
    ignore_readonly_files := ReadBool (prof_section, ignore_readonly_key, True);
    ignore_system_files := ReadBool (prof_section, ignore_system_key, True);
    ignore_zero_length_files := ReadBool (prof_section, ignore_zero_length_key, True);
    use_recycle_bin := ReadBool (prof_section, use_recycle_bin_key, True);
    end;

  // create lists of wildcards for file search, preloading from the registry
  // there's no real need to have these sorted, but doing so means that
  // duplicate entries in the registry will be ingored
  normal_names := TStringList.Create;
  with normal_names do
    begin
    Sorted := True;
    Duplicates := dupIgnore;
    CommaText := registry_data.ReadString (prof_section, normal_names_key, normal_list);
    end;

  extra_names := TStringList.Create;
  with extra_names do
    begin
    Sorted := True;
    Duplicates := dupIgnore;
    CommaText := registry_data.ReadString (prof_section, extra_names_key, extra_list);
    end;

  // create the Keep and Delete lists, which may be present on the command line
  // these lists have to be sorted to allow the TStringList.Find method to work.
  keep_list := TStringList.Create;
  with keep_list do
    begin
    Sorted := True;
    Duplicates := dupIgnore;
    end;

  delete_list := TStringList.Create;
  with delete_list do
    begin
    Sorted := True;
    Duplicates := dupIgnore;
    end;

  // we can't use shell file operations unless on Windows 95 or NT 4
  // so check the version of Windows we're running under
  FillChar (v, SizeOf (v), 0);
  v.dwOsVersionInfoSize := SizeOf (v);
  GetVersionEx (v);
  if (v.dwPlatformID = VER_PLATFORM_WIN32_WINDOWS) or   // we're on Win95 or
     ((v.dwPlatformID = VER_PLATFORM_WIN32_NT) and      // we're on Windows NT
     (v.dwMajorVersion >= 4))                           // and NT 4 at that
  then                                                  // it's OK to run
  else
    begin
    ShowMessage ('WinTidy requires at least Windows 95 or NT 4 to run.' + #13 +
                 'Your system does not appear to meet this requirement');
    Exit;   // give up trying to initialise
    end;

  // see if there was anything on the command line
  if ParamCount > 0 then
    begin
    params := ParamStr (1);
    edtDirectory.Text := params;  // command line value overwrites registry data
    if ParamCount > 1 then
      for param := 2 to ParamCount do
        begin
        params := LowerCase (ParamStr (param));
        if params = '-auto_delete' then auto_delete := True;
        equals := Pos ('-keep=', params);
        if equals = 1 then
          // fill the keep list with real file names (from wildcards)
          fill_list_from_wildcard (keep_list, Copy (Params, 7, 999));
        equals := Pos ('-delete=', params);
        if equals = 1 then
          // fill the delete list with real file names
          fill_list_from_wildcard (delete_list, Copy (Params, 9, 999));
        end;
    end;

  // to prevent the form being made too small, record the present size
  suggested_min_x := Width;
  suggested_min_y := Height;

  initialised_ok := True;

  // trigger the timer, will look at drives to select local hard disks
  // and trigger search if there are command-line parameters
  Timer1.Enabled := True;
end;

procedure TForm1.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  // save the current settings in the registry
  with registry_data do
    begin
    WriteString (prof_section, normal_names_key, normal_names.CommaText);
    WriteString (prof_section, extra_names_key, extra_names.CommaText);
    WriteString (prof_section, dir_mask_key, edtDirectory.Text);
    WriteBool (prof_section, ignore_hidden_key, ignore_hidden_files);
    WriteBool (prof_section, ignore_readonly_key, ignore_readonly_files);
    WriteBool (prof_section, ignore_system_key, ignore_system_files);
    WriteBool (prof_section, ignore_zero_length_key, ignore_zero_length_files);
    WriteBool (prof_section, use_recycle_bin_key, use_recycle_bin);
    end;
  registry_data.Free;

  keep_list.Free;
  delete_list.Free;
end;

procedure TForm1.Timer1Timer(Sender: TObject);
var
  i: integer;
  drive_root: string;      // e.g. A:\
  drive_type: integer;     // fixed, removable etc.
  drive_list: string;
begin
  // this is one-shot code, so stop the timer from repeating!
  Timer1.Enabled := False;

  // cheekily, steal the list of disks from the DriveComboBox
  // and use it to fill the drive list box
  drive_list := '';
  for i := 0 to DriveComboBox1.Items.Count-1 do
    begin
    lbxDrives.Items.AddObject (
      DriveComboBox1.Items.Strings [i], DriveComboBox1.Items.Objects [i]);
    drive_root := Copy (DriveComboBox1.Items.Strings [i], 1, 2);
    drive_type := GetDriveType (PChar (drive_root + '\'));
    if drive_type = DRIVE_FIXED then
      begin
      lbxDrives.Selected [lbxDrives.Items.Count-1] := True;
      drive_list := drive_list + UpperCase (drive_root) + ' ';
      end;
    end;

  // and tell the user what drives we will search by default
  Form1.Caption := 'WinTidy - ' + drive_list;

  // if a command-line paramter was specified, start the scan
  if ParamCount <> 0 then
    begin
    btnFind.Click;
    // and if the user _really_ trusts us, try and delete the files & exit
    if auto_delete then
      begin
      btnSelectAll.Click;
      btnDelete.Click;
      Close;
      end;
    end;
end;

procedure TForm1.fill_list_from_wildcard (list: TStringList;  wildcard: string);
// This procedure converts a multiple wildcard file specification
// (e.g. C:\temp\*.GIF,"c:\temp files\*.tmp")
// into a list a actual complete file names in the LIST parameter.
// Note that the TStringList CommaText property is used to do the parsing
var
  s: TSearchRec;
  element: integer;
begin
  with TStringList.Create do   // get a temporary list to parse the wildcard
    begin
    Sorted := True;
    Duplicates := dupIgnore;
    CommaText := wildcard;        // convert comma separated to a TStringList
    for element := 0 to Count - 1 do     // and loop over the parsed elements
      begin                              // doing a directory scan on each
      if FindFirst (Strings [element], faAnyFile, s) = 0 then
      repeat
        list.Add (ExtractFilePath (Strings [element]) + s.Name);
      until FindNext (s) <> 0;
      FindClose (s);
      end;
    Free;
  end;
end;

procedure TForm1.lbxDrivesExit(Sender: TObject);
// procedure to keep the main form caption up-to-date with the drive list
var
  i: integer;
  drive_list: string;
begin
  drive_list := '';
  for i := 0 to lbxDrives.Items.Count-1 do
    begin
    if lbxDrives.Selected [i] then
      drive_list :=
      drive_list + UpperCase (Copy (lbxDrives.Items.Strings [i], 1, 2)) + ' ';
    end;
  Form1.Caption := 'WinTidy - ' + drive_list;
end;

procedure TForm1.btnExitClick(Sender: TObject);
begin
  stop_requested := True;        // to stop the scanning loop
  Close;
end;

procedure TForm1.btnSelectAllClick(Sender: TObject);
begin
//  for item := 0 to ListBox1.Items.Count-1 do
//   ListBox1.Selected [item] := True;
  // following suggestion from Thomas Gravaard (jesus@diku.dk)
  SendMessage (ListBox1.Handle, LB_SETSEL, 1, -1); // select all items
  btnDelete.Enabled := ListBox1.SelCount <> 0;
end;

procedure TForm1.ListBox1Click(Sender: TObject);
begin
  btnDelete.Enabled := ListBox1.SelCount <> 0;
end;

procedure TForm1.btnFindClick(Sender: TObject);
var
  root: string;
  i: integer;
begin
  if searching
  then stop_requested := True
  else
    begin
    searching := True;
    btnFind.Caption := 'Stop';
    Find1.Caption := 'Stop';
    btnDelete.Enabled := False;
    btnSelectAll.Enabled := False;
    lblSize.Caption := '';
    Label1.Caption := '';
    ListBox1.Clear;

    // preload the delete list box with the user's -delete paramter
    for i := 0 to delete_list.Count - 1 do
      ListBox1.Items.Add (' -forced- ' + delete_list.Strings [i]);

    stop_requested := False;
    lbxDrives.Enabled := False;
    files_found := 0;
    KB_found := 0;
    unwanted_attributes := 0;

    // get a directory-only string from the directory mask edit control
    with edtDirectory do
      begin
      // ensure a trailing "/" is present
      if Text [Length (Text)] <> '\' then Text := Text + '\';
      // strip any drive specification
      if Text [2] = ':' then Text := Copy (Text, 3, 999);
      // and ensure that _something_ is present!
      if Text = '' then Text := '\';
      // convert it to a directory specification
      Text := ExtractFileDir (Text);
      end;

    // set the attributes mask from the user's options
    if ignore_readonly_files
      then unwanted_attributes := unwanted_attributes or faReadOnly;
    if ignore_system_files
      then unwanted_attributes := unwanted_attributes or faSysFile;
    if ignore_hidden_files
      then unwanted_attributes := unwanted_attributes or faHidden;

    // for each selected drive, search for matching files
    try
      for i := 0 to lbxDrives.Items.Count-1 do
        if lbxDrives.Selected [i] then
          begin
          root := UpperCase (Copy (lbxDrives.Items.Strings [i], 1, 2));
          root := root + edtDirectory.Text;
          if find_extra_files
            then scan_tree (root, extra_names)
            else scan_tree (root, normal_names);
          end;
    finally
    end;

    searching := False;
    lbxDrives.Enabled := True;
    btnFind.Caption := 'Find';
    Find1.Caption := 'Find';
    btnSelectAll.Enabled := True;
    if files_found <> 0
    then
      begin
      Label1.Caption := 'Candidates for deleting ....';
      StatusBar1.SimpleText := '';
      btnSelectAll.Enabled := True;
      end
    else
      begin
      Label1.Caption := '';
      StatusBar1.SimpleText := 'No files found';
      btnSelectAll.Enabled := False;
      end;
    end;
end;

procedure TForm1.scan_tree (root: string;  unwanted_list: TStringList);
var
  test_name: string;
  full_name: string;
  attributes: string;
  s: TSearchRec;
  unwanted: integer;
  index: integer;
  KB: integer;
begin
  if stop_requested then Exit;
  StatusBar1.SimpleText := 'Searching ' + root + '...';
  if root [Length (root)] <> '\' then root := root + '\';

  for unwanted := 0 to unwanted_list.Count - 1 do
    begin
    Application.ProcessMessages;
    test_name := root + unwanted_list.Strings [unwanted];
    if FindFirst (test_name, faAnyFile, s) = 0 then
    repeat
      if stop_requested then Exit;
      with s do
        begin
        if (Attr <> faDirectory) and ((Attr and unwanted_attributes) = 0) and
           ((Size <> 0) or (not ignore_zero_length_files)) then
          begin
          full_name := root + Name;
          // is this on the keep list?  if not, continue
          if not keep_list.Find (full_name, index) then
            begin
            KB := (Size + 1023) div 1024;
            Inc (files_found);
            Inc (KB_found, KB);
            lblSize.Caption := IntToStr (KB_found) + ' KB';
            attributes := '';
            if (Attr and faHidden) <> 0 then attributes := attributes + 'H';
            if (Attr and faReadOnly) <> 0 then attributes := attributes + 'R';
            if (Attr and faSysFile) <> 0 then attributes := attributes + 'S';
            if attributes <> '' then attributes := '[' + attributes + ']';
            ListBox1.Items.Add (Format ('%5d  ', [KB]) +
                                attributes + '  ' + full_name);
            end;
          end;
        end;
    until FindNext (s) <> 0;
    FindClose (s);
    end;

  test_name := root + '*.*';
  if FindFirst (test_name, faAnyFile, s) = 0 then
  repeat
    with s do
      if ((Attr and faDirectory) <> 0) and ((Name <> '.') and (Name <> '..'))
        then scan_tree (root + Name, unwanted_list);
  until FindNext (s) <> 0;
  FindClose (s);
end;

procedure TForm1.Panel1Resize(Sender: TObject);
begin
  btnFind.Top := Panel1.Height - 16 - btnFind.Height;
  btnDelete.Top := btnFind.Top;
  btnSelectAll.Top := btnFind.Top;
  btnExit.Top := btnFind.Top;
  ListBox1.Height := btnFind.Top - 16 - ListBox1.Top;
  edtDirectory.Top := btnFind.Top - 16 - edtDirectory.Height;
  Label3.Top := edtDirectory.Top - 16;
  lbxDrives.Height := Label3.Top - lbxDrives.Top - 12;
end;

procedure TForm1.Panel2Resize(Sender: TObject);
begin
  btnExit.Left := Panel2.Width - 16 - btnExit.Width;
  ListBox1.Width := Panel2.Width - 16 - ListBox1.Left;
end;

procedure TForm1.GetMinMaxInfo (var info: TWMGetMinMaxInfo);
begin
  with info.MinMaxInfo.ptMinTrackSize do
    begin
    x := suggested_min_x;
    y := suggested_min_y;
    end;
end;

procedure TForm1.btnDeleteClick(Sender: TObject);

  function get_just_filename (const filename: string): string;
  var
    s: string;
  begin
    s := Trim (filename);
    while s [1] <> ' ' do delete (s, 1, 1);
    s := Trim (s);
    if s [2] <> ':' then      // we have an attribute string
      begin
      while s [1] <> ' ' do delete (s, 1, 1);
      s := Trim (s);
      end;
    Result := s;
  end;

var
  item: integer;
  f: file;
  filename: string;
  SHFileOp: TSHFileOpStruct;
begin
  if use_recycle_bin
  then
    begin
    filename := '';
    for item := ListBox1.Items.Count-1 downto 0 do
      if ListBox1.Selected [item] then
        filename := filename + get_just_filename (ListBox1.Items.Strings [item]) + #0;
    FillChar (SHFileOp, SizeOf (SHFileOp), 0);
    with SHFileOp do
      begin
      wFunc := FO_DELETE;
      pFrom := PChar (filename);
      fFlags := FOF_ALLOWUNDO or FOF_NOCONFIRMATION;
      end;
    if (SHFileOperation (SHFileOp) = 0) and (not SHFileOp.fAnyOperationsAborted)
    then
      for item := ListBox1.Items.Count-1 downto 0 do
        if ListBox1.Selected [item] then ListBox1.Items.Delete (item)
    else;
    end
  else
    for item := ListBox1.Items.Count-1 downto 0 do
      if ListBox1.Selected [item] then
        begin
        filename := get_just_filename (ListBox1.Items.Strings [item]);
        AssignFile (f, filename);
        {$I-}  Erase (f);  {$I+}
        if IOResult = 0
        then ListBox1.Items.Delete (item)
        else
          begin
          filename := 'Unable to delete the file: '#13#10#13#10 + filename +
                      #13#10#13#10'Perhaps this file is still in use, ' +
                      'or is write-protected ?' + #0;
          MessageDlg (filename, mtWarning, [mbIgnore], 0);
          end;
        end;

  if ListBox1.Items.Count = 0 then
    begin
    Label1.Caption := '';
    btnDelete.Enabled := False;
    btnSelectAll.Enabled := False;
    end;

  lblSize.Caption := 'KBytes';
end;

procedure TForm1.Exit1Click(Sender: TObject);
begin
  btnExit.Click;    // reflect menu choice as a button click
end;

procedure TForm1.Find1Click(Sender: TObject);
begin
  btnFind.Click;    // reflect menu choice as a button click
end;

procedure TForm1.About1Click(Sender: TObject);
begin
  with AboutBox1 do   // fill in the details and display the dialog
    begin
    Caption := 'About David''s Windows Tidy Application';
    ProductName.Caption := product_name;
    Version.Caption := product_version;
    Copyright.Caption := product_copyright;
    Comments.Caption := product_comments;
    ProgramIcon.Picture.Icon := Application.Icon;
    ShowModal;
    end;
end;

procedure TForm1.Options1Click(Sender: TObject);
begin
  if not searching then    // don't alter options while searching
    with frmOptions do
      begin
      // set Options display according to current values
      chkHidden.Checked := ignore_hidden_files;
      chkReadonly.Checked := ignore_readonly_files;
      chkSystem.Checked := ignore_system_files;
      chkZeroLength.Checked := ignore_zero_length_files;

      // these are mutually exclusive
      if find_extra_files
      then btnExtraFiles.Checked := True
      else btnMostFiles.Checked := True;

      // these are mutually exclusive
      if use_recycle_bin
      then btnUseRecycleBin.Checked := True
      else btnTraditionalDelete.Checked := True;

      // if the user clicks OK, update the current values
      if ShowModal = mrOK then
        begin
        find_extra_files := btnExtraFiles.Checked;
        use_recycle_bin := btnUseRecycleBin.Checked;
        ignore_hidden_files := chkHidden.Checked;
        ignore_readonly_files := chkReadOnly.Checked;
        ignore_system_files := chkSystem.Checked;
        ignore_zero_length_files := chkZeroLength.Checked;
        end;
      end;
end;

procedure TForm1.lbxDrivesDrawItem(Control: TWinControl; Index: Integer;
  Rect: TRect; State: TOwnerDrawState);
var
  Bitmap: TBitmap;      // temporary variable for the item's bitmap
  Offset: Integer;      // text offset width
  bmpWidth: Integer;
begin
  // draw on the control canvas, not on the form
  with (Control as TListBox).Canvas do
    begin
    FillRect (Rect);    // clear the rectangle
    Offset := 2;        // provide default offset

    // get the bitmap
    Bitmap := TBitmap ((Control as TListBox).Items.Objects[Index]);

    if Bitmap <> nil then
      begin
      bmpWidth := Bitmap.Width;      // render the bitmap
      BrushCopy (Bounds (Rect.Left + 2,
                 (Rect.Top + Rect.Bottom - Bitmap.Height) div 2,
                 Bitmap.Width, Bitmap.Height),
                 Bitmap, Bounds (0, 0, Bitmap.Width, Bitmap.Height),
                 Bitmap.Canvas.Pixels [0, Bitmap.Height - 1]);
      // add four pixels between bitmap and text
      Offset := bmpWidth + 6;
      end;
      // display the text
      TextOut (Rect.Left + Offset, Rect.Top, (Control as TListbox).Items[Index])
    end;
  end;

procedure TForm1.FormActivate(Sender: TObject);
begin
  if not initialised_ok then Close;
end;

end.

