/** \file CON_Input.h
    \brief Input devices: Keyboard, Mouse, Joystick 

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.  
Permission is granted to use and copy this file for non-commercial use only.  
Please contact the author concerning commercial usage. 
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_CON_INPUT
#define H_CON_INPUT

#include <CON_Bitmap.h>

/** A joystick class, created automatically (if available) by the console */
class Joystick : public SystemInterface
{
public:
  /** Must be called to poll the current joystick status */
  virtual long update() = 0;
  /** Left-Right axis position */
  virtual int  getX() = 0;
  /** Up down axis position */
  virtual int  getY() = 0;
  /** Z Axe, not available in all joysticks. */
  virtual int  getZ() = 0;
  /** Get the status of a button (main one is usually 0) */
  virtual int  button(int Num=0) = 0;
  /** Get the status of a Point of view control */
  virtual int  POV(int Num=0) = 0;
  /** Returns the number of axes this joystick has. */
  virtual int  getAxeNumber() = 0;
  /** Returns the number of buttons this joystick has. */
  virtual int  getButtonNumber() = 0;
  /** Returns the number of Point of view controls this joystick has. */
  virtual int  getPOVNumber() = 0;
};

/** This class represents the system mouse.  It doesn't use the windows
    pointer, but rather an internal one.
    This class offers two way of accessing the mouse.
    Polling, which is less accurate, but sometimes enough.
    Events, which record all mouse state changes, for easy read.
    These ways should not be mixed.  Use only one. 
*/
class Mouse : public SystemInterface
{
public:
  /** Render the mouse pointer to the screen (screen must be unlocked) */
  virtual void render() = 0;

  /** Old version.  Here for compatibility. */
  virtual void render(Screen* S) = 0;

  /** Set the bitmap used for the mouse pointer. 
      Bitmap object is not retained, but rather copied, so you have to
      release it.
  */
  virtual void setPointer(Bitmap* Pointer, int HotSpotX=0, int HotSpotY=0) = 0;

  /** Set the mouse sensitivity, controling its speed of motion. */
  virtual void setSensitivity(float Sensitivity=1.0f) = 0;

  /** This must be called to update (poll) the mouse state
      The parameter rectangle is the bounding box for the mouse pointer. 
      It is used in the polling method (see above) but can be used to force
      the mouse to move to a certain location.
  */
  virtual long update(int x1, int y1, int x2, int y2) = 0;

  /** These methods are used when polling to check the previous state.
      This is the only way (except saving this data externally)
      to know if there was a state change. */

  /** Get polled last X state */
  virtual int  getXLast() = 0; 

  /** Get polled last Y state */
  virtual int  getYLast() = 0;

  /** Get polled last left button state */
  virtual int  leftButtonLast() = 0;

  /** Get polled last right button state */
  virtual int  rightButtonLast() = 0;

  /** Get polled last button (any) state */
  virtual int  buttonLast(int Num) = 0;


  /** These methods are used both in polling and events to get the mouse state. */
  
  /** Returns current mouse X coordinate. */
  virtual int  getX() = 0;
  /** Returns current mouse Y coordinate. */
  virtual int  getY() = 0;
  /** Returns non zero if the left button (0) is pressed */
  virtual int  leftButton() = 0;
  /** Returns non zero if the right button (1) is pressed */
  virtual int  rightButton() = 0;
  /** Returns non zero if specific (can be non-standard) button is pressed.
      Buttons number ranges 0 - 3 */
  virtual int  button(int Num) = 0;

  /** This must be called to get a mouse event.
      Returns 0 if no event, 1 if mouse moved, 2..5 if buttons (0..3) changed.
      The parameter rectangle is the bounding box for the mouse pointer. */
  virtual int  getEvent(int x1, int y1, int x2, int y2) = 0;  
};

/** This class gives a representation of the system keyboard.
    It also provides two methods of access, but these may be combined.
    The polling is good for games that need to know if a certain key is pressed NOW.
    The event is good for applications need to get any key press/release 
    information for input of text for example.
*/
class Keyboard : public SystemInterface
{
public:
  /** This must be called for polling (using the [] operator) */
  virtual long update() = 0;
  /** Get the current press state of a certain key. */
  virtual int  operator[] (int ID) = 0;
  /** Get the current press state of a certain key. */
  virtual int  press      (int ID) = 0;
  /** Check if this key was pressed last time update was called. */
  virtual int  constPress(int ID) = 0;

  /** Get a key event.
      Returns 1 if key was pressed or released, Modifys "ID" & "Pressed" accordingly. */
  virtual int  getKey(int& ID, int& Pressed) = 0; 

  /** Sets the state of the keyboard leds. 
      Doesn't work yet.
  */
  virtual void setLeds(int NumLock, int CapsLock, int ScrollLock) = 0;
};


////////////////////////////////////////////////////////////
// Keyboard Functions
////////////////////////////////////////////////////////////
#ifndef FUNC_KEYBOARD
#define FUNC_KEYBOARD


/** \relates Keyboard
    Translate key ID into the character they represent on the keyboard. */
char DLLExport getKeyChar(int Key, int Shift=0);


#endif // FUNC_KEYBOARD


#endif // H_CON_INPUT