/** \file CON_Sound.h
    Sound Objects (Clip & Stream) 

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.  
Permission is granted to use and copy this file for non-commercial use only.  
Please contact the author concerning commercial usage. 
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
class Sound;
class SoundBuffer;

#ifndef H_CON_SOUND
#define H_CON_SOUND

#include <CON_3D.h>
#include <UTIL_Misc.h>

/** Main sound object.  It is created by Console */
class Sound : public SystemInterface
{
public:
  /** Volume must be between -10000 (silence) and 0 (Full) */
  virtual long setVolume(int Volume) = 0;

  /** Returns non zero if an error has occured. */
  virtual long status() = 0;
};


/** Flags passed in the HW parameter when creating sounds
*/
enum {
  SoundAccel=1,  // Enable hardware acceleration
  Sound3D=2      // Enable 3D sound
};

/** Generic SoundBuffer class implements the sound generics
    3D methods are not a must.  Standard mono / stereo can be played too. */
class SoundBuffer : public Interface
{
public:
  /** Sets the Position in the 3D World. */
  virtual long setPosition(float x, float y, float z) = 0;

  /** Sets the direction and speed in the 3D World. */
  virtual long setDirection(float x, float y, float z, float Speed) = 0;

  /** Set 3D Buffer parameters */
  virtual long set3DParameters(float MinDistance=15.0f, float MaxDistance=5000.0f, 
                               float ConeInside=60, float ConeOutside=360,
                               int   VolumeOutside=-5000) = 0;

  /** Determines whether this sound is processed by the 3D sound system.
      Note: if a buffer was not created with the Sound3D flag
            it cannot be set to 3D mode.*/
  virtual long set3DMode(int Mode) = 0;

  /** Set the delay (in millisecs) of a sound stream.  
      A delay is needed to avoid sound clicks resulting 
      from non-realtime threads.
      This defaults to 30ms */
  virtual long setStreamDelay(ushort Delay) = 0;

  /** Detaches the sound buffer, and lets it play until it stops.
      Sound buffer will then be deleted!!! 
      Not recommended for loop playing sound clips */
  virtual long destroyOnStop() = 0;

  /** Returns non zero if the buffer is currently playing. */
  virtual int  isPlaying() const = 0;

  /** Volume must be between -10000 (silence) and 0 (Full) */
  virtual long setVolume(int Volume) = 0;

  /** Play the sound. If Loop is non zero, it will play over and over */
  virtual long play(int Loop=0) = 0;

  /** Stop playing */
  virtual long stop() = 0;

  /** Returns non zero if an error occured. */
  virtual long status() const = 0;

};

typedef SoundBuffer SoundClip;
typedef SoundBuffer SoundStream;


////////////////////////////////////////////////////////////
// Sound Functions
////////////////////////////////////////////////////////////
#ifndef FUNC_SOUND
#define FUNC_SOUND

#include <CON_Resource.h>


/** \relates SoundBuffer
    Create a new sound clip from raw data.  Size is in bytes. */
DLLExport SoundClip* newSoundClip(char* AudioData, int Size,    int Freq=44100, int Bits=16, int Channels=2, int HW=SoundAccel);

/** \relates SoundBuffer
    Create a new sound clip from raw data (taken from a binary stream).  Size is in bytes. */
DLLExport SoundClip* newSoundClip(istream* AudioData, int Size, int Freq=44100, int Bits=16, int Channels=2, int HW=SoundAccel);

/** \relates SoundBuffer
    Create a new sound stream from raw data.  
    Stream must remain valid until this object is destroyed.
    Data will be streamed to the sound card continuously.
*/
DLLExport SoundStream* newSoundStream(ResourceStream* RS, istream* AudioData, int Freq=44100, int Bits=16, int Channels=2, int HW=0);

/** \relates SoundBuffer
    Create a new sound stream from raw data.  
    Stream (AudioData) will not be deleted when SoundStream object is destroyed.
    User must provide AudioData disposal.
    Data will be streamed to the sound card continuously.
*/
DLLExport SoundStream* newSoundStream(InputFilter* AudioData, int Freq=44100, int Bits=16, int Channels=2, int HW=0);

/** \relates SoundBuffer
    Loads a wave from a binary input stream.
    This function accepts only plain PCM files.
    All compressed types must be streamed, using  streamAudioFile*/
DLLExport SoundClip* loadWaveFile(istream* is, int HW=SoundAccel);

/** \relates SoundBuffer
    Loads a wave from the default resource stream (if it's set), or a file. 
    This function accepts only plain PCM files.
    All compressed types must be streamed, using  streamAudioFile*/
DLLExport SoundClip* loadWaveFile(const char* Name, int HW=SoundAccel);

/** \relates SoundBuffer
    Open a large wave file for streaming.
    The ResourceStream must remain alive until the SoundStream is released.
    It will be used to release the data input stream.
*/
DLLExport SoundStream* streamWaveFile(ResourceStream* RS, const char* Name, int HW=0);

/** \relates SoundBuffer
    Open an audio file for streaming.
    This requires windows to have DirectShow installed, and support the file format.
    Uses only files, no resources yet.
*/
DLLExport SoundStream* streamAudioFile(const char* Name, int HW=0);


#endif // FUNC_SOUND


#endif // H_CON_SOUND