/** \file WO_Collision.h
    Generic collision detection control.

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.
Permission is granted to use and copy this file for non-commercial use only.  
Please contact the author concerning commercial usage. 
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_WO_COLLISION
#define H_WO_COLLISION

#include <WO_World.h>


class Collidable_World;  // forward ref


/* Collidable Objects may be subclassed from this type */

class Collidable_Object : public World_Object, public Event_Receiver
{
public:
   Collidable_Object();

   /** this checks to make sure that any Object added to a Collidable_Object
       is derived from Collidable_Object.  Returns -1 if dynamic_cast fails. */
   virtual long add(Object* O);

  /** This checks two Collidable_Objects for collision. */
  virtual int  isCollision(Collidable_Object* O)=0;

  /** This is called to notify this Object of a collision with O. */
  virtual long collision(Collidable_Object* O)=0;

  /** This processes events and calls the above collision() for each collsion event.*/
  virtual long advance(float Fraction);

  /** This can be called to determine if the Collidable_Object is to be included
      in the CollisionDetection set of Collidable_Objects. */
  virtual int  isCollidable()=0;
};


/** 
  Collision interaction between two objects.  It is created by the collision
  detection object, and can be retrieved by using the 
  CollisionDetection::getCollision() method.
*/
class Collision : public Serial_Containable
{
public:
  Collision(Collidable_Object* O1, Collidable_Object* O2) : m_O1(O1), m_O2(O2) {}
  Collidable_Object* m_O1;
  Collidable_Object* m_O2;
};


/** This class provides mechanizm of checking for interactions between objects.
*/
class CollisionDetection
{
public:
  /** Construct a collision detection object */
  CollisionDetection();

  /** Destroys a collision detection object */
  virtual ~CollisionDetection();

  /** Readys the detection mechanizm, by clearing old results. */
  long       clear();

  // Add an object to be checked for collision. */
  long       add(Collidable_Object* O);

  /** This method performs the actual checks for collisions
      It returns the number of collisions detected. */
  long       check();

  /** After check was run, query for collision by serial number starting
      at 0.  A non NULL value will return a collision object, to be processed.
      NULL means that there are no more collisions. */
  Collision* getCollision(int Number);

  /** After a check is run, this function may be called to post collision events
      to each Collidable_Object involved in a collision. */
  void       postEvents();

protected:
  Vector          m_Objects;
  Vector          m_Results;
};


class Collidable_World : public World
{
public:
   Collidable_World() : World() {}

   /** this checks to make sure that any Object added to a Collidable_World
       is derived from Collidable_Object.  Returns -1 if dynamic_cast fails. */
   virtual long add(Object* O);

   /** This method is called once after the world is advanced.  Default implementation
      calls check(), postEvents(), and clear(). */
   virtual long postAdvance(float Fraction);

   /** This method is called once per object after it is advanced.  Default 
      implementation calls m_ColDet.add() if the Object isCollidable(). */
   virtual long postObjectAdvance(World_Object* O, float Fraction);

   CollisionDetection m_ColDet;
};

#endif // H_WO_COLLISION