/** \file WO_Property.h
    Generic property.  Association between a name and some data.

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.
Permission is granted to use and copy this file for non-commercial use only.
Please contact the author concerning commercial usage.
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_WO_PROPERTY
#define H_WO_PROPERTY

#include <DS_Hashtable.h>

#pragma comment(lib, "libWO.lib")

/** This is a generic association that is used to store any information
    for an object.  The key is the name, and the data is a void pointer */
class Property : public String_Containable
{
public:
  /** Constructs a new Property.  The name should be a unique key in an object.
      The value can be anything, and is only stored unless Destroy is used.
      If Destroy is not 0, the Value is assumed to be dynamically allocated,
      and will be 'delete'd when the property is destroyed.
      A property that is set in an Object is automatically destroyed when
      the Object is destroyed.
  */
  Property(String Name, void* Value, int Destroy=0);
  virtual ~Property();

  /** Returns the name of the property. 
      Implementing String_Containable */
  String& getString() const;

  /** Returns the data value. */
  void*   getData() const;

  /** Set a different data value. */
  void    setData(void* Value);

  /** Hash value for property.  Uses the Name's hash value. */
  size_t  hashValue() const;
protected:
  String m_Name;
  void*  m_Value;
  size_t m_HashValue;
  int    m_Destroy;
};


/** This is a "container" for properties.  Classes that have properties may 
    inherit from this class to implement them.
    Note: this is a Container::DESTRUCTIVE.
*/
class Property_Set {
public:
   Property_Set();

   /** Fetch a property. */
   long getProperty(Property& P);

   /** Sets a property.  Property object must be dynamically allocated */
   long setProperty(Property* P);

private:
   Hashtable m_Properties;
};

#endif // H_WO_PROPERTY
