//
//                  ooo        ooooo
//                  `88.       .888'
//                   888b     d'888   .oooo.   oooo    ooo
//                   8 Y88. .P  888  `P  )88b   `88b..8P'
//                   8  `888'   888   .oP"888     Y888'
//                   8    Y     888  d8(  888   .o8"'88b
//                  o8o        o888o `Y888""8o o88'   888o
//
//                          Across the event horizon...
//
//                       Microsoft Windows 95/98/NT Version
//
//  Copyright (c) 1994-1999 by Dan Higdon, Tim Little, and Chuck Walbourn
//
//
//
// This file and all associated files are subject to the terms of the
// GNU Lesser General Public License version 2 as published by the
// Free Software Foundation (http://www.gnu.org).   They remain the
// property of the authors: Dan Higdon, Tim Little, and Chuck Walbourn.
// See LICENSE.TXT in the distribution for a copy of this license.
//
// THE AUTHORS MAKE NO WARRANTIES, EXPRESS OR IMPLIED, AS TO THE CORRECTNESS
// OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE IT.  THE AUTHORS
// PROVIDE THE CODE ON AN "AS-IS" BASIS AND EXPLICITLY DISCLAIMS ANY
// LIABILITY, INCLUDING CONSEQUENTIAL AND INCIDENTAL DAMAGES FOR ERRORS,
// OMISSIONS, AND OTHER PROBLEMS IN THE CODE.
//
//
//
//                        http://www.mythos-engine.org/
//
//
//
// Created by Dan Higdon
//
// maxdevs.cpp
//
// Event Devices
//
//      This module defines the physical devices used by the event
//  system.  Each device has a unique interface, suited to the device.
//
//  This module defines a "block" of devices (mouse, kb, joystick), and
//  the interface used to communicate with them.  Two implementations
//  are provided, one using Win32 and the other using DirectInput.
//
//

//
//
//                                Includes
//
//

#include <assert.h>
#include <string.h>

#define WIN32_LEAN_AND_MEAN
#include <windows.h>

#include <ivory.hpp>

#include "maxdevs.hpp"
#include "maxjoy.hpp"
#include "maxkb.hpp"
#include "maxmouse.hpp"

#include <iostream.h>

//
//
//                                Equates
//
//

//
//
//                               Structures
//
//


//
//
//                               Routines
//
//

//
//
//                                 Data
//
//

//
//
//                              Static Data
//
//

//
//
//                                 Code
//
//

//
//
//                      MaxDevice Member Functions
//
//

//
//  MaxDevices Constructor
//
//
MaxDevices::MaxDevices():
    theKeyboard (0),
    theMouse    (0),
    theJoystick (0),
    snapshot_mx (0),
    snapshot_my (0),
    snapshot_mdx (0),
    snapshot_mdy (0),
    snapshot_mbuttons (0)
{
    // Joystick state info
    memset (&snapshot_jstat, 0, sizeof (snapshot_jstat));
}


//
//  MaxDevices Destructor
//
//
MaxDevices::~MaxDevices()
{
    if (theKeyboard)
    {
        delete theKeyboard;
        theKeyboard = 0;
    }

    if (theMouse)
    {
        delete theMouse;
        theMouse = 0;
    }

    if (theJoystick)
    {
        delete theJoystick;
        theJoystick = 0;
    }
}


//
//  MaxDevices::update
//  Update the internal values for the various devices
//
void MaxDevices::update ()
{
    // Capture the mouse state
    if (is_mouse_present())
    {
        theMouse->update();
        theMouse->get_movement (&snapshot_mdx, &snapshot_mdy);
        theMouse->get_pos      (&snapshot_mx, &snapshot_my);
        snapshot_mbuttons = theMouse->get_buttons ();
    }

    // Capture the keyboard state.
    if (is_keyboard_present())
    {
        theKeyboard->update();
    }

    // Capture the joystick state.
    if (is_joystick_present())
    {
        theJoystick->get_stat (&snapshot_jstat);
    }
}


//
//  MaxDevices::wm_clear
//  Clears all devices before processing wm_ members
//
void MaxDevices::wm_clear ()
{
    if (theKeyboard)
        theKeyboard->clear();

    if (theMouse)
        theMouse->clear();

    if (theJoystick)
        theJoystick->clear();
}


//
//  MaxDevices::init
//  Initializes all devices
//
void MaxDevices::init ()
{
    if (theKeyboard)
        theKeyboard->init();

    if (theMouse)
        theMouse->init();

    if (theJoystick)
        theJoystick->init();
}


//
//  MaxDevices::uninit
//  Uninitializes all devices
//
void MaxDevices::uninit ()
{
    if (theKeyboard)
        theKeyboard->uninit();

    if (theMouse)
        theMouse->uninit();

    if (theJoystick)
        theJoystick->uninit();
}


//
//  MaxDevices::enable_keyboard
//
//
int MaxDevices::enable_keyboard (int enabled)
{
    if (theKeyboard)
    {
        if (enabled)
        {
            if (!theKeyboard->is_initialized())
                theKeyboard->init ();
        }
        else
        {
            if (theKeyboard->is_initialized())
                theKeyboard->uninit();
        }
    }

    return theKeyboard != 0;
}


//
//  MaxDevices::enable_mouse
//
//
int MaxDevices::enable_mouse (int enabled)
{
    if (theMouse)
    {
        if (enabled)
        {
            if (!theMouse->is_initialized())
                theMouse->init();
        }
        else
        {
            if (theMouse->is_initialized())
                theMouse->uninit();
        }
    }

    return theMouse != 0;
}


//
//  MaxDevices::enable_joystick
//
//
int MaxDevices::enable_joystick (int enabled)
{
    if (theJoystick)
    {
        if (enabled)
        {
            if (!theJoystick->is_initialized())
                theJoystick->init ();
        }
        else
        {
            if (theJoystick->is_initialized())
                theJoystick->uninit ();
        }
    }

    return theJoystick != 0;
}


//
//  MaxDevices::get_mouse_movement
//
//
void MaxDevices::get_mouse_movement (short *mdx, short *mdy) const
{
    *mdx = snapshot_mdx;
    *mdy = snapshot_mdy;
}


//
//  MaxDevices::get_mouse_pos
//
//
void MaxDevices::get_mouse_pos (short *mx, short *my) const
{
    *mx = snapshot_mx;
    *my = snapshot_my;
}


//
//  MaxDevices::set_mouse_pos
//
//
void MaxDevices::set_mouse_pos (short mx, short my)
{
    if (theMouse)
        theMouse->set_pos (snapshot_mx = mx,
                           snapshot_my = my);
}


//
//  MaxDevices::get_mouse_buttons
//
//
byte MaxDevices::get_mouse_buttons () const
{
    return snapshot_mbuttons;
}


//
//  MaxDevices::check_keyboard_key
//
//
int MaxDevices::check_keyboard_key (int key) const
{
    return theKeyboard ? theKeyboard->check_key (key): 0;
}


//
//  MaxDevices::get_joystick
//
//
void MaxDevices::get_joystick (JOYINFOEX *js) const
{
    *js = snapshot_jstat;
}


//
//  MaxDevices::is_mouse_present
//
//
bool MaxDevices::is_mouse_present () const
{
    return theMouse != 0 && theMouse->is_present();
}


//
//  MaxDevices::is_keyboard_present
//
//
bool MaxDevices::is_keyboard_present () const
{
    return theKeyboard != 0
           && theKeyboard->is_present();
}


//
//  MaxDevices::is_joystick_present
//
//
bool MaxDevices::is_joystick_present () const
{
    return theJoystick != 0
           && theJoystick->is_present();
}


//
//  MaxDevices::is_mouse_enabled
//
//
bool MaxDevices::is_mouse_enabled () const
{
    return theMouse != 0
           && theMouse->is_present()
           && theMouse->is_initialized();
}


//
//  MaxDevices::is_keyboard_enabled
//
//
bool MaxDevices::is_keyboard_enabled () const
{
    return theKeyboard != 0
           && theKeyboard->is_present()
           && theKeyboard->is_initialized();
}


//
//  MaxDevices::is_joystick_enabled
//
//
bool MaxDevices::is_joystick_enabled () const
{
    return theJoystick != 0
           && theJoystick->is_present()
           && theJoystick->is_initialized();
}


//
//
//                  MaxDevicesWin32 Member Functions
//
//

//
//  MaxDevicesWin32 Constructor
//
//
MaxDevicesWin32::MaxDevicesWin32 (HWND _hwnd):
    MaxDevices (),
    hwndClient (_hwnd)
{
    // Create the Keyboard object
    theKeyboard = new MaxKeyboardWin32 ();
    if (!theKeyboard->is_present())
    {
        delete theKeyboard;
        theKeyboard = 0;
    }

    // Create the Mouse object
    theMouse = new MaxMouseWin32 ();
    if (!theMouse->is_present())
    {
        delete theMouse;
        theMouse = 0;
    }

    // Create the Joystick object
    theJoystick = new MaxJoystickWin32 ();
    if (!theJoystick->is_present())
    {
        delete theJoystick;
        theJoystick = 0;
    }
}


//
//  MaxDevicesWin32 Destructor
//
//
MaxDevicesWin32::~MaxDevicesWin32()
{
}



//
//
//                   MaxDevicesDirectX Member Functions
//
//

//
//  MaxDevicesDirectX Constructor
//
//
MaxDevicesDirectX::MaxDevicesDirectX (HWND _hwnd):
    MaxDevices (),
    hwndClient (_hwnd),
    pdi (0)
{
    if (DirectInputCreate ((HMODULE)GetWindowLong (_hwnd, GWL_HINSTANCE),
                           DIRECTINPUT_VERSION,
                           &pdi,
                           NULL) != DI_OK)
        pdi = 0;

    if (pdi)
    {
        // Create the Keyboard object
        theKeyboard = new MaxKeyboardDirectX (pdi, hwndClient);
        if (!theKeyboard->is_present())
        {
            delete theKeyboard;
            theKeyboard = 0;
        }

        // Create the Mouse object
        theMouse = new MaxMouseDirectX (pdi, hwndClient);
        if (!theMouse->is_present())
        {
            delete theMouse;
            theMouse = 0;
        }

        // Create the Joystick object
        theJoystick = new MaxJoystickDirectX (pdi, hwndClient);
        if (!theJoystick->is_present())
        {
            delete theJoystick;
            theJoystick = 0;
        }
    }
}


//
//  MaxDevicesDirectX Destructor
//
//
MaxDevicesDirectX::~MaxDevicesDirectX()
{
    // Must perform these deletes before the base class gets the chance
    if (theKeyboard)
    {
        delete theKeyboard;
        theKeyboard = 0;
    }

    if (theMouse)
    {
        delete theMouse;
        theMouse = 0;
    }

    if (theJoystick)
    {
        delete theJoystick;
        theJoystick = 0;
    }

    // Now, clean up the master DInput interface
    if (pdi)
    {
        pdi->Release();
        pdi = 0;
    }
}

// End of module - maxdevs.cpp 
