/* PhotoOrganizer 
 * Copyright (C) 1999-2000 Dmitry Rogatkin.  All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 *  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package photoorganizer;
import java.io.*;
import java.net.*;
import java.awt.Image;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.Hashtable;
import java.util.ResourceBundle;

import javax.swing.JOptionPane;
import javax.swing.Icon;

import rogatkin.*;
import photoorganizer.renderer.*;
import photoorganizer.formats.*;
import photoorganizer.ftp.*;
import photoorganizer.album.*;

// TODO: add HTML encoding for all replaced lexems

/** The class parses HTML template and it's looking for special tags
!%loop - start of a loop
!%tnof - substitute original file name
!%tnfn - substitute thumbnail file name
!%tntt - tooltip for thumbnail
!%tnc  - commentary under
!%tnw  - width of thumbnail image (optional)
!%tnh  - height of thumbnail image (optional)
!%tn+  - advance to next thumbnail
!%endl - end of the loop

!%tnif - original image file name 
!%cprt - copyright note with link 
!%alnm - album name 
!%webp - related path on Web site to original images

!%cloop - loop of child albums
!%praln - parent album name
!%pralr - parent album reference in URL
!%ch+   - next child
!%cendl - end loop of child albums refs marker
!%chan  - child album name
!%chap  - child album URL
!%cendl - end marker of child loop
!%chan  - child album name
!%chap  - child album web path

Notice, all tags can be enclosed to HTML comment
*/
public class HtmlProducer {
    
    private final int HTML           = 1;
    private final int END            = 32;
    private final int RECORD_LOOP    = 33;
    
    // lexem codes
    final static int PLAIN_HTML              = 0;
    final static int IMAGE_LOOP_START_MARKER = 1;
    final static int IMAGE_FILE_NAME         = 2;
    final static int THUMB_WEB_PATH          = 3;
    final static int THUMB_TOOLTIP           = 4;
    final static int MIDRES_IMAGE_FILE_NAME  = 5;
    final static int LOWRES_IMAGE_FILE_NAME  = 6;
    final static int COMMENTARY              = 7;
    final static int WIDTH_OF_THUMB          = 8;
    final static int HEIGHT_OF_THUMB         = 9;
    final static int TO_NEXT_IMAGE           = 10;
    final static int IMAGE_LOOP_END_MARKER   = 11;
    final static int IMAGE_WEB_DIR           = 12;
    final static int IMAGE_FILE_PATH         = 13;
    final static int COPYRIGHT               = 14;
    final static int ALBUM_NAME              = 15;
    final static int PARENT_ALBUM_NAME       = 16;
    final static int PARENT_ALBUM_WEB_PATH   = 17;
    final static int ALBUM_LOOP_START_MARKER = 18;
    final static int ALBUM_LOOP_END_MARKER   = 19;
    final static int CHILD_ALBUM_NAME        = 20;
    final static int CHILD_ALBUM_WEB_PATH    = 21;
    final static int TO_NEXT_CHILD_ALBUM     = 22;
    final static int END_DOCUMENT            = -1;

    static final String [] LEXEM_MENMONICS = {
        "html",     // PLAIN_HTML
	    "loop", // IMAGE_LOOP_START_MARKER
	    "tnif", // IMAGE_FILE_NAME
	    "tnfn", // THUMB_WEB_PATH
	    "tntt", // THUMB_TOOLTIP
	    "tnhr", // MIDRES_IMAGE_FILE_NAME
	    "tnlr", // LOWRES_IMAGE_FILE_NAME
	    "tnc",  // COMMENTARY
	    "tnw",  // WIDTH_OF_THUMB
	    "tnh",  // HEIGHT_OF_THUMB
	    "tn+",  // TO_NEXT_IMAGE
	    "endl", // IMAGE_LOOP_END_MARKER
	    "webp", // IMAGE_WEB_DIR
	    "tnof", // IMAGE_FILE_PATH
	    "cprt", // COPYRIGHT
	    "alnm", // ALBUM_NAME
	    "praln",// PARENT_ALBUM_NAME
	    "pralr",// PARENT_ALBUM_WEB_PATH
	    "cloop",// ALBUM_LOOP_START_MARKER
	    "cendl",// ALBUM_LOOP_END_MARKER
	    "chan", // CHILD_ALBUM_NAME
	    "chap", // CHILD_ALBUM_WEB_PATH
	    "ch+"   // TO_NEXT_CHILD_ALBUM
    };

    final static char CPERCENT = '%';
    final static char CEXLAM = '!';
    final static char[] LEX_MARKER = {CEXLAM, CPERCENT};

    // simplify parsing code, adding lexer returning pair lexem:code
	public HtmlProducer(Controller controller) {
		this.controller = controller;
		s = controller.getSerializer();
		album_pane = (AlbumPane)controller.component(Controller.COMP_ALBUMPANEL);
		try {
			album_name = album_pane.getSelectionPath().toString();
		} catch(NullPointerException ne) {
			album_name = "";
		}
		html = new StringBuffer(512);
		statusbar = (StatusBar)controller.component(Controller.COMP_STATUSBAR);
	}
    
    public void produce(Courier courier, int albumId, boolean recursevly) throws IOException {
        produce(courier, albumId, -1, recursevly);
    }
    
    public void produce(Courier courier, int albumId, int parentAlbumId, boolean recursevly) throws IOException {
		if (courier == null)
			return;
		courier.init();
		Access access = album_pane.getAccess();
		String albumName = access.getNameOfAlbum(albumId);
		try {
			statusbar.displayInfo(Resources.INFO_WEBPUBLISHING);
			Lexer lexer = new Lexer((String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTMLTEMPL));
			Vector lexems = new Vector();
			int state = HTML;
			Lex l;
			do {
				l = lexer.getNextLex(); 
				switch(l.code) {
				case IMAGE_LOOP_START_MARKER:
					if (state == HTML) {
						lexems.removeAllElements();
						state = RECORD_LOOP;
					} else // loop inside loop is not allowed
						state = END;
					break;
				case IMAGE_LOOP_END_MARKER:
					// do the loop
					processImageLoop(courier, access.getAlbumContents(albumId), lexems, albumId);
					state = HTML;
					break;  
				case COPYRIGHT:
					if (state == HTML)
						html.append(PhotoOrganizer.COPYRIGHT);
					break;              
				case ALBUM_NAME:
					if (state == HTML)
						html.append(albumName);
					break;             
				case PARENT_ALBUM_NAME:
					if (parentAlbumId >= 0)
						html.append(access.getNameOfAlbum(parentAlbumId));
					break;      
				case PARENT_ALBUM_WEB_PATH:
					if (parentAlbumId >= 0)
						html.append(access.getNameOfAlbum(parentAlbumId)+Resources.EXT_HTML);
					break;  
				case ALBUM_LOOP_START_MARKER:
					if (state == HTML) {
						lexems.removeAllElements();
						state = RECORD_LOOP;
					} else // loop inside loop is not allowed
						state = END;
					break;
				case ALBUM_LOOP_END_MARKER:
					processAlbumLoop(courier, albumId, lexems);
					state = HTML;
					break;  
				case END_DOCUMENT:
					state = END;
					html.append(l.text);
					break;
				default:
					if (state == HTML)
						html.append(l.text);
					else if (state == RECORD_LOOP)
						lexems.addElement(l);
				}
			} while(state != END);
			courier.deliver(html, albumName+Resources.EXT_HTML, "text/html", null); // don't use encoding for a while
			html = new StringBuffer(512); // html.setLength(0); can't be used cause a bug in Sun's libs
			if (images_to_copy != null) {
				String imagePath = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.IMAGEPATH);
				if (imagePath == null)
					imagePath = "";
				else
					courier.checkForDestPath(imagePath);
				statusbar.clearProgress();
				statusbar.displayInfo(Resources.INFO_COPYING);
				statusbar.setProgress(images_to_copy.size());
				for (int i=0; i<images_to_copy.size(); i++) {
					courier.deliver((String)images_to_copy.elementAt(i), imagePath);
					statusbar.tickProgress();
				}
			}
		} finally {
			statusbar.clearInfo();
			statusbar.clearProgress();
			courier.done();
		}
		if (recursevly) {
			int[] childs = album_pane.getAccess().getAlbumsId(albumId);
			for (int i=0; i<childs.length; i++)
				produce(courier, childs[i], albumId, true);
		}
	}

	void processImageLoop(Courier courier, File[] files, Vector lexems, int albumId) throws IOException {
		String thumbsPath = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.TNWEBPATH);
		if (thumbsPath == null)
			thumbsPath = "";
		else
			courier.checkForDestPath(thumbsPath);
		String destPath = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.WEBROOT);
		if (destPath == null)
			destPath = "";
		String imagePath = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.IMAGEPATH);
		if (imagePath == null)
			imagePath = "";
		else {
			courier.checkForDestPath(imagePath);
			if (imagePath.length() > 0 && imagePath.charAt(imagePath.length()-1) != '/' && imagePath.charAt(imagePath.length()-1) != '\\')
				imagePath += '/'; //File.separatorChar;
		}

        boolean useUrl = s.getInt(s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.USEURLPATH), 0) == 1;
        String imageURL = null;
        if (useUrl) {
            imageURL = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.IMAGEURL);
            if (imageURL == null)
                imageURL = "";
        }
		if (s.getInt(s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.CPYPICS), 0) == 1)
			images_to_copy = new Vector(10);
		statusbar.setProgress(files.length);
		int ii = 0; // image index
imageloop:
		while(ii < files.length) {
			BasicJpeg format = new BasicJpeg(files[ii]);
			for (int i=0; i<lexems.size(); i++) {
				Lex l = (Lex)lexems.elementAt(i);
				switch(l.code) {
				case PLAIN_HTML:
					html.append(l.text);
					break;
				case IMAGE_FILE_NAME:
					html.append(format.getImageInfo().getName());
					break;        
				case THUMB_WEB_PATH:
					String m = (String)s.getProperty(ThumbnailsOptionsTab.SECNAME, ThumbnailsOptionsTab.FILEMASK);
					if (m == null || m.length() == 0)
						m = PhotoCollectionPanel.DEFTNMASK;
					html.append(courier.deliver(format, thumbsPath));
					break;         
				case THUMB_TOOLTIP:
					String ttm = s.arrayToString(s.getProperty(ThumbnailsOptionsTab.SECNAME, ThumbnailsOptionsTab.TOOLTIPMASK));
					if (ttm != null)
						html.append(new FileNameFormat(ttm).format(format));
					break;          
				case MIDRES_IMAGE_FILE_NAME:
					break; 
				case LOWRES_IMAGE_FILE_NAME:
					break; 
				case COMMENTARY:
					String cmt = 
						album_pane.getAccess().getPictureComment(albumId, format.getLocationName());
					// remove leading <html> probably
					if (cmt == null || cmt.length() == 0) {
						cmt = (String)s.getProperty(ThumbnailsOptionsTab.SECNAME, ThumbnailsOptionsTab.LABELMASK);
						if (cmt != null)
							html.append(new FileNameFormat(cmt).format(format));
					} else
						html.append(cmt);
					break;             
				case WIDTH_OF_THUMB:
					try {
						html.append(format.getImageInfo().getThumbnailIcon(format).getIconWidth());
					} catch(NullPointerException npe) {
					}
					break;         
				case HEIGHT_OF_THUMB:
					try {
						html.append(format.getImageInfo().getThumbnailIcon(format).getIconHeight());
					} catch(NullPointerException npe) {
					}
					break;        
				case TO_NEXT_IMAGE:
					ii++;
					if (ii>=files.length) {
						break imageloop;
					} else {
						format = new BasicJpeg(files[ii]);
					}
					statusbar.tickProgress();
					break;          
				case IMAGE_WEB_DIR:
					if (useUrl)
						html.append(imageURL);
					break;          
				case IMAGE_FILE_PATH:
					String ifn;
					try {
						ifn = format.getImageInfo().getName();
					} catch(NullPointerException npe) {
						break; // ifn = new File(format.getLocationName()).getName();
					}
					if (useUrl) // it means 
						html.append(ifn);
					else {
						// what should be in the place of URL of the image?
						//
						//            | local | ftp(r)|  http | e-mail |  svg  |
						// copy image | PATH  | RPATH | RPATH | IMGID  | RPATH |
						// use URL    | URL+N | URL+N | URL+N | URL+N  | URL+N |
						// copy + URL | URL+N | URL+N | URL+N | URL+N  | URL+N |
						// none       | PATH  | RPATH | RPATH | IMGID  | RPATH |
						if (images_to_copy != null) {
							if (courier.isLocal()) {
								try {
									html.append(new URL("file", "localhost/", new File(new File(destPath, imagePath), ifn).getAbsolutePath()).toString());
								} catch(MalformedURLException mfpe) {
									html.append(format.getLocationName());
								}
							} else {
								if (courier.isContentIncluded())
									html.append("cid:").append(format.getLocationName());
								else
									html.append(imagePath).append(ifn);
							}
						} else {
							try {
								html.append(new URL("file", "localhost/", format.getLocationName()).toString());
							} catch(MalformedURLException mfpe) {
								html.append(format.getLocationName());
							}
						}
					}
					if (images_to_copy != null) 
						images_to_copy.addElement(format.getLocationName());
					break;        
				}
			}
			ii++; // advance to next image automatically at the and of the loop
		} 
    }

    void processAlbumLoop(Courier courier, int albumId, Vector lexems) throws IOException {
        Access access = album_pane.getAccess();
        int[] childs = album_pane.getAccess().getAlbumsId(albumId);
	int ai = 0;
albumloop:
	while(ai < childs.length) {
	    for (int i=0; i<lexems.size(); i++) {
		Lex l = (Lex)lexems.elementAt(i);
		switch(l.code) {
		case PLAIN_HTML:
		    html.append(l.text);
		    break;
		case CHILD_ALBUM_NAME:
                    html.append(access.getNameOfAlbum(childs[ai]));
                    break;
		case CHILD_ALBUM_WEB_PATH:
                    html.append(access.getNameOfAlbum(childs[ai])+Resources.EXT_HTML);
		    break;
		case TO_NEXT_CHILD_ALBUM:
		    ai++;
		    if (ai>=childs.length) {
			break albumloop;
		    } else {
			
		    }
		    break;
		}
	    }
            ai++;  // advance to next child album automatically at the and of the loop
	}
    }

    public void produce(Courier courier, File[] files) throws IOException {
	produce(Resources.LABEL_NO_ALBUMNAME, files, courier);
    }

    public void produce(String desthtmlname, File[] files, Courier courier) throws IOException {
	if (courier == null)
	    return;
        courier.init();
        String albumName = Resources.LABEL_CURRENT_SELECTION;
        try {
	    statusbar.displayInfo(Resources.INFO_WEBPUBLISHING);
            Lexer lexer = new Lexer((String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTMLTEMPL));
            Vector lexems = new Vector();
	    int state = HTML;
	    Lex l;
	    do {
		l = lexer.getNextLex(); 
		switch(l.code) {
		case IMAGE_LOOP_START_MARKER:
		    if (state == HTML) {
			lexems.removeAllElements();
			state = RECORD_LOOP;
		    } else // loop inside loop is not allowed
			state = HTML;
		    break;
		case IMAGE_LOOP_END_MARKER:
		    // do the loop
		    processImageLoop(courier, files, lexems, -1);
		    state = HTML;
		    break;  
		case COPYRIGHT:
		    if (state == HTML)
			html.append(PhotoOrganizer.COPYRIGHT);
		    break;              
		case ALBUM_NAME:
		    if (state == HTML)
			html.append(albumName);
		    break;             
		case PARENT_ALBUM_NAME:
		    break;
		case PARENT_ALBUM_WEB_PATH:
		    break;  
		case ALBUM_LOOP_START_MARKER:
		    if (state == HTML) {
			lexems.removeAllElements();
			state = RECORD_LOOP;
		    } else // loop inside loop is not allowed
			state = END;
		    break;
		case ALBUM_LOOP_END_MARKER:
                    // no album processing for this mode
		    state = HTML;
		    break;  
		case END_DOCUMENT:
		    state = END;
                    html.append(l.text);
		    break;
		default:
		    if (state == HTML)
			html.append(l.text);
		    else if (state == RECORD_LOOP)
			lexems.addElement(l);
		}
	    } while(state != END);            
	    courier.deliver(html, desthtmlname, "text/html", null); // don't use encoding for a while
	    if (images_to_copy != null) {
		String imagePath = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.IMAGEPATH);
		if (imagePath == null)
		    imagePath = "";
		else
		    courier.checkForDestPath(imagePath);
		statusbar.clearProgress();
		statusbar.displayInfo(Resources.INFO_COPYING);
		statusbar.setProgress(images_to_copy.size());
		for (int i=0; i<images_to_copy.size(); i++) {
		    courier.deliver((String)images_to_copy.elementAt(i), imagePath);
		    statusbar.tickProgress();
		}
	    }
	} finally {
	    statusbar.clearInfo();
	    statusbar.clearProgress();
	    courier.done();
	}
    }

    Controller controller;
    Serializer s;
    String album_name;
    AlbumPane album_pane;
    Ftp ftp_connection;
    StringBuffer html;
    Vector images_to_copy;
    StatusBar statusbar;
}

class Lex {
    String text;
    int code;
    Lex(String text, int code) {
        this.text = text;
        this.code = code;
	//System.err.print("Lex: "+code+":"+text);
    }
}

class Lexer {
	final static char CPERCENT = '%';
	final static char CEXLAM = '!';
	final static char[] LEX_MARKER = {CEXLAM, CPERCENT};

	char[] stream;
	int p;
	int state;
	ResourceBundle mnemonicsBundle;
	Hashtable mnemonics;
	
	Lexer(String url) {
		InputStream is = null;
		// note: content length and file length are not used here to 
		// improve relaiability of the code
		try {
			// try it like URL
			try {
				is = new URL(url).openStream();
			} catch(MalformedURLException mfe) {
			}
			if (is == null) { // OK, try it as a file
				is = new FileInputStream(url);
			}
			byte[] buffer = new byte[4096];
			StringBuffer sb = new StringBuffer(buffer.length);
			do {
				int cl = is.read(buffer);
				if (cl < 0)
					break;
				sb.append(new String(buffer, 0, cl));
			} while(true);
			stream = sb.toString().toCharArray();
			is.close();
		} catch(IOException ioe) {
			stream = Resources.TMPL_NOPUBLISHTEMPLATE.toCharArray();
		}
		mnemonics = new Hashtable(HtmlProducer.LEXEM_MENMONICS.length);
		for (int i=0; i<HtmlProducer.LEXEM_MENMONICS.length; i++)
			mnemonics.put(HtmlProducer.LEXEM_MENMONICS[i], new Integer(i));
		state = STATE_IN_HTML;
	}
	
	final static int STATE_IN_HTML = 0;
	final static int STATE_IN_MARKER = 1;
	final static int STATE_IN_LEXEM = 2;
	final static int STATE_IN_ = 3;
	final static String DELIMS = " -/,\\()<>\"'"+LEX_MARKER[0];

	Lex getNextLex() {
		int m = p;
		int ms = LEX_MARKER.length;
		do {
			if (p >= stream.length)
				return new Lex(new String(stream, m, stream.length-m), HtmlProducer.END_DOCUMENT);
			switch(state) {
			case STATE_IN_HTML:
				if (ms > 0) {
					if (stream[p] == LEX_MARKER[LEX_MARKER.length - ms]) {
						state = STATE_IN_MARKER;
						ms--;
						return new Lex(new String(stream, m, p-m), HtmlProducer.PLAIN_HTML);
					}
				}
				break;
			case STATE_IN_MARKER:
				if (ms == 0) { // 
					state = STATE_IN_LEXEM;
					ms = LEX_MARKER.length;
				} else if (stream[p] == LEX_MARKER[LEX_MARKER.length - ms])
					ms--;
				else {
					state = STATE_IN_HTML;
					ms = LEX_MARKER.length;
				}
				break;
			case STATE_IN_LEXEM:
				if (DELIMS.indexOf(stream[p]) >= 0) {
					String text = new String(stream, m+LEX_MARKER.length, p-m-LEX_MARKER.length).toLowerCase();
					Integer li = (Integer)mnemonics.get(text);
					state = STATE_IN_HTML;
					if (li != null)
						return new Lex(text, li.intValue());
				}
			}
			p++;
		} while(true);
	}
}