/* Quat - A 3D fractal generation program */ 
/* Copyright (C) 1997,98 Dirk Meyer */ 
/* (email: dirk.meyer@studbox.uni-stuttgart.de) */ 
/* mail:  Dirk Meyer */ 
/*        Marbacher Weg 29 */ 
/*        D-71334 Waiblingen */ 
/*        Germany */ 
/* */ 
/* This program is free software; you can redistribute it and/or */ 
/* modify it under the terms of the GNU General Public License */ 
/* as published by the Free Software Foundation; either version 2 */ 
/* of the License, or (at your option) any later version. */ 
/* */ 
/* This program is distributed in the hope that it will be useful, */ 
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */ 
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */ 
/* GNU General Public License for more details. */ 
/* */ 
/* You should have received a copy of the GNU General Public License */ 
/* along with this program; if not, write to the Free Software */ 
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. */ 
 
#include "kcolordlg.h"
#include "ColorEditor.moc"
#include "main.h"

#define Inherited ColorEditorData
#define prec 6
// prec...precision of numbers in edit controls

extern QCmds *cmds;

ColorEditor::ColorEditor(QWidget* parent, const char* name, bool nochange) 
   : Inherited( parent, name )
{
   setCaption( "Color Editor" );
   connect(this, SIGNAL(weightingChanged(const char *)), edit_weighting, SLOT(setText(const char *)) );
   connect(this, SIGNAL(red1Changed(const char *)), edit_red1, SLOT(setText(const char *)) );
   connect(this, SIGNAL(green1Changed(const char *)), edit_green1, SLOT(setText(const char *)) );
   connect(this, SIGNAL(blue1Changed(const char *)), edit_blue1, SLOT(setText(const char *)) );
   connect(this, SIGNAL(red2Changed(const char *)), edit_red2, SLOT(setText(const char *)) );
   connect(this, SIGNAL(green2Changed(const char *)), edit_green2, SLOT(setText(const char *)) );
   connect(this, SIGNAL(blue2Changed(const char *)), edit_blue2, SLOT(setText(const char *)) );
   connect(this, SIGNAL(formulaChanged(const char*)), edit_formula, SLOT(setText(const char *)) );

   if (nochange)
   {
      button_sel1->setEnabled(FALSE);
      button_sel2->setEnabled(FALSE);
      button_copy1->setEnabled(FALSE);
      button_copy2->setEnabled(FALSE);
      button_paste1->setEnabled(FALSE);
      button_paste2->setEnabled(FALSE);
      button_add->setEnabled(FALSE);
      button_del->setEnabled(FALSE);
      edit_weighting->setEnabled(FALSE);
      edit_red1->setEnabled(FALSE);
      edit_red2->setEnabled(FALSE);
      edit_green1->setEnabled(FALSE);
      edit_green2->setEnabled(FALSE);
      edit_blue1->setEnabled(FALSE);
      edit_blue2->setEnabled(FALSE);
      edit_formula->setEnabled(FALSE);
      combo_formula->setEnabled(FALSE);
   }
}

void ColorEditor::SetData(struct realpal_struct *p, const char *colscheme)
{
   pal = *p;
   scroll->setRange(1, pal.colnum);
   setPosition(1);
   color_clipboard->SetColor(255, 255, 255);
   setFormula(colscheme);
}

void ColorEditor::GetData(struct realpal_struct *p, char *colscheme)
{  
   char *c;
 
   *p = pal;
   strncpy(colscheme, (const char *)formula, 251);
   c = strchr(colscheme, '#');
   if (c) *c = 0;
}

void ColorEditor::Add()
{
   int z;

   if (pal.colnum<30)
   {
      pal.colnum++;
      for (z=pal.colnum-1; z>idx-1; z--) pal.cols[z] = pal.cols[z-1];
      if (idx>1) for (z=0; z<3; z++)
         pal.cols[idx-1].col1[z] = pal.cols[idx-2].col2[z];
      else for (z=0; z<3; z++) pal.cols[idx-1].col1[z] = pal.cols[idx-1].col2[z];
      for (z=0; z<3; z++) pal.cols[idx-1].col2[z] = pal.cols[idx].col1[z];
      scroll->setRange(1, pal.colnum);
      setPosition(idx | 128);
   }
}

void ColorEditor::Delete()
{
   int z;

   if (pal.colnum>1)
   {
      for (z=idx-1; z<pal.colnum-1; z++) pal.cols[z] = pal.cols[z+1];
      pal.colnum--;
      if (idx-1>=pal.colnum) idx--;
      scroll->setRange(1, pal.colnum);
      setPosition(idx | 128);
   }
}

void ColorEditor::Sel1()
{
   QColor c;
   QString s;

   setCursor(waitCursor);
   if (KColorDialog::getColor(c))
   {
      s.setNum((double)c.red()/255, 'g', prec);
      setRed1((const char*)s);
      s.setNum((double)c.green()/255, 'g', prec);
      setGreen1((const char*)s);
      s.setNum((double)c.blue()/255, 'g', prec);
      setBlue1((const char*)s);
   }
   setCursor(arrowCursor);
}

void ColorEditor::Copy1()
{
   unsigned char r, g, b;

   r = (int)(pal.cols[idx-1].col1[0]*255);
   g = (int)(pal.cols[idx-1].col1[1]*255);
   b = (int)(pal.cols[idx-1].col1[2]*255);
   color_clipboard->SetColor(r, g, b);
}

void ColorEditor::Paste1()
{
   unsigned char r, g, b;
   QString s;

   color_clipboard->GetColor(&r, &g, &b);
   s.setNum((double)r/255, 'g', prec); setRed1((const char*)s);
   s.setNum((double)g/255, 'g', prec); setGreen1((const char*)s);
   s.setNum((double)b/255, 'g', prec); setBlue1((const char*)s);
}

void ColorEditor::Sel2()
{
   QColor c;
   QString s;

   setCursor(waitCursor);
   if (KColorDialog::getColor(c))
   {
      s.setNum((double)c.red()/255, 'g', prec);
      setRed2((const char*)s);
      s.setNum((double)c.green()/255, 'g', prec);
      setGreen2((const char*)s);
      s.setNum((double)c.blue()/255, 'g', prec);
      setBlue2((const char*)s);
   }
   setCursor(arrowCursor);
}

void ColorEditor::Copy2()
{
   unsigned char r, g, b;

   r = (int)(pal.cols[idx-1].col2[0]*255);
   g = (int)(pal.cols[idx-1].col2[1]*255);
   b = (int)(pal.cols[idx-1].col2[2]*255);
   color_clipboard->SetColor(r, g, b);
}

void ColorEditor::Paste2()
{
   unsigned char r, g, b;
   QString s;

   color_clipboard->GetColor(&r, &g, &b);
   s.setNum((double)r/255, 'g', prec); setRed2((const char*)s);
   s.setNum((double)g/255, 'g', prec); setGreen2((const char*)s);
   s.setNum((double)b/255, 'g', prec); setBlue2((const char*)s);
}

void ColorEditor::setPosition(int i)
{
   QString s;

   if (i!=idx || (i & 128) != 0)
   {
      idx = i & 63;
      s.setNum(pal.cols[idx-1].weight, 'g', prec);
      setWeighting((const char *)s);
      s.setNum(pal.cols[idx-1].col1[0], 'g', prec);
      setRed1((const char *)s);
      s.setNum(pal.cols[idx-1].col1[1], 'g', prec);
      setGreen1((const char *)s);
      s.setNum(pal.cols[idx-1].col1[2], 'g', prec);
      setBlue1((const char *)s);
      s.setNum(pal.cols[idx-1].col2[0], 'g', prec);
      setRed2((const char *)s);
      s.setNum(pal.cols[idx-1].col2[1], 'g', prec);
      setGreen2((const char *)s);
      s.setNum(pal.cols[idx-1].col2[2], 'g', prec);
      setBlue2((const char *)s);
      s.setNum(idx);
      label_position->setText((const char *)s);
      color_preview->Update(&pal, idx);
   }
}

void ColorEditor::setWeighting(const char *c)
{
   double d;
 
   weighting.setStr(c);
   d = weighting.toDouble(NULL);
   if (strcmp(c, edit_weighting->text()) != 0) emit weightingChanged(c);
   if (d != pal.cols[idx-1].weight)
   {
      pal.cols[idx-1].weight = d;
      color_preview->Update(&pal, idx);
   }
}

void ColorEditor::setRed1(const char *c)
{
   double d;

   red1.setStr(c);
   d = red1.toDouble(NULL);
   if (strcmp(c, edit_red1->text()) != 0) emit red1Changed(c);
   if (pal.cols[idx-1].col1[0] != d)
   {
      pal.cols[idx-1].col1[0] = d;
      color_preview->Update(&pal, idx);
   }
}

void ColorEditor::setGreen1(const char *c)
{
   double d;

   green1.setStr(c);
   d = green1.toDouble(NULL);
   if (strcmp(c, edit_green1->text()) != 0) emit green1Changed(c);
   if (pal.cols[idx-1].col1[1] != d)
   {
      pal.cols[idx-1].col1[1] = d;
      color_preview->Update(&pal, idx);
   }
}

void ColorEditor::setBlue1(const char *c)
{
   double d;

   blue1.setStr(c);
   d = blue1.toDouble(NULL);
   if (strcmp(c, edit_blue1->text()) != 0) emit blue1Changed(c);
   if (pal.cols[idx-1].col1[2] != d)
   {
      pal.cols[idx-1].col1[2] = d;
      color_preview->Update(&pal, idx);
   }
}

void ColorEditor::setRed2(const char *c)
{
   double d;

   red2.setStr(c);
   d = red2.toDouble(NULL);
   if (strcmp(c, edit_red2->text()) != 0) emit red2Changed(c);
   if (pal.cols[idx-1].col2[0] != d)
   {
      pal.cols[idx-1].col2[0] = d;
      color_preview->Update(&pal, idx);
   }
}

void ColorEditor::setGreen2(const char *c)
{
   double d;

   green2.setStr(c);
   d = green2.toDouble(NULL);
   if (strcmp(c, edit_green2->text()) != 0) emit green2Changed(c);
   if (pal.cols[idx-1].col2[1] != d)
   {
      pal.cols[idx-1].col2[1] = d;
      color_preview->Update(&pal, idx);
   }
}

void ColorEditor::setBlue2(const char *c)
{
   double d;
 
   blue2.setStr(c);
   d = blue2.toDouble(NULL);
   if (strcmp(c, edit_blue2->text()) != 0) emit blue2Changed(c);
   if (pal.cols[idx-1].col2[2] != d)
   {
      pal.cols[idx-1].col2[2] = d;
      color_preview->Update(&pal, idx);
   }
}

void ColorEditor::setFormula(const char *c)
{
   formula.setStr(c);
   if (strcmp(c, edit_formula->text()) != 0) emit formulaChanged(c);
}

void ColorEditor::SaveAs()
{
   cmds->Backup();
   cmds->realpal = pal;
   strcpy(cmds->colscheme, (const char*)formula);
   cmds->DoSaveParamsAs(".col", PS_COL);
   cmds->Restore();
   return;
}

ColorEditor::~ColorEditor()
{
}
