/* PhotoOrganizer 
 * Copyright (C) 1999 Dmitry Rogatkin.  All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 *  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package photoorganizer;

import java.util.Hashtable;
//import java.util.Enumeration;
import java.util.Properties;
import java.util.Vector;
import java.util.Locale;
import java.awt.event.*;
//import java.awt.Component;
import java.awt.Image;
import java.awt.Toolkit;
import java.awt.FlowLayout;
import java.awt.PrintJob;
import java.awt.Color;
//import java.awt.Container;
//import java.awt.print.PrinterJob;
import java.awt.Graphics;
import java.awt.Dimension;
import java.awt.image.ImageObserver;
import java.io.File;
//import java.io.IOException;
//import java.io.BufferedInputStream;
//import java.io.BufferedOutputStream;
//import java.io.FileInputStream;
//import java.io.FileOutputStream;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.filechooser.FileFilter;

import rogatkin.*;
import photoorganizer.directory.*;
import photoorganizer.renderer.*;
import photoorganizer.formats.*;

public class Controller extends BaseController implements ActionListener {
    // component constraint
    public final static Integer COMP_IMAGEPANEL       = new Integer(1);
    public final static Integer COMP_DIRTREE          = new Integer(2);
    public final static Integer COMP_THUMBPANEL       = new Integer(3);
    public final static Integer COMP_FORMATS          = new Integer(4);
    public final static Integer COMP_COLLECTION       = new Integer(5);
    public final static Integer COMP_THUMBCOLLCTPANEL = new Integer(6);
    public final static Integer COMP_IMAGECOLLCTPANEL = new Integer(7);
    public final static Integer COMP_ALBUMPANEL       = new Integer(8);
    public final static Integer COMP_STATUSBAR        = new Integer(9);
    public final static Integer COMP_IMAGEALBUMPANEL  = new Integer(10);
    public final static Integer COMP_ALBUMTHUMBPANEL  = new Integer(11);
    public final static Integer COMP_WEBALBUMPANEL    = new Integer(12);

    public SampleJpeg sampleJpeg;

    public Controller(PhotoOrganizer photoorganizer) {
	super(photoorganizer);
        sampleJpeg = new SampleJpeg();
        filter     = new HTMLFilter();
        uiupdater  = new UiUpdater();
        mainicon   = getResourceIcon(Resources.IMG_PHOTOORG).getImage();
        if (java2 && System.getProperty("USE_JAI") != null) {
            try {
                advancedimage = (AdvancedImage)Class.forName("photoorganizer.jai.AdvancedImageImpl").newInstance();
            } catch(ClassNotFoundException e) {
                System.err.println(e);
            } catch(NoClassDefFoundError e) {
                System.err.println(e);
            } catch(InstantiationException e) {
                System.err.println(e);
            } catch(IllegalAccessException e) {
                System.err.println(e);
            }
        }
        serializer.load();
    }   
    
    public void actionPerformed(ActionEvent a) {
        String cmd = a.getActionCommand();
        if (cmd.equals(Resources.MENU_OPTIONS)) {
            if (options == null) {
                options = new OptionsFrame(this);
            } else 
                options.setVisible(true);
        } else if (cmd.equals(Resources.MENU_ABOUT)) {
            JOptionPane.showMessageDialog(getMainFrame(), "<html><i>"+PhotoOrganizer.PROGRAMNAME+"\n"+
                PhotoOrganizer.VERSION+'.'+PhotoOrganizer.BUILD+'\n'+
                PhotoOrganizer.COPYRIGHT+'\n'+
                "Java "+System.getProperty("java.version")+
                (java2?(" spec "+System.getProperty("java.specification.name")):"")+'\n'+
                "JVM "+System.getProperty("java.vendor")+
                " OS "+System.getProperty("os.name")+' '+System.getProperty("os.version")+' '+System.getProperty("os.arch")+'\n'+
                (java2?("Available "+(Runtime.getRuntime().totalMemory()/1024/1024)+"MB, free "+(Runtime.getRuntime().freeMemory()/1024/1024)+"MB\n"):"")+
                Locale.getDefault().getDisplayName()+
                (advancedimage!=null?"\nJava Advanced Imaging":""),
                Resources.MENU_ABOUT, JOptionPane.PLAIN_MESSAGE,
                getResourceIcon(Resources.IMG_LOGO));
        } else if (cmd.equals(Resources.MENU_CONTENTS)) {
            showUrl(Resources.URL_HELP);
        } else if (cmd.equals(Resources.MENU_VIEW_HTML)) {
            String d = (String)serializer.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.WEBROOT);
            if (d == null || d.length() == 0)
                d = serializer.getHomeDirectory();
            
	    Object[] options = {Resources.CMD_OPEN, Resources.CMD_CANCEL, Resources.CMD_BROWSE};
	    
	    JOptionPane pane = new JOptionPane(Resources.LABEL_INPUT_URL, 
		JOptionPane.QUESTION_MESSAGE,
		JOptionPane.OK_CANCEL_OPTION,
		null, options, options[0]);
	    pane.setWantsInput(true);
	    pane.setInitialSelectionValue("http://"+serializer.getProperty(WebPublishOptionsTab.SECNAME,
		WebPublishOptionsTab.HOSTNAME)+'/'+PhotoCollectionPanel.LastHtmlName);
	    pane.selectInitialValue();
	    
	    JDialog dialog = pane.createDialog(getMainFrame(), Resources.TITLE_URL_SEL);
	    dialog.show();
	    String selectedValue = (String) pane.getValue();
	    
	    if (selectedValue == JOptionPane.UNINITIALIZED_VALUE ||
		options[0].equals(selectedValue)) {
		// user entered URL and selected open
		showUrl((String) pane.getInputValue());
	    } else if (options[1].equals(selectedValue)) {
		// user selected cancel
	    } else if (selectedValue != null) {
		// user wants to open a file
                JFileChooser fc = new JFileChooser(d);
                fc.addChoosableFileFilter(filter);
                fc.setFileFilter(filter);
                fc.setDialogTitle(Resources.TITLE_HTML_TEMPL);
                if (fc.showOpenDialog(getMainFrame()) == JFileChooser.APPROVE_OPTION)
                    showUrl(fc.getSelectedFile().getAbsolutePath());
	    }
        } else if (cmd.equals(Resources.MENU_EXIT)) {
            close();
            getMainFrame().dispose();
            System.exit(0);
        }
    }

    public synchronized void print(File[] files) {
        final PrintJob pj = Toolkit.getDefaultToolkit().getPrintJob(getMainFrame(),
            Resources.TITLE_PRINT, new Properties());
        final StatusBar statusbar = (StatusBar)component(COMP_STATUSBAR);
        statusbar.displayInfo(Resources.INFO_PRINTING);
        if (files != null)
            statusbar.setProgress(files.length*2);
        final Graphics g = pj.getGraphics();
        g.setColor(Color.white);
        final Dimension pd = pj.getPageDimension();
        for (int i=0; files != null && i<files.length; i++) {
            g.setClip(0,i*pd.height, pd.width, pd.height);
            g.drawRect(0,20, pd.width-10, pd.height-20-10);
            Image im = new BasicJpeg(files[i]).getImage();
            final boolean lastimage = i == (files.length-1);
            if (im == null)
                continue;
            final File file = files[i];
            if (!g.drawImage(im, 0,0,
                new ImageObserver() {
                    public boolean imageUpdate(Image img, int infoflags, int x, int y,
                        int width, int height) {
                        if (y == height/2 && height != 0)
                            statusbar.tickProgress();
                        if ((infoflags & (ImageObserver.ALLBITS | ImageObserver.ERROR)) == 0)
                            return true;
                        else {
                            g.drawImage(img, 0,15, pd.width-30, (pd.width-30)*height/width, null);
                            img.flush();
                            g.drawString(file.getAbsolutePath(), 15, (pd.width-30)*height/width+30);
                            statusbar.tickProgress();
                            if (lastimage) {
                                g.dispose();
                                pj.end();
                                statusbar.clearInfo();
                                statusbar.clearProgress();
                            }
                            return false;
                        }
                    }
                }
            )
        ) // if
        ;
        }
    }
    
    public UiUpdater getUiUpdater() {
        return uiupdater;
    }

    public Image getMainIcon() {
        return mainicon;
    }

    public void updateCaption(String str) {
        if (str == null)
            str = "";
	// TODO: check for null
        getMainFrame().setTitle(str);
    }

    public static AdvancedImage getAdvancedImage() {
        return advancedimage;
    }

    public int adjustMenuY(int y, int h) {
        if (y > screen_height-h)
            return screen_height-h;
        else
            return y; 
    }

    public FileFilter getHtmlFilter() {
        return filter;
    }

    public void close() {
        save();
        main.save();
        serializer.save();
    }

    public static JPanel createButtonPanel(ActionListener al) {
        JButton btn;
        JPanel result = new JPanel();
        result.setLayout(new FlowLayout(FlowLayout.RIGHT));
        result.add(btn = new JButton(Resources.CMD_OK));
        btn.addActionListener(al);
        result.add(btn = new JButton(Resources.CMD_APPLY));
        btn.addActionListener(al);
        result.add(btn = new JButton(Resources.CMD_CANCEL));
        btn.addActionListener(al);
        return result;
    }

    public static JMenu createTransformMenu(ActionListener listener) {
        JMenu result = new JMenu(Resources.MENU_TRANSFORM);
        JMenuItem item;
        result.add(item = new JMenuItem(Resources.MENU_NONE));
        item.addActionListener(listener);
        result.add(item = new JMenuItem(Resources.MENU_ROTATE90));
        item.addActionListener(listener);
        result.add(item = new JMenuItem(Resources.MENU_ROTATE180));
        item.addActionListener(listener);
        result.add(item = new JMenuItem(Resources.MENU_ROTATE270));
        item.addActionListener(listener);
        result.add(item = new JMenuItem(Resources.MENU_FLIP_H));
        item.addActionListener(listener);
        result.add(item = new JMenuItem(Resources.MENU_FLIP_V));
        item.addActionListener(listener);
        result.add(item = new JMenuItem(Resources.MENU_TRANSPOSE));
        item.addActionListener(listener);
        result.add(item = new JMenuItem(Resources.MENU_TRANSVERSE));
        item.addActionListener(listener);
	result.addSeparator();
        result.add(item = new JMenuItem(Resources.MENU_COMMENT));
        item.addActionListener(listener);
        return result;
    }
    
    public static int convertCmdToTrnasformOp(String cmd) {
        int op = -1;
        if (cmd.equals(Resources.MENU_ROTATE90)) {
            op = BasicJpeg.ROT_90;
        } else if (cmd.equals(Resources.MENU_ROTATE180)) {
            op = BasicJpeg.ROT_180;
        } else if (cmd.equals(Resources.MENU_ROTATE270)) {
            op = BasicJpeg.ROT_270;
        } else if (cmd.equals(Resources.MENU_FLIP_H)) {
            op = BasicJpeg.FLIP_H;
        } else if (cmd.equals(Resources.MENU_FLIP_V)) {
            op = BasicJpeg.FLIP_V;
        } else if (cmd.equals(Resources.MENU_TRANSPOSE)) {
            op = BasicJpeg.TRANSPOSE;
        } else if (cmd.equals(Resources.MENU_TRANSVERSE)) {
            op = BasicJpeg.TRANSVERSE;
        } else if (cmd.equals(Resources.MENU_NONE)) {
            op = BasicJpeg.NONE;
        } else if (cmd.equals(Resources.MENU_COMMENT)) {
            op = BasicJpeg.COMMENT;
        }
        return op;
    }

    class HTMLFilter extends FileFilter  {
        final String [] extensions = new String[] {"html", "htm", "htmt"};
        
        // Accept all directories and (gif || jpg || tiff) files.
        public boolean accept(File f) {
            if(f.isDirectory()) {
                return true;
            }
            String s = f.getName();
            int i = s.lastIndexOf('.');
            if(i > 0 &&  i < s.length() - 1) {
                String extension = s.substring(i+1).toLowerCase();
                for (i=0; i<extensions.length; i++) {
                    if (extensions[i].equals(extension)) {
                        return true;
                    } else {
                        return false;
                    }
                }
            }
            return false;
        }
        
        // The desctiption of this filter
        public String getDescription() {
            return Resources.LABEL_HTML_FILES;
        }
    }

    private Image mainicon;
    private String transformtargetdir;
    private OptionsFrame options;
    private HTMLFilter filter;
    private UiUpdater uiupdater;
    private static AdvancedImage advancedimage;
    /*
        synchronized(pj) {
            pj.notify();
        }
        synchronized(pj) {
            
            try {
                pj.wait(10*60*1000);
            } catch(InterruptedException e) {
            }
        }*/
   
}

