/* PhotoOrganizer - HTTPCourier 
 * Copyright (C) 1999-2000 Dmitry Rogatkin.  All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 *  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package photoorganizer.courier;

import java.io.*;
import java.net.*;
import java.util.*;

import rogatkin.*;
import photoorganizer.*;
import photoorganizer.renderer.*;
import photoorganizer.formats.*;
import photoorganizer.Courier;

// TODO: remove calculation content length, HttpURLConnection does it

public class HTTPCourier implements Courier {

    final static String CONTENT_TYPE = "Content-Type";
    final static String CONTENT_LENGTH = "Content-Length";
    final static String SET_COOKIE = "Set-Cookie";

    final static String CONTENT_DISP = "Content-Disposition: form-data; name=\"";
    final static String FILENAME = "\"; filename=\"";
    final static String CONTENT_ENCODING = "Content-Encoding: ";
    final static String CONTENT_TYPE_ = CONTENT_TYPE+": \"";
    final static String MULTIPART = "multipart/form-data, boundary=";
    final static String POST_ENCODING = "application/x-www-form-urlencoded";
    final static String DEFAULT_CONTENTTYPE = "application/octet-stream";
    final static String SEP = "--";
    final static String COOKIE = "Cookie";

    final static int AUTH_COOKIE = 1;
    final static int METHOD_GET  = 0;
    final static int METHOD_POST = 1;

    final static int BUFSIZE = 1024;

    Controller controller;
    Serializer s;

	public HTTPCourier(Controller controller) {
		this.controller = controller;
	}

	public void deliver(StringBuffer buf, String destPath, String contentType, String encoding) throws IOException {
		HttpURLConnection con = getConnectedToPublish();
		String boundary = genBoundary();
		con.setRequestProperty(CONTENT_TYPE, MULTIPART+boundary);
		// compute content length
		int len = (boundary.length()+2)*3+SEP.length()+2;
		len += CONTENT_DISP.length();
		String newName = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.UPL_DEST_NAME);
		if (newName == null || newName.length() == 0)
			newName = WebPublishOptionsTab.UPL_DEST_NAME;
		len += newName.length()+1+2+2;
		len += destPath.length()+2;
		String dataName = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.UPL_DATA_NAME);
		if (dataName == null || dataName.length() == 0)
			dataName = WebPublishOptionsTab.UPL_DATA_NAME;
		len += CONTENT_DISP.length();
		len += dataName.length()+1;
		len += FILENAME.length();
		len += destPath.length()+1+2;
		if (contentType != null && contentType.length() > 0)
			len += contentType.length() + CONTENT_TYPE_.length()+1+2;
		if (encoding != null && encoding.length() > 0) // do not use for a while
			;
		len += 2;
		len += buf.length();
		//con.setRequestProperty(CONTENT_LENGTH, ""+len);
		PrintWriter osw = new PrintWriter(con.getOutputStream());
		osw.println(boundary);
		osw.print(CONTENT_DISP);
		osw.print(newName); osw.println('"');
		osw.println();
		osw.println(destPath);
		osw.println(boundary);
		osw.print(CONTENT_DISP);
		osw.print(dataName); osw.print(FILENAME); osw.print(destPath); osw.println('"');
		if (contentType != null && contentType.length() > 0) {
			osw.print(CONTENT_TYPE_);
			osw.print(contentType); osw.println('"');
		}
		if (encoding != null && encoding.length() > 0) // do not use for a while 
			;
		osw.println();
		osw.println(buf);
		osw.println(boundary+SEP);
		osw.println();
		osw.close();
		System.err.println("Done html "+con.getContent());
		System.err.println("Result :"+con.getResponseCode()+"/"+con.getResponseMessage());
		con.disconnect();
	}

	public void deliver(String srcPath, String destPath) throws IOException {
		HttpURLConnection con = getConnectedToPublish();
		String boundary = genBoundary();
		con.setRequestProperty(CONTENT_TYPE, MULTIPART+boundary);
		// compute content length
		int len = (boundary.length()+2)*3+SEP.length()+2;
		len += CONTENT_DISP.length();
		String newName = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.UPL_DEST_NAME);
		if (newName == null || newName.length() == 0)
			newName = WebPublishOptionsTab.UPL_DEST_NAME;
		len += newName.length()+1+2+2;
		String name = new File(srcPath).getName();
		if (destPath == null)
			destPath = "";
		if (destPath.length() > 0)
			if (destPath.charAt(destPath.length()-1) != '/' && destPath.charAt(destPath.length()-1) != '\\')
				destPath += '/';        
		destPath += name;
		len += destPath.length()+2;
		String dataName = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.UPL_DATA_NAME);
		if (dataName == null || dataName.length() == 0)
			dataName = WebPublishOptionsTab.UPL_DATA_NAME;
		len += CONTENT_DISP.length();
		len += dataName.length()+1;
		len += FILENAME.length();
		len += srcPath.length()+1+2;
		String contentType = null;
		try {
			contentType = URLConnection.getFileNameMap().getContentTypeFor(srcPath);
		} catch(Throwable t) {
			// JDK 1.1
		}
		if (contentType == null && contentType.length() == 0)
			contentType = DEFAULT_CONTENTTYPE;
		len += contentType.length() + CONTENT_TYPE_.length()+1+2;
		len += 2;
		File sf = new File(srcPath);
		len += (int)sf.length()+2;
		//con.setRequestProperty(CONTENT_LENGTH, ""+len);
		OutputStream os;
		PrintWriter osw = new PrintWriter(os = con.getOutputStream());
		osw.println(boundary);
		osw.print(CONTENT_DISP);
		osw.print(newName); osw.println('"');
		osw.println();
		osw.println(destPath);
		osw.println(boundary);
		osw.print(CONTENT_DISP);
		osw.print(dataName); osw.print(FILENAME); osw.print(srcPath); osw.println('"');
		osw.print(CONTENT_TYPE_);
		osw.print(contentType); osw.println('"');
		osw.println();
		osw.flush();
		Controller.copyFile(sf, os);
		os.flush();
		osw.println();
		osw.println(boundary+SEP);
		osw.println();
		osw.close();
		System.err.println("Done image "+con.getContent());
		System.err.println("Result :"+con.getResponseCode()+"/"+con.getResponseMessage());
		con.disconnect();
	}
	
	
	public void checkForDestPath(String path) throws IOException {
	}
	
	public String deliver(BasicJpeg format, String destPath) throws IOException {
		HttpURLConnection con = getConnectedToPublish();
		String boundary = genBoundary();
		con.setRequestProperty(CONTENT_TYPE, MULTIPART+boundary);
		// compute content length
		int len = (boundary.length()+2)*3+SEP.length()+2;
		len += CONTENT_DISP.length();
		String newName = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.UPL_DEST_NAME);
		if (newName == null || newName.length() == 0)
			newName = WebPublishOptionsTab.UPL_DEST_NAME;
		len += newName.length()+1+2+2;
		AbstractImageInfo ii = format.getImageInfo();
		if (ii == null)
			return "";
		String name = FileNameFormat.makeValidPathName(new FileNameFormat(mask, true).format(format), ii.getThumbnailExtension());
		if (destPath == null)
			destPath = "";
		if (destPath.length() > 0)
			if (destPath.charAt(destPath.length()-1) != '/' && destPath.charAt(destPath.length()-1) != '\\')
				destPath += '/';        
		destPath += name;
		len += destPath.length()+2;
		String dataName = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.UPL_DATA_NAME);
		if (dataName == null || dataName.length() == 0)
			dataName = WebPublishOptionsTab.UPL_DATA_NAME;
		len += CONTENT_DISP.length();
		len += dataName.length()+1;
		len += FILENAME.length();
		len += name.length()+1+2;
		String contentType = "image/"+ii.getThumbnailExtension();
		len += contentType.length() + CONTENT_TYPE_.length()+1+2;
		ByteArrayOutputStream os = new ByteArrayOutputStream(8192);
		ii.saveThumbnailImage(format, os);
		len += os.size()+2;
		//con.setRequestProperty(CONTENT_LENGTH, ""+len);
		OutputStream os2;
		PrintWriter osw = new PrintWriter(os2 = con.getOutputStream());
		osw.println(boundary);
		osw.print(CONTENT_DISP);
		osw.print(newName); osw.println('"');
		osw.println();
		osw.println(destPath);
		osw.println(boundary);
		osw.print(CONTENT_DISP);
		osw.print(dataName); osw.print(FILENAME); osw.print(name); osw.println('"');
		osw.print(CONTENT_TYPE_);
		osw.print(contentType); osw.println('"');
		osw.println();
		osw.flush();
		os2.write(os.toByteArray());
		os2.flush();
		osw.println();
		osw.println(boundary+SEP);
		osw.println();
		osw.close();
		System.err.println("Done thumbnail "+con.getContent());
		System.err.println("Result :"+con.getResponseCode()+"/"+con.getResponseMessage());
		con.disconnect();
		return destPath;
	}

	public void init() throws IOException {
		s = controller.getSerializer();
		// do login if authentication requested
		authMode = Serializer.getInt(s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTP_AUTH_SHC), 0);
		if (authMode == AUTH_COOKIE) {
			HttpURLConnection con = null;
			String query = URLEncoder.encode((String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTPLOGIN_NAME));
			query += '=';
			query += URLEncoder.encode((String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTPLOGIN));
			query += '&';
			query += URLEncoder.encode((String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTPPASSWORD_NAME));
			query += '=';
			String sp = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTPPASSWORD);
			if (sp != null) {
				try {
					query += URLEncoder.encode(Controller.encryptXor(
						new String(BaseController.hexToBytes(sp), BaseController.ISO_8859_1)));
				} catch(UnsupportedEncodingException uee) {
				}
			}
			query += '&';
			// no URL encode for additional part, TODO: make a tokenizer and do encode
			query += (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTPSTATICQUERY);
			if (Serializer.getInt(s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTPLOGINMETHOD), METHOD_GET) == METHOD_GET) {
				try {
					con = (HttpURLConnection)new URL((String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTPLOGINURL)+'?'+query).openConnection();
				} catch(MalformedURLException mue) {
				}
			} else { // POST
				try {
					con = (HttpURLConnection)new URL((String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTPLOGINURL)).openConnection();
					con.setDoOutput(true);
					con.setUseCaches(false);
					con.setAllowUserInteraction(false);
					// note: not necessary to set method, content type, and lenght
					// HttpURLConnection does it authomatically (
					query += "\r\n";
					// send parameters
					PrintWriter out = new PrintWriter(con.getOutputStream());
					out.print(query);
					out.flush();
					out.close();
				} catch(MalformedURLException mue) {
				}
			}
			if (con != null) {
				//con.getContentType();
				//System.err.println("Content:\n"+con.getContent());
				int respCode = con.getResponseCode();
				if (respCode == HttpURLConnection.HTTP_OK) {
					InputStream in = con.getInputStream();
					byte []buf = new byte[2048];
					byte []im = new byte[0];
					try {
						int cl;
						do {
							cl = in.read(buf);
							if (cl < 0)
								break;
							byte []wa = new byte[im.length+cl];
							System.arraycopy(im, 0, wa, 0, im.length);
							System.arraycopy(buf, 0, wa, im.length, cl);
							im = wa;
						} while(true);
					} finally {
						in.close();
					}
					System.err.println("Content:\n"+new String(im));
				} else if (respCode == HttpURLConnection.HTTP_NO_CONTENT) {
					System.err.println("The request is OK, but server returned no data");
				} else {
					System.err.println("Server return error code "+respCode);
				}
				
				System.err.println("Cookie from srv "+con.getHeaderField(SET_COOKIE));
				con.disconnect();
			}
		}
		try {
			publisherURL = new URL((String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.UPL_SERVLET_URL));
		} catch(MalformedURLException mue) {
		} catch(NullPointerException npe) {
		}

		mask = (String)s.getProperty(ThumbnailsOptionsTab.SECNAME, ThumbnailsOptionsTab.FILEMASK);
		if (mask == null || mask.length() == 0)
			mask = PhotoCollectionPanel.DEFTNMASK;
	}

	public void done() {

	}

	public boolean isLocal() {
		return false;
	}
	
	public boolean isContentIncluded() {
		return false;
	}

	private String genBoundary() {
		return "---------------------------"+Long.toHexString(new Random().nextLong());
	}

	private HttpURLConnection getConnectedToPublish() throws IOException {
		HttpURLConnection result = (HttpURLConnection)publisherURL.openConnection();
		result.setDoOutput(true);
		result.setDoInput(true);
		result.setUseCaches(false);
		result.setAllowUserInteraction(false);
		if (authMode == AUTH_COOKIE && cookieValue != null)
			result.setRequestProperty(COOKIE, cookieValue);
		return result;
	}
    
    URL publisherURL;
    String mask;
    int authMode;
    String cookieValue;
}