/*Data Access Library. v. 1.1.
 *Data Access SQL Generator. 
 *AUTHOR:  Alexander Jaremenko <jarem@altavista.net>
 *RELEASE DATE: 
 */

package JProjects.eab.data;

/**Generates SQL for Manager classes.
 *@author Alexander Jaremenko  <address><a href="mailto:jarem@altavista.net">&lt jarem@altavista.net &gt</a></address>
 */
public class DASQLGenerator implements java.io.Serializable {
    private String _tblName;
    private String[] _pkCols;
    private String[] _cols;
    private String[] _mpkCols;

    /** Public constructor.
     * @param tbl table's name
     * @param pkCols column's names for columns that constitute primary key
     * @param cols other column's names of the table. May be null.
     * @param mPk primary key column's names that can be modified. May be null.
     */
    public DASQLGenerator(String tbl,String[] pkCols,String[] cols,String[] mPk) {
	_tblName = tbl;
	_pkCols = pkCols;
	_cols = ((cols==null)?new String[0]:cols);
	_mpkCols = mPk;
    }

    /** Generates parametrized SQL update statement for table. Assumes that at 
     *	least one element of modified array is true.
     * @param tblQual qualifier of the table
     * @param modified array of columns modification indicators. Sequance must
     * correspond those that in constructor's cols arguments.
     */
    public String genUpdate(String tblQual, boolean[] modified) {
	String res = "update "+((tblQual!=null && tblQual.length()>0)?(tblQual + "."):"")+_tblName;
	res+= " " + genUpdCols(_cols,modified) + " " + genWhereClause();
	return res;
    }

    /** Generates parametrized SQL insert statement for table. 
     * @param tblQual qualifier of the table.
     *@return - generated SQL statement.
     */
    public String genInsert(String tblQual) {
	String res="insert into "+((tblQual!=null && tblQual.length()>0)?(tblQual+"."):"")+_tblName;
	String[] allCols = new String[_cols.length + ((_mpkCols!=null)?_mpkCols.length:0)];
	boolean[] sel = new boolean[allCols.length];
	for (int i=0;i<sel.length;i++)
	    sel[i]=true;
	if (_mpkCols!=null) {
	    System.arraycopy(_mpkCols,0,allCols,0,_mpkCols.length);
	    System.arraycopy(_cols,0,allCols,_mpkCols.length,_cols.length);
	} else
	    System.arraycopy(_cols,0,allCols,0,_cols.length);
	res+="("+getColList(allCols,sel,"",",")+") ";
	res+="values (";
	for (int i=0;i<sel.length-1;i++)
	    res+="?,";
	res+="?)";
	return res;
    }

    /** Generates parametrized SQL delete statement for table. 
     * @param tblQual qualifier of the table.
     *@return - generated SQL statement.
     */
    public String genDelete(String tblQual) {
	String res="delete from "+((tblQual!=null && tblQual.length()>0)?(tblQual+"."):"")+_tblName;
	boolean sel[] = new boolean[_pkCols.length];
	for (int i=0;i<sel.length;i++)
	    sel[i]=true;
	res+=" where "+getColList(_pkCols,sel," =? "," and ");
	return res;
    }

    /** Generates parametrized SQL update statement for table's primary key. 
     * Assumes that at least one element of modified array is true.
     * @param tblQual qualifier of the table
     * @param modified array of columns modification indicators. Sequance must
     * correspond those that in constructor's cols arguments.
     */
    public String genUpdateDataId(String tblQual, boolean[] modified) {
	String res = "update "+((tblQual!=null && tblQual.length()>0)?(tblQual + "."):"")+_tblName;
	String[] allCols = new String[_cols.length + ((_mpkCols!=null)?_mpkCols.length:0)];
	boolean[] sel = new boolean[allCols.length];
	if (_mpkCols!=null) {
	    System.arraycopy(_mpkCols,0,allCols,0,_mpkCols.length);
	    for (int i=0;i<_mpkCols.length;i++)
		sel[i] = true;
	    if (modified!=null) {
		System.arraycopy(_cols,0,allCols,_mpkCols.length,_cols.length);
		System.arraycopy(modified,0,sel,_mpkCols.length,modified.length);
	    }
	}
	else {
	    System.arraycopy(_cols,0,allCols,0,_cols.length);
	    System.arraycopy(modified,0,sel,0,modified.length);
	}
	res+= " " + genUpdCols(allCols,sel) + " " + genWhereClause();
	return res;
    }

    /** Generates parametrized update statement for positioned update.
     * @param tblQual qualifier of the table
     * @param idModified array of columns modification indicators. Sequance must
     * correspond those that in constructor's mPk argument.
     * @param modified array of columns modification indicators. Sequance must
     * correspond those that in constructor's cols argument.
     */
    public String genUpdateFetched(String tblQual, boolean[] idModified,boolean[] modified) {
	int lastIdx = 0;
	String res = "update "+((tblQual!=null && tblQual.length()>0)?(tblQual + "."):"")+_tblName;
	String[] allCols = new String[_cols.length + ((_mpkCols!=null)?_mpkCols.length:0)];
	boolean[] sel = new boolean[allCols.length];
	if (_mpkCols!=null) {
	    lastIdx = _mpkCols.length;
	    System.arraycopy(_mpkCols,0,allCols,0,lastIdx);
	    System.arraycopy(idModified,0,sel,0,lastIdx);
	}
	if (modified != null) {
	    System.arraycopy(_cols,0,allCols,lastIdx,_cols.length);
	    System.arraycopy(modified,0,sel,lastIdx,modified.length);
	}
	res+= " "+genUpdCols(allCols,sel) + " where current of ";
	return res;
    }

    /** Generates parametrized update statement for positioned update.
     * @param tblQual qualifier of the table
     * @param idModified array of columns modification indicators. Sequance must
     * correspond those that in constructor's mPk argument.
     */
    public String genUpdateDataIdFetched(String tblQual, boolean[] idModified) {
	String res = "update "+((tblQual!=null && tblQual.length()>0)?(tblQual + "."):"")+_tblName;
	res+= " "+genUpdCols(_mpkCols,idModified) + " where current of ";
	return res;
    }

    private String genUpdCols(String[] cols,boolean[] sel) {
	String res = "set (" + getColList(cols,sel,"",",") + ") = ";
	for (int i=0,vals=0; i < sel.length; i++)
	    if (sel[i]) {
		res+=((vals==0)?"(?":",?");
		vals++;
	    }
	res+= ")";
	return res;
    }

    private String getColList(String[] cols,boolean[] selector,String sfx,String separator) {
	boolean isFst = true;
	String res = "";
	for (int i=0;i<selector.length;i++) {
	    if (selector[i])
		if (isFst) {
		    isFst = false;
		    res = new String(cols[i] + sfx);
		}
		else res+= separator + cols[i] + sfx;
	}
	return res;
    }

    private String genWhereClause() {
	String res = "where "+ _pkCols[0]+"= ?";
	for (int i=1; i < _pkCols.length; i++) 
	    res+=" and "+_pkCols[i]+"= ?";
	return res+" ";
    }

}
