package com.tolstoy.imagemeister;

import com.jconfig.*;
import java.io.*;
import java.awt.*;
import java.util.*;

/**
This is the main class.

It owns a BrowserDisplay object which represents the GUI portion.

It owns the image and info viewers currently displayed, and it keeps these two sets of viewers in separate vectors.

This class directs which objects are shown in the main window in response to user actions.

This class implements the BrowserDisplayOwnerI interface, and as the user clicks buttons and objects in the GUI,
methods of this object are called back by the BrowserDisplay object.

This class implements the VisitURLRecipient interface; when the visitURLButtonClicked() method is called,
we launch the URL 'kURLString' in the user's browser. You'll probably call this method in your about box,
and you'll probably want to go to your own URL.

This class implements the DDCommandRecipient interface, so we can accept commands from external programs,
such as our shell app.

This class implements the ImageViewerOwner and InfoViewerOwner interfaces, which lets this class know
when image and info viewers are cloaed.
 
@author Copyright (c) 1997,1998,1999,2000 Samizdat Productions. All Rights Reserved.
*/

class Browser implements BrowserDisplayOwnerI,
							WindowOwner,
							VisitURLRecipient,
							ImageViewerOwner,
							InfoViewerOwner,
							DDCommandRecipient {
	private static final String copyright="ImageMeister Copyright (c) 1998,1999,2000 Samizdat Productions. All Rights Reserved. ImageMeister is a Trademark of Samizdat Productions.";

	private static final String		kURLString = "http://www.tolstoy.com";
	private static final String		kFactoriesFileName = "plugins.txt";
	private static final String		kImageFolderName = "images";
	private static final String		kTopLevelTextString = "<Top Level>";
	private static final String		kDDSinkFileName = "ddout.txt";
	private static final String		kDDSourceFileName = "ddin.txt";

	private static final int		kMaxFilesToShow = 1000;
	private static final int		kStandardFilterFlags = ( DiskFilter.IGNORE_NAME_LOCKED |
																DiskFilter.IGNORE_HIDDEN |
																DiskFilter.IGNORE_ALIASES );

	private AboutBox			about;
	private BrowserDisplay		display;
	private DDSource			ees;
	private Vector				theImageViewers, theInfoViewers;
	private DiskObject			currentContainer;
	private String				dataFolder;
	private boolean				bShowInNewWindow, bShowAllFiles;

/**
Create the BrowserDisplay object, and start out by showing the mounted volumes.

@param df full path to ImageMeister's 'data' folder
*/

	Browser( String df ) {
		dataFolder = df;
		if ( !dataFolder.endsWith( File.separator ) )
			dataFolder += File.separator;

		PluginManager.initialize( dataFolder + kFactoriesFileName );

		theImageViewers = new Vector( 5, 5 );
		theInfoViewers = new Vector( 5, 5 );
		bShowInNewWindow = true;
		bShowAllFiles = false;

		display = new BrowserDisplay( dataFolder + kImageFolderName + File.separator, this );
		display.setCheckboxValue( kCheckboxNewWindow, bShowInNewWindow );
		display.setCheckboxValue( kCheckboxShowAllFiles, bShowAllFiles );

		currentContainer = null;
		about = null;
		display.setItemText( 0, kTopLevelTextString );
		refreshCurrentDisplay();
		display.setVisibility( true );
		display.setItems( FileRegistry.getVolumes() );

		openDDSink( dataFolder + kDDSinkFileName );
		openDDSource( dataFolder + kDDSourceFileName );
	}

	boolean openDDSink( String eePath ) {
		try {
			DDSink.init( eePath );
		}
		catch ( Exception e ) {
			Trace.println( "openDDSink: couldn't init with " + eePath );
			return false;
		}

		return true;
	}

	boolean openDDSource( String eePath ) {
		Thread		t;

		try {
			ees = new DDSource( eePath, this );
			t = new Thread( ees );
			t.setDaemon( true );
			t.start();
		}
		catch ( Exception e ) {
			Trace.println( "openDDSource: couldn't init with " + eePath );
			return false;
		}

		return true;
	}

/**
Called when the About Box's URL button is clicked.
*/

	public void visitURLButtonClicked() {
		FileRegistry.launchURL( kURLString, FileRegistry.APP_MOVE_TOFRONT, null );
	}

/**
From the ImageViewerOwner interface.
*/

	public void closeImageViewer( ImageViewerI viewer ) {
		Frame			fm;
		
		theImageViewers.removeElement( viewer );

		fm = viewer.getFrame();
		fm.hide();
		fm.dispose();
	}

/**
From the InfoViewerOwner interface.
*/

	public void closeInfoViewer( InfoViewerI viewer ) {
		Frame			fm;
		
		theInfoViewers.removeElement( viewer );

		fm = viewer.getFrame();
		fm.hide();
		fm.dispose();
	}

/**
From the WindowOwner interface.
*/

	public void closeWindow( Window w ) {
		if ( w == null )
			return;

		if ( w == about ) {
			about.hide();
			about.dispose();
			about = null;
		}
	}

/**
Handle commands sent from the DDSource class.
*/

	public boolean processEvent( Event e ) {
		DiskObject		diskObj;
		File			fl;

		if ( e.id == DDCommandRecipient.kAboutCommand ) {
			doAboutButton();
			return true;
		}
		else if ( e.id == DDCommandRecipient.kOpenDoc ) {
			try {
				fl = new File( (String) e.arg );
				diskObj = FileRegistry.createDiskObject( fl, FileRegistry.ALIAS_NO_UI );
			}
			catch ( Exception excep ) {
				diskObj = null;
			}
			
			if ( diskObj != null )
				showImageViewerForFile( diskObj );

			return true;
		}

		return false;
	}

	public void browserCloseButtonClicked() {
		doQuitButton();
	}

	private void doQuitButton() {
		System.exit( 0 );
	}

	private void doAboutButton() {
		if ( about != null )
			return;

		about = new AboutBox( this, this, dataFolder + kImageFolderName + File.separator,
								display.getMainFrame(), true );
		about.show();
	}

/**
Display the currently selected object.
*/

	private void doShowButton() {
		int			curItem;
		
		curItem = display.getSelectedItemIndex();
		if ( curItem >= 0 )
			browserItemClicked( curItem );
	}

/**
Display info about the currently selected object.
*/

	private void doInfoButton() {
		InfoViewerI			infoViewer;
		FileSpecifier		specifier;
		DiskObject			dobj;
		Rectangle			rect;
		int					curItem;
		
		curItem = display.getSelectedItemIndex();
		if ( curItem < 0 )
			return;

		dobj = display.getItem( curItem );
		if ( dobj == null )
			return;

		showInfoViewerForFile( dobj );
	}

	private void doShowAllFilesCheckbox() {
		bShowAllFiles = display.getCheckboxValue( BrowserDisplayOwnerI.kCheckboxShowAllFiles );
		refreshCurrentDisplay();
	}

	private void doNewWindowCheckbox() {
		bShowInNewWindow = display.getCheckboxValue( BrowserDisplayOwnerI.kCheckboxNewWindow );
	}

/**
Show the parent folder of the current folder.
*/

	void doUpALevelButton() {
		if ( currentContainer != null ) {
			try {
				currentContainer = currentContainer.getContainer();
				if ( currentContainer == null )
					display.setItemText( 0, kTopLevelTextString );
				else
					display.setItemText( 0, currentContainer.getDisplayName() );
			}
			catch ( Exception e ) {
				Trace.println( "can't get container for " + currentContainer.getName() + ":" + e );
			}
		}

		refreshCurrentDisplay();
	}

/**
From the BrowserDisplayOwnerI interface.
The user clicked one of the disk objects; if it's a volume or directory, show its contents.
Otherwise, it's a file, so try to show an image viewer for the file.
*/

	public void browserItemClicked( int which ) {
		DiskObject			prospect;
		boolean				bIsDirectory, bIsVolume;
		
		prospect = display.getItem( which );
		if ( prospect == null ) {
			Trace.println( "can't set null as container" );
			return;
		}
		
		bIsVolume = ( prospect instanceof DiskVolume );

		bIsDirectory = false;
		if ( ( prospect instanceof DiskFile ) )
			if ( ( prospect.getFlags() & DiskFile.FILE_DIR ) != 0 )
				bIsDirectory = true;

		if ( bIsDirectory || bIsVolume )
			tryToSetAsContainer( prospect );
		else
			showImageViewerForFile( prospect );
	}

	public boolean browserKeyPressed( int which, int key ) {
		if ( which < 0 || key != 'i' )
			return false;
		
		doInfoButton();
		
		return true;
	}

/**
Try to display 'dobj'
*/

	private void showImageViewerForFile( DiskObject dobj ) {
		ImageViewerI		newImageViewer, oldImageViewer;
		FileSpecifier		specifier;
		OwnedFrame			frame;
		Rectangle			rect;
		int					curItem;
		
		specifier = IMUtils.makeFileSpecifier( dobj );
		rect = null;
		frame = null;
		oldImageViewer = null;

		if ( !bShowInNewWindow ) {
			if ( theImageViewers.size() > 0 ) {
				oldImageViewer = (ImageViewerI) theImageViewers.elementAt( 0 );
				frame = oldImageViewer.getFrame();
				if ( frame != null )
					rect = frame.bounds();
			}
		}

		if ( rect == null )
			rect = new Rectangle( 100, 100, 400, 200 );

		newImageViewer = PluginManager.createImageViewer( frame, this, specifier, rect );
		if ( newImageViewer == null )
			return;

		if ( oldImageViewer != null )
			theImageViewers.setElementAt( newImageViewer, 0 );
		else
			theImageViewers.addElement( newImageViewer );

		newImageViewer.setVisible( true );
	}
	
/**
Try to display info about 'dobj'
*/

	private void showInfoViewerForFile( DiskObject dobj ) {
		InfoViewerI			newInfoViewer, oldInfoViewer;
		FileSpecifier		specifier;
		OwnedFrame			frame;
		Rectangle			rect;
		int					curItem;

		specifier = IMUtils.makeFileSpecifier( dobj );
		rect = null;
		frame = null;
		oldInfoViewer = null;

		if ( !bShowInNewWindow ) {
			if ( theInfoViewers.size() > 0 ) {
				oldInfoViewer = (InfoViewerI) theInfoViewers.elementAt( 0 );
				frame = oldInfoViewer.getFrame();
				if ( frame != null )
					rect = frame.bounds();
			}
		}

		if ( rect == null )
			rect = new Rectangle( 100, 100, 400, 200 );

		newInfoViewer = PluginManager.createInfoViewer( frame, this, specifier, rect );
		if ( newInfoViewer == null )
			return;

		if ( oldInfoViewer != null )
			theInfoViewers.setElementAt( newInfoViewer, 0 );
		else
			theInfoViewers.addElement( newInfoViewer );

		newInfoViewer.setVisible( true );
	}
	
/**
Update the display. If we're at the top level, use the FileRegistry to get a list of the
mounted volumes.
*/

	public void refreshCurrentDisplay() {
		DiskVolume			mountedVolumes[];
		int					i;

		if ( currentContainer != null ) {
			tryToSetAsContainer( currentContainer );
			return;
		}

		mountedVolumes = FileRegistry.getVolumes();
		if ( mountedVolumes == null ) {
			Trace.println( "can't get mounted volumes" );
			return;
		}

		display.setItems( mountedVolumes );
	}

/**
Try to set the given object as the current container. First, we try to iterate on its contents; if
that fails ( e.g., if it's a file ), we do nothing. Otherwise, set the current container, and
initialize the panels with the contents of the current container.
*/

	void tryToSetAsContainer( DiskObject newContainer ) {
		DiskObject			iteratedItems[];

		iteratedItems = PluginManager.runDiskFilter( newContainer, kStandardFilterFlags,
														kMaxFilesToShow, bShowAllFiles );

		if ( iteratedItems == null )
			return;

		currentContainer = newContainer;

		display.setItems( iteratedItems );
		display.setItemText( 0, currentContainer.getDisplayName() );
	}

/**
From the BrowserDisplayOwnerI interface.
*/

	public void browserButtonClicked( int which ) {
		switch ( which ) {
			case BrowserDisplayOwnerI.kButtonQuit:
				doQuitButton();
			break;

			case BrowserDisplayOwnerI.kButtonAbout:
				doAboutButton();
			break;

			case BrowserDisplayOwnerI.kButtonInfo:
				doInfoButton();
			break;

			case BrowserDisplayOwnerI.kButtonShow:
				doShowButton();
			break;

			case BrowserDisplayOwnerI.kButtonUpALevel:
				doUpALevelButton();
			break;

			case BrowserDisplayOwnerI.kCheckboxShowAllFiles:
				doShowAllFilesCheckbox();
			break;

			case BrowserDisplayOwnerI.kCheckboxNewWindow:
				doNewWindowCheckbox();
			break;
		}
	}
}

