/*  
 * Copyright (c) 2002-2003 MIIK Ltd. All rights reserved.  
 *  
 * Use is subject to license terms.  
 *   
 * The complete licence text can be found at   
 * http://www.jniwrapper.com/license.jsp?prod=winpack  
 */
package com.jniwrapper.win32.io;

import com.jniwrapper.*;
import com.jniwrapper.win32.FunctionName;
import com.jniwrapper.win32.Handle;
import com.jniwrapper.win32.Kernel32;
import com.jniwrapper.win32.LastError;

import java.io.File;
import java.io.FileFilter;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Class WinNTWatcherStrategy implements watcher strategy based on wrapping ReadDirectoryChangesW
 * api function.
 *
 * Platforms: WinNT (SP3 or later), Win2000, WinXP
 *
 * @author Serge Piletsky
 */

class WinNTWatcherStrategy extends WatcherStrategy
{
    static final FunctionName FUNCTION_CREATE_FILE = new FunctionName("CreateFile");
    static final String FUNCTION_READ_DIRECTORY_CHANGES = "ReadDirectoryChangesW";

    static final int FILE_LIST_DIRECTORY = 0x001;
    static final int FILE_SHARE_READ = 0x00000001;
    static final int FILE_SHARE_DELETE = 0x00000004;
    static final int OPEN_EXISTING = 3;
    static final int FILE_FLAG_BACKUP_SEMANTICS = 0x02000000;

    static final int FUFFER_SIZE = 1024;
    static final int INVALID_HANDLE_VALUE = -1;

    private Handle _directory = new Handle();
    private Thread _watcherThread;

    public WinNTWatcherStrategy(FileSystemWatcher fileSystemWatcher)
    {
        super(fileSystemWatcher);
    }

    public void start() throws FileSystemException
    {
        final FileSystemWatcher watcher = getFileSystemWatcher();
        final Kernel32 kernel32 = Kernel32.getInstance();
        String path = watcher.getPath();
        final Function functionCreateFile = kernel32.getFunction(FUNCTION_CREATE_FILE.toString());
        functionCreateFile.invoke(_directory, new Parameter[]
        {
            kernel32.stringParam(path),
            new UInt32(FILE_LIST_DIRECTORY),
            new UInt32(FILE_SHARE_DELETE | FILE_SHARE_READ),
            new Pointer(null, true),
            new UInt32(OPEN_EXISTING),
            new UInt32(FILE_FLAG_BACKUP_SEMANTICS),
            new Pointer(null, true)
        });
        final long handleValue = _directory.getValue();
        if (_directory.isNull() || handleValue == INVALID_HANDLE_VALUE)
            throw new FileSystemException();

        final PrimitiveArray buffer = new PrimitiveArray(UInt8.class, FUFFER_SIZE);
        final Pointer bufferPointer = new Pointer(buffer);
        final boolean watchSubtree = watcher.isWatchSubree();
        final int notifyFilter = (int)watcher.getOptions().getFlags();
        final UInt32 bytesReturned = new UInt32();

        _watcherThread = new Thread(new Runnable()
        {
            public void run()
            {
                setWatching(true);
                while (isWatching())
                {
                    final Function functionReadDirectoryChanges = kernel32.getFunction(FUNCTION_READ_DIRECTORY_CHANGES);
                    Int returnValue = new Int();
                    functionReadDirectoryChanges.invoke(returnValue, new Parameter[]
                    {
                        _directory,
                        bufferPointer,
                        new UInt32(FUFFER_SIZE),
                        new Bool(watchSubtree),
                        new UInt32(notifyFilter),
                        new Pointer(bytesReturned),
                        new Pointer(null, true), //null pointer to an overlapped structure
                        new Pointer(null, true)  //null pointer to completion callback
                    });
                    final long value = returnValue.getValue();
                    if (value == 0)
                        throw new IllegalStateException("Function ReadDirectoryChangesW failed. Last error: " + LastError.getValue());
                    else
                    {
                        retrieveEvents(buffer.getBytes());
                    }
                }
                setWatching(false);
            }
        });
        _watcherThread.start();
    }

    class FileActionInfo
    {
        private FileInfo _fileInfo;
        private int _action;

        public FileActionInfo(FileInfo fileInfo, int action)
        {
            _fileInfo = fileInfo;
            _action = action;
        }

        public FileInfo getFileInfo()
        {
            return _fileInfo;
        }

        public int getAction()
        {
            return _action;
        }
    }

    /**
     * Parses passed buffer according to FILE_NOTIFY_INFORMATION structure
     * @param buffer
     */
    private void retrieveEvents(byte[] buffer)
    {
        if (!isWatching())
            return;

        final FileSystemWatcher watcher = getFileSystemWatcher();
        int index = 0, nextEntryIndex = 0;
        boolean lastEvent = false;
        List actionInfos = new ArrayList();
        while (index < FUFFER_SIZE && !lastEvent)
        {
            // NextEntryOffset
            int nextEntryOffset = readDWORD(buffer, index);
            nextEntryIndex += nextEntryOffset;
            index += 4;
            lastEvent = nextEntryOffset == 0;
            // Action
            int action = readDWORD(buffer, index);
            index += 4;
            // FileNameLength
            int fileNameLength = readDWORD(buffer, index);
            index += 4;
            // FileName
            StringBuffer fileName = new StringBuffer(watcher.getPath());
            fileName.append(File.separatorChar);
            for (int i = 0; i < fileNameLength >> 1; i++)
            {
                char c = readWCHAR(buffer, index);
                index += 2;
                fileName.append(c);
            }
            actionInfos.add(new FileActionInfo(new FileInfo(fileName.toString(), 0, 0, 0), action));
            index = nextEntryIndex;
        }

        final int actionInfoCount = actionInfos.size();
        List events = new ArrayList(actionInfoCount);
        for (int i = 0; i < actionInfoCount; i++)
        {
            FileSystemEvent event = null;
            FileActionInfo actionInfo = (FileActionInfo) actionInfos.get(i);
            if (actionInfo.getAction() == FileSystemEvent.FILE_RENAMED)
            {
                if (i + 1 < actionInfoCount)
                {
                    FileActionInfo newActionInfo = (FileActionInfo) actionInfos.get(++i);
                    event = new FileSystemEvent(watcher, actionInfo.getAction(), actionInfo.getFileInfo(), newActionInfo.getFileInfo());
                }
            }
            else
            {
                event = new FileSystemEvent(watcher, actionInfo.getAction(), actionInfo.getFileInfo());
            }
            events.add(event);
        }

        final FileFilter fileFilter = watcher.getFileFilter();
        for (Iterator i = events.iterator(); i.hasNext();)
        {
            FileSystemEvent event = (FileSystemEvent) i.next();
            if (fileFilter != null)
            {
                final File file1 = new File(event.getFileInfo().getFileName());
                File file2 = null;
                if (event.getOldFileInfo() != null)
                {
                    file2 = new File(event.getOldFileInfo().getFileName());
                }

                if (fileFilter.accept(file1) || (file2 != null && fileFilter.accept(file2)))
                {
                    watcher.fireFileSystemEvent(event);
                }
            }
            else
            {
                watcher.fireFileSystemEvent(event);
            }
        }
    }

    private int readDWORD(byte[] buffer, int offset)
    {
        UInt32 result = new UInt32();
        result.read(buffer, offset);
        return (int)result.getValue();
    }

    private char readWCHAR(byte[] buffer, int offset)
    {
        WideChar result = new WideChar();
        result.read(buffer, offset);
        return result.getValue();
    }

    public void stop() throws FileSystemException
    {
        setWatching(false);
    }
}
