/*  
 * Copyright (c) 2002-2003 MIIK Ltd. All rights reserved.  
 *  
 * Use is subject to license terms.  
 *   
 * The complete licence text can be found at   
 * http://www.jniwrapper.com/license.jsp?prod=winpack  
 */
package com.jniwrapper.win32.shell;

import com.jniwrapper.Bool;
import com.jniwrapper.Function;
import com.jniwrapper.Int32;
import com.jniwrapper.Pointer;
import com.jniwrapper.win32.FunctionName;
import com.jniwrapper.win32.LastErrorException;
import com.jniwrapper.win32.Point;
import com.jniwrapper.win32.gdi.Cursor;
import com.jniwrapper.win32.gdi.Icon;
import com.jniwrapper.win32.ui.*;
import com.jniwrapper.win32.ui.Timer;

import java.util.*;

/**
 * Class TrayIcon provides ability to add/modify tray icon and its properties such as hint,
 * icon, tooltip, callout.
 *
 * You can register TrayIcon listeners for receiving mouse events (WM_MOUSEMOVE, WM_LBUTTONDOWN etc.)
 *
 * @author Alexander Kireyev
 */
public class TrayIcon
{
    static final FunctionName FUNCTION_SHELL_NOTIFY_ICON = new FunctionName("Shell_NotifyIcon");
    /*
     * Operation constants for adding, modifying and removing icon in tray
     */
    // operation ID for adding tray icon to tray
    static final int NIM_ADD = 0x00000000;
    // operation ID for modifying icon in the tray
    static final int NIM_MODIFY = 0x00000001;
    // operation ID for removig icon from the tray
    static final int NIM_DELETE = 0x00000002;

    /*
     * Constants for configuring tray icon
     */
    static final int NIF_MESSAGE = 0x00000001;
    static final int NIF_ICON = 0x00000002;
    static final int NIF_TIP = 0x00000004;
    static final int NIF_STATE = 0x00000008;
    static final int NIF_INFO = 0x00000010;

    /*
     * Extended tray icon attributes
     */
    static final int NIS_HIDDEN = 0x00000001;
    static final int NIS_SHAREDICON = 0x00000002;
    static final int TIMER_ID = 1;

    public static final int WM_TRAY = Msg.WM_USER + 1;

    public static final String CLASS_NAME = "JW_TrayWindowClassName";

    private static Map _messageHandlers = new HashMap();
    private static Object _trayWindowLock = new Object();
    private static int _curID = 0;
    private static long _hWnd;

    private final int _trayID;
    private boolean _disposed = false;
    private List _listeners = new ArrayList();

    public TrayIcon()
    {
        this(null);
    }

    /**
     * Creates an instance of TrayIcon with specified icon
     * @param icon
     */
    public TrayIcon(Icon icon)
    {
        _trayID = _curID++;
        ensureEventProcessing();
        _messageHandlers.put(new Integer(_trayID), this);
        NotifyIconData notifyicondata = new NotifyIconData(_hWnd, _trayID);
        notifyicondata.setCallbackMessage(WM_TRAY);
        notifyicondata.setFlags(NIF_ICON | NIF_MESSAGE | NIF_TIP | NIF_INFO);
        notify(NIM_ADD, notifyicondata);
        setIcon(icon);
    }

    /**
     * @param operation specifies the action to be taken.
     * @param notifyIconData is a prepared structure for notify function
     */
    private void notify(int operation, NotifyIconData notifyIconData)
    {
        if (_disposed)
        {
            throw new IllegalStateException("Already disposed");
        }
        Function function = Shell32.getInstance().getFunction(FUNCTION_SHELL_NOTIFY_ICON.toString());
        Bool result = new Bool();
        function.invoke(result, new Int32(operation), new Pointer(notifyIconData));
        if (!result.getValue())
        {
            throw new LastErrorException("Icon operation failed.");
        }
    }

    private static void ensureEventProcessing()
    {
        synchronized (_trayWindowLock)
        {
            if (_hWnd == 0)
            {
                new Thread()
                {
                    public void run()
                    {
                        createEmptyNativeWindow();
                        Wnd.eventLoop(_hWnd);
                    }
                }.start();
                try
                {
                    _trayWindowLock.wait();
                }
                catch (InterruptedException e)
                {
                    e.printStackTrace();
                }
                if (_hWnd == 0)
                {
                    throw new RuntimeException("Event processing window creation failed");
                }
            }
        }
    }

    /**
     * Sets an icon in the system tray
     * @param icon
     */
    public void setIcon(Icon icon)
    {
        if (icon != null && !icon.isNull())
        {
            NotifyIconData notifyicondata = new NotifyIconData(_hWnd, _trayID);
            notifyicondata.setIcon(icon);
            notifyicondata.setFlags(NIF_ICON);
            notify(NIM_MODIFY, notifyicondata);
        }
    }

    /**
     * Sets and shows a balloon message
     * @param value is a container of ballon message atttibutes
     */
    public void showMessage(TrayMessage value)
    {
        NotifyIconData notifyicondata = new NotifyIconData(_hWnd, _trayID);
        notifyicondata.setFlags(NIF_INFO);
        notifyicondata.setInfoTitle(value.getTitle());
        notifyicondata.setInfo(value.getMessage());
        notifyicondata.setTimeout(value.getTimeout() * 1000);
        notifyicondata.setInfoFlags(value.getIconType());
        notify(NIM_MODIFY, notifyicondata);
    }

    /**
     * Disposes tray icon
     */
    public void dispose()
    {
        if (!_disposed)
        {
            notify(NIM_DELETE, new NotifyIconData(_hWnd, _trayID));
            _messageHandlers.remove(new Integer(_trayID));
            _disposed = true;
        }
    }

    /**
     * Setups tooltip for tray icon
     * @param tip is message to be shown in tooltip
     */
    public void setToolTip(String tip)
    {
        NotifyIconData notifyicondata = new NotifyIconData(_hWnd, _trayID);
        notifyicondata.setToolTip(tip);
        notifyicondata.setFlags(NIF_TIP);
        notify(NIM_MODIFY, notifyicondata);
    }

    /**
     * Adds specified tray icon listener
     * @param listener
     */
    public void addTrayListener(TrayIconListener listener)
    {
        _listeners.add(listener);
    }


    /**
     * Removes specified listener
     * @param listener
     */
    public void removeTrayListener(TrayIconListener listener)
    {
        _listeners.remove(listener);
    }

    private void onIconMessage(long message, int x, int y)
    {
        for (Iterator i = _listeners.iterator(); i.hasNext();)
        {
            TrayIconListener listener = (TrayIconListener) i.next();
            listener.trayActionPerformed(message, x, y);
        }
    }

    /**
     * Shows or hides icon in the system tray
     * @param visible
     */
    public void setVisible(boolean visible)
    {
        NotifyIconData notifyicondata = new NotifyIconData(_hWnd, _trayID);
        notifyicondata.setFlags(NIF_STATE);
        notifyicondata.setState(visible?0:NIS_HIDDEN);
        notifyicondata.setStateMask(NIS_HIDDEN);
        notify(NIM_MODIFY, notifyicondata);
    }

    private static void createEmptyNativeWindow()
    {
        synchronized (_trayWindowLock)
        {
            WndClass wndClass = new WndClass(new TrayIconWindowProc(), CLASS_NAME);
            wndClass.register();

            Wnd hWnd = Wnd.createWindow(CLASS_NAME);
            _hWnd = hWnd.getValue();
            _trayWindowLock.notify();
        }
    }

    private static class TrayIconWindowProc extends WindowProc
    {
        boolean _timerRuning = false;
        boolean _mouseIn = false;
        int _mouseX = 0;
        int _mouseY = 0;
        Timer _timer = null;
        TrayIcon _handler = null;

        Timer getTimer()
        {
            if (_timer == null)
            {
                _timer = new Timer(new Wnd(_hWnd), TIMER_ID, 150, new TimeOutCallback());
            }
            return _timer;
        }

        public void callback()
        {
            final int msg = (int) _msg.getValue();
            switch (msg)
            {
                case WM_TRAY:
                    final long lParam = _lParam.getValue();
                    final long wParam = _wParam.getValue();

                    final Point cursorPos = Cursor.getCursorPosition();
                    int mx = (int)cursorPos.getX();
                    int my = (int)cursorPos.getY();

                    Integer id = new Integer((int) wParam);
                    _handler = (TrayIcon) _messageHandlers.get(id);

                    if (!_mouseIn)
                    {
                        if ((_handler != null) && ((mx != _mouseX) || (my != _mouseY)))
                            _handler.onIconMessage(Msg.WM_MOUSEHOVER, mx, my);
                    }
                    if (lParam == Msg.WM_MOUSEMOVE)
                    {
                        final Timer timer = getTimer();
                        if (_timerRuning)
                        {
                            timer.stop();
                            _timerRuning = false;
                        }
                        timer.start();
                        _timerRuning = true;
                    }

                    _mouseX = mx;
                    _mouseY = my;
                    if (_handler != null)
                    {
                        _handler.onIconMessage(lParam, _mouseX, _mouseY);
                    }
                    _mouseIn = true;
                    _lResult.setValue(0);
                    break;
                default:
                    super.callback();
                    break;
            }
        }

        private class TimeOutCallback extends Timer.Callback
        {
            public void callback()
            {
                final Timer timer = getTimer();
                if (_timerRuning)
                {
                    timer.stop();
                    _timerRuning = false;
                }
                final Point cursorPosition = Cursor.getCursorPosition();
                int mx = (int)cursorPosition.getX();
                int my = (int)cursorPosition.getY();
                if (_handler != null)
                {
                    if ((mx != _mouseX) || (my != _mouseY))
                    {
                        _handler.onIconMessage(Msg.WM_MOUSELEAVE, mx, my);
                        _mouseIn = false;
                        _mouseY = 0;
                        _mouseY = 0;
                    }
                }
            }
        }
    }
}
