/*
 *    The term 'program' used in the copyright and the license below means
 *    PReP Version 1.0 
 *
 *    Copyright (C) 1998, 1999 DELECis
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *    This copyright notice must appear in all copies and modifications.
 *
 *    Contact: prep@delec.com
 *
 *    Read the manual: http://www.delec.com/is/products/prep/manual/jprep/
 */

/**
 * PReP Version 1.0
 *
 * SQLServe: has the connection to the database and defines it. 
 *           connects and execute SQL Statements. It stores the
 *           result in a Store object
 *
 * History: - Creation nh
 *
 * @author nh
 * @version 1.0
 */

package com.delec.prep;

import java.sql.*;
import com.delec.global.dbms.*;

public class SQLServe
{
	public static final String SESSION_NAME = "SQLServe"; 

	/** connection object to the database for all classes that use the db connection
	 * 
	 * this has been made static, coz the JDBC Driver repeatedly reported errors
	 * when the connection was opened and closed all the time
	 *
	 */
	//  protected Connection con = null;
	//  protected ConnectionPool connectionPool = null;

	public static Connection getConnection(ConnectionPool connectionPool)
	{
		return connectionPool.getConnection();
	}

	public static void recoverConnection()
	{
	}

	public static Store execStmtStore(String sqlStatement, Session session, ExceptionHandler exceptionHandler)
	{
		return execStmtStore(sqlStatement, 0, session, exceptionHandler);
	}

	/**
	 * returns the Store object with the specified maximum of rows
	 *
	 * @param String SQL statement 
	 * @param int maximum of rows
	 * @return the Store object
	 */

	public static Store execStmtStore(String sqlStatement, int iMaxRows, Session session, ExceptionHandler exceptionHandler)
	{
		ConnectionPool cp = (ConnectionPool)session.get(ConnectionPool.SESSION_NAME);

		if ( cp == null )
		{
			// no database connection was defined!
			System.err.println("no database connection defined. Please use the DB tag or enter PReP for a help text");
			System.exit(1);
		}

		Connection dbmsCon = cp.getConnection();
		Store store = null;

		try
		{
			int i;  

			Statement stmt = dbmsCon.createStatement();
			ResultSet rs = stmt.executeQuery(sqlStatement); 

			ResultSetMetaData rsmd = rs.getMetaData();
			int numCols = rsmd.getColumnCount();    

			store = new Store(numCols);

			for ( i = 1; i <= numCols; i++ )
			{
				store.addName(rsmd.getColumnLabel(i), rsmd.getColumnType(i));
			} 

			iMaxRows--;

			while ( rs.next() )
			{
				for ( i = 1; i <= numCols; i++ )
				{
					PRePTagObject prepTagObject = new PRePTagObject("dummy");
					prepTagObject.setValue(rs.getObject(i));
					store.addValue(prepTagObject);
				}
				if ( iMaxRows > 0 )
				{
					iMaxRows--;
				}
				else if ( iMaxRows == 0 )
				{
					break;
				}
			} 
			rs.close();
			stmt.close();
		}
		catch ( Exception e )
		{
			exceptionHandler.handleException(e);
		}

		cp.close(dbmsCon);
		return store;
	}

	public static Store execSpStore(String strSP, Session session, ExceptionHandler exceptionHandler)
	{
		ConnectionPool cp = (ConnectionPool)session.get(ConnectionPool.SESSION_NAME);
		Connection dbmsCon = cp.getConnection();

		Store store = null;
		try
		{
			int i;  
			Statement stmt = dbmsCon.createStatement();
			ResultSet rs = null;
			boolean bStmt = stmt.execute(strSP);
			int iCount = -1;
			if ( bStmt == false ) iCount = stmt.getUpdateCount();
			if ( !(!bStmt && (iCount == -1)) )
			{
				rs = stmt.getResultSet();

				ResultSetMetaData rsmd = rs.getMetaData();
				int numCols = rsmd.getColumnCount();

				store = new Store(numCols);

				for ( i = 1; i <= numCols; i++ )
				{
					store.addName(rsmd.getColumnLabel(i), rsmd.getColumnType(i));
				}

				while ( rs.next() )
				{
					for ( i = 1; i <= numCols; i++ )
					{
						PRePTagObject prepTagObject = new PRePTagObject("dummy");
						prepTagObject.setValue(rs.getObject(i));
						store.addValue(prepTagObject);
					}
				}
				rs.close();
			}
			stmt.close();
		}
		catch ( SQLException e )
		{
			exceptionHandler.handleException(e);
		}
		cp.close(dbmsCon);
		return store;
	}

	public static int execUpdate(String sqlStatement, Session session, ExceptionHandler exceptionHandler)
	{
		ConnectionPool cp = (ConnectionPool)session.get(ConnectionPool.SESSION_NAME);
		Connection dbmsCon = cp.getConnection();

		int count = 0;

		try
		{
			Statement stmt = dbmsCon.createStatement();
			count = stmt.executeUpdate(sqlStatement);
			stmt.close();
		}
		catch ( SQLException e )
		{
			exceptionHandler.handleException(e);
		}
		cp.close(dbmsCon);
		return count;
	}

	public static CallableStatement getCallableStatement(String strCallableStmt, Connection dbmsCon)
	{
		CallableStatement callableStmt = null;
		try
		{
			callableStmt = dbmsCon.prepareCall(strCallableStmt);
		}
		catch ( SQLException e )
		{
		}
		return callableStmt;
	}

	public static CallableStatement loopThroughResultSet(CallableStatement callableStmt, boolean hasResultSet, ResultSet rs)
	{
		while ( true )
		{
			try
			{
				int updateCount = callableStmt.getUpdateCount(); 

				if ( updateCount != -1 && hasResultSet ) rs = callableStmt.getResultSet();

				// If there are no more results or update counts, we're done 	
				if ( rs == null && updateCount == -1 )
				{
					break; 
				}

				// Check to see if there is a ResultSet 	
				if ( rs != null )
				{

					while ( rs.next() )
					{
					} 

				} // Otherwise, there will be an update count 
				else
				{
					callableStmt.getUpdateCount(); 
				} 

				hasResultSet = callableStmt.getMoreResults();
				if ( !hasResultSet ) break;
			}
			catch ( SQLException e )
			{
			}
		}
		return callableStmt;
	}


	public static float executePriceCallableStmt(String strINUserID, String strINPartno, String strCallableStmt, Session session)
	{
		ConnectionPool cp = (ConnectionPool)session.get(ConnectionPool.SESSION_NAME);
		Connection dbmsCon = cp.getConnection();

		CallableStatement callableStmt = null;
		double dResult = 0;

		try
		{
			callableStmt = getCallableStatement(strCallableStmt, dbmsCon);
			callableStmt.registerOutParameter(1, Types.FLOAT);
			Integer intUserID = new Integer(strINUserID);
			callableStmt.setInt(2, intUserID.intValue());
			Integer intPartno = new Integer(strINPartno);
			callableStmt.setInt(3, intPartno.intValue());


			DatabaseMetaData dmd = dbmsCon.getMetaData();   
			ResultSet rs = dmd.getProcedureColumns(null, null,  "SpGet" , null);

			while ( rs.next() )
			{
			}

			boolean hasResultSet = callableStmt.execute(); 

			callableStmt = loopThroughResultSet(callableStmt, hasResultSet, rs);
			dResult = callableStmt.getDouble(1);
			callableStmt.close();
		}
		catch ( Exception e )
		{
			//ExceptionHandler.handleException(e);
		}
		cp.close(dbmsCon);
		return(float)dResult;
	}

	public static float[] executeDetailPriceCallableStmt(String strINUserID, String strINPartno, String strINDistID, String strCallableStmt, Session session)
	{

		ConnectionPool cp = (ConnectionPool)session.get(ConnectionPool.SESSION_NAME);
		Connection dbmsCon = cp.getConnection();

		CallableStatement callableStmt = null;

		double dQ1 = 0;
		double dEPUS = 0;
		double dEPCH = 0;      

		try
		{
			callableStmt = getCallableStatement(strCallableStmt, dbmsCon);

			callableStmt.registerOutParameter(1, Types.FLOAT);
			callableStmt.registerOutParameter(2, Types.FLOAT);
			callableStmt.registerOutParameter(3, Types.FLOAT);

			Integer intUserID = new Integer(strINUserID);
			callableStmt.setInt(4, intUserID.intValue());
			Integer intPartno = new Integer(strINPartno);
			callableStmt.setInt(5, intPartno.intValue());
			Integer intDistID = new Integer(strINDistID);
			callableStmt.setInt(6, intDistID.intValue());


			DatabaseMetaData dmd = dbmsCon.getMetaData();
			ResultSet rs = dmd.getProcedureColumns(null, null, "SpGetPriceDetails", null);

			while ( rs.next() )
			{
			}

			boolean hasResultSet = callableStmt.execute();
			callableStmt = loopThroughResultSet(callableStmt, hasResultSet, rs);

			dQ1 = callableStmt.getDouble(1);
			dEPUS = callableStmt.getDouble(2);
			dEPCH = callableStmt.getDouble(3);

			callableStmt.close();
		}
		catch ( Exception e )
		{
			session.getExceptionList().addElement(new PRePException(null, "'SQLException: '"+e));
		}
		cp.close(dbmsCon);
		float[] fResult = {(float)dQ1, (float)dEPUS, (float)dEPCH};

		return fResult;
	}  

	public static float[] executeDeltaCallableStmt(String strUserID, String strPartno, String strCallableStmt, Session session)
	{
		ConnectionPool cp = (ConnectionPool)session.get(ConnectionPool.SESSION_NAME);
		Connection dbmsCon = cp.getConnection();

		CallableStatement callableStmt = null;

		float fCHEPCH = 0;
		float fCHEPUS = 0;
		float fUSEPCH = 0;      
		float fUSEPUS = 0;
		float fDCH = 0;
		float fDUS = 0;

		try
		{
			callableStmt = getCallableStatement(strCallableStmt, dbmsCon);

			callableStmt.registerOutParameter(1, Types.FLOAT);
			callableStmt.registerOutParameter(2, Types.FLOAT);
			callableStmt.registerOutParameter(3, Types.FLOAT);
			callableStmt.registerOutParameter(4, Types.FLOAT);
			callableStmt.registerOutParameter(5, Types.FLOAT);
			callableStmt.registerOutParameter(6, Types.FLOAT);

			Integer intUserID = new Integer(strUserID);
			callableStmt.setInt(7, intUserID.intValue());
			Integer intPartno = new Integer(strPartno);
			callableStmt.setInt(8, intPartno.intValue());

			DatabaseMetaData dmd = dbmsCon.getMetaData();
			ResultSet rs = dmd.getProcedureColumns(null, null, "SpGetDelta", null);

			while ( rs.next() )
			{
			}

			boolean hasResultSet = callableStmt.execute();
			callableStmt = loopThroughResultSet(callableStmt, hasResultSet, rs);

			fCHEPCH = callableStmt.getFloat(1);
			fCHEPUS = callableStmt.getFloat(2);
			fUSEPCH = callableStmt.getFloat(3);
			fUSEPUS = callableStmt.getFloat(4);
			fDCH = callableStmt.getFloat(5);
			fDUS = callableStmt.getFloat(6);

			callableStmt.close();
		}
		catch ( Exception e )
		{
			session.getExceptionList().addElement(new PRePException(null, "'SQLException: '"+e));
		}
		cp.close(dbmsCon);
		float[] fResult = {fCHEPCH, fCHEPUS, fUSEPCH, fUSEPUS, fDCH, fDUS};

		return fResult;
	}  


	/*public static void main(String[] argv)
	  {
	  SQLServe sqlserve = new SQLServe();
	  sqlserve.setDatabaseConnection("sun.jdbc.odbc.JdbcOdbcDriver", "jdbc:odbc:DOC", "sa", "");
	  sqlserve.connect(null);
	  CallableStatement callableStmt = null;
	  
	  try
	  {
	  callableStmt = sqlserve.getCallableStatement("{call SpGetPrice(?, ?, ?)}");
	  
	  callableStmt.registerOutParameter(1, Types.FLOAT);
	  Integer intUserID = new Integer("15595");
	  callableStmt.setInt(2, intUserID.intValue());
	  Integer intPartno = new Integer("10410");
	  callableStmt.setInt(3, intPartno.intValue());
	  }
	  catch (SQLException e)
	  {
	  }
	  
	  String dPrice = sqlserve.executeCallableStmt(callableStmt, null);
	  System.out.println("dPrice: "+dPrice);
	  }
	  */

}
