
/*
 * Copyright (c) 1998, 1999 Semiotek Inc. All Rights Reserved.
 *
 * This software is the confidential intellectual property of
 * of Semiotek Inc.; it is copyrighted and licensed, not sold.
 * You may use it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation. If you 
 * do not want to use the GPL, you may still use the software after
 * purchasing a proprietary developers license from Semiotek Inc.
 *
 * This software is provided "as is", with NO WARRANTY, not even the 
 * implied warranties of fitness to purpose, or merchantability. You
 * assume all risks and liabilities associated with its use.
 *
 * See the attached License.html file for details, or contact us
 * by e-mail at info@semiotek.com to get a copy.
 */


package org.webmacro.engine;

import org.webmacro.util.java2.*;
import java.util.*;
import java.io.*;
import org.webmacro.util.*;

/**
  * This directive is used to set variables in the context data map 
  * if a variable contains a dot operator the variable will be resolved
  * accordingly. Expects the following syntax:
  * #set $abc = Term 
  */
final class SetDirective implements Macro
{

   final private Variable myVariable;
   final private Object myValue;

   /**
     *  Constuctor for SetDirective.
     */
   SetDirective(Object inVariable, Object inValue) {
      myVariable = (Variable) inVariable;
      myValue= inValue;
   }


   /**
     * Parse the directive. Expect format "set $person = @customer"
     * etc.
     * @return SetDirective
     * @exception ParseException on unrecoverable parse error
     * @exception IOException on failure to read from parseTool
     */
   static public final Object parse(ParseTool in) 
      throws IOException, ParseException
   {

      Variable tempSetVariable;

      // look for "set" key word  
      if ((in.ttype !=  in.TT_WORD) && (!in.sval.equals("set"))) {
         if (Engine.debug) {
            Engine.log.debug("Set: " + in.sval + " unrecognized.");
         }
         return null;
      } 
     
      in.nextToken(); // advance token by one  
      if (! in.parseSpaces()) {  // expects to be at space advance till no space
         if (Engine.debug) {
	    Engine.log.debug("Set: ttype:" + (char)in.ttype);
	    Engine.log.debug("Set: sval:" + in.sval);
         }
         throw new ParseException(in,
               "(1) expected spaces after keyword, got: " + (char) in.ttype);
      }

      if (in.ttype != '$'){ // do not eat dollar sign check only 
         throw new ParseException(in,
               "expected $ after set key word, got: " + (char) in.ttype);
      }
      
      // get Variable
      tempSetVariable = (Variable) Variable.parse(in);
      if (tempSetVariable == null) {
         throw new ParseException(in,
               "expected TT_WORD, got: " + (char) in.ttype);
      }    
      if (Engine.debug) {
         Engine.log.debug("Set: (2) char at:" + (char) in.ttype);
      }

      in.parseSpaces();

      if (! in.parseChar('=') ) {  
         // expects to be at space adv till no space 
         throw new ParseException(in,
               "(2) expected spaces after keyword, got: " + (char) in.ttype);
      }

      in.parseSpaces(); 

      Object outValue = Term.parse(in); 

      in.parseSpaces(); 

      if (in.parseToEOL()) {
         throw new ParseException(in,
              "extraneous text after Term in SetDirective");  
      }
      return new SetDirective(tempSetVariable, outValue); 
   }

   /**
     * Interpret the directive and write it out
     * @exception InvalidContextException is required data is missing
     * @exception IOException if could not write to output stream
     */
   public final void write(Writer out, Object context) 
      throws InvalidContextException, IOException
   {
       
      Object errorString;
      // on fail evaluate returns a String
      if ((errorString = evaluate(context)) != null) {
         out.write(errorString.toString());  
      } 
       
   }

   /**
     * Evaluate the current macro. The evaluate function for the list
     * directive
     */ 
   public final Object evaluate(Object context)
   {
      // do nothing
      try {
         if (myValue instanceof Macro) {
            myVariable.setValue(context, ((Macro) myValue).evaluate(context));
         } else {
            myVariable.setValue(context, myValue);
         }
      } catch (InvalidContextException e) {
         Engine.log.exception(e);
         Engine.log.error("Set: Unable to set value: " + myVariable);
         return ("<!--\n Unable to set value: " + myVariable + " \n-->");
      } 
      return null;
   }  


   /** test harness */
   public static void main(String arg[]) {
   
      Log.setLevel(Log.DEBUG);
      Log.traceExceptions(true);
      Map m = new HashMap();
      m.put("a", "paper");
      String tests[] = {
         "set $a = \"wood\"", 
         "set $b = \"cake\"", 
      };
  
      System.out.println("- - - - - - - - - - - - - - - - -");
      for (int i = 0; i < tests.length; i++) {
         System.out.println("TESTING: " + tests[i] + " :");
         try {
            System.out.println("before set variable a: " + m.get("a"));
            System.out.println("befoer set variable b: " + m.get("b"));
            ParseTool in = new ParseTool("string",new StringReader(tests[i]));
            in.nextToken();
            SetDirective id = (SetDirective) SetDirective.parse(in);

            if (id == null) {
               System.out.println("FAILED--null returned from parse().");
            } else {
               Object result = id.evaluate(m);
               System.out.println("after set variable a: " + m.get("a"));
               System.out.println("after set variable a: " + m.get("b"));
            }
         } catch (Exception e) {
            e.printStackTrace();
         }
         System.out.println("- - - - - - - - - - - - - - - - -");
      }
      System.out.println();
      System.out.println("Done.");
   }
}


