
/*
 * Copyright (c) 1998, 1999 Semiotek Inc. All Rights Reserved.
 *
 * This software is the confidential intellectual property of
 * of Semiotek Inc.; it is copyrighted and licensed, not sold.
 * You may use it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation. If you 
 * do not want to use the GPL, you may still use the software after
 * purchasing a proprietary developers license from Semiotek Inc.
 *
 * This software is provided "as is", with NO WARRANTY, not even the 
 * implied warranties of fitness to purpose, or merchantability. You
 * assume all risks and liabilities associated with its use.
 *
 * See the attached License.html file for details, or contact us
 * by e-mail at info@semiotek.com to get a copy.
 */


package org.webmacro.engine;

import org.webmacro.util.java2.*;
import java.util.*;
import java.io.*;
import org.webmacro.util.*;


/**
  * Enclose a block of text that is to be inrepreted by a different
  * interpreter than WebMacro's template engine. Initially the only
  * other available interpreter is the "text" interpreter which 
  * includes text as it is, without parsing it at all. In the future
  * we will support a variety of script langauges as well.
  * <p>
  * Here is an example:<pre>
  *
  *     #use text until END
  *        $notParsed here at all
  *     END
  *
  *     #use text
  *        the default close is the word #end.
  *     #end
  *
  * </pre>In the future many languages will be supported, for now 
  * you can use this to include unquoted text.
  * that should not be interpreted at all.
  */
final class UseDirective implements Macro
{

   /**
     * What language this script represents
     */
   final private String myLangDefn;

   /**
     * The text to be parsed by the alternate language
     */
   final private String myContent;

   /**
     * The languages that are supported: later we can add more, and
     * preferably we should be able to add them dynamically.
     */
   private final static Hashtable languages = new Hashtable();
   static {
      languages.put("text","text");
   }

   /**
     * If an end token is not supplied, used this instead
     */
   final public static String DEFAULT_END_TOKEN = "#end";

   UseDirective(String langDefn, String content)
   { 
       myLangDefn = (String) langDefn;
       myContent = content;
   }

   /**
     * Begins parsing at the word "use", we assume the '#' has already
     * been eaten. The first argument is the name of the language to use 
     * instead of WebMacro. Subsequently there may optionally be an 
     * attribute specifying what ends the unparsed block, the default 
     * is "#end".
     * @exception ParseException on unrecoverable parse error
     * @exception IOException on failure to read from parseTool
    */
   static public final Object parse(ParseTool in) 
      throws IOException, ParseException
   {
      String endToken;
      String outLangDefn;
      
      // detect use keyword 
      if (! in.parseWord("use")) {
         return null;
      }

      // if at space remove spaces 
      in.parseSpaces();

      // get type of language 
      if ((in.ttype=='\"') || (in.ttype=='\'')) {
         outLangDefn =  QuotedText.parseQuotedText(in);
      } else if (in.ttype ==in.TT_WORD)  {
         outLangDefn = in.sval; 
         in.nextToken();
      } else {
          throw new ParseException(in, 
             "UseDirective: unexpected character '" + (char)in.ttype  +
             "' found, expecting a word or a quoted string."); 
      }

      if (outLangDefn == null) {
         throw new ParseException (in,
               "UseDirective: unable to determine language type" +
               ", which should be the first argument after #use.");
      }
      if ( languages.get(outLangDefn) == null ) {
         throw new ParseException (in,
            "UseDirective: " + outLangDefn + " is not a supported language."
            + " Valid values are: " 
            + (new DictionaryTool(languages)).keyString());
      }

      // detect key word until
      in.parseSpaces();
      if (!in.parseWord("until")) {
         endToken = DEFAULT_END_TOKEN;
      } else {
         in.parseSpaces();
         if ((in.ttype=='\"') || (in.ttype=='\'')) {
            endToken =  QuotedText.parseQuotedText(in);
         } else if (in.ttype==in.TT_WORD) {
            endToken = in.sval;
            in.nextToken();
         } else {
             throw new ParseException(in, 
                   "UseDirective: until not followed by boundary token"); 
         }
      }
      in.parseSpaces();

      // newline must exist to indicate where use text actually begins 
      if (in.ttype != '\n') { // parseTool makes this portable
         if (Engine.debug) {
	    Engine.log.debug("UseDirective: in.ttype: " + (char) in.ttype);
         }
         throw new ParseException(in, 
           "Unexpected content (" + (char) in.ttype + ") at end of line in #use");
      }
      in.nextToken();

      String line = in.parseLine();

      StringBuffer str = new StringBuffer(1024);
      // advance until all tokens until endToken matched or end of line
      // get line will advance to the next line

      LOOP: while (in.ttype != in.TT_EOF) 
      {
         if (Engine.debug) {
	    Engine.log.debug("UseDirective read line: " + line);
         }
         if (line.indexOf(endToken) != -1) {
            if (line.trim().equals(endToken.trim())) {
               break LOOP; 
            }
         }
         str.append(line);
         line = in.parseLine();
      }

      if (Engine.debug) {
         Engine.log.debug("UseDirective: DONE PARSING");
      }

      // if endToken was not encountered
      if (line.indexOf(endToken) == -1) {
         throw new ParseException(in, 
	     "Expected endToken before EOF");
      } 

      return new UseDirective(outLangDefn, str.toString());
   }


   /**
     * Evaluate the current macro and return it as a string. Same
     * basic operation as calling write.
     */ 
   public Object evaluate(Object data)
   {
      return myContent;
   }  


   /**
     * Interpret the contents according to the language type and 
     * write the result out ot the stream.
     * @exception InvalidContextException is required data is missing
     * @exception IOException if could not write to output stream
     */
   public void write(Writer out, Object data) 
      throws InvalidContextException, IOException
   {
      out.write(myContent);
   }

   /**
     * Test harness
     */
   public static void main(String arg[]) {
      
      
      System.out.println("Testing UseDirective:");

      Log.setLevel(Log.DEBUG);
      Log.traceExceptions(true);
   
   
      Map m = new HashMap();
   
      //m.put("name","some content");


      String tests[] = {
            "use",
            "use_default"
         };
   
      
      System.out.println("- - - - - - - - - - - - - - - - -");
      for (int i = 0; i < tests.length; i++) {
         File inFile = new File("examples",tests[i] + ".wm");
         File outFile = new File("examples",tests[i] + ".html");
         System.out.println("TESTING input file: " + inFile);
         try {
            ParseTool iin = new ParseTool(inFile);
            iin.nextToken();
            Block id = (Block) Block.parse(iin);
            //UseDirective id = (UseDirective) UseDirective.parse(iin);
            if (id == null) {
               System.out.println("FAILED--null returned from parse().");
            } else {
               Writer out = new FileWriter(outFile);
               id.write(out, m); 
	       out.close();
            }
         } catch (Exception e) {
            System.out.println("FAILED--threw exception:");
            e.printStackTrace();
            System.out.println();
         }
         System.out.println("- - - - - - - - - - - - - - - - -");
      }
      System.out.println();
      System.out.println("Done.");

   } 
  
}


