/** PhotoOrganizer - MailCourier 
 * Copyright (C) 1999 Dmitriy Rogatkin.  All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 *  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *  $Id: MailCourier.java,v 1.2 2001/06/27 20:22:44 rogatkin Exp $
 */
package photoorganizer.courier;

import java.io.*;

import javax.mail.*;
import javax.mail.internet.MimeMultipart;
import javax.mail.internet.MimeBodyPart;
import javax.activation.DataHandler;
import javax.activation.DataSource;

import rogatkin.*;
import photoorganizer.*;
import photoorganizer.formats.*;
import photoorganizer.renderer.*;
import photoorganizer.ftp.*;

public class MailCourier implements Courier {
    Multipart multipart, alternative;
    Controller controller;
    int counter;
    
	public MailCourier(Controller controller, Multipart multipart, Multipart alternative) {
		this.controller = controller;
		this.multipart = multipart;
		this.alternative = alternative;
	}

	public void deliver(StringBuffer buf, String destPath, String contentType, String encoding) throws IOException {
		try {
			MimeBodyPart bp = new MimeBodyPart();
			bp.setDataHandler(new DataHandler(new TextDataSource(buf, contentType, encoding)));
			alternative.addBodyPart(bp);
		}  catch(MessagingException me) {
			throw new IOException(""+me);
		}
	}

	public void deliver(String srcPath, String destPath) throws IOException {
		try {
			//multipart.addBodyPart(new MimeBodyPart(new FileInputStream((String)images_to_copy.elementAt(i))));
			MimeBodyPart bp = new MimeBodyPart();
			bp.setDataHandler(new DataHandler(new ImageDataSource(srcPath)));
			bp.addHeader("Content-ID", "<"+srcPath+'>');
			multipart.addBodyPart(bp);
		}  catch(MessagingException me) {
			throw new IOException(""+me);
		}
	}
	
	public String deliver(AbstractFormat format, String destPath) throws IOException {
		try {
			AbstractInfo ii = format.getInfo();
			if (ii == null)
				return "";
			ThumbnailImageDataSource ids = 
				new ThumbnailImageDataSource("media"+counter++, format.getThumbnailType());
			if (ii instanceof AbstractImageInfo)
				((AbstractImageInfo)ii).saveThumbnailImage((BasicJpeg)format, ids.getOutputStream());
			else 
				ids.getOutputStream().write(format.getThumbnailData(null/*get desired dimension from props*/));
			MimeBodyPart bp = new MimeBodyPart();
			bp.setDataHandler(new DataHandler(ids));
			bp.addHeader("Content-ID", "<"+ids.getName()+'>');
			//bp.setContentType(ids.getContentType());
			multipart.addBodyPart(bp);
			return "cid:"+ids.getName();
		} catch(MessagingException me) {
			throw new IOException(""+me);
		}
	}
	
	public void checkForDestPath(String path) throws IOException {
	}
	
	public void init() throws IOException {
	}

	public void done() {

	}

	public boolean isLocal() {
		return false;
	}

	public boolean isContentIncluded() {
		return true;
	}
}

class TextDataSource implements DataSource {
	StringBuffer html;
	String contentType, encoding;
	TextDataSource(StringBuffer html, String contentType, String encoding) {
		this.html = html;
		this.contentType = contentType;
		this.encoding = encoding;
	}

	public InputStream getInputStream()	throws IOException {
		if (encoding != null && encoding.length()>0)
			return new ByteArrayInputStream(html.toString().getBytes(encoding));
		else
			return new ByteArrayInputStream(html.toString().getBytes());
	}
	
	public OutputStream getOutputStream() throws IOException {
		return null;
	}

	public String getContentType() {
		return contentType+((encoding != null && encoding.length()>0)?("; charset="+encoding):"");
	}

	public String getName() {
		return "Album";
	}
}

class ImageDataSource implements DataSource {
	String path;
	
	ImageDataSource(String path) {
		this.path = path;
	}

	synchronized public InputStream getInputStream() throws IOException {
		return new FileInputStream(path);
	}
	
	public OutputStream getOutputStream() throws IOException {
		return null;
	}

	public String getContentType() {
		String name = getName();
		int dp = name.lastIndexOf('.');
		String ext;
		if (dp < 0)
			ext = Resources.EXT_JPEG;
		else
			ext = name.substring(dp+1);
		if ("jpg".equalsIgnoreCase(ext))
			ext = Resources.EXT_JPEG;
		else if (MP3.TYPE.equalsIgnoreCase(ext))
			return "audio/mpeg";
		return "image/"+ext;
	}

	public String getName() {
		return new File(path).getName();
	}
	
/*	protected void finalize() throws Throwable {
		super.finalize();
	}*/
}

class ThumbnailImageDataSource implements DataSource {
	ByteArrayOutputStream os;
	String name, ext;
	ThumbnailImageDataSource(String name, String ext) {
		this.name = name;
		this.ext = ext;
		os = new ByteArrayOutputStream(8096);
	}

	public InputStream getInputStream()	throws IOException {
		return new ByteArrayInputStream(os.toByteArray());
	}
	
	public OutputStream getOutputStream() throws IOException {
		return os;
	}

	public String getContentType() {
		return "image/"+ext;
	}

	public String getName() {
		return name;
	}
}
