/* MediaChest - $RCSfile: RipperPanel.java,v $
 * Copyright (C) 2001 Dmitriy Rogatkin.  All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 *  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 *  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 *  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 *  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 *  SUCH DAMAGE.
 *
 *  Visit http://mediachest.sourceforge.net to get the latest infromation
 *  about Rogatkin's products.
 *  $Id: RipperPanel.java,v 1.22 2001/09/22 00:39:40 rogatkin Exp $
 */

package photoorganizer.renderer;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import java.awt.datatransfer.*;
import java.text.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;
import javax.swing.tree.*;
import javax.swing.table.*;

import javax.xml.parsers.*;
import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import rogatkin.*;
import photoorganizer.formats.*;
import photoorganizer.*;

import photoorganizer.album.*;
import photoorganizer.directory.*;
import photoorganizer.media.*;

public /*final*/class RipperPanel extends JSplitPane implements ActionListener, Persistable {
    public static final String SECNAME = "RipperPanel";
    static final DataFlavor MEDIA_FORMAT_DF = new DataFlavor(AbstractFormat.class, "application/x-java-jvm-local-objectref");
    static final DataFlavor FILE_FORMAT_DF = new DataFlavor(File.class, "application/x-java-jvm-local-objectref");
    static final DataFlavor ALBUM_NODE_INDEX_DF = new DataFlavor(Integer.class, "application/x-java-jvm-local-objectref");
    static final DataFlavor FILE_LIST_DF = DataFlavor.javaFileListFlavor;
    static final DataFlavor MIXED_FILE_MEDIA_LIST_DF = new DataFlavor(java.util.ArrayList.class, "application/x-java-jvm-local-objectref");
    final static int TOO_MANY_WINDOWS = 20;
    
    protected Controller controller;
    protected DirectoryTable folderTable;
    protected AlbumTable albumTable;
    protected RipperTable ripperTable;
    protected CollectionTable collectionTable;
    protected SAXParserFactory saxParserFactory;
    
    public RipperPanel(Controller controller) {
        super(VERTICAL_SPLIT);
        this.controller = controller;
        setOneTouchExpandable(true);
        access = new MediaAccess(controller);
        JTabbedPane mediaSourceTabs = new JTabbedPane(SwingConstants.TOP);
        folderTable = new DirectoryTable(new FolderTableModel(controller));
        mediaSourceTabs.insertTab(Resources.TAB_DIRECTORY, (Icon)null,
        new JSplitPane(HORIZONTAL_SPLIT,
        new OneThirdScroll(new DirectoryTree(folderTable)), new JScrollPane(folderTable)),
        Resources.TTIP_DIRECTORYTAB, mediaSourceTabs.getTabCount());
        albumTable = new AlbumTable(new AlbumTableModel(controller));
        mediaSourceTabs.insertTab(Resources.TAB_ALBUM, (Icon)null,
        new JSplitPane(HORIZONTAL_SPLIT,
        new OneThirdScroll(new AlbumTree(albumTable)),
        new JScrollPane(albumTable)),
        Resources.TTIP_ALBUM, mediaSourceTabs.getTabCount());
        mediaSourceTabs.insertTab(Resources.TAB_COLLECTION, (Icon)null,
        new JScrollPane(collectionTable = new CollectionTable()),
        Resources.TTIP_COLLECTLIST, mediaSourceTabs.getTabCount());
        setTopComponent(mediaSourceTabs);
        ripperTable = new RipperTable();
        setBottomComponent(new JSplitPane(HORIZONTAL_SPLIT,
        new OneThirdScroll(ripper = new RipperTree(ripperTable)), new JScrollPane(ripperTable)));
        setDividerLocation(0.5);
    }
    
    public void save() {
        controller.saveTableColumns(folderTable, SECNAME, folderTable.getClass().getName());
        controller.saveTableColumns(albumTable, SECNAME, albumTable.getClass().getName());
        controller.saveTableColumns(ripperTable, SECNAME, ripperTable.getClass().getName());
        controller.saveTableColumns(collectionTable, SECNAME, collectionTable.getClass().getName());
    }
    
    public void load() {
        controller.loadTableColumns(folderTable, SECNAME, folderTable.getClass().getName());
        controller.loadTableColumns(albumTable, SECNAME, albumTable.getClass().getName());
        controller.loadTableColumns(ripperTable, SECNAME, ripperTable.getClass().getName());
        controller.loadTableColumns(collectionTable, SECNAME, collectionTable.getClass().getName());
    }
    
    public void actionPerformed(ActionEvent a) {
        String cmd = a.getActionCommand();
        
        if (cmd.equals(Resources.MENU_RECORD_DISK)) {
            ((RipperModel)ripper.getModel()).doDisk(controller);
        }
    }
    
    class DirectoryTree extends JTree implements ActionListener {
        DirectoryTable folderTable;
        DirectoryTree(DirectoryTable folderTable) {
            super(new TreeDesktopModel() /*FileSystemModel(controller)*/);
            this.folderTable = folderTable;
            setCellRenderer(((TreeDesktopModel)getModel()).adoptCellRenderer(getCellRenderer()));
            setRootVisible(false);
            addTreeSelectionListener(new TreeSelectionListener() {
                public void valueChanged(TreeSelectionEvent e) {
                    ((FolderTableModel)DirectoryTree.this.folderTable.getModel()).updateModel(((File)e.getPath().getLastPathComponent()).listFiles());
                    if (DirectoryTree.this.folderTable.getRowCount() > 0)
                        DirectoryTree.this.folderTable.removeRowSelectionInterval(0, DirectoryTree.this.folderTable.getRowCount()-1);
                    DirectoryTree.this.folderTable.revalidate();
                }
            });
            addMouseListener(new MouseInputAdapter() {
                public void mouseClicked(MouseEvent e) {
                    if ((e.getModifiers() & InputEvent.BUTTON3_MASK) > 0)
                        getRMouseMenu().show(DirectoryTree.this, e.getX(), e.getY());
                }
            });
            setDragEnabled(true);
        }
        
        public void actionPerformed(ActionEvent a) {
            String cmd = a.getActionCommand();
            if (cmd.equals(Resources.MENU_REFRESH)) {
                ((TreeDesktopModel)getModel()).reset(null);
                // invalidate();
            }
        }
        
        JPopupMenu getRMouseMenu() {
            JPopupMenu result = new JPopupMenu();
            JMenuItem item;
            result.add(item = new JMenuItem(Resources.MENU_REFRESH));
            item.addActionListener(this);
            result.addSeparator();
            result.add(item = new JMenuItem(Resources.MENU_PROPERTIES));
            item.addActionListener(this);
            return result;
        }
    }
    
    class DirectoryTable extends JTable implements ActionListener {
        DirectoryTable(FolderTableModel tableModel) {
            super(tableModel);
            addMouseListener(new MouseInputAdapter() {
                public void mouseClicked(MouseEvent e) {
                    if ((e.getModifiers() & InputEvent.BUTTON3_MASK) > 0)
                        getRMouseMenu().show(DirectoryTable.this, e.getX(), e.getY());
                    else if (e.getClickCount() == 2)
                        actionPerformed(new ActionEvent(this, 0, Resources.MENU_SHOW));
                }
            });
            addKeyListener(new KeyAdapter() {
                public void keyReleased(KeyEvent e) {
                    if (e.getKeyCode() == e.VK_DELETE) {
						actionPerformed(new ActionEvent(this, 3, Resources.MENU_DELETE));
					}
				}
				});
            setDragEnabled(true);
            setTransferHandler(new MediaTransferHandler());
        }
        
        public void actionPerformed(ActionEvent a) {
            String cmd = a.getActionCommand();
            int[] cols = getSelectedRows() ;
            
            FolderTableModel model = (FolderTableModel)getModel();
            if (cmd.equals(Resources.MENU_SHOW)) {
                Object []medias = new Object[cols.length];
                for (int i=0; i<cols.length; i++) {
                    medias[i] = model.getElementAt(cols[i]);
                }
                if (medias.length == 1 && medias[0] != null && medias[0] instanceof File
                && ((File)medias[0]).isDirectory()) {
                } else
                    new PlaybackRequest(medias, controller.getSerializer()).playList(controller);
            } else if (cmd.equals(Resources.MENU_ADDTOCOLLECT)) {
				PhotoCollectionPanel collectionpanel = (PhotoCollectionPanel)controller.component(Controller.COMP_COLLECTION);
				File []medias = new File[cols.length];
				for (int i=0; i<cols.length; i++)
					medias[i] = model.getElementAt(cols[i]) instanceof File?(File)model.getElementAt(cols[i]):
																			((AbstractFormat)model.getElementAt(cols[i])).getFile();
				collectionpanel.add(medias);
            } else if (cmd.equals(Resources.MENU_ADDTOALBUM)) {
				// TODO: there is some code duplication from others similar places
				AlbumPane albumpane = (AlbumPane)controller.component(Controller.COMP_ALBUMPANEL);
				AlbumSelectionDialog asd = albumpane.getSelectionDialog();
				asd.setTitle(Resources.TITLE_SELECT_ALBUM+":"+cols.length);
				asd.setVisible(true);
				TreePath[] tps = asd.getSelectedAlbums();
				if (tps != null) {
					AbstractFormat []medias = new AbstractFormat[cols.length];
					for (int i=0; i<cols.length; i++)
						medias[i] = model.getElementAt(cols[i]) instanceof AbstractFormat?
									(AbstractFormat)model.getElementAt(cols[i]):
									// TODO: need check for not null and valid?
									MediaFormatFactory.createMediaFormat((File)model.getElementAt(cols[i]));
					albumpane.addToAlbum(medias, tps);
				}
            } else if (cmd.equals(Resources.MENU_DELETE)) {
				Object media;
				boolean requestConfirm = true;
				for (int i=0; i<cols.length; i++) {
					media = model.getElementAt(cols[i]);
					File file = null;
					if (media instanceof File)
						file = (File)media;
					else if (media instanceof AbstractFormat) 
						file = ((AbstractFormat)media).getFile();
					if (requestConfirm) {
						int res = JOptionPane.showOptionDialog(this,
															   file.getName()+Resources.LABEL_CONFIRM_DEL, Resources.TITLE_DELETE,
															   0, JOptionPane.WARNING_MESSAGE, null, new Object[] {Resources.CMD_YES, Resources.CMD_YES_ALL, Resources.CMD_NO, Resources.CMD_CANCEL}, Resources.CMD_NO);
						if (res == 2)
							continue;
						else if (res == 1)
							requestConfirm = false;
						else if (res == 3)
							break;
					}
					if (file != null && !file.delete())
						((StatusBar)controller.component(Controller.COMP_STATUSBAR)).flashInfo("Cannot delete "+file, true);
				}
				revalidate();
			} else if (cmd.equals(Resources.MENU_PROPERTIES)) {
                for (int i=0; i<cols.length && i<TOO_MANY_WINDOWS; i++) {
                    Object o = model.getElementAt(cols[i]);
                    if (o instanceof AbstractFormat)
                        PropertiesPanel.showProperties((AbstractFormat)o, controller);
                    else if (o instanceof File) {
                        AbstractFormat format = MediaFormatFactory.createMediaFormat((File)o);
                        if (format != null && format.isValid())
                            PropertiesPanel.showProperties(format, controller);
                    }
                }
            }
        }
        
        JPopupMenu getRMouseMenu() {
            JPopupMenu result = new JPopupMenu();
            JMenuItem item;
            result.add(item = new JMenuItem(Resources.MENU_SHOW));
            item.addActionListener(this);
            result.addSeparator();
            result.add(item = new JMenuItem(Resources.MENU_ADDTOCOLLECT));
            item.addActionListener(this);
            result.add(item = new JMenuItem(Resources.MENU_ADDTOALBUM));
            item.addActionListener(this);
            result.add(item = new JMenuItem(Resources.MENU_DELETE));
            item.addActionListener(this);
            result.addSeparator();
            result.add(item = new JMenuItem(Resources.MENU_PROPERTIES));
            item.addActionListener(this);
            return result;
        }
    }
    
    class AlbumTree extends JTree {
        AlbumTable albumTable;
        AlbumTree(AlbumTable albumTable) {
            this.albumTable = albumTable;
            setModel(new AlbumModel(access));
            getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
            addTreeSelectionListener(new TreeSelectionListener() {
                public void valueChanged(TreeSelectionEvent e) {
                    ((AlbumTableModel)AlbumTree.this.albumTable.getModel()).updateModel(access.getAlbumContents(access.getAlbumId(e.getPath())));
                    if (AlbumTree.this.albumTable.getRowCount() > 0)
                        AlbumTree.this.albumTable.removeRowSelectionInterval(0, AlbumTree.this.albumTable.getRowCount()-1);
                    AlbumTree.this.albumTable.revalidate();
                }
            });
            setDragEnabled(true);
            setTransferHandler(new MediaTransferHandler());
        }
    }
    

	// TODO: merge with folder table using some common interface for different models
    class MediaTable extends JTable implements ActionListener {
        MediaTable(TableModel tableModel) {
            super(tableModel);
            addMouseListener(new MouseInputAdapter() {
                public void mouseClicked(MouseEvent e) {
                    if ((e.getModifiers() & InputEvent.BUTTON3_MASK) > 0)
                        getRMouseMenu().show(MediaTable.this, e.getX(), e.getY());
                    else if (e.getClickCount() == 2)
                        actionPerformed(new ActionEvent(this, 0, Resources.MENU_SHOW));
                }
            });
        }
        
        public void actionPerformed(ActionEvent a) {
            String cmd = a.getActionCommand();
            int[] cols = getSelectedRows() ;
            AlbumTableModel model = (AlbumTableModel)getModel();
            if (cmd.equals(Resources.MENU_SHOW)) {
                Object []medias = new Object[cols.length];
                for (int i=0; i<cols.length; i++)
                    medias[i] = model.getElementAt(cols[i]);
                // TODO: do it only for MP3 and display images in a free panel
                new PlaybackRequest(medias, controller.getSerializer()).playList(controller);
            } else if (cmd.equals(Resources.MENU_ADDTOCOLLECT)) {
				PhotoCollectionPanel collectionpanel = (PhotoCollectionPanel)controller.component(Controller.COMP_COLLECTION);
				File []medias = new File[cols.length];
				for (int i=0; i<cols.length; i++)
					medias[i] = model.getElementAt(cols[i]) instanceof File?(File)model.getElementAt(cols[i]):
																			((AbstractFormat)model.getElementAt(cols[i])).getFile();
				collectionpanel.add(medias);
            } else if (cmd.equals(Resources.MENU_ADDTOALBUM)) {
				// TODO: there is some code duplication from others similar places
				AlbumPane albumpane = (AlbumPane)controller.component(Controller.COMP_ALBUMPANEL);
				AlbumSelectionDialog asd = albumpane.getSelectionDialog();
				asd.setTitle(Resources.TITLE_SELECT_ALBUM+":"+cols.length);
				asd.setVisible(true);
				TreePath[] tps = asd.getSelectedAlbums();
				if (tps != null) {
					AbstractFormat []medias = new AbstractFormat[cols.length];
					for (int i=0; i<cols.length; i++)
						medias[i] = model.getElementAt(cols[i]) instanceof AbstractFormat?
									(AbstractFormat)model.getElementAt(cols[i]):
									// TODO: need check for not null and valid?
									MediaFormatFactory.createMediaFormat((File)model.getElementAt(cols[i]));
					albumpane.addToAlbum(medias, tps);
				}
            } else if (cmd.equals(Resources.MENU_PROPERTIES)) {
                for (int i=0; i<cols.length && i<TOO_MANY_WINDOWS; i++) {
                    Object o = model.getElementAt(cols[i]);
                    if (o instanceof AbstractFormat)
                        PropertiesPanel.showProperties((AbstractFormat)o, controller);
                    else if (o instanceof File) {
                        AbstractFormat format = MediaFormatFactory.createMediaFormat((File)o);
                        if (format != null && format.isValid())
                            PropertiesPanel.showProperties(format, controller);
                    }
                }
            }
        }
        
        JPopupMenu getRMouseMenu() {
            JPopupMenu result = new JPopupMenu();
            JMenuItem item;
            result.add(item = new JMenuItem(Resources.MENU_SHOW));
            item.addActionListener(this);
            result.addSeparator();
            result.add(item = new JMenuItem(Resources.MENU_ADDTOCOLLECT));
            item.addActionListener(this);
            result.add(item = new JMenuItem(Resources.MENU_ADDTOALBUM));
            item.addActionListener(this);
            result.addSeparator();
            result.add(item = new JMenuItem(Resources.MENU_PROPERTIES));
            item.addActionListener(this);
            return result;
        }
    }
    
    class AlbumTable extends MediaTable {
        AlbumTable(AlbumTableModel albumTableModel) {
            super(albumTableModel);
            setDragEnabled(true);
            setTransferHandler(new MediaTransferHandler());
        }
    }
    
    class CollectionTable extends JTable implements ActionListener {
        CollectionTable() {
            super(((PhotoCollectionPanel)controller.component(Controller.COMP_COLLECTION)).getModel());
            addMouseListener(new MouseInputAdapter() {
                public void mouseClicked(MouseEvent e) {
                    if ((e.getModifiers() & InputEvent.BUTTON3_MASK) > 0)
                        getRMouseMenu().show(CollectionTable.this, e.getX(), e.getY());
                    else if (e.getClickCount() == 2)
                        actionPerformed(new ActionEvent(this, 0, Resources.MENU_SHOW));
                }
            });
            setDragEnabled(true);
            setTransferHandler(new MediaTransferHandler());
        }
        
        public void actionPerformed(ActionEvent a) {
            String cmd = a.getActionCommand();
            int[] cols = getSelectedRows() ;
            
            ListModel model = (ListModel)getModel();
            if (cmd.equals(Resources.MENU_SHOW)) {
                Object []medias = new Object[cols.length];
                for (int i=0; i<cols.length; i++)
                    medias[i] = ((Thumbnail)model.getElementAt(cols[i])).getFormat();
                new PlaybackRequest(medias, controller.getSerializer()).playList(controller);
            } else if (cmd.equals(Resources.MENU_PROPERTIES)) {
                for (int i=0; i<cols.length && i<TOO_MANY_WINDOWS; i++) {
                    Thumbnail tn = (Thumbnail)model.getElementAt(cols[i]);
                    if (tn != null) // TODO: investigate why it can happen
                        PropertiesPanel.showProperties(tn.getFormat(), controller);
                }
            }
        }
        
        JPopupMenu getRMouseMenu() {
            JPopupMenu result = new JPopupMenu();
            JMenuItem item;
            result.add(item = new JMenuItem(Resources.MENU_SHOW));
            item.addActionListener(this);
            result.addSeparator();
            result.add(item = new JMenuItem(Resources.MENU_PROPERTIES));
            item.addActionListener(this);
            return result;
        }
    }
    
    class RipperTree extends JTree implements ActionListener {
        RipperTable ripperTable;
        RipperTree(RipperTable ripperTable) {
            super (new RipperModel(Serializer.getInt(controller.getSerializer().getProperty(MediaOptionsTab.SECNAME, MediaOptionsTab.PLAYLIST_TYPE)
            , MediaOptionsTab.AUDIO_MEDIA) == MediaOptionsTab.AUDIO_MEDIA));
            this.ripperTable = ripperTable;
            addMouseListener(new MouseInputAdapter() {
                public void mouseClicked(MouseEvent e) {
                    if ((e.getModifiers() & InputEvent.BUTTON3_MASK) > 0)
                        getRMouseMenu().show(RipperTree.this, e.getX(), e.getY());
                    else if (e.getClickCount() == 2)
                        actionPerformed(new ActionEvent(this, 0, Resources.MENU_SHOW));
                }
            });
            
            setTransferHandler(new MediaTransferHandler());
            
            addTreeSelectionListener(new TreeSelectionListener() {
                public void valueChanged(TreeSelectionEvent e) {
                    ((AlbumTableModel)RipperTree.this.ripperTable.getModel()).updateModel((VirtualFolder)e.getPath().getLastPathComponent());
                    if (RipperTree.this.ripperTable.getRowCount() > 0)
                        RipperTree.this.ripperTable.removeRowSelectionInterval(0, RipperTree.this.ripperTable.getRowCount()-1);
                    RipperTree.this.ripperTable.tableChanged(new TableModelEvent(RipperTree.this.ripperTable.getModel()));
                }
            });
            setEditable(true);
        }
        
        public void actionPerformed(ActionEvent a) {
            String cmd = a.getActionCommand();
            
            if (cmd.equals(Resources.MENU_NEW_FOLDER)) {
                Object lc = getSelectionPath()==null || getSelectionPath().getLastPathComponent() == null?getModel():getSelectionPath().getLastPathComponent();
                ((java.util.List)lc).add(new VirtualFolder(Resources.LABEL_NEW_FOLDER+((java.util.List)lc).size()));
                ((RipperModel)getModel()).fireTreeStructureChanged(this, new Object[]{lc}, null, null);
            } else if (cmd.equals(Resources.MENU_RENAME)) {
                startEditingAtPath(getSelectionPath());
            } else if (cmd.equals(Resources.MENU_DELETE)) {
                Object lc = getSelectionPath();
                if (lc != null) {
                    Object pc = ((TreePath)lc).getParentPath();
                    if (pc != null)
						pc = ((TreePath)pc).getLastPathComponent();
                    if (pc != null && pc instanceof java.util.List) {
                        ((java.util.List)pc).remove(((TreePath)lc).getLastPathComponent());
                        ((RipperModel)getModel()).fireTreeStructureChanged(this, new Object[]{pc}, null, null);
                    }
                }
            } else if (cmd.equals(Resources.MENU_NEW_LAYOUT)) {
                //setModel(new RipperModel(Serializer.getInt(controller.getSerializer().getProperty(MediaOptionsTab.SECNAME, MediaOptionsTab.PLAYLIST_TYPE)
                //										   , MediaOptionsTab.AUDIO_MEDIA) == MediaOptionsTab.AUDIO_MEDIA));
                ((AlbumTableModel)ripperTable.getModel()).updateModel((VirtualFolder)null);
                ((RipperModel)getModel()).reset(Serializer.getInt(controller.getSerializer().getProperty(MediaOptionsTab.SECNAME, MediaOptionsTab.PLAYLIST_TYPE)
                , MediaOptionsTab.AUDIO_MEDIA) == MediaOptionsTab.AUDIO_MEDIA);
                ((RipperModel)getModel()).fireTreeStructureChanged(this, new Object[]{getModel()}, null, null);
                ((StatusBar)controller.component(Controller.COMP_STATUSBAR)).displayMetric("");
            } else if (cmd.equals(Resources.MENU_LOAD_LAYOUT)) {
                JFileChooser fc = new XmlFileChooser(false);
                if(fc.showOpenDialog(this) == JFileChooser.APPROVE_OPTION)
                    try {
                        getSaxParserFactory().newSAXParser().parse(fc.getSelectedFile(),
                        ((XmlExposable)getModel()).getXmlHandler(null, null,
                        ((XmlExposable)getModel()).getNameSpacePrefix(), null, null));
                        ((RipperModel)getModel()).fireTreeStructureChanged(this, new Object[]{getModel()}, null, null);
                        ((StatusBar)controller.component(Controller.COMP_STATUSBAR)).displayMetric("");
                        // update view
                    } catch(javax.xml.parsers.ParserConfigurationException pce) {
                        System.err.println("Can not obtain XML parser "+pce);
                    } catch(IOException ioe) {
                        System.err.println("Can not read file "+fc.getSelectedFile()+' '+ioe);
                    } catch(org.xml.sax.SAXException se) {
                        System.err.println("Parser failed for "+fc.getSelectedFile()+' '+se);
                        if (se.getException() != null)
                            se.getException().printStackTrace();
                        else if (se instanceof SAXParseException) {
                            SAXParseException spe = (SAXParseException)se;
                            System.err.println(" in "+spe.getLineNumber()+':'+spe.getColumnNumber());
                        }
                    }                    
            } else if (cmd.equals(Resources.MENU_SAVE_LAYOUT)) {
                // TODO: it can be FTP either
                JFileChooser fc = new XmlFileChooser(true);
                if(fc.showOpenDialog(this) == JFileChooser.APPROVE_OPTION) {
                    if (false == fc.getSelectedFile().exists() ||
                    JOptionPane.YES_OPTION == JOptionPane.showConfirmDialog(this,
                    fc.getSelectedFile().getName()+Resources.LABEL_CONFIRM_OVERWRITE, Resources.TITLE_OVERWRITE,
                    JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE))
                        try {
                            // TODO: add encoding
                            // TODO: preserve name of saved/loaded layout
                            // TODO: keep latest visited directory?
                            Writer w = new FileWriter(fc.getSelectedFile());
                            w.write(((XmlExposable)getModel()).toXmlString());
                            w.flush();
                            w.close();
                            //System.err.println("XML:"+((XmlExposable)getModel()).toXmlString());
                        } catch(IOException ioe) {
                            System.err.println("Can not write XML coded ripper layout to :"+fc.getSelectedFile()+' '+ioe);
                        }
                }
            } else if (cmd.equals(Resources.MENU_PROPERTIES)) {
            }
        }
        
        synchronized protected SAXParserFactory getSaxParserFactory() {
            if (saxParserFactory == null)
                saxParserFactory = SAXParserFactory.newInstance();
            return saxParserFactory;
        }
        
        class XmlFileChooser extends JFileChooser {
            XmlFileChooser(boolean writeRequest) {
                super();
                if (System.getProperty(PhotoOrganizer.PROGRAMNAME+Serializer.HOMEDIRSUFX) != null) {
                    setCurrentDirectory(new File(System.getProperty(PhotoOrganizer.PROGRAMNAME+Serializer.HOMEDIRSUFX)).getAbsoluteFile());
                    rescanCurrentDirectory();
                    //System.err.println("Absolute "+new File(System.getProperty(PhotoOrganizer.PROGRAMNAME+Serializer.HOMEDIRSUFX)).getAbsoluteFile());
                }
                setDialogTitle(writeRequest?Resources.MENU_SAVE_LAYOUT:Resources.MENU_LOAD_LAYOUT);
                setDialogType(writeRequest?SAVE_DIALOG:OPEN_DIALOG );
                if (writeRequest)
                    setSelectedFile(new File(getCurrentDirectory(), "Layout1.xml"));
                //setApproveButtonText();
                //setApproveButtonToolTipText();
                addChoosableFileFilter(new javax.swing.filechooser.FileFilter() {
                    public String getDescription() {
                        return Resources.LABEL_XML_LAYOUT_FILES;
                    }
                    public boolean accept(File f) {
                        return XmlFileChooser.this.accept(f);
                    }
                });
            }
            
            public boolean accept(File f) {
                //System.err.println("Accept for "+f+" is "+(f.isDirectory() || f.getName().toLowerCase().endsWith(Resources.EXT_XML)));
                return f.isDirectory() || f.getName().toLowerCase().endsWith(Resources.EXT_XML);
            }
            
            public File getSelectedFile() {
                File result = super.getSelectedFile();
                if (result != null && !accept(result))
                    result = new File(result.getParent(), result.getName()+Resources.EXT_XML);
                return result;
            }
        }
        
        JPopupMenu getRMouseMenu() {
            JPopupMenu result = new JPopupMenu();
            JMenuItem item;
            result.add(item = new JMenuItem(Resources.MENU_NEW_FOLDER));
            item.addActionListener(this);
            result.add(item = new JMenuItem(Resources.MENU_RENAME));
            item.addActionListener(this);
            result.add(item = new JMenuItem(Resources.MENU_DELETE));
            item.addActionListener(this);
            result.addSeparator();
            result.add(item = new JMenuItem(Resources.MENU_PROPERTIES));
            item.addActionListener(this);
            result.addSeparator();
            result.add(item = new JMenuItem(Resources.MENU_NEW_LAYOUT));
            item.addActionListener(this);
            result.addSeparator();
            result.add(item = new JMenuItem(Resources.MENU_LOAD_LAYOUT));
            item.addActionListener(this);
            result.add(item = new JMenuItem(Resources.MENU_SAVE_LAYOUT));
            item.addActionListener(this);
            return result;
        }
    }
    
    class RipperTable extends MediaTable {
        RipperTable() {
            super(new AlbumTableModel(controller));
            addKeyListener(new KeyAdapter() {
                public void keyReleased(KeyEvent e) {
                    if (e.getKeyCode() == e.VK_DELETE) {
                        ((AlbumTableModel)getModel()).delete(getSelectedRows());
                        RipperModel model = (RipperModel)ripper.getModel();
                        ((StatusBar)controller.component(Controller.COMP_STATUSBAR)).displayMetric(model.music?MP3.convertTime(model.getTime()):BasicIo.convertLength(model.getLength()));
                    }
                }
            });
        }
    }
    
    class TransferableMedias implements Transferable {
        java.util.List medias;
        Integer albumId;
        
        TransferableMedias(java.util.List medias) {
            this.medias = medias;
        }
        
        TransferableMedias(int albumId) {
            this.albumId = new Integer(albumId);
        }
        
        public Object getTransferData(DataFlavor flavor) throws UnsupportedFlavorException {
            if (flavor.equals(MIXED_FILE_MEDIA_LIST_DF))
                return medias;
            else if (flavor.equals(ALBUM_NODE_INDEX_DF))
                return albumId;
            throw new UnsupportedFlavorException(flavor);
        }
        
        public java.awt.datatransfer.DataFlavor[] getTransferDataFlavors() {
            return new DataFlavor[] {MIXED_FILE_MEDIA_LIST_DF, ALBUM_NODE_INDEX_DF};
        }
        
        public boolean isDataFlavorSupported(DataFlavor flavor) {
            return (medias != null && flavor.equals(MIXED_FILE_MEDIA_LIST_DF))
            || (flavor.equals(ALBUM_NODE_INDEX_DF) && albumId != null);
        }   
    }
    
    class MediaTransferHandler extends TransferHandler {
        
        MediaTransferHandler() {
        }
        
        public int getSourceActions(JComponent c) {
            return COPY;
        }
        
        protected Transferable createTransferable(JComponent c) {
            if (c instanceof JTable) {
                JTable t = (JTable)c;
                int [] selections = t.getSelectedRows();
                java.util.List medias = new ArrayList();
                if (c instanceof CollectionTable) {
                    ListModel model = (ListModel)t.getModel();
                    for (int i = 0; i < selections.length; i++)
                        medias.add(((Thumbnail)model.getElementAt(selections[i])).getFormat());
                    return new TransferableMedias(medias);
                } else if(c instanceof AlbumTable) {
                    AlbumTableModel model = (AlbumTableModel)t.getModel();
                    for (int i = 0; i < selections.length; i++)
                        medias.add(model.getElementAt(selections[i]));
                    return new TransferableMedias(medias);
                } else if(c instanceof DirectoryTable) {
                    FolderTableModel model = (FolderTableModel)t.getModel();
                    for (int i = 0; i < selections.length; i++)
                        medias.add(model.getElementAt(selections[i]));
                    return new TransferableMedias(medias);
                }
            } else if(c instanceof AlbumTree) {
                return new TransferableMedias(access.getAlbumId(((AlbumTree)c).getSelectionPath()));
            }
            
            return null;
        }
        
        // TODO: do not drop non audio files on music layout
        public boolean importData(JComponent comp, Transferable t) {
            if (comp instanceof RipperTree) {
                RipperTree rt = (RipperTree)comp;
                RipperModel model = (RipperModel)rt.getModel();
                TreePath tp = rt.getSelectionPath() ;
                Object lc = tp==null?null :tp.getLastPathComponent();
                VirtualFolder dropHost = (lc != null && lc instanceof VirtualFolder)?
                (VirtualFolder)lc:
                    (VirtualFolder)model;
                    try {
                        if (t.isDataFlavorSupported(MIXED_FILE_MEDIA_LIST_DF)) {
                            java.util.List medias = (java.util.List)t.getTransferData(MIXED_FILE_MEDIA_LIST_DF);
                            Iterator i = medias.iterator();
                            while (i.hasNext()) {
                                Object media = i.next();
                                if (media instanceof AbstractFormat)
                                    dropHost.addMedia((AbstractFormat)media);
                                else if (media instanceof File)
                                    dropHost.addMedia((File)media);
                            }
                        } else if (t.isDataFlavorSupported(FILE_LIST_DF)) {
                            Iterator i = ((java.util.List)t.getTransferData(FILE_LIST_DF)).iterator();
                            // TODO: create virtual folders if file is directory
                            while(i.hasNext())
                                dropHost.addMedia(i.next());
                        } else if (t.isDataFlavorSupported(ALBUM_NODE_INDEX_DF)) {
                            // TODO: make it recursive
                            Integer albumNode = (Integer)t.getTransferData(ALBUM_NODE_INDEX_DF);
                            VirtualFolder vf;
                            vf = new VirtualFolder(access.getNameOfAlbum(albumNode.intValue()));
                            dropHost.add(vf);
                            File []files = access.getAlbumContents(albumNode.intValue());
                            for (int i=0; i < files.length; i++)
                                vf.addMedia(files[i]);
                        }
                        rt.ripperTable.revalidate();
                        ((StatusBar)controller.component(Controller.COMP_STATUSBAR)).displayMetric(model.music?MP3.convertTime(model.getTime()):BasicIo.convertLength(model.getLength()));
                        return true;
                    } catch(Exception e) {
                        e.printStackTrace();
                    }
                    
            }
            return false;
        }
        
        public boolean canImport(JComponent comp, DataFlavor[] transferFlavors) {
            return true;
        }
    }
    
    protected RipperTree ripper;
    protected MediaAccess access;
}

class OneThirdScroll extends JScrollPane {
    protected boolean vertical;
    OneThirdScroll(Component view) {
        super(view);
        this.vertical = vertical;
    }
    public Dimension getPreferredSize() {
        Dimension result = super.getPreferredSize();
        result .height = getParent().getSize().height;
        result.width
        = getParent().getSize().width / 3 + 1;
        return result;
    }
}

// TODO: do sync on using more methods
class MediaList extends ArrayList implements java.util.List {
    protected long length;
    protected int time;
    
    public boolean add(Object o) {
        boolean beAdded = false;
        if (o != null) {
            // TODO: consider conversion to AbstractFormat on adding
            if (o instanceof AbstractFormat) {
                AbstractFormat af = (AbstractFormat)o;
                length += af.getFileSize();
                if (af.getType().equals(MP3.MP3))
                    time += af.getInfo().getLongAttribute(AbstractInfo.LENGTH);
            } else if (o instanceof File) {
                length += ((File)o).length();
                // TODO: do only for audio
                AbstractFormat af = MediaFormatFactory.createMediaFormat((File)o);
                if (af != null && af.getType().equals(MP3.MP3))
                    time += af.getInfo().getLongAttribute(AbstractInfo.LENGTH);
            }
        }
        return super.add(o); // af
    }
    
    public boolean remove(Object o) {
        if (o != null) {
            if (o instanceof AbstractFormat) {
                AbstractFormat af = (AbstractFormat)o;
                length -= af.getFileSize();
                if (af.getType().equals(MP3.MP3))
                    time -= af.getInfo().getLongAttribute(AbstractInfo.LENGTH);
            } else if (o instanceof File) {
                length -= ((File)o).length();
                AbstractFormat af = MediaFormatFactory.createMediaFormat((File)o);
                if (af != null && af.getType().equals(MP3.MP3))
                    time -= af.getInfo().getLongAttribute(AbstractInfo.LENGTH);
            }
        }
        return super.remove(o); // af
    }
    
    public Object set(int index, Object o) {
        if (o != null) {
            // TODO: consider conversion to AbstractFormat on setting
            if (o instanceof AbstractFormat) {
                AbstractFormat af = (AbstractFormat)o;
                length += af.getFileSize();
                if (af.getType().equals(MP3.MP3))
                    time += af.getInfo().getLongAttribute(AbstractInfo.LENGTH);
            } else if (o instanceof File) {
                length += ((File)o).length();
                // TODO: do only for audio
                AbstractFormat af = MediaFormatFactory.createMediaFormat((File)o);
                if (af != null && af.getType().equals(MP3.MP3))
                    time += af.getInfo().getLongAttribute(AbstractInfo.LENGTH);
            }
        }
        o = super.set(index, o);
        if (o != null) {
            if (o instanceof AbstractFormat) {
                AbstractFormat af = (AbstractFormat)o;
                length -= af.getFileSize();
                if (af.getType().equals(MP3.MP3))
                    time -= af.getInfo().getLongAttribute(AbstractInfo.LENGTH);
            } else if (o instanceof File) {
                length -= ((File)o).length();
                AbstractFormat af = MediaFormatFactory.createMediaFormat((File)o);
                if (af != null && af.getType().equals(MP3.MP3))
                    time -= af.getInfo().getLongAttribute(AbstractInfo.LENGTH);
            }
        }
        return o;
    }
    
    public void clear() {
        super.clear();
        length = 0;
        time = 0;
    }
    
    public long getLength() {
        return length;
    }
    
    public int getTime() {
        return time;
    }
}

class VirtualFolder extends ArrayList implements XmlExposable {
    String name;
    MediaList medias;
    
    VirtualFolder(String name) {
        this.name = name;
    }
    
    void addMedia(AbstractFormat format) {
        if (medias == null)
            medias = new MediaList();
        medias.add(format);
    }
    
    void addMedia(File file) {
        if (medias == null)
            medias = new MediaList();
        medias.add(file);
    }
    
    void addMedia(Object media) {
        if (medias == null)
            medias = new MediaList();
        medias.add(media);
    }
    
    java.util.List getContent() {
        return medias;
    }
    
    public void setName(String name) {
        this.name = name;
    }
    
    public String toString() {
        return name;
    }
    
    public long getLength() {
        return medias!=null?medias.getLength():0;
    }
    
    public int getTime() {
        return medias!=null?medias.getTime():0;
    }
    
    public String toXmlString() {
        StringBuffer xmlBuffer = new StringBuffer();
        if (medias != null) {
            Iterator i = medias.iterator();
            while(i.hasNext()) {
                Object o = i.next();
                if (o instanceof XmlExposable)
                    xmlBuffer.append(((XmlExposable)o).toXmlString());
                else if (o instanceof AbstractFormat)
                    XmlHelper.appendTag(xmlBuffer, Resources.TAG_MEDIA,
                    XmlHelper.appendAttr(
                    XmlHelper.appendAttr(null, Resources.ATTR_MEDIATYPE, ((AbstractFormat)o).getType()),
                    Resources.ATTR_NAME, ((AbstractFormat)o).getType()), HttpUtils.htmlEncode(((AbstractFormat)o).getFile().getPath()), getNameSpacePrefix());
                else if (o instanceof File)
                    XmlHelper.appendTag(xmlBuffer, Resources.TAG_FILE, null, HttpUtils.htmlEncode(((File)o).getPath()), getNameSpacePrefix());
            }
        }
        Iterator i = iterator();
        while(i.hasNext()) {
            Object o = i.next();
            if (o instanceof XmlExposable)
                xmlBuffer.append(((XmlExposable)o).toXmlString());
        }
        return XmlHelper.getTag(Resources.TAG_FOLDER,
        XmlHelper.appendAttr(null, Resources.ATTR_NAME, name),
        xmlBuffer == null?null:xmlBuffer.toString(), getNameSpacePrefix());
    }
    
    public String getNameSpacePrefix() {
        return PhotoOrganizer.PROGRAMNAME;
    }
    public String getNameSpaceUri() {
        return PhotoOrganizer.BASE_URL;
    }
    
    public DefaultHandler getXmlHandler(final ContentHandler parent,
    String namespaceURI,
    String localName,
    String qName,
    Attributes atts) {
        
        return new DefaultHandler() {
            ContentHandler handler, parentHandler; {
                parentHandler = parent;
            }
            String file;
            boolean open;
            public void startElement(String uri,
            String localName,
            String qName,
            Attributes attributes)
            throws SAXException {                
                if (handler != null)
                    handler.startElement(uri, localName, qName, attributes);
                else {
                    if((getNameSpacePrefix()+':'+Resources.TAG_FOLDER).equals(qName)) {
                        if (!open) {
                            setName(attributes.getValue(Resources.ATTR_NAME));
                            open = true;
                        } else {
                            VirtualFolder folder = new VirtualFolder(attributes.getValue(Resources.ATTR_NAME));
                            add(folder);
                            handler = folder.getXmlHandler(this, uri, localName, qName, attributes);
                            handler.startElement(uri, localName, qName, attributes);
                        }
                    } else if ((getNameSpacePrefix()+':'+Resources.TAG_FILE).equals(qName)) {
                        file = null;
                    }
                }
            }
            public void endElement(String namespaceURI,
            String localName,
            String qName)
            throws SAXException {
                if (handler != null)
                    handler.endElement(namespaceURI, localName, qName);
                else {
                    if((getNameSpacePrefix()+':'+Resources.TAG_FOLDER).equals(qName)) {
                        if (parentHandler != null)
                            parentHandler.endDocument();
                        open = false;
                    } else if ((getNameSpacePrefix()+':'+Resources.TAG_FILE).equals(qName)) {
                        if (handler != null)
                            handler.endElement(namespaceURI, localName, qName);
                        else
                            addMedia(new File(HttpUtils.htmlDecode(file)));                        
                    }
                }
            }
            public void characters(char buf [], int offset, int len)
            throws SAXException {
                if (handler != null)
                    handler.characters(buf, offset, len);
                else {
                    if (file == null)
                        file = new String(buf, offset, len);
                    else
                        file += new String(buf, offset, len);
                }
            }
            public void endDocument()
                 throws SAXException {
                     handler = null;
            }
        };
    }
}

class RipperModel extends VirtualFolder implements TreeModel/*, Comparable */{
    EventListenerList listenerList = new EventListenerList();
    boolean music;
    public RipperModel(boolean music) {
        super(""+(Math.round(Math.random()*10000)));
        this.music = music;
    }
    
    public void reset(boolean music) {
        this.music = music;
        clear();
        medias = null;
    }
    
    public Object getChild(Object parent, int index) {   
        if (parent instanceof java.util.List)
            return ((java.util.List)parent).get(index);
        return null;
    }
    
    public java.lang.Object getRoot() {
        return this;
    }
    
    public int getChildCount(Object parent) {
        if (parent instanceof java.util.List)
            return ((java.util.List)parent).size();
        return 0;
    }
    
    public boolean isLeaf(Object node) {
        return false; //getChildCount(node) == 0;
    }
    
    public void valueForPathChanged(TreePath path, Object newValue) {
        if (newValue != null) {
            VirtualFolder target = (VirtualFolder)(path==null?this:path.getLastPathComponent());
            target.setName(newValue.toString());
        }
        fireTreeStructureChanged(this, path==null?new Object[]{this}:path.getPath(), null, null);
    }
    
    public int getIndexOfChild(Object parent, Object child) {
        if (parent instanceof java.util.List)
            return ((java.util.List)parent).indexOf(child);
        return -1;
    }
    
    public void addTreeModelListener(TreeModelListener l) {
        listenerList.add(TreeModelListener.class, l);
    }
    
    public void removeTreeModelListener(TreeModelListener l) {
        listenerList.remove(TreeModelListener.class, l);
    }
    
    public void fireTreeStructureChanged(Object source, Object[] path,
    int[] childIndices,
    Object[] children) {
        Object[] listeners = listenerList.getListenerList();
        TreeModelEvent e = null;
        for (int i = 0; i<listeners.length-1; i+=2) {
            if (listeners[i] == TreeModelListener.class) {
                if (e == null)
                    e = new TreeModelEvent(source, path,
                    childIndices, children);
                ((TreeModelListener)listeners[i+1]).treeStructureChanged(e);
            }
        }
    }
    
    public void fireTreeNodesChanged(Object[] path) {
        Object[] listeners = listenerList.getListenerList();
        TreeModelEvent e = null;
        for (int i = listeners.length-2; i>=0; i-=2) {
            if (listeners[i] == TreeModelListener.class) {
                if (e == null)
                    e = new TreeModelEvent(this, path, null, null);
                ((TreeModelListener)listeners[i+1]).treeNodesChanged(e);
            }
        }
    }
    
    public boolean add(Object o) {
        if (!music)
            if (o != null)
                if(o instanceof VirtualFolder)
                    return super.add(o);
                else
                    throw new ClassCastException("Only VirtualFolder can be added in this way");
            else
                throw new IllegalArgumentException("Can not add null folder");
        else
            throw new UnsupportedOperationException("New folder can not be added to music layout");
    }
    
    public String toString() {
        if (size() == 0 && (medias == null || medias.size() == 0))
            return MessageFormat.format(Resources.LABEL_EMPTY_LAYOUT
            , new String[] {music?
            Resources.LABEL_MUSIC:Resources.LABEL_DATA});
            return MessageFormat.format(Resources.LABEL_LAYOUT_
            , new Object[] {music?
            Resources.LABEL_MUSIC:Resources.LABEL_DATA,
            name,
            new Integer(size()),
            new Integer(medias==null?0:medias.size())});
    }
    
    public void doDisk(Controller controller) {
        Serializer s = controller.getSerializer();
        String destpath = (String)s.getProperty(MediaOptionsTab.SECNAME, MediaOptionsTab.RIPPER_FOLDER);
        if (destpath == null)
            return; // TODO: warning box
        final File destDir = new File(destpath);
        if (!destDir.exists()) {
            if(!destDir.mkdirs())
                return; // TODO: warning box
        }
        final StatusBar statusBar = (StatusBar)controller.component(Controller.COMP_STATUSBAR);
        if (music) {
            
            new Thread(new Runnable() {
                public void run() {
                    AbstractFormat format;
                    // use clone().
                    Iterator i = medias.iterator();
                    while(i.hasNext()) {
                        Object o = i.next();
                        if (o instanceof File)
                            format = MediaFormatFactory.createMediaFormat((File)o);
                        else if (o instanceof AbstractFormat)
                            format = (AbstractFormat)o;
                        else
                            format = null;
                        if (format != null && format.isValid() && MP3.TYPE.equals(format.getType())) {
                            String name = format.getName();
                            int ep = name.lastIndexOf('.');
                            if (ep > 0)
                                name = name.substring(0, ep)+".wav";
                            File dest = new File(destDir, name);
                            try {
                                Controller.convertToWav(format, dest.getPath(), statusBar);
                            } catch(Exception e) {
                                System.err.println("Exception at convertion of "+format+" to "+dest+':'+e);
                            }
                        }
                    }
                }
            }, "Convert to wav").start();
        }
    }    
}

class AlbumTableModel extends BaseConfigurableTableModel {
    VirtualFolder medias;
    AlbumTableModel(Controller controller) {
        updateView(controller.getSerializer());
    }
    
    synchronized public void updateModel(VirtualFolder medias) {
        this.medias = medias;
    }
    
    synchronized public void updateModel(File[] medias) {
        if (medias == null) {
            this.medias = new VirtualFolder("Empty");
            return;
        }
        clear();
        for (int i=0; i<medias.length; i++)
            if (medias[i].isDirectory())
                this.medias.add(new VirtualFolder(medias[i].getName()));
            else
                this.medias.addMedia(medias[i]);
        
    }
    
    synchronized public void clear() {
        if (this.medias == null) {
            this.medias = new VirtualFolder("Content");
        } else {
            java.util.List content = medias.getContent();
            if (content != null)
                content.clear();
            this.medias.clear();
        }
    }
    
    synchronized public void updateModel(java.util.List medias) {
        if (medias == null) {
            this.medias = new VirtualFolder("Empty");
            return;
        }
        clear();
        for (int i=0; i<medias.size(); i++) {
            if (medias.get(i) instanceof AbstractFormat)
                this.medias.addMedia(((AbstractFormat)medias.get(i)));
            else {
                File file = (File)medias.get(i);
                if (file.isDirectory())
                    this.medias.add(new VirtualFolder(file.getName()));
                else
                    this.medias.addMedia(file);
            }
        }
    }
    
    // folders temporary not supported
    synchronized public void delete(int[] rows) {
        int deleted = 0;
        int di;
        java.util.List list = medias.getContent();
        int maxr = list.size() - 1, minr = 0;
        for (int i=0; i<rows.length; i++) {
            di = rows[i];
            if (di<list.size() && di >= 0) {
                list.set(di, null);
                deleted++;
                if (di > maxr)
                    maxr = di;
                else if (di < minr)
                    minr = di;
            }
        }
        if (deleted == 0)
            return;
        while(list.remove(null));
        fireTableRowsDeleted(minr, maxr);
    }
    
    public int getRowCount() {
        return medias==null?0:(medias.getContent()==null?0:medias.getContent().size());
    }
    
    
    public Object getValueAt(int row, int column) {
        if (medias != null && row >= 0 && row < medias.getContent().size() ) {
            Object o = medias.getContent().get(row);
            if (o instanceof AbstractFormat)
                return getValueAt(((AbstractFormat)o).getFile(), ((AbstractFormat)o).getInfo(), column);
            else if (o instanceof File) {
                AbstractFormat media = MediaFormatFactory.createMediaFormat((File)o);
                if (media != null && media.isValid())
                    return getValueAt((File)o, media.getInfo(), column);
            }
        }
        return null;
    }
    
    public Object getElementAt(int row) {
        return medias.getContent().get(row);
    }
    
    protected int getDescriptionIndex() {
        return AppearanceOptionsTab.COLLECT_VIEW;
    }
}

class FolderTableModel extends BaseConfigurableTableModel {
    File[] files;
	photoorganizer.directory.FileSystem fileSystem;
	
	FolderTableModel(Controller controller) {
		fileSystem = new photoorganizer.directory.FileSystem();
		updateView(controller.getSerializer());
	}
    
    public void updateModel(File[] files) {
        this.files = files;
    }
    
    public File getElementAt(int row) {
        if (files != null && row >= 0 && row < files.length)
            return files[row];
        return null;
    }
    
    public int getRowCount() {
        return files==null?0:files.length;
    }
    
    public Object getValueAt(int row, int column) {
		if (column == 0)
			return fileSystem.getSystemIcon(files[row]);
		column--;
        if (files != null && row >= 0 && row < files.length) {
            return getValueAt(files[row], null, column);
        }
        return null;
    }
    
    protected int getDescriptionIndex() {
        return AppearanceOptionsTab.BROWSE_VIEW;
    }
	
	public int getColumnCount() {
		return super.getColumnCount()+1;
	}

	public String getColumnName(int column) {
		if (column == 0)
			return " ";
		return super.getColumnName(column-1);
	}

	public Class getColumnClass(int column) {
		if (column == 0)
			return Icon.class;
		return super.getColumnClass(column-1);
	}
}
