#include <string.h>
#if !defined WM_NULL
  #include <windows.h>
#endif

/* AniSprite does not provide a native file format of alpha and luma
 * tables, so you can choose any convenient format. This example uses
 * a simple uncompressed file format that resembles the file formats
 * used by Autodesk Animator Pro.
 *
 * Future versions of EGI (a frame animation engine) may support this
 * format natively. In any case, the "type" value have been chosen so
 * that they do not conflict with EGI or Autodesk Animator Pro.
 */

typedef struct {
  DWORD size;           /* size of file including this header */
  WORD  type;           /* 0xAF06 for alpha level files */
  WORD  levels;         /* number of alpha level tables in the file */
  WORD  reserved[4];    /* future extension */
} ALPHA_HEADER;

typedef struct {
  DWORD size;           /* size of the alpha table including this header */
  WORD  type;           /* 0xF106 for an alpha table */
  WORD  reserved[5];    /* future extension */
} ALPHA_TABLE;

BOOL LoadAlphaLevels(ASBOARD Board, LPSTR Filename, LPINT Number)
{
  int f;
  ALPHA_HEADER header;
  ALPHA_TABLE table;
  LPVOID lpAlpha;
  int i, num;

  if (Number != NULL)
    *Number = 0;

  if ((f = _lopen(Filename,OF_READ|OF_SHARE_DENY_WRITE)) == HFILE_ERROR)
    return FALSE;

  _lread(f, &header, sizeof header);
  if (header.type != 0xaf06) {
    _lclose(f);
    return FALSE;       /* invalid file type */
  } /* if */
  num = header.levels;

  for (i = 1; i <= num; i++) {
    _lread(f, &table, sizeof table);

    if (table.type != 0xf106 || table.size != 65536L + sizeof table) {
      _lclose(f);
      return FALSE;       /* file is damaged */
    } /* if */

    if ((lpAlpha = as_AllocResource(0x10000L)) == NULL) {
      _lclose(f);
      return FALSE;       /* file is insufficient memory */
    } /* if */

    /* read in two chunks; maximum integer size is 65535 in
     * 16-bit mode, and we must read one byte more.
     */
    _lread(f, lpAlpha, 0x8000u);
    _lread(f, (LPBYTE)lpAlpha + 0x8000u, 0x8000u);

    as_SetBoardData(Board, AS_DATA_ALPHA_LEVEL, i, lpAlpha);

    if (Number != NULL)
      *Number += 1;
  } /* for */

  _lclose(f);
  return TRUE;
}

BOOL SaveAlphaLevels(ASBOARD Board, LPSTR Filename, int Number)
{
  int f;
  ALPHA_HEADER header;
  ALPHA_TABLE table;
  LPVOID lpAlpha;
  int i;

  if ((f = _lcreat(Filename,0)) == HFILE_ERROR)
    return FALSE;

  memset(&header, 0, sizeof header);
  header.size = Number * (65536L + sizeof table) + sizeof header;
  header.type = 0xaf06;         /* type for alpha level files */
  header.levels = Number;
  _lwrite(f, (LPVOID)&header, sizeof header);

  for (i = 1; i <= Number; i++) {
    memset(&table, 0, sizeof table);
    table.size = 65536L + sizeof table;
    table.type = 0xf106;        /* type for alpha tables */
    _lwrite(f, (LPVOID)&table, sizeof table);

    lpAlpha = as_GetBoardData(Board, AS_DATA_ALPHA_LEVEL, i);
    /* write in two chunks; maximum integer size is 65535 in
     * 16-bit mode, and we must write one byte more.
     */
    _lwrite(f, lpAlpha, 0x8000u);
    _lwrite(f, (LPBYTE)lpAlpha + 0x8000u, 0x8000u);
  } /* for */

  _lclose(f);
  return TRUE;
}

