/*
 * ANIMATE.C
 *
 * Simple sprite animation with two images for a single sprite.
 *
 * This example shows:
 * o  Changing the appearance of a sprite.
 *
 * Compiling:
 *   Borland C++, 16-bit
 *      bcc -W animate.c as16.lib
 *
 *   Borland C++, 32-bit
 *      bcc32 -tW animate.c as32b.lib
 *
 *   Microsoft C/C++, 16 bit
 *      cl -Gw animate.c as16.lib slibcew.lib libw.lib example.def
 *      rc example.rc animate.exe
 *
 *   Microsoft C/C++, 32 bit
 *      cl -GA animate.c as32m.lib user32.lib gdi32.lib
 *
 *   Watcom C/C++ 11.0, 32-bit
 *      wcl386 /l=nt_win animate.c as32w.lib
 *
 * Copyright (c) 1997-1999, ITB CompuPhase. You may use/modify/distribute
 * this file or portions of it. It is provided as an example for the use of
 * the AniSprite API. There are no waranties on the correct behavior of this
 * program.
 */
#include <stdlib.h>
#include <windows.h>
#include "..\include\anispri.h"

#define BOARD_WIDTH     640
#define BOARD_HEIGHT    480
#define TIMER_INTERVAL  50
#define NUM_IMAGES      2       /* 2 images for this sprite */

LRESULT CALLBACK AnimWinFunc(HWND hwnd, unsigned message,
                             WPARAM wParam, LPARAM lParam);


int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
                   LPSTR lpCmdLine, int nCmdShow)
{
  MSG msg;
  WNDCLASS wc;
  RECT rect;
  DWORD dwStyle;

  wc.style = 0;
  wc.lpfnWndProc = (WNDPROC)AnimWinFunc;
  wc.cbClsExtra = 0;
  wc.cbWndExtra = 0;
  wc.hInstance = hInstance;
  wc.hIcon = LoadIcon(hInstance, (LPSTR)"anim_icon");
  wc.hCursor = LoadCursor(NULL, IDC_ARROW);
  wc.hbrBackground = GetStockObject(LTGRAY_BRUSH);
  wc.lpszMenuName = (LPSTR)NULL;
  wc.lpszClassName = "Animation";
  if (!RegisterClass(&wc))
    return FALSE;

  /* creat a window with the right size for the board */
  SetRect(&rect, 0, 0, BOARD_WIDTH, BOARD_HEIGHT);
  dwStyle = WS_POPUPWINDOW | WS_CAPTION | WS_VISIBLE;
  AdjustWindowRect(&rect, dwStyle, FALSE);
  CreateWindow("Animation", "AniSprite: Animated sprite", dwStyle,
               50, 50, rect.right - rect.left, rect.bottom - rect.top,
               0, 0, hInstance, NULL);

  while (GetMessage(&msg, NULL, 0, 0)) {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
  } /* while */

  return msg.wParam;
}

LRESULT CALLBACK AnimWinFunc(HWND hwnd, unsigned message,
                             WPARAM wParam, LPARAM lParam)
{
static COLORREF crTransColor[] = {PALETTERGB(0xff, 0x00, 0xff)};  /* magenta */
static ASBOARD Board;
static ASPRITE Sprite;
static LPVOID lpSpriteImages[NUM_IMAGES], lpSpriteMasks[NUM_IMAGES];
static int stepx, stepy, count, face;
  LPBITMAPINFO lpBitsInfo;
  PAINTSTRUCT ps;
  HDC hdc;
  int x, y, width, height;
  int i;

  switch (message) {
  case WM_CREATE:
    /* create the board (with an identity palette) */
    Board = as_LoadBoard("maze.bmp", AS_MODE_PAL_COLORS, NULL, NULL);

    /* lpBitsInfo is allocated temporarily */
    lpBitsInfo = as_AllocResource(sizeof(BITMAPINFO) + 256 * sizeof(RGBQUAD));

    /* load boath images for the sprite, create both masks */
    lpSpriteImages[0] = as_LoadDIB("sprite1.bmp", lpBitsInfo, NULL, NULL);
    lpSpriteMasks[0] = as_CreateMask(lpBitsInfo, lpSpriteImages[0],
                                     TRUE, crTransColor, 1);
    lpSpriteImages[1] = as_LoadDIB("sprite2.bmp", lpBitsInfo, NULL, NULL);
    lpSpriteMasks[1] = as_CreateMask(lpBitsInfo, lpSpriteImages[1],
                                     TRUE, crTransColor, 1);

    /* create the sprite
     * This example takes advantage of the knowledge that both images
     * for the sprite have the same size. AniSprite does not require
     * this.
     */
    face = 0;
    Sprite = as_Create((int)lpBitsInfo->bmiHeader.biWidth,
                       (int)lpBitsInfo->bmiHeader.biHeight,
                       lpSpriteMasks[face], AS_MASK_CUTOUT,
                       lpSpriteImages[face], TRUE);

    /* assign and show the sprite */
    as_Assign(Sprite, Board, 0);
    as_Show(Sprite, TRUE);

    /* clean up */
    as_FreeResource(lpBitsInfo);

    /* Create a timer to move the sprite, set initial direction */
    as_SetTimer(Board, hwnd, TIMER_INTERVAL);
    stepx = 5;
    stepy = 3;
    count = 0;
    break;

  case WM_DESTROY:
    as_Assign(Sprite, Board, -1);
    /* Do not delete the image of the sprite; it is easier to
     * free both image ourselves than to check which one would
     * have been implicitly removed.
     */
    as_Delete(Sprite, FALSE);
    for (i = 0; i < NUM_IMAGES; i++) {
      as_FreeResource(lpSpriteImages[i]);
      as_FreeResource(lpSpriteMasks[i]);
    } /* for */
    as_DeleteBoard(Board, TRUE);
    PostQuitMessage(0);
    break;

  case AS_TIMER:
    /* get current position */
    x = as_GetValue(Sprite, AS_VALUE_XPOS);
    y = as_GetValue(Sprite, AS_VALUE_YPOS);
    width = as_GetValue(Sprite,AS_VALUE_WIDTH);
    height = as_GetValue(Sprite,AS_VALUE_HEIGHT);

    /* Move the image around a bit, check against borders */
    x += stepx;
    y += stepy;
    if (x < 0) {
      x = 0;
      stepx = -stepx;
    } else if (x + width > BOARD_WIDTH) {
      x = BOARD_WIDTH - width;
      stepx = -stepx;
    } /* if */
    if (y < 0) {
      y = 0;
      stepy = -stepy;
    } else if (y + height > BOARD_HEIGHT) {
      y = BOARD_HEIGHT - height;
      stepy = -stepy;
    } /* if */

    as_SetPos(Sprite, x, y);

    /* animate the sprite */
    count = (count + 1) % 5;
    if (count == 0) {
      face = (face+1) % NUM_IMAGES;
      as_Animate(Sprite, width, height,
                 lpSpriteMasks[face], AS_MASK_CUTOUT,
                 lpSpriteImages[face], TRUE, NULL);
    } /* if */

    /* repaint the board */
    hdc = GetDC(hwnd);
    as_PaintBoard(hdc, Board, 0, 0, FALSE);
    ReleaseDC(hwnd, hdc);
    break;

  case WM_PAINT:
    hdc=BeginPaint(hwnd, &ps);
    as_PaintBoard(hdc, Board, 0, 0, TRUE);
    EndPaint(hwnd, &ps);
    break;

  case WM_ERASEBKGND:
  case WM_PALETTECHANGED:
  case WM_QUERYNEWPALETTE:
    return as_ForwardMessage(Board, hwnd, message, wParam, lParam);

  default:
    return DefWindowProc(hwnd, message, wParam, lParam);
  } /* switch */

  return 0L;
}

