/*
 * CUTOUT.C
 *
 * Simple sprite animation with a cutout mask.
 *
 * This example shows:
 * o  Loading a board from a BMP image.
 * o  Loading a sprite from a BMP image
 * o  Doing simple animation.
 * o  Cleaning up (deleting the sprite and the board).
 *
 * Compiling:
 *   Borland C++, 16-bit
 *      bcc -W cutout.c as16.lib
 *
 *   Borland C++, 32-bit
 *      bcc32 -tW cutout.c as32b.lib
 *
 *   Microsoft C/C++, 16 bit
 *      cl -Gw cutout.c as16.lib slibcew.lib libw.lib example.def
 *      rc example.rc cutout.exe
 *
 *   Microsoft C/C++, 32 bit
 *      cl -GA cutout.c as32m.lib user32.lib gdi32.lib
 *
 *   Watcom C/C++ 11.0, 32-bit
 *      wcl386 /l=nt_win cutout.c as32w.lib
 *
 *   LCC-Win32
 *      lcc cutout.c
 *      lcclnk cutout.obj ..\lib\as32l.lib
 *
 * Copyright (c) 1997-1999, ITB CompuPhase. You may use/modify/distribute
 * this file or portions of it. It is provided as an example for the use of
 * the AniSprite API. There are no waranties on the correct behavior of this
 * program.
 */
#include <windows.h>
#include "../include/anispri.h"

#define BOARD_WIDTH     640
#define BOARD_HEIGHT    480
#define TIMER_INTERVAL  50

LRESULT CALLBACK AnimWinFunc(HWND hwnd, unsigned message,
                             WPARAM wParam, LPARAM lParam);


int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
                   LPSTR lpCmdLine, int nCmdShow)
{
  MSG msg;
  WNDCLASS wc;
  RECT rect;
  DWORD dwStyle;

  wc.style = 0;
  wc.lpfnWndProc = (WNDPROC)AnimWinFunc;
  wc.cbClsExtra = 0;
  wc.cbWndExtra = 0;
  wc.hInstance = hInstance;
  wc.hIcon = LoadIcon(hInstance, (LPSTR)"anim_icon");
  wc.hCursor = LoadCursor(NULL, IDC_ARROW);
  wc.hbrBackground = GetStockObject(LTGRAY_BRUSH);
  wc.lpszMenuName = (LPSTR)NULL;
  wc.lpszClassName = "Animation";
  if (!RegisterClass(&wc))
    return FALSE;

  /* create a window with the right size for the board */
  SetRect(&rect, 0, 0, BOARD_WIDTH, BOARD_HEIGHT);
  dwStyle = WS_POPUPWINDOW | WS_CAPTION | WS_VISIBLE;
  AdjustWindowRect(&rect, dwStyle, FALSE);
  CreateWindow("Animation", "AniSprite: Cutout mask", dwStyle,
               50, 50, rect.right - rect.left, rect.bottom - rect.top,
               0, 0, hInstance, NULL);

  while (GetMessage(&msg, NULL, 0, 0)) {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
  } /* while */

  return msg.wParam;
}

LRESULT CALLBACK AnimWinFunc(HWND hwnd, unsigned message,
                             WPARAM wParam, LPARAM lParam)
{
static ASBOARD Board;
static ASPRITE Sprite;
static int stepx, stepy;
  PAINTSTRUCT ps;
  HDC hdc;
  int x, y, width, height;

  switch (message) {
  case WM_CREATE:
    /* load the board, use "palette mode" for optimal performace */
    Board = as_LoadBoard("maze.bmp", AS_MODE_PAL_COLORS, NULL, 0);
    /* load a sprite (with a cutout mask) */
    Sprite = as_Load("sprite1.bmp", PALETTERGB(0xff, 0x00, 0xff), NULL, 0);

    /* assign and show the sprite */
    as_Assign(Sprite, Board, 0);
    as_Show(Sprite, TRUE);

    /* Create a timer to move the sprite, set initial direction */
    as_SetTimer(Board, hwnd, TIMER_INTERVAL);
    // ??? SetTimer(hwnd,0,TIMER_INTERVAL,NULL);        // for debugging
    stepx = 5;
    stepy = 3;
    break;

  case WM_DESTROY:
    as_Assign(Sprite, Board, -1);
    as_Delete(Sprite, TRUE);
    as_DeleteBoard(Board, TRUE);
    PostQuitMessage(0);
    break;

  case WM_TIMER:
  case AS_TIMER:
    /* get current position */
    x = as_GetValue(Sprite, AS_VALUE_XPOS);
    y = as_GetValue(Sprite, AS_VALUE_YPOS);
    width = as_GetValue(Sprite,AS_VALUE_WIDTH);
    height = as_GetValue(Sprite,AS_VALUE_HEIGHT);

    /* Move the image around a bit, check against borders */
    x += stepx;
    y += stepy;
    if (x < 0) {
      x = 0;
      stepx = -stepx;
    } else if (x + width > BOARD_WIDTH) {
      x = BOARD_WIDTH - width;
      stepx = -stepx;
    } /* if */
    if (y < 0) {
      y = 0;
      stepy = -stepy;
    } else if (y + height > BOARD_HEIGHT) {
      y = BOARD_HEIGHT - height;
      stepy = -stepy;
    } /* if */

    as_SetPos(Sprite, x, y);

    /* repaint the board */
    hdc = GetDC(hwnd);
    as_PaintBoard(hdc, Board, 0, 0, FALSE);
    ReleaseDC(hwnd, hdc);
    break;

  case WM_PAINT:
    hdc=BeginPaint(hwnd, &ps);
    as_PaintBoard(hdc, Board, 0, 0, TRUE);
    EndPaint(hwnd, &ps);
    break;

  case WM_ERASEBKGND:
  case WM_PALETTECHANGED:
  case WM_QUERYNEWPALETTE:
    return as_ForwardMessage(Board, hwnd, message, wParam, lParam);

  default:
    return DefWindowProc(hwnd, message, wParam, lParam);
  } /* switch */

  return 0L;
}

