// ReportSummaryTask.cpp: implementation of the ReportSummaryTask class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "resource.h"
#include "ReportSummaryTask.h"

//////////////////////////////////////////////////////////////////////
// Static Definitions
//////////////////////////////////////////////////////////////////////

static const char* summary_text_format[] =
	{"%1d times (%d ms)",
	 "%1d times",
	 "%1d times (%d ms)"};

static const char* state_name[] = {"Thinking", "Waiting", "Eating"};
static const char philosopher_name_format[] = "Philosopher %1d :";

static int summary_control_ID(PhilosopherId id, int state_ix)
{
	return IDC_SUMMARYTEXT + nPhilosopherState * id + state_ix;
}

static int summary_control_ID(PhilosopherId id, PhilosopherState state)
{
	return summary_control_ID(id, (int)state);
}

//////////////////////////////////////////////////////////////////////
// WindowClass Definitions
//////////////////////////////////////////////////////////////////////

class ReportSummaryTask::WindowClass
{
public:
	BOOL registered;
	HMODULE instance;
	TCHAR name[128];

	WindowClass(WNDPROC lpfnWndProc);
};

ReportSummaryTask::WindowClass::WindowClass(WNDPROC lpfnWndProc)
{
	WNDCLASSEX wcex;

	this->instance = GetModuleHandle(NULL);
	LoadString(instance, IDC_DININGPHILOSOPHERS_S, this->name, 128);

	wcex.cbSize = sizeof(WNDCLASSEX); 
	wcex.style = CS_HREDRAW | CS_VREDRAW | CS_NOCLOSE;
	wcex.lpfnWndProc = lpfnWndProc;
	wcex.cbClsExtra = 0;
	wcex.cbWndExtra = 0;
	wcex.hInstance = this->instance;
	wcex.hIcon = LoadIcon(this->instance, 
							MAKEINTRESOURCE(IDI_DININGPHILOSOPHERS));
	wcex.hCursor = LoadCursor(NULL, IDC_ARROW);
	wcex.hbrBackground = (HBRUSH)(COLOR_MENU+1);
	wcex.lpszMenuName = NULL;
	wcex.lpszClassName = this->name;
	wcex.hIconSm = LoadIcon(this->instance, MAKEINTRESOURCE(IDI_SMALL));

	this->registered = RegisterClassEx(&wcex);
}

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

ReportSummaryTask::ReportSummaryTask()
{
	this->activate();
}

ReportSummaryTask::~ReportSummaryTask()
{
	this->terminate();
	DeleteObject(this->font);
}

//////////////////////////////////////////////////////////////////////
// Member Functions
//////////////////////////////////////////////////////////////////////

void ReportSummaryTask::body()
{
	Summary summary;

	// Initialize the summary
	for (PhilosopherId id = 0; id < N_PHILOSOPHERS; id++) {
		for (int state_ix = 0; state_ix < nPhilosopherState; state_ix++) {
			PhilosopherStateSummary &total = summary[id][state_ix];

			total.count = 0;
			total.time = 0;
		}
	}

	if (!initialize_window(SW_SHOWNORMAL)) {
		MessageBox(NULL, "Error creating summary window", 
					"Error", MB_ICONERROR | MB_OK);
		return;
	}

	message_loop(summary);
}

void ReportSummaryTask::message_loop(Summary summary)
{
	bool done = false;
	
	while (!done) {
		// Start rendezvous
		Any_Entry* pe_table[4] = {&update, &msg_entry, &end, NULL};
		Rendezvous accept(pe_table);
		switch (accept.entry_selector()) {
		case 0: {
			// Accepted update
			UpdateParams &act_params = update.actual_parameters(accept);
			PhilosopherStateSummary &total 
				= summary[act_params.id][act_params.state];
			
			total.count += 1;
			total.time += act_params.time;
			update_control(act_params.id, act_params.state, total);
			break;
				}
		case 1: {
			// Accepted msg_entry
			MSG &msg = msg_entry.actual_msg(accept);
			
			if (msg.message == WM_QUIT) {
				done = true;
			}
			else {
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
			break;
				}
		case 2: {
			// Accepted end
			DestroyWindow(this->hWnd);
			break;
				}
		}
		// End Rendezvous
	}
}

bool ReportSummaryTask::initialize_window(int cmdShow)
{
	// Register the window class, if this hasn't been done yet
	// There is only one window_class object for the task type, shared by all
	// task objects
	static ReportSummaryTask::WindowClass volatile window_class(window_proc);
	if (!window_class.registered) {
		return false;
	}
	this->hInstance = window_class.instance;

	// Create the window
	TCHAR window_title[128];
	LoadString(window_class.instance, IDS_SUMMARY_TITLE, window_title, 128);
	this->hWnd = CreateWindow
		((TCHAR*)window_class.name, window_title, 
		 WS_OVERLAPPED, 
		 CW_USEDEFAULT, 0, 480, 30*N_PHILOSOPHERS + 70, 
		 NULL, NULL, this->hInstance, NULL);
	if (this->hWnd == NULL) {
		return false;
	}

	this->font = 
		CreateFont(13, 0, 0, 0, 
					FW_NORMAL, 
					false, false, false, 
					ANSI_CHARSET, 
					OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, 
					DEFAULT_QUALITY,
					DEFAULT_PITCH | FF_SWISS, 
					NULL);
	SendMessage(this->hWnd, WM_SETFONT, (WPARAM)this->font, FALSE);
	initialize_window_controls();

	ShowWindow(this->hWnd, cmdShow);
	UpdateWindow(this->hWnd);
	return true;
}

void ReportSummaryTask::initialize_window_controls()
{
	PhilosopherId id;
	int state_ix;
	char text_buf[128];
	HWND hControl;
	
	// Create the static fields
	for (id = 0; id <N_PHILOSOPHERS; id++) {
		_snprintf(text_buf, 128, philosopher_name_format, id);
		hControl = CreateWindow
			("STATIC", text_buf, 
			WS_CHILD | WS_VISIBLE | SS_SIMPLE,
			14, 45 + 30*id, 80, 20, this->hWnd, (HMENU)IDC_STATIC, 
			this->hInstance, NULL);
		SendMessage(hControl, WM_SETFONT, (WPARAM)this->font, FALSE);
	}
	for (state_ix = 0; state_ix < nPhilosopherState; state_ix++) {
		hControl = CreateWindow
			("STATIC", state_name[state_ix], 
			WS_CHILD | WS_VISIBLE | SS_SIMPLE,
			120 + 120*state_ix, 15, 80, 16, this->hWnd, (HMENU)IDC_STATIC, 
			this->hInstance, NULL);
		SendMessage(hControl, WM_SETFONT, (WPARAM)this->font, TRUE);
	}
	// Create the controls
	for (id = 0; id < N_PHILOSOPHERS; id++) {
		for (state_ix = 0; state_ix < nPhilosopherState; state_ix++) {
			hControl = CreateWindow
				("EDIT", "", 
				WS_CHILD | WS_VISIBLE | ES_LEFT | ES_READONLY | ES_NOHIDESEL,
				120 + 120*state_ix, 45 + 30*id, 100, 20, this->hWnd, 
				(HMENU)summary_control_ID(id, (PhilosopherState)state_ix),
				this->hInstance, NULL);
			SendMessage(hControl, WM_SETFONT, (WPARAM)this->font, TRUE);
		}
	}
}

void ReportSummaryTask::update_control
		(PhilosopherId id, PhilosopherState state, PhilosopherStateSummary &total)
{
	char text_buf[128];

	_snprintf(text_buf, 128, summary_text_format[state], total.count, total.time);
	SetDlgItemText(this->hWnd, summary_control_ID(id, state), text_buf);

	for (int state_ix = 0; state_ix < nPhilosopherState; state_ix++) {
		HWND hControl = 
			GetDlgItem(this->hWnd, summary_control_ID(id, state_ix));

		if (state_ix == state) {
			SendMessage(hControl, EM_SETSEL, 0, -1);
		}
		else {
			SendMessage(hControl, EM_SETSEL, -1, -1);
		}
	}
}

Priority ReportSummaryTask::priority()
{
	return high_Priority;
}

size_t ReportSummaryTask::virtual_sizeof()
{
	return sizeof(ReportSummaryTask);
}

// Window message handler
LRESULT CALLBACK ReportSummaryTask::window_proc
		(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
	case WM_DESTROY:
		PostQuitMessage(0);
		break;
	default:
		return DefWindowProc(hWnd, message, wParam, lParam);
	}
	return 0;
}

