/** \file 2D_Object2D.h
    A 2D World Object.

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.
Permission is granted to use and copy this file for non-commercial use only.  
Please contact the author concerning commercial usage. 
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_2D_OBJECT2D
#define H_2D_OBJECT2D

#include <WO_Collision.h>
#include <2D_Types.h>
#include <CON_Math2D.h>

class World2D;

/** 2D World Object.  Connects the concepts of a 2D World and the bitmap.
    Provides movement and collision detection for sprites.
*/
class Object2D : public SpriteSelector, public Collidable_Object
{
public:
  /** Construct a 2D Object.  */
  Object2D();

  /** Destruct the object. */
  ~Object2D();

   /** this checks to make sure that any Object added to a Object2D
       is derived from Object2D.  Returns -1 if dynamic_cast fails. */
  virtual long add(Object* O);

  /** Calculate the object's global position.  The global position
      is relative to the world, whereas the object's (local) position
      is relative to its parent.
      Non-zero AllTheWay means that the calculation will recursively
      go up the tree all the way to the root.  Otherwise, the
      calculation will only go up one level.
  */
  virtual long calcGlobalPosition(int AllTheWay=0);

  /** Render the sprite to the 2D view. */
  virtual long render(View& view);

  /** Set the local (relative to parent) position. */
  long       setPosition(fVector2D P);

  /** Retrieve the object's global position.  The global position
      is relative to the world, whereas the object's (local) position
      is relative to its parent.
  */
  Vector2D&  getGlobalPosition();

  /** Retrieve the object's dimensions.  If a sprite is available,
      it will return its size, otherwize returns zero size. */
  virtual Vector2D getSize();

  /** Retrieve the object's local position. */
  fVector2D& getPosition();

  /** Retrieve the mid point of the object.  
      This point is relative to the parent.
  */
  fVector2D  getMiddle();

// Set the sprite to display for this object */
  void setSprite(Sprite* sprite);

  /** Retrieve the sprite associated with this object.
      Implements the abstract SpriteSelector class method. */
  virtual Sprite* getSprite();

/** implement the abstract Collidable_Object functions */

  /** collision is called to indicate a collision between this and O. 
      It may return a World_Object::RETCODE to indicate that it ('this' Object)
      should be removed/deleted. */
  virtual long collision(Collidable_Object* O);

  /** called by CollisionDetection.  Determines if a collision occurred
      bewteen this and O */
  virtual int isCollision(Collidable_Object* O);

  /** called to determine if the Object is to participate in collision
      detection.  Default implementation returns non-zero if the sprite
      assigned to this is solid. */
  virtual int isCollidable();

protected:
  fVector2D        m_Position;
  Vector2D         m_GlobalPosition;
  Sprite*          m_Sprite;
};

#pragma comment(lib, "lib2D.lib")

#endif // H_2D_OBJECT2D